# -*- coding: utf-8 -*-
# @Time    : 13/12/2024 16.53
# @Author  : ljc
# @FileName: convol.py
# @Software: PyCharm
# Update:  2025/11/26 20:38:04


# ======================================================================
# 1. Introduction
# ======================================================================
r"""Python implementation of resolution degradation for LASP-CurveFit.

1.1 Purpose
-----------
Implement spectral resolution degradation using convolution method
based on IDL convol core algorithm.

1.2 Functions
-------------
1) convol: Called by 'uly_fit/uly_fit_conv_weight_poly.py' to degrade
   model spectral resolution.

1.3 Explanation
---------------
The convol function performs convolution operation on input high
resolution spectrum to degrade its resolution, matching the resolution
of observed spectrum.
Steps:
    1) Determine whether convolution kernel length is even or odd.
    2) Pad input spectrum edges based on kernel length.
    3) Convolve padded spectrum with the kernel.
    4) Return degraded resolution spectrum.

1.4 Notes
---------
- This is a Python-specific rewrite and optimization, not a complete
  port of all IDL features, applicable to LASP-CurveFit.
- This module uses 'warnings.filterwarnings("ignore")' to globally
  suppress Python warnings for cleaner large-scale runs. If you
  rely on warnings from NumPy/SciPy or other libraries, please
  comment out or adjust this line in your environment.

"""


# ======================================================================
# 2. Import libraries
# ======================================================================
import numpy as np
from scipy.signal import convolve
import warnings

warnings.filterwarnings("ignore")


# ======================================================================
# 3. Type definitions for better code readability
# ======================================================================
ArrayLike = np.ndarray


# ======================================================================
# 4. Spectral resolution degradation function
# ======================================================================
def convol(models: ArrayLike, losvd: ArrayLike) -> ArrayLike:
    r"""Degrade model spectral resolution using convolution method.

    Parameters
    ----------
    models : np.ndarray
        shape (n,) or (n, 1)
        high-resolution spectrum, i.e., model spectrum to be degraded.
    losvd : np.ndarray
        shape (k,)
        Sliding window generated by Gaussian convolution kernel
        function, used to convolve with model spectrum.

    Returns
    -------
    result : np.ndarray
        shape (n,) or (n, 1)
        Degraded resolution spectrum with the same length as input
        spectrum.

    Notes
    -----
    - LASP uses Gaussian convolution kernel determined by parameters
      mu and sigma, where mu can be used to calculate RV, and sigma
      contains instrumental broadening and rotation effects.
    - Edge padding uses 'edge' mode, i.e., pad with boundary values
      to reduce edge effects.
    - Convolution uses 'valid' mode to ensure output spectrum length
      matches input.

    Examples
    --------
    >>> models = np.array([1.0, 2.0, 3.0, 4.0, 5.0])
    >>> losvd = np.array([0.0, 1.0, 1.2])
    >>> result = convol(models, losvd)
    >>> print(result.shape)
    (5, 1)

    >>> models = np.array([2.0, 1.0, 1.1, 6.0, 2.1]).reshape(-1, 1)
    >>> losvd = np.array([1.0, 2.0, 1.2])
    >>> result = convol(models, losvd)
    >>> print(result.shape)
    (5, 1)

    """

    # ------------------------------------------------------------------
    # 4.1 Determine padding length
    # ------------------------------------------------------------------
    # 4.1.1 Check if kernel length is even or odd
    if len(losvd) % 2 == 0:
        pad_left = len(losvd) // 2
        pad_right = len(losvd) // 2 - 1
    else:
        pad_left = pad_right = (len(losvd) - 1) // 2

    # ------------------------------------------------------------------
    # 4.2 Pad spectrum edges
    # ------------------------------------------------------------------
    padded = np.pad(models.reshape(-1), (pad_left, pad_right), mode="edge")

    # ------------------------------------------------------------------
    # 4.3 Perform convolution to degrade resolution
    # ------------------------------------------------------------------
    result = convolve(padded, losvd[::-1], mode="valid").reshape(-1, 1)

    # ------------------------------------------------------------------
    # 4.4 Return degraded resolution spectrum
    # ------------------------------------------------------------------
    return result
