// Copyright 2022 HE Boliang
// All rights reserved.

package gofa_test

import (
	"testing"

	"github.com/hebl/gofa"
)

func TestBi00(t *testing.T) {
	var dpsibi, depsbi, dra float64

	gofa.Bi00(&dpsibi, &depsbi, &dra)

	vvd(t, dpsibi, -0.2025309152835086613e-6, 1e-12, "Bi00", "dpsibi")
	vvd(t, depsbi, -0.3306041454222147847e-7, 1e-12, "Bi00", "depsbi")
	vvd(t, dra, -0.7078279744199225506e-7, 1e-12, "Bi00", "dra")
}
func TestBp00(t *testing.T) {
	var rb, rp, rbp [3][3]float64

	gofa.Bp00(2400000.5, 50123.9999, &rb, &rp, &rbp)

	vvd(t, rb[0][0], 0.9999999999999942498, 1e-12, "Bp00", "rb11")
	vvd(t, rb[0][1], -0.7078279744199196626e-7, 1e-16, "Bp00", "rb12")
	vvd(t, rb[0][2], 0.8056217146976134152e-7, 1e-16, "Bp00", "rb13")
	vvd(t, rb[1][0], 0.7078279477857337206e-7, 1e-16, "Bp00", "rb21")
	vvd(t, rb[1][1], 0.9999999999999969484, 1e-12, "Bp00", "rb22")
	vvd(t, rb[1][2], 0.3306041454222136517e-7, 1e-16, "Bp00", "rb23")
	vvd(t, rb[2][0], -0.8056217380986972157e-7, 1e-16, "Bp00", "rb31")
	vvd(t, rb[2][1], -0.3306040883980552500e-7, 1e-16, "Bp00", "rb32")
	vvd(t, rb[2][2], 0.9999999999999962084, 1e-12, "Bp00", "rb33")

	vvd(t, rp[0][0], 0.9999995504864048241, 1e-12, "Bp00", "rp11")
	vvd(t, rp[0][1], 0.8696113836207084411e-3, 1e-14, "Bp00", "rp12")
	vvd(t, rp[0][2], 0.3778928813389333402e-3, 1e-14, "Bp00", "rp13")
	vvd(t, rp[1][0], -0.8696113818227265968e-3, 1e-14, "Bp00", "rp21")
	vvd(t, rp[1][1], 0.9999996218879365258, 1e-12, "Bp00", "rp22")
	vvd(t, rp[1][2], -0.1690679263009242066e-6, 1e-14, "Bp00", "rp23")
	vvd(t, rp[2][0], -0.3778928854764695214e-3, 1e-14, "Bp00", "rp31")
	vvd(t, rp[2][1], -0.1595521004195286491e-6, 1e-14, "Bp00", "rp32")
	vvd(t, rp[2][2], 0.9999999285984682756, 1e-12, "Bp00", "rp33")

	vvd(t, rbp[0][0], 0.9999995505175087260, 1e-12, "Bp00", "rbp11")
	vvd(t, rbp[0][1], 0.8695405883617884705e-3, 1e-14, "Bp00", "rbp12")
	vvd(t, rbp[0][2], 0.3779734722239007105e-3, 1e-14, "Bp00", "rbp13")
	vvd(t, rbp[1][0], -0.8695405990410863719e-3, 1e-14, "Bp00", "rbp21")
	vvd(t, rbp[1][1], 0.9999996219494925900, 1e-12, "Bp00", "rbp22")
	vvd(t, rbp[1][2], -0.1360775820404982209e-6, 1e-14, "Bp00", "rbp23")
	vvd(t, rbp[2][0], -0.3779734476558184991e-3, 1e-14, "Bp00", "rbp31")
	vvd(t, rbp[2][1], -0.1925857585832024058e-6, 1e-14, "Bp00", "rbp32")
	vvd(t, rbp[2][2], 0.9999999285680153377, 1e-12, "Bp00", "rbp33")
}
func TestBp06(t *testing.T) {

	var rb, rp, rbp [3][3]float64

	gofa.Bp06(2400000.5, 50123.9999, &rb, &rp, &rbp)

	vvd(t, rb[0][0], 0.9999999999999942497, 1e-12, "Bp06", "rb11")
	vvd(t, rb[0][1], -0.7078368960971557145e-7, 1e-14, "Bp06", "rb12")
	vvd(t, rb[0][2], 0.8056213977613185606e-7, 1e-14, "Bp06", "rb13")
	vvd(t, rb[1][0], 0.7078368694637674333e-7, 1e-14, "Bp06", "rb21")
	vvd(t, rb[1][1], 0.9999999999999969484, 1e-12, "Bp06", "rb22")
	vvd(t, rb[1][2], 0.3305943742989134124e-7, 1e-14, "Bp06", "rb23")
	vvd(t, rb[2][0], -0.8056214211620056792e-7, 1e-14, "Bp06", "rb31")
	vvd(t, rb[2][1], -0.3305943172740586950e-7, 1e-14, "Bp06", "rb32")
	vvd(t, rb[2][2], 0.9999999999999962084, 1e-12, "Bp06", "rb33")

	vvd(t, rp[0][0], 0.9999995504864960278, 1e-12, "Bp06", "rp11")
	vvd(t, rp[0][1], 0.8696112578855404832e-3, 1e-14, "Bp06", "rp12")
	vvd(t, rp[0][2], 0.3778929293341390127e-3, 1e-14, "Bp06", "rp13")
	vvd(t, rp[1][0], -0.8696112560510186244e-3, 1e-14, "Bp06", "rp21")
	vvd(t, rp[1][1], 0.9999996218880458820, 1e-12, "Bp06", "rp22")
	vvd(t, rp[1][2], -0.1691646168941896285e-6, 1e-14, "Bp06", "rp23")
	vvd(t, rp[2][0], -0.3778929335557603418e-3, 1e-14, "Bp06", "rp31")
	vvd(t, rp[2][1], -0.1594554040786495076e-6, 1e-14, "Bp06", "rp32")
	vvd(t, rp[2][2], 0.9999999285984501222, 1e-12, "Bp06", "rp33")

	vvd(t, rbp[0][0], 0.9999995505176007047, 1e-12, "Bp06", "rbp11")
	vvd(t, rbp[0][1], 0.8695404617348208406e-3, 1e-14, "Bp06", "rbp12")
	vvd(t, rbp[0][2], 0.3779735201865589104e-3, 1e-14, "Bp06", "rbp13")
	vvd(t, rbp[1][0], -0.8695404723772031414e-3, 1e-14, "Bp06", "rbp21")
	vvd(t, rbp[1][1], 0.9999996219496027161, 1e-12, "Bp06", "rbp22")
	vvd(t, rbp[1][2], -0.1361752497080270143e-6, 1e-14, "Bp06", "rbp23")
	vvd(t, rbp[2][0], -0.3779734957034089490e-3, 1e-14, "Bp06", "rbp31")
	vvd(t, rbp[2][1], -0.1924880847894457113e-6, 1e-14, "Bp06", "rbp32")
	vvd(t, rbp[2][2], 0.9999999285679971958, 1e-12, "Bp06", "rbp33")
}
func TestBpn2xy(t *testing.T) {
	var rbpn [3][3]float64
	var x, y float64

	rbpn[0][0] = 9.999962358680738e-1
	rbpn[0][1] = -2.516417057665452e-3
	rbpn[0][2] = -1.093569785342370e-3

	rbpn[1][0] = 2.516462370370876e-3
	rbpn[1][1] = 9.999968329010883e-1
	rbpn[1][2] = 4.006159587358310e-5

	rbpn[2][0] = 1.093465510215479e-3
	rbpn[2][1] = -4.281337229063151e-5
	rbpn[2][2] = 9.999994012499173e-1

	gofa.Bpn2xy(rbpn, &x, &y)

	vvd(t, x, 1.093465510215479e-3, 1e-12, "Bpn2xy", "x")
	vvd(t, y, -4.281337229063151e-5, 1e-12, "Bpn2xy", "y")

}
func TestC2i00a(t *testing.T) {
	var rc2i [3][3]float64

	gofa.C2i00a(2400000.5, 53736.0, &rc2i)

	vvd(t, rc2i[0][0], 0.9999998323037165557, 1e-12, "C2i00a", "11")
	vvd(t, rc2i[0][1], 0.5581526348992140183e-9, 1e-12, "C2i00a", "12")
	vvd(t, rc2i[0][2], -0.5791308477073443415e-3, 1e-12, "C2i00a", "13")

	vvd(t, rc2i[1][0], -0.2384266227870752452e-7, 1e-12, "C2i00a", "21")
	vvd(t, rc2i[1][1], 0.9999999991917405258, 1e-12, "C2i00a", "22")
	vvd(t, rc2i[1][2], -0.4020594955028209745e-4, 1e-12, "C2i00a", "23")

	vvd(t, rc2i[2][0], 0.5791308472168152904e-3, 1e-12, "C2i00a", "31")
	vvd(t, rc2i[2][1], 0.4020595661591500259e-4, 1e-12, "C2i00a", "32")
	vvd(t, rc2i[2][2], 0.9999998314954572304, 1e-12, "C2i00a", "33")
}
func TestC2i00b(t *testing.T) {

	var rc2i [3][3]float64

	gofa.C2i00b(2400000.5, 53736.0, &rc2i)

	vvd(t, rc2i[0][0], 0.9999998323040954356, 1e-12, "C2i00b", "11")
	vvd(t, rc2i[0][1], 0.5581526349131823372e-9, 1e-12, "C2i00b", "12")
	vvd(t, rc2i[0][2], -0.5791301934855394005e-3, 1e-12, "C2i00b", "13")

	vvd(t, rc2i[1][0], -0.2384239285499175543e-7, 1e-12, "C2i00b", "21")
	vvd(t, rc2i[1][1], 0.9999999991917574043, 1e-12, "C2i00b", "22")
	vvd(t, rc2i[1][2], -0.4020552974819030066e-4, 1e-12, "C2i00b", "23")

	vvd(t, rc2i[2][0], 0.5791301929950208873e-3, 1e-12, "C2i00b", "31")
	vvd(t, rc2i[2][1], 0.4020553681373720832e-4, 1e-12, "C2i00b", "32")
	vvd(t, rc2i[2][2], 0.9999998314958529887, 1e-12, "C2i00b", "33")
}
func TestC2i06a(t *testing.T) {

	var rc2i [3][3]float64

	gofa.C2i06a(2400000.5, 53736.0, &rc2i)

	vvd(t, rc2i[0][0], 0.9999998323037159379, 1e-12, "C2i06a", "11")
	vvd(t, rc2i[0][1], 0.5581121329587613787e-9, 1e-12, "C2i06a", "12")
	vvd(t, rc2i[0][2], -0.5791308487740529749e-3, 1e-12, "C2i06a", "13")

	vvd(t, rc2i[1][0], -0.2384253169452306581e-7, 1e-12, "C2i06a", "21")
	vvd(t, rc2i[1][1], 0.9999999991917467827, 1e-12, "C2i06a", "22")
	vvd(t, rc2i[1][2], -0.4020579392895682558e-4, 1e-12, "C2i06a", "23")

	vvd(t, rc2i[2][0], 0.5791308482835292617e-3, 1e-12, "C2i06a", "31")
	vvd(t, rc2i[2][1], 0.4020580099454020310e-4, 1e-12, "C2i06a", "32")
	vvd(t, rc2i[2][2], 0.9999998314954628695, 1e-12, "C2i06a", "33")
}
func TestC2ibpn(t *testing.T) {

	var rbpn, rc2i [3][3]float64

	rbpn[0][0] = 9.999962358680738e-1
	rbpn[0][1] = -2.516417057665452e-3
	rbpn[0][2] = -1.093569785342370e-3

	rbpn[1][0] = 2.516462370370876e-3
	rbpn[1][1] = 9.999968329010883e-1
	rbpn[1][2] = 4.006159587358310e-5

	rbpn[2][0] = 1.093465510215479e-3
	rbpn[2][1] = -4.281337229063151e-5
	rbpn[2][2] = 9.999994012499173e-1

	gofa.C2ibpn(2400000.5, 50123.9999, rbpn, &rc2i)

	vvd(t, rc2i[0][0], 0.9999994021664089977, 1e-12, "C2ibpn", "11")
	vvd(t, rc2i[0][1], -0.3869195948017503664e-8, 1e-12, "C2ibpn", "12")
	vvd(t, rc2i[0][2], -0.1093465511383285076e-2, 1e-12, "C2ibpn", "13")

	vvd(t, rc2i[1][0], 0.5068413965715446111e-7, 1e-12, "C2ibpn", "21")
	vvd(t, rc2i[1][1], 0.9999999990835075686, 1e-12, "C2ibpn", "22")
	vvd(t, rc2i[1][2], 0.4281334246452708915e-4, 1e-12, "C2ibpn", "23")

	vvd(t, rc2i[2][0], 0.1093465510215479000e-2, 1e-12, "C2ibpn", "31")
	vvd(t, rc2i[2][1], -0.4281337229063151000e-4, 1e-12, "C2ibpn", "32")
	vvd(t, rc2i[2][2], 0.9999994012499173103, 1e-12, "C2ibpn", "33")
}

func TestC2ixy(t *testing.T) {

	var rc2i [3][3]float64
	var x, y float64

	x = 0.5791308486706011000e-3
	y = 0.4020579816732961219e-4

	gofa.C2ixy(2400000.5, 53736, x, y, &rc2i)

	vvd(t, rc2i[0][0], 0.9999998323037157138, 1e-12, "C2ixy", "11")
	vvd(t, rc2i[0][1], 0.5581526349032241205e-9, 1e-12, "C2ixy", "12")
	vvd(t, rc2i[0][2], -0.5791308491611263745e-3, 1e-12, "C2ixy", "13")

	vvd(t, rc2i[1][0], -0.2384257057469842953e-7, 1e-12, "C2ixy", "21")
	vvd(t, rc2i[1][1], 0.9999999991917468964, 1e-12, "C2ixy", "22")
	vvd(t, rc2i[1][2], -0.4020579110172324363e-4, 1e-12, "C2ixy", "23")

	vvd(t, rc2i[2][0], 0.5791308486706011000e-3, 1e-12, "C2ixy", "31")
	vvd(t, rc2i[2][1], 0.4020579816732961219e-4, 1e-12, "C2ixy", "32")
	vvd(t, rc2i[2][2], 0.9999998314954627590, 1e-12, "C2ixy", "33")

}

func TestC2ixys(t *testing.T) {

	var rc2i [3][3]float64
	var x, y, s float64

	x = 0.5791308486706011000e-3
	y = 0.4020579816732961219e-4
	s = -0.1220040848472271978e-7

	gofa.C2ixys(x, y, s, &rc2i)

	vvd(t, rc2i[0][0], 0.9999998323037157138, 1e-12, "C2ixys", "11")
	vvd(t, rc2i[0][1], 0.5581984869168499149e-9, 1e-12, "C2ixys", "12")
	vvd(t, rc2i[0][2], -0.5791308491611282180e-3, 1e-12, "C2ixys", "13")

	vvd(t, rc2i[1][0], -0.2384261642670440317e-7, 1e-12, "C2ixys", "21")
	vvd(t, rc2i[1][1], 0.9999999991917468964, 1e-12, "C2ixys", "22")
	vvd(t, rc2i[1][2], -0.4020579110169668931e-4, 1e-12, "C2ixys", "23")

	vvd(t, rc2i[2][0], 0.5791308486706011000e-3, 1e-12, "C2ixys", "31")
	vvd(t, rc2i[2][1], 0.4020579816732961219e-4, 1e-12, "C2ixys", "32")
	vvd(t, rc2i[2][2], 0.9999998314954627590, 1e-12, "C2ixys", "33")
}
func TestC2t00a(t *testing.T) {
	var tta, ttb, uta, utb, xp, yp float64
	var rc2t [3][3]float64

	tta = 2400000.5
	uta = 2400000.5
	ttb = 53736.0
	utb = 53736.0
	xp = 2.55060238e-7
	yp = 1.860359247e-6

	gofa.C2t00a(tta, ttb, uta, utb, xp, yp, &rc2t)

	vvd(t, rc2t[0][0], -0.1810332128307182668, 1e-12, "C2t00a", "11")
	vvd(t, rc2t[0][1], 0.9834769806938457836, 1e-12, "C2t00a", "12")
	vvd(t, rc2t[0][2], 0.6555535638688341725e-4, 1e-12, "C2t00a", "13")

	vvd(t, rc2t[1][0], -0.9834768134135984552, 1e-12, "C2t00a", "21")
	vvd(t, rc2t[1][1], -0.1810332203649520727, 1e-12, "C2t00a", "22")
	vvd(t, rc2t[1][2], 0.5749801116141056317e-3, 1e-12, "C2t00a", "23")

	vvd(t, rc2t[2][0], 0.5773474014081406921e-3, 1e-12, "C2t00a", "31")
	vvd(t, rc2t[2][1], 0.3961832391770163647e-4, 1e-12, "C2t00a", "32")
	vvd(t, rc2t[2][2], 0.9999998325501692289, 1e-12, "C2t00a", "33")

}
func TestC2t00b(t *testing.T) {

	var tta, ttb, uta, utb, xp, yp float64
	var rc2t [3][3]float64

	tta = 2400000.5
	uta = 2400000.5
	ttb = 53736.0
	utb = 53736.0
	xp = 2.55060238e-7
	yp = 1.860359247e-6

	gofa.C2t00b(tta, ttb, uta, utb, xp, yp, &rc2t)

	vvd(t, rc2t[0][0], -0.1810332128439678965, 1e-12, "C2t00b", "11")
	vvd(t, rc2t[0][1], 0.9834769806913872359, 1e-12, "C2t00b", "12")
	vvd(t, rc2t[0][2], 0.6555565082458415611e-4, 1e-12, "C2t00b", "13")

	vvd(t, rc2t[1][0], -0.9834768134115435923, 1e-12, "C2t00b", "21")
	vvd(t, rc2t[1][1], -0.1810332203784001946, 1e-12, "C2t00b", "22")
	vvd(t, rc2t[1][2], 0.5749793922030017230e-3, 1e-12, "C2t00b", "23")

	vvd(t, rc2t[2][0], 0.5773467471863534901e-3, 1e-12, "C2t00b", "31")
	vvd(t, rc2t[2][1], 0.3961790411549945020e-4, 1e-12, "C2t00b", "32")
	vvd(t, rc2t[2][2], 0.9999998325505635738, 1e-12, "C2t00b", "33")
}
func TestC2t06a(t *testing.T) {

	var tta, ttb, uta, utb, xp, yp float64
	var rc2t [3][3]float64
	tta = 2400000.5
	uta = 2400000.5
	ttb = 53736.0
	utb = 53736.0
	xp = 2.55060238e-7
	yp = 1.860359247e-6

	gofa.C2t06a(tta, ttb, uta, utb, xp, yp, &rc2t)

	vvd(t, rc2t[0][0], -0.1810332128305897282, 1e-12, "C2t06a", "11")
	vvd(t, rc2t[0][1], 0.9834769806938592296, 1e-12, "C2t06a", "12")
	vvd(t, rc2t[0][2], 0.6555550962998436505e-4, 1e-12, "C2t06a", "13")

	vvd(t, rc2t[1][0], -0.9834768134136214897, 1e-12, "C2t06a", "21")
	vvd(t, rc2t[1][1], -0.1810332203649130832, 1e-12, "C2t06a", "22")
	vvd(t, rc2t[1][2], 0.5749800844905594110e-3, 1e-12, "C2t06a", "23")

	vvd(t, rc2t[2][0], 0.5773474024748545878e-3, 1e-12, "C2t06a", "31")
	vvd(t, rc2t[2][1], 0.3961816829632690581e-4, 1e-12, "C2t06a", "32")
	vvd(t, rc2t[2][2], 0.9999998325501747785, 1e-12, "C2t06a", "33")
}
func TestC2tcio(t *testing.T) {

	var era float64
	var rc2i, rpom, rc2t [3][3]float64

	rc2i[0][0] = 0.9999998323037164738
	rc2i[0][1] = 0.5581526271714303683e-9
	rc2i[0][2] = -0.5791308477073443903e-3

	rc2i[1][0] = -0.2384266227524722273e-7
	rc2i[1][1] = 0.9999999991917404296
	rc2i[1][2] = -0.4020594955030704125e-4

	rc2i[2][0] = 0.5791308472168153320e-3
	rc2i[2][1] = 0.4020595661593994396e-4
	rc2i[2][2] = 0.9999998314954572365

	era = 1.75283325530307

	rpom[0][0] = 0.9999999999999674705
	rpom[0][1] = -0.1367174580728847031e-10
	rpom[0][2] = 0.2550602379999972723e-6

	rpom[1][0] = 0.1414624947957029721e-10
	rpom[1][1] = 0.9999999999982694954
	rpom[1][2] = -0.1860359246998866338e-5

	rpom[2][0] = -0.2550602379741215275e-6
	rpom[2][1] = 0.1860359247002413923e-5
	rpom[2][2] = 0.9999999999982369658

	gofa.C2tcio(rc2i, era, rpom, &rc2t)

	vvd(t, rc2t[0][0], -0.1810332128307110439, 1e-12, "C2tcio", "11")
	vvd(t, rc2t[0][1], 0.9834769806938470149, 1e-12, "C2tcio", "12")
	vvd(t, rc2t[0][2], 0.6555535638685466874e-4, 1e-12, "C2tcio", "13")

	vvd(t, rc2t[1][0], -0.9834768134135996657, 1e-12, "C2tcio", "21")
	vvd(t, rc2t[1][1], -0.1810332203649448367, 1e-12, "C2tcio", "22")
	vvd(t, rc2t[1][2], 0.5749801116141106528e-3, 1e-12, "C2tcio", "23")

	vvd(t, rc2t[2][0], 0.5773474014081407076e-3, 1e-12, "C2tcio", "31")
	vvd(t, rc2t[2][1], 0.3961832391772658944e-4, 1e-12, "C2tcio", "32")
	vvd(t, rc2t[2][2], 0.9999998325501691969, 1e-12, "C2tcio", "33")
}

func TestC2teqx(t *testing.T) {

	var gst float64
	var rbpn, rpom, rc2t [3][3]float64

	rbpn[0][0] = 0.9999989440476103608
	rbpn[0][1] = -0.1332881761240011518e-2
	rbpn[0][2] = -0.5790767434730085097e-3

	rbpn[1][0] = 0.1332858254308954453e-2
	rbpn[1][1] = 0.9999991109044505944
	rbpn[1][2] = -0.4097782710401555759e-4

	rbpn[2][0] = 0.5791308472168153320e-3
	rbpn[2][1] = 0.4020595661593994396e-4
	rbpn[2][2] = 0.9999998314954572365

	gst = 1.754166138040730516

	rpom[0][0] = 0.9999999999999674705
	rpom[0][1] = -0.1367174580728847031e-10
	rpom[0][2] = 0.2550602379999972723e-6

	rpom[1][0] = 0.1414624947957029721e-10
	rpom[1][1] = 0.9999999999982694954
	rpom[1][2] = -0.1860359246998866338e-5

	rpom[2][0] = -0.2550602379741215275e-6
	rpom[2][1] = 0.1860359247002413923e-5
	rpom[2][2] = 0.9999999999982369658

	gofa.C2teqx(rbpn, gst, rpom, &rc2t)

	vvd(t, rc2t[0][0], -0.1810332128528685730, 1e-12, "C2teqx", "11")
	vvd(t, rc2t[0][1], 0.9834769806897685071, 1e-12, "C2teqx", "12")
	vvd(t, rc2t[0][2], 0.6555535639982634449e-4, 1e-12, "C2teqx", "13")

	vvd(t, rc2t[1][0], -0.9834768134095211257, 1e-12, "C2teqx", "21")
	vvd(t, rc2t[1][1], -0.1810332203871023800, 1e-12, "C2teqx", "22")
	vvd(t, rc2t[1][2], 0.5749801116126438962e-3, 1e-12, "C2teqx", "23")

	vvd(t, rc2t[2][0], 0.5773474014081539467e-3, 1e-12, "C2teqx", "31")
	vvd(t, rc2t[2][1], 0.3961832391768640871e-4, 1e-12, "C2teqx", "32")
	vvd(t, rc2t[2][2], 0.9999998325501691969, 1e-12, "C2teqx", "33")
}
func TestC2tpe(t *testing.T) {
	var tta, ttb, uta, utb, dpsi, deps, xp, yp float64
	var rc2t [3][3]float64

	tta = 2400000.5
	uta = 2400000.5
	ttb = 53736.0
	utb = 53736.0
	deps = 0.4090789763356509900
	dpsi = -0.9630909107115582393e-5
	xp = 2.55060238e-7
	yp = 1.860359247e-6

	gofa.C2tpe(tta, ttb, uta, utb, dpsi, deps, xp, yp, &rc2t)

	vvd(t, rc2t[0][0], -0.1813677995763029394, 1e-12, "C2tpe", "11")
	vvd(t, rc2t[0][1], 0.9023482206891683275, 1e-12, "C2tpe", "12")
	vvd(t, rc2t[0][2], -0.3909902938641085751, 1e-12, "C2tpe", "13")

	vvd(t, rc2t[1][0], -0.9834147641476804807, 1e-12, "C2tpe", "21")
	vvd(t, rc2t[1][1], -0.1659883635434995121, 1e-12, "C2tpe", "22")
	vvd(t, rc2t[1][2], 0.7309763898042819705e-1, 1e-12, "C2tpe", "23")

	vvd(t, rc2t[2][0], 0.1059685430673215247e-2, 1e-12, "C2tpe", "31")
	vvd(t, rc2t[2][1], 0.3977631855605078674, 1e-12, "C2tpe", "32")
	vvd(t, rc2t[2][2], 0.9174875068792735362, 1e-12, "C2tpe", "33")
}

func TestC2txy(t *testing.T) {

	var tta, ttb, uta, utb, x, y, xp, yp float64
	var rc2t [3][3]float64

	tta = 2400000.5
	uta = 2400000.5
	ttb = 53736.0
	utb = 53736.0
	x = 0.5791308486706011000e-3
	y = 0.4020579816732961219e-4
	xp = 2.55060238e-7
	yp = 1.860359247e-6

	gofa.C2txy(tta, ttb, uta, utb, x, y, xp, yp, &rc2t)

	vvd(t, rc2t[0][0], -0.1810332128306279253, 1e-12, "C2txy", "11")
	vvd(t, rc2t[0][1], 0.9834769806938520084, 1e-12, "C2txy", "12")
	vvd(t, rc2t[0][2], 0.6555551248057665829e-4, 1e-12, "C2txy", "13")

	vvd(t, rc2t[1][0], -0.9834768134136142314, 1e-12, "C2txy", "21")
	vvd(t, rc2t[1][1], -0.1810332203649529312, 1e-12, "C2txy", "22")
	vvd(t, rc2t[1][2], 0.5749800843594139912e-3, 1e-12, "C2txy", "23")

	vvd(t, rc2t[2][0], 0.5773474028619264494e-3, 1e-12, "C2txy", "31")
	vvd(t, rc2t[2][1], 0.3961816546911624260e-4, 1e-12, "C2txy", "32")
	vvd(t, rc2t[2][2], 0.9999998325501746670, 1e-12, "C2txy", "33")

}
func TestEo06a(t *testing.T) {

	eo := gofa.Eo06a(2400000.5, 53736.0)

	vvd(t, eo, -0.1332882371941833644e-2, 1e-15, "Eo06a", "")

}

func TestEors(t *testing.T) {
	var rnpb [3][3]float64
	var s, eo float64

	rnpb[0][0] = 0.9999989440476103608
	rnpb[0][1] = -0.1332881761240011518e-2
	rnpb[0][2] = -0.5790767434730085097e-3

	rnpb[1][0] = 0.1332858254308954453e-2
	rnpb[1][1] = 0.9999991109044505944
	rnpb[1][2] = -0.4097782710401555759e-4

	rnpb[2][0] = 0.5791308472168153320e-3
	rnpb[2][1] = 0.4020595661593994396e-4
	rnpb[2][2] = 0.9999998314954572365

	s = -0.1220040848472271978e-7

	eo = gofa.Eors(rnpb, s)

	vvd(t, eo, -0.1332882715130744606e-2, 1e-14, "Eors", "")
}

func TestFw2m(t *testing.T) {
	var gamb, phib, psi, eps float64
	var r [3][3]float64

	gamb = -0.2243387670997992368e-5
	phib = 0.4091014602391312982
	psi = -0.9501954178013015092e-3
	eps = 0.4091014316587367472

	gofa.Fw2m(gamb, phib, psi, eps, &r)

	vvd(t, r[0][0], 0.9999995505176007047, 1e-12, "Fw2m", "11")
	vvd(t, r[0][1], 0.8695404617348192957e-3, 1e-12, "Fw2m", "12")
	vvd(t, r[0][2], 0.3779735201865582571e-3, 1e-12, "Fw2m", "13")

	vvd(t, r[1][0], -0.8695404723772016038e-3, 1e-12, "Fw2m", "21")
	vvd(t, r[1][1], 0.9999996219496027161, 1e-12, "Fw2m", "22")
	vvd(t, r[1][2], -0.1361752496887100026e-6, 1e-12, "Fw2m", "23")

	vvd(t, r[2][0], -0.3779734957034082790e-3, 1e-12, "Fw2m", "31")
	vvd(t, r[2][1], -0.1924880848087615651e-6, 1e-12, "Fw2m", "32")
	vvd(t, r[2][2], 0.9999999285679971958, 1e-12, "Fw2m", "33")
}
func TestFw2xy(t *testing.T) {
	var gamb, phib, psi, eps, x, y float64

	gamb = -0.2243387670997992368e-5
	phib = 0.4091014602391312982
	psi = -0.9501954178013015092e-3
	eps = 0.4091014316587367472

	gofa.Fw2xy(gamb, phib, psi, eps, &x, &y)

	vvd(t, x, -0.3779734957034082790e-3, 1e-14, "Fw2xy", "x")
	vvd(t, y, -0.1924880848087615651e-6, 1e-14, "Fw2xy", "y")
}

func TestLtp(t *testing.T) {
	var epj float64
	var rp [3][3]float64

	epj = 1666.666

	gofa.Ltp(epj, &rp)

	vvd(t, rp[0][0], 0.9967044141159213819, 1e-14, "Ltp", "rp11")
	vvd(t, rp[0][1], 0.7437801893193210840e-1, 1e-14, "Ltp", "rp12")
	vvd(t, rp[0][2], 0.3237624409345603401e-1, 1e-14, "Ltp", "rp13")
	vvd(t, rp[1][0], -0.7437802731819618167e-1, 1e-14, "Ltp", "rp21")
	vvd(t, rp[1][1], 0.9972293894454533070, 1e-14, "Ltp", "rp22")
	vvd(t, rp[1][2], -0.1205768842723593346e-2, 1e-14, "Ltp", "rp23")
	vvd(t, rp[2][0], -0.3237622482766575399e-1, 1e-14, "Ltp", "rp31")
	vvd(t, rp[2][1], -0.1206286039697609008e-2, 1e-14, "Ltp", "rp32")
	vvd(t, rp[2][2], 0.9994750246704010914, 1e-14, "Ltp", "rp33")
}

func TestLtpb(t *testing.T) {
	var epj float64
	var rpb [3][3]float64

	epj = 1666.666

	gofa.Ltpb(epj, &rpb)

	vvd(t, rpb[0][0], 0.9967044167723271851, 1e-14, "Ltpb", "rpb11")
	vvd(t, rpb[0][1], 0.7437794731203340345e-1, 1e-14, "Ltpb", "rpb12")
	vvd(t, rpb[0][2], 0.3237632684841625547e-1, 1e-14, "Ltpb", "rpb13")
	vvd(t, rpb[1][0], -0.7437795663437177152e-1, 1e-14, "Ltpb", "rpb21")
	vvd(t, rpb[1][1], 0.9972293947500013666, 1e-14, "Ltpb", "rpb22")
	vvd(t, rpb[1][2], -0.1205741865911243235e-2, 1e-14, "Ltpb", "rpb23")
	vvd(t, rpb[2][0], -0.3237630543224664992e-1, 1e-14, "Ltpb", "rpb31")
	vvd(t, rpb[2][1], -0.1206316791076485295e-2, 1e-14, "Ltpb", "rpb32")
	vvd(t, rpb[2][2], 0.9994750220222438819, 1e-14, "Ltpb", "rpb33")
}

func TestLtpecl(t *testing.T) {
	var epj float64
	var vec [3]float64

	epj = -1500.0

	gofa.Ltpecl(epj, &vec)

	vvd(t, vec[0], 0.4768625676477096525e-3, 1e-14, "Ltpecl", "vec1")
	vvd(t, vec[1], -0.4052259533091875112, 1e-14, "Ltpecl", "vec2")
	vvd(t, vec[2], 0.9142164401096448012, 1e-14, "Ltpecl", "vec3")
}

func TestLtpequ(t *testing.T) {
	var epj float64
	var veq [3]float64

	epj = -2500.0

	gofa.Ltpequ(epj, &veq)

	vvd(t, veq[0], -0.3586652560237326659, 1e-14, "Ltpequ", "veq1")
	vvd(t, veq[1], -0.1996978910771128475, 1e-14, "Ltpequ", "veq2")
	vvd(t, veq[2], 0.9118552442250819624, 1e-14, "Ltpequ", "veq3")
}

func TestNum00a(t *testing.T) {
	var rmatn [3][3]float64

	gofa.Num00a(2400000.5, 53736.0, &rmatn)

	vvd(t, rmatn[0][0], 0.9999999999536227949, 1e-12, "Num00a", "11")
	vvd(t, rmatn[0][1], 0.8836238544090873336e-5, 1e-12, "Num00a", "12")
	vvd(t, rmatn[0][2], 0.3830835237722400669e-5, 1e-12, "Num00a", "13")

	vvd(t, rmatn[1][0], -0.8836082880798569274e-5, 1e-12, "Num00a", "21")
	vvd(t, rmatn[1][1], 0.9999999991354655028, 1e-12, "Num00a", "22")
	vvd(t, rmatn[1][2], -0.4063240865362499850e-4, 1e-12, "Num00a", "23")

	vvd(t, rmatn[2][0], -0.3831194272065995866e-5, 1e-12, "Num00a", "31")
	vvd(t, rmatn[2][1], 0.4063237480216291775e-4, 1e-12, "Num00a", "32")
	vvd(t, rmatn[2][2], 0.9999999991671660338, 1e-12, "Num00a", "33")

}

func TestNum00b(t *testing.T) {
	var rmatn [3][3]float64

	gofa.Num00b(2400000.5, 53736, &rmatn)

	vvd(t, rmatn[0][0], 0.9999999999536069682, 1e-12, "Num00b", "11")
	vvd(t, rmatn[0][1], 0.8837746144871248011e-5, 1e-12, "Num00b", "12")
	vvd(t, rmatn[0][2], 0.3831488838252202945e-5, 1e-12, "Num00b", "13")

	vvd(t, rmatn[1][0], -0.8837590456632304720e-5, 1e-12, "Num00b", "21")
	vvd(t, rmatn[1][1], 0.9999999991354692733, 1e-12, "Num00b", "22")
	vvd(t, rmatn[1][2], -0.4063198798559591654e-4, 1e-12, "Num00b", "23")

	vvd(t, rmatn[2][0], -0.3831847930134941271e-5, 1e-12, "Num00b", "31")
	vvd(t, rmatn[2][1], 0.4063195412258168380e-4, 1e-12, "Num00b", "32")
	vvd(t, rmatn[2][2], 0.9999999991671806225, 1e-12, "Num00b", "33")

}

func TestNum06a(t *testing.T) {
	var rmatn [3][3]float64

	gofa.Num06a(2400000.5, 53736, &rmatn)

	vvd(t, rmatn[0][0], 0.9999999999536227668, 1e-12, "Num06a", "11")
	vvd(t, rmatn[0][1], 0.8836241998111535233e-5, 1e-12, "Num06a", "12")
	vvd(t, rmatn[0][2], 0.3830834608415287707e-5, 1e-12, "Num06a", "13")

	vvd(t, rmatn[1][0], -0.8836086334870740138e-5, 1e-12, "Num06a", "21")
	vvd(t, rmatn[1][1], 0.9999999991354657474, 1e-12, "Num06a", "22")
	vvd(t, rmatn[1][2], -0.4063240188248455065e-4, 1e-12, "Num06a", "23")

	vvd(t, rmatn[2][0], -0.3831193642839398128e-5, 1e-12, "Num06a", "31")
	vvd(t, rmatn[2][1], 0.4063236803101479770e-4, 1e-12, "Num06a", "32")
	vvd(t, rmatn[2][2], 0.9999999991671663114, 1e-12, "Num06a", "33")

}

func TestNumat(t *testing.T) {
	var epsa, dpsi, deps float64

	var rmatn [3][3]float64

	epsa = 0.4090789763356509900
	dpsi = -0.9630909107115582393e-5
	deps = 0.4063239174001678826e-4

	gofa.Numat(epsa, dpsi, deps, &rmatn)

	vvd(t, rmatn[0][0], 0.9999999999536227949, 1e-12, "Numat", "11")
	vvd(t, rmatn[0][1], 0.8836239320236250577e-5, 1e-12, "Numat", "12")
	vvd(t, rmatn[0][2], 0.3830833447458251908e-5, 1e-12, "Numat", "13")

	vvd(t, rmatn[1][0], -0.8836083657016688588e-5, 1e-12, "Numat", "21")
	vvd(t, rmatn[1][1], 0.9999999991354654959, 1e-12, "Numat", "22")
	vvd(t, rmatn[1][2], -0.4063240865361857698e-4, 1e-12, "Numat", "23")

	vvd(t, rmatn[2][0], -0.3831192481833385226e-5, 1e-12, "Numat", "31")
	vvd(t, rmatn[2][1], 0.4063237480216934159e-4, 1e-12, "Numat", "32")
	vvd(t, rmatn[2][2], 0.9999999991671660407, 1e-12, "Numat", "33")

}

func TestNut00a(t *testing.T) {
	var dpsi, deps float64
	gofa.Nut00a(2400000.5, 53736.0, &dpsi, &deps)

	vvd(t, dpsi, -0.9630909107115518431e-5, EPS12, "Nut00a", "dpsi")
	vvd(t, deps, 0.4063239174001678710e-4, EPS12, "Nut00a", "deps")
}

func TestNut00b(t *testing.T) {
	var dpsi, deps float64

	gofa.Nut00b(2400000.5, 53736.0, &dpsi, &deps)

	vvd(t, dpsi, -0.9632552291148362783e-5, 1e-13, "Nut00b", "dpsi")
	vvd(t, deps, 0.4063197106621159367e-4, 1e-13, "Nut00b", "deps")

}

func TestNut06a(t *testing.T) {
	var dpsi, deps float64

	gofa.Nut06a(2400000.5, 53736.0, &dpsi, &deps)

	vvd(t, dpsi, -0.9630912025820308797e-5, 1e-13, "Nut06a", "dpsi")
	vvd(t, deps, 0.4063238496887249798e-4, 1e-13, "Nut06a", "deps")
}

func TestNut80(t *testing.T) {
	var dpsi, deps float64

	gofa.Nut80(2400000.5, 53736.0, &dpsi, &deps)

	vvd(t, dpsi, -0.9643658353226563966e-5, 1e-13, "Nut80", "dpsi")
	vvd(t, deps, 0.4060051006879713322e-4, 1e-13, "Nut80", "deps")
}

func TestNutm80(t *testing.T) {
	var rmatn [3][3]float64

	gofa.Nutm80(2400000.5, 53736.0, &rmatn)

	vvd(t, rmatn[0][0], 0.9999999999534999268, 1e-12, "Nutm80", "11")
	vvd(t, rmatn[0][1], 0.8847935789636432161e-5, 1e-12, "Nutm80", "12")
	vvd(t, rmatn[0][2], 0.3835906502164019142e-5, 1e-12, "Nutm80", "13")

	vvd(t, rmatn[1][0], -0.8847780042583435924e-5, 1e-12, "Nutm80", "21")
	vvd(t, rmatn[1][1], 0.9999999991366569963, 1e-12, "Nutm80", "22")
	vvd(t, rmatn[1][2], -0.4060052702727130809e-4, 1e-12, "Nutm80", "23")

	vvd(t, rmatn[2][0], -0.3836265729708478796e-5, 1e-12, "Nutm80", "31")
	vvd(t, rmatn[2][1], 0.4060049308612638555e-4, 1e-12, "Nutm80", "32")
	vvd(t, rmatn[2][2], 0.9999999991684415129, 1e-12, "Nutm80", "33")

}

func TestObl06(t *testing.T) {
	vvd(t, gofa.Obl06(2400000.5, 54388.0), 0.4090749229387258204, 1e-14, "Obl06", "")
}

func TestObl80(t *testing.T) {

	eps0 := gofa.Obl80(2400000.5, 54388.0)

	vvd(t, eps0, 0.4090751347643816218, 1e-14, "Obl80", "")
}

func TestPb06(t *testing.T) {
	var bzeta, bz, btheta float64

	gofa.Pb06(2400000.5, 50123.9999, &bzeta, &bz, &btheta)

	vvd(t, bzeta, -0.5092634016326478238e-3, 1e-12, "Pb06", "bzeta")
	vvd(t, bz, -0.3602772060566044413e-3, 1e-12, "Pb06", "bz")
	vvd(t, btheta, -0.3779735537167811177e-3, 1e-12, "Pb06", "btheta")
}

func TestPfw06(t *testing.T) {
	var gamb, phib, psib, epsa float64

	gofa.Pfw06(2400000.5, 50123.9999, &gamb, &phib, &psib, &epsa)

	vvd(t, gamb, -0.2243387670997995690e-5, 1e-16, "Pfw06", "gamb")
	vvd(t, phib, 0.4091014602391312808, 1e-12, "Pfw06", "phib")
	vvd(t, psib, -0.9501954178013031895e-3, 1e-14, "Pfw06", "psib")
	vvd(t, epsa, 0.4091014316587367491, 1e-12, "Pfw06", "epsa")
}
func TestPmat00(t *testing.T) {
	var rbp [3][3]float64

	gofa.Pmat00(2400000.5, 50123.9999, &rbp)

	vvd(t, rbp[0][0], 0.9999995505175087260, 1e-12, "Pmat00", "11")
	vvd(t, rbp[0][1], 0.8695405883617884705e-3, 1e-14, "Pmat00", "12")
	vvd(t, rbp[0][2], 0.3779734722239007105e-3, 1e-14, "Pmat00", "13")

	vvd(t, rbp[1][0], -0.8695405990410863719e-3, 1e-14, "Pmat00", "21")
	vvd(t, rbp[1][1], 0.9999996219494925900, 1e-12, "Pmat00", "22")
	vvd(t, rbp[1][2], -0.1360775820404982209e-6, 1e-14, "Pmat00", "23")

	vvd(t, rbp[2][0], -0.3779734476558184991e-3, 1e-14, "Pmat00", "31")
	vvd(t, rbp[2][1], -0.1925857585832024058e-6, 1e-14, "Pmat00", "32")
	vvd(t, rbp[2][2], 0.9999999285680153377, 1e-12, "Pmat00", "33")
}
func TestPmat06(t *testing.T) {
	var rbp [3][3]float64

	gofa.Pmat06(2400000.5, 50123.9999, &rbp)

	vvd(t, rbp[0][0], 0.9999995505176007047, 1e-12, "Pmat06", "11")
	vvd(t, rbp[0][1], 0.8695404617348208406e-3, 1e-14, "Pmat06", "12")
	vvd(t, rbp[0][2], 0.3779735201865589104e-3, 1e-14, "Pmat06", "13")

	vvd(t, rbp[1][0], -0.8695404723772031414e-3, 1e-14, "Pmat06", "21")
	vvd(t, rbp[1][1], 0.9999996219496027161, 1e-12, "Pmat06", "22")
	vvd(t, rbp[1][2], -0.1361752497080270143e-6, 1e-14, "Pmat06", "23")

	vvd(t, rbp[2][0], -0.3779734957034089490e-3, 1e-14, "Pmat06", "31")
	vvd(t, rbp[2][1], -0.1924880847894457113e-6, 1e-14, "Pmat06", "32")
	vvd(t, rbp[2][2], 0.9999999285679971958, 1e-12, "Pmat06", "33")
}

func TestPmat76(t *testing.T) {

	var rmatp [3][3]float64

	gofa.Pmat76(2400000.5, 50123.9999, &rmatp)

	vvd(t, rmatp[0][0], 0.9999995504328350733, 1e-12, "Pmat76", "11")
	vvd(t, rmatp[0][1], 0.8696632209480960785e-3, 1e-14, "Pmat76", "12")
	vvd(t, rmatp[0][2], 0.3779153474959888345e-3, 1e-14, "Pmat76", "13")

	vvd(t, rmatp[1][0], -0.8696632209485112192e-3, 1e-14, "Pmat76", "21")
	vvd(t, rmatp[1][1], 0.9999996218428560614, 1e-12, "Pmat76", "22")
	vvd(t, rmatp[1][2], -0.1643284776111886407e-6, 1e-14, "Pmat76", "23")

	vvd(t, rmatp[2][0], -0.3779153474950335077e-3, 1e-14, "Pmat76", "31")
	vvd(t, rmatp[2][1], -0.1643306746147366896e-6, 1e-14, "Pmat76", "32")
	vvd(t, rmatp[2][2], 0.9999999285899790119, 1e-12, "Pmat76", "33")

}

func TestPn00(t *testing.T) {
	var dpsi, deps, epsa float64
	var rb, rp, rbp, rn, rbpn [3][3]float64

	dpsi = -0.9632552291149335877e-5
	deps = 0.4063197106621141414e-4

	gofa.Pn00(2400000.5, 53736.0, dpsi, deps,
		&epsa, &rb, &rp, &rbp, &rn, &rbpn)

	vvd(t, epsa, 0.4090791789404229916, 1e-12, "Pn00", "epsa")

	vvd(t, rb[0][0], 0.9999999999999942498, 1e-12, "Pn00", "rb11")
	vvd(t, rb[0][1], -0.7078279744199196626e-7, 1e-18, "Pn00", "rb12")
	vvd(t, rb[0][2], 0.8056217146976134152e-7, 1e-18, "Pn00", "rb13")

	vvd(t, rb[1][0], 0.7078279477857337206e-7, 1e-18, "Pn00", "rb21")
	vvd(t, rb[1][1], 0.9999999999999969484, 1e-12, "Pn00", "rb22")
	vvd(t, rb[1][2], 0.3306041454222136517e-7, 1e-18, "Pn00", "rb23")

	vvd(t, rb[2][0], -0.8056217380986972157e-7, 1e-18, "Pn00", "rb31")
	vvd(t, rb[2][1], -0.3306040883980552500e-7, 1e-18, "Pn00", "rb32")
	vvd(t, rb[2][2], 0.9999999999999962084, 1e-12, "Pn00", "rb33")

	vvd(t, rp[0][0], 0.9999989300532289018, 1e-12, "Pn00", "rp11")
	vvd(t, rp[0][1], -0.1341647226791824349e-2, 1e-14, "Pn00", "rp12")
	vvd(t, rp[0][2], -0.5829880927190296547e-3, 1e-14, "Pn00", "rp13")

	vvd(t, rp[1][0], 0.1341647231069759008e-2, 1e-14, "Pn00", "rp21")
	vvd(t, rp[1][1], 0.9999990999908750433, 1e-12, "Pn00", "rp22")
	vvd(t, rp[1][2], -0.3837444441583715468e-6, 1e-14, "Pn00", "rp23")

	vvd(t, rp[2][0], 0.5829880828740957684e-3, 1e-14, "Pn00", "rp31")
	vvd(t, rp[2][1], -0.3984203267708834759e-6, 1e-14, "Pn00", "rp32")
	vvd(t, rp[2][2], 0.9999998300623538046, 1e-12, "Pn00", "rp33")

	vvd(t, rbp[0][0], 0.9999989300052243993, 1e-12, "Pn00", "rbp11")
	vvd(t, rbp[0][1], -0.1341717990239703727e-2, 1e-14, "Pn00", "rbp12")
	vvd(t, rbp[0][2], -0.5829075749891684053e-3, 1e-14, "Pn00", "rbp13")

	vvd(t, rbp[1][0], 0.1341718013831739992e-2, 1e-14, "Pn00", "rbp21")
	vvd(t, rbp[1][1], 0.9999990998959191343, 1e-12, "Pn00", "rbp22")
	vvd(t, rbp[1][2], -0.3505759733565421170e-6, 1e-14, "Pn00", "rbp23")

	vvd(t, rbp[2][0], 0.5829075206857717883e-3, 1e-14, "Pn00", "rbp31")
	vvd(t, rbp[2][1], -0.4315219955198608970e-6, 1e-14, "Pn00", "rbp32")
	vvd(t, rbp[2][2], 0.9999998301093036269, 1e-12, "Pn00", "rbp33")

	vvd(t, rn[0][0], 0.9999999999536069682, 1e-12, "Pn00", "rn11")
	vvd(t, rn[0][1], 0.8837746144872140812e-5, 1e-16, "Pn00", "rn12")
	vvd(t, rn[0][2], 0.3831488838252590008e-5, 1e-16, "Pn00", "rn13")

	vvd(t, rn[1][0], -0.8837590456633197506e-5, 1e-16, "Pn00", "rn21")
	vvd(t, rn[1][1], 0.9999999991354692733, 1e-12, "Pn00", "rn22")
	vvd(t, rn[1][2], -0.4063198798559573702e-4, 1e-16, "Pn00", "rn23")

	vvd(t, rn[2][0], -0.3831847930135328368e-5, 1e-16, "Pn00", "rn31")
	vvd(t, rn[2][1], 0.4063195412258150427e-4, 1e-16, "Pn00", "rn32")
	vvd(t, rn[2][2], 0.9999999991671806225, 1e-12, "Pn00", "rn33")

	vvd(t, rbpn[0][0], 0.9999989440499982806, 1e-12, "Pn00", "rbpn11")
	vvd(t, rbpn[0][1], -0.1332880253640848301e-2, 1e-14, "Pn00", "rbpn12")
	vvd(t, rbpn[0][2], -0.5790760898731087295e-3, 1e-14, "Pn00", "rbpn13")

	vvd(t, rbpn[1][0], 0.1332856746979948745e-2, 1e-14, "Pn00", "rbpn21")
	vvd(t, rbpn[1][1], 0.9999991109064768883, 1e-12, "Pn00", "rbpn22")
	vvd(t, rbpn[1][2], -0.4097740555723063806e-4, 1e-14, "Pn00", "rbpn23")

	vvd(t, rbpn[2][0], 0.5791301929950205000e-3, 1e-14, "Pn00", "rbpn31")
	vvd(t, rbpn[2][1], 0.4020553681373702931e-4, 1e-14, "Pn00", "rbpn32")
	vvd(t, rbpn[2][2], 0.9999998314958529887, 1e-12, "Pn00", "rbpn33")
}

func TestPn00a(t *testing.T) {

	var dpsi, deps, epsa float64
	var rb, rp, rbp, rn, rbpn [3][3]float64

	gofa.Pn00a(2400000.5, 53736.0,
		&dpsi, &deps, &epsa, &rb, &rp, &rbp, &rn, &rbpn)

	vvd(t, dpsi, -0.9630909107115518431e-5, 1e-12, "Pn00a", "dpsi")
	vvd(t, deps, 0.4063239174001678710e-4, 1e-12, "Pn00a", "deps")
	vvd(t, epsa, 0.4090791789404229916, 1e-12, "Pn00a", "epsa")

	vvd(t, rb[0][0], 0.9999999999999942498, 1e-12, "Pn00a", "rb11")
	vvd(t, rb[0][1], -0.7078279744199196626e-7, 1e-16, "Pn00a", "rb12")
	vvd(t, rb[0][2], 0.8056217146976134152e-7, 1e-16, "Pn00a", "rb13")

	vvd(t, rb[1][0], 0.7078279477857337206e-7, 1e-16, "Pn00a", "rb21")
	vvd(t, rb[1][1], 0.9999999999999969484, 1e-12, "Pn00a", "rb22")
	vvd(t, rb[1][2], 0.3306041454222136517e-7, 1e-16, "Pn00a", "rb23")

	vvd(t, rb[2][0], -0.8056217380986972157e-7, 1e-16, "Pn00a", "rb31")
	vvd(t, rb[2][1], -0.3306040883980552500e-7, 1e-16, "Pn00a", "rb32")
	vvd(t, rb[2][2], 0.9999999999999962084, 1e-12, "Pn00a", "rb33")

	vvd(t, rp[0][0], 0.9999989300532289018, 1e-12, "Pn00a", "rp11")
	vvd(t, rp[0][1], -0.1341647226791824349e-2, 1e-14, "Pn00a", "rp12")
	vvd(t, rp[0][2], -0.5829880927190296547e-3, 1e-14, "Pn00a", "rp13")

	vvd(t, rp[1][0], 0.1341647231069759008e-2, 1e-14, "Pn00a", "rp21")
	vvd(t, rp[1][1], 0.9999990999908750433, 1e-12, "Pn00a", "rp22")
	vvd(t, rp[1][2], -0.3837444441583715468e-6, 1e-14, "Pn00a", "rp23")

	vvd(t, rp[2][0], 0.5829880828740957684e-3, 1e-14, "Pn00a", "rp31")
	vvd(t, rp[2][1], -0.3984203267708834759e-6, 1e-14, "Pn00a", "rp32")
	vvd(t, rp[2][2], 0.9999998300623538046, 1e-12, "Pn00a", "rp33")

	vvd(t, rbp[0][0], 0.9999989300052243993, 1e-12, "Pn00a", "rbp11")
	vvd(t, rbp[0][1], -0.1341717990239703727e-2, 1e-14, "Pn00a", "rbp12")
	vvd(t, rbp[0][2], -0.5829075749891684053e-3, 1e-14, "Pn00a", "rbp13")

	vvd(t, rbp[1][0], 0.1341718013831739992e-2, 1e-14, "Pn00a", "rbp21")
	vvd(t, rbp[1][1], 0.9999990998959191343, 1e-12, "Pn00a", "rbp22")
	vvd(t, rbp[1][2], -0.3505759733565421170e-6, 1e-14, "Pn00a", "rbp23")

	vvd(t, rbp[2][0], 0.5829075206857717883e-3, 1e-14, "Pn00a", "rbp31")
	vvd(t, rbp[2][1], -0.4315219955198608970e-6, 1e-14, "Pn00a", "rbp32")
	vvd(t, rbp[2][2], 0.9999998301093036269, 1e-12, "Pn00a", "rbp33")

	vvd(t, rn[0][0], 0.9999999999536227949, 1e-12, "Pn00a", "rn11")
	vvd(t, rn[0][1], 0.8836238544090873336e-5, 1e-14, "Pn00a", "rn12")
	vvd(t, rn[0][2], 0.3830835237722400669e-5, 1e-14, "Pn00a", "rn13")

	vvd(t, rn[1][0], -0.8836082880798569274e-5, 1e-14, "Pn00a", "rn21")
	vvd(t, rn[1][1], 0.9999999991354655028, 1e-12, "Pn00a", "rn22")
	vvd(t, rn[1][2], -0.4063240865362499850e-4, 1e-14, "Pn00a", "rn23")

	vvd(t, rn[2][0], -0.3831194272065995866e-5, 1e-14, "Pn00a", "rn31")
	vvd(t, rn[2][1], 0.4063237480216291775e-4, 1e-14, "Pn00a", "rn32")
	vvd(t, rn[2][2], 0.9999999991671660338, 1e-12, "Pn00a", "rn33")

	vvd(t, rbpn[0][0], 0.9999989440476103435, 1e-12, "Pn00a", "rbpn11")
	vvd(t, rbpn[0][1], -0.1332881761240011763e-2, 1e-14, "Pn00a", "rbpn12")
	vvd(t, rbpn[0][2], -0.5790767434730085751e-3, 1e-14, "Pn00a", "rbpn13")

	vvd(t, rbpn[1][0], 0.1332858254308954658e-2, 1e-14, "Pn00a", "rbpn21")
	vvd(t, rbpn[1][1], 0.9999991109044505577, 1e-12, "Pn00a", "rbpn22")
	vvd(t, rbpn[1][2], -0.4097782710396580452e-4, 1e-14, "Pn00a", "rbpn23")

	vvd(t, rbpn[2][0], 0.5791308472168152904e-3, 1e-14, "Pn00a", "rbpn31")
	vvd(t, rbpn[2][1], 0.4020595661591500259e-4, 1e-14, "Pn00a", "rbpn32")
	vvd(t, rbpn[2][2], 0.9999998314954572304, 1e-12, "Pn00a", "rbpn33")
}

func TestPn00b(t *testing.T) {

	var dpsi, deps, epsa float64
	var rb, rp, rbp, rn, rbpn [3][3]float64

	gofa.Pn00b(2400000.5, 53736.0, &dpsi, &deps, &epsa,
		&rb, &rp, &rbp, &rn, &rbpn)

	vvd(t, dpsi, -0.9632552291148362783e-5, 1e-12, "Pn00b", "dpsi")
	vvd(t, deps, 0.4063197106621159367e-4, 1e-12, "Pn00b", "deps")
	vvd(t, epsa, 0.4090791789404229916, 1e-12, "Pn00b", "epsa")

	vvd(t, rb[0][0], 0.9999999999999942498, 1e-12, "Pn00b", "rb11")
	vvd(t, rb[0][1], -0.7078279744199196626e-7, 1e-16, "Pn00b", "rb12")
	vvd(t, rb[0][2], 0.8056217146976134152e-7, 1e-16, "Pn00b", "rb13")

	vvd(t, rb[1][0], 0.7078279477857337206e-7, 1e-16, "Pn00b", "rb21")
	vvd(t, rb[1][1], 0.9999999999999969484, 1e-12, "Pn00b", "rb22")
	vvd(t, rb[1][2], 0.3306041454222136517e-7, 1e-16, "Pn00b", "rb23")

	vvd(t, rb[2][0], -0.8056217380986972157e-7, 1e-16, "Pn00b", "rb31")
	vvd(t, rb[2][1], -0.3306040883980552500e-7, 1e-16, "Pn00b", "rb32")
	vvd(t, rb[2][2], 0.9999999999999962084, 1e-12, "Pn00b", "rb33")

	vvd(t, rp[0][0], 0.9999989300532289018, 1e-12, "Pn00b", "rp11")
	vvd(t, rp[0][1], -0.1341647226791824349e-2, 1e-14, "Pn00b", "rp12")
	vvd(t, rp[0][2], -0.5829880927190296547e-3, 1e-14, "Pn00b", "rp13")

	vvd(t, rp[1][0], 0.1341647231069759008e-2, 1e-14, "Pn00b", "rp21")
	vvd(t, rp[1][1], 0.9999990999908750433, 1e-12, "Pn00b", "rp22")
	vvd(t, rp[1][2], -0.3837444441583715468e-6, 1e-14, "Pn00b", "rp23")

	vvd(t, rp[2][0], 0.5829880828740957684e-3, 1e-14, "Pn00b", "rp31")
	vvd(t, rp[2][1], -0.3984203267708834759e-6, 1e-14, "Pn00b", "rp32")
	vvd(t, rp[2][2], 0.9999998300623538046, 1e-12, "Pn00b", "rp33")

	vvd(t, rbp[0][0], 0.9999989300052243993, 1e-12, "Pn00b", "rbp11")
	vvd(t, rbp[0][1], -0.1341717990239703727e-2, 1e-14, "Pn00b", "rbp12")
	vvd(t, rbp[0][2], -0.5829075749891684053e-3, 1e-14, "Pn00b", "rbp13")

	vvd(t, rbp[1][0], 0.1341718013831739992e-2, 1e-14, "Pn00b", "rbp21")
	vvd(t, rbp[1][1], 0.9999990998959191343, 1e-12, "Pn00b", "rbp22")
	vvd(t, rbp[1][2], -0.3505759733565421170e-6, 1e-14, "Pn00b", "rbp23")

	vvd(t, rbp[2][0], 0.5829075206857717883e-3, 1e-14, "Pn00b", "rbp31")
	vvd(t, rbp[2][1], -0.4315219955198608970e-6, 1e-14, "Pn00b", "rbp32")
	vvd(t, rbp[2][2], 0.9999998301093036269, 1e-12, "Pn00b", "rbp33")

	vvd(t, rn[0][0], 0.9999999999536069682, 1e-12, "Pn00b", "rn11")
	vvd(t, rn[0][1], 0.8837746144871248011e-5, 1e-14, "Pn00b", "rn12")
	vvd(t, rn[0][2], 0.3831488838252202945e-5, 1e-14, "Pn00b", "rn13")

	vvd(t, rn[1][0], -0.8837590456632304720e-5, 1e-14, "Pn00b", "rn21")
	vvd(t, rn[1][1], 0.9999999991354692733, 1e-12, "Pn00b", "rn22")
	vvd(t, rn[1][2], -0.4063198798559591654e-4, 1e-14, "Pn00b", "rn23")

	vvd(t, rn[2][0], -0.3831847930134941271e-5, 1e-14, "Pn00b", "rn31")
	vvd(t, rn[2][1], 0.4063195412258168380e-4, 1e-14, "Pn00b", "rn32")
	vvd(t, rn[2][2], 0.9999999991671806225, 1e-12, "Pn00b", "rn33")

	vvd(t, rbpn[0][0], 0.9999989440499982806, 1e-12, "Pn00b", "rbpn11")
	vvd(t, rbpn[0][1], -0.1332880253640849194e-2, 1e-14, "Pn00b", "rbpn12")
	vvd(t, rbpn[0][2], -0.5790760898731091166e-3, 1e-14, "Pn00b", "rbpn13")

	vvd(t, rbpn[1][0], 0.1332856746979949638e-2, 1e-14, "Pn00b", "rbpn21")
	vvd(t, rbpn[1][1], 0.9999991109064768883, 1e-12, "Pn00b", "rbpn22")
	vvd(t, rbpn[1][2], -0.4097740555723081811e-4, 1e-14, "Pn00b", "rbpn23")

	vvd(t, rbpn[2][0], 0.5791301929950208873e-3, 1e-14, "Pn00b", "rbpn31")
	vvd(t, rbpn[2][1], 0.4020553681373720832e-4, 1e-14, "Pn00b", "rbpn32")
	vvd(t, rbpn[2][2], 0.9999998314958529887, 1e-12, "Pn00b", "rbpn33")
}

func TestPn06(t *testing.T) {

	var dpsi, deps, epsa float64
	var rb, rp, rbp, rn, rbpn [3][3]float64

	dpsi = -0.9632552291149335877e-5
	deps = 0.4063197106621141414e-4

	gofa.Pn06(2400000.5, 53736.0, dpsi, deps,
		&epsa, &rb, &rp, &rbp, &rn, &rbpn)

	vvd(t, epsa, 0.4090789763356509926, 1e-12, "Pn06", "epsa")

	vvd(t, rb[0][0], 0.9999999999999942497, 1e-12, "Pn06", "rb11")
	vvd(t, rb[0][1], -0.7078368960971557145e-7, 1e-14, "Pn06", "rb12")
	vvd(t, rb[0][2], 0.8056213977613185606e-7, 1e-14, "Pn06", "rb13")

	vvd(t, rb[1][0], 0.7078368694637674333e-7, 1e-14, "Pn06", "rb21")
	vvd(t, rb[1][1], 0.9999999999999969484, 1e-12, "Pn06", "rb22")
	vvd(t, rb[1][2], 0.3305943742989134124e-7, 1e-14, "Pn06", "rb23")

	vvd(t, rb[2][0], -0.8056214211620056792e-7, 1e-14, "Pn06", "rb31")
	vvd(t, rb[2][1], -0.3305943172740586950e-7, 1e-14, "Pn06", "rb32")
	vvd(t, rb[2][2], 0.9999999999999962084, 1e-12, "Pn06", "rb33")

	vvd(t, rp[0][0], 0.9999989300536854831, 1e-12, "Pn06", "rp11")
	vvd(t, rp[0][1], -0.1341646886204443795e-2, 1e-14, "Pn06", "rp12")
	vvd(t, rp[0][2], -0.5829880933488627759e-3, 1e-14, "Pn06", "rp13")

	vvd(t, rp[1][0], 0.1341646890569782183e-2, 1e-14, "Pn06", "rp21")
	vvd(t, rp[1][1], 0.9999990999913319321, 1e-12, "Pn06", "rp22")
	vvd(t, rp[1][2], -0.3835944216374477457e-6, 1e-14, "Pn06", "rp23")

	vvd(t, rp[2][0], 0.5829880833027867368e-3, 1e-14, "Pn06", "rp31")
	vvd(t, rp[2][1], -0.3985701514686976112e-6, 1e-14, "Pn06", "rp32")
	vvd(t, rp[2][2], 0.9999998300623534950, 1e-12, "Pn06", "rp33")

	vvd(t, rbp[0][0], 0.9999989300056797893, 1e-12, "Pn06", "rbp11")
	vvd(t, rbp[0][1], -0.1341717650545059598e-2, 1e-14, "Pn06", "rbp12")
	vvd(t, rbp[0][2], -0.5829075756493728856e-3, 1e-14, "Pn06", "rbp13")

	vvd(t, rbp[1][0], 0.1341717674223918101e-2, 1e-14, "Pn06", "rbp21")
	vvd(t, rbp[1][1], 0.9999990998963748448, 1e-12, "Pn06", "rbp22")
	vvd(t, rbp[1][2], -0.3504269280170069029e-6, 1e-14, "Pn06", "rbp23")

	vvd(t, rbp[2][0], 0.5829075211461454599e-3, 1e-14, "Pn06", "rbp31")
	vvd(t, rbp[2][1], -0.4316708436255949093e-6, 1e-14, "Pn06", "rbp32")
	vvd(t, rbp[2][2], 0.9999998301093032943, 1e-12, "Pn06", "rbp33")

	vvd(t, rn[0][0], 0.9999999999536069682, 1e-12, "Pn06", "rn11")
	vvd(t, rn[0][1], 0.8837746921149881914e-5, 1e-14, "Pn06", "rn12")
	vvd(t, rn[0][2], 0.3831487047682968703e-5, 1e-14, "Pn06", "rn13")

	vvd(t, rn[1][0], -0.8837591232983692340e-5, 1e-14, "Pn06", "rn21")
	vvd(t, rn[1][1], 0.9999999991354692664, 1e-12, "Pn06", "rn22")
	vvd(t, rn[1][2], -0.4063198798558931215e-4, 1e-14, "Pn06", "rn23")

	vvd(t, rn[2][0], -0.3831846139597250235e-5, 1e-14, "Pn06", "rn31")
	vvd(t, rn[2][1], 0.4063195412258792914e-4, 1e-14, "Pn06", "rn32")
	vvd(t, rn[2][2], 0.9999999991671806293, 1e-12, "Pn06", "rn33")

	vvd(t, rbpn[0][0], 0.9999989440504506688, 1e-12, "Pn06", "rbpn11")
	vvd(t, rbpn[0][1], -0.1332879913170492655e-2, 1e-14, "Pn06", "rbpn12")
	vvd(t, rbpn[0][2], -0.5790760923225655753e-3, 1e-14, "Pn06", "rbpn13")

	vvd(t, rbpn[1][0], 0.1332856406595754748e-2, 1e-14, "Pn06", "rbpn21")
	vvd(t, rbpn[1][1], 0.9999991109069366795, 1e-12, "Pn06", "rbpn22")
	vvd(t, rbpn[1][2], -0.4097725651142641812e-4, 1e-14, "Pn06", "rbpn23")

	vvd(t, rbpn[2][0], 0.5791301952321296716e-3, 1e-14, "Pn06", "rbpn31")
	vvd(t, rbpn[2][1], 0.4020538796195230577e-4, 1e-14, "Pn06", "rbpn32")
	vvd(t, rbpn[2][2], 0.9999998314958576778, 1e-12, "Pn06", "rbpn33")
}

func TestPn06a(t *testing.T) {

	var dpsi, deps, epsa float64
	var rb, rp, rbp, rn, rbpn [3][3]float64

	gofa.Pn06a(2400000.5, 53736.0, &dpsi, &deps, &epsa,
		&rb, &rp, &rbp, &rn, &rbpn)

	vvd(t, dpsi, -0.9630912025820308797e-5, 1e-12, "Pn06a", "dpsi")
	vvd(t, deps, 0.4063238496887249798e-4, 1e-12, "Pn06a", "deps")
	vvd(t, epsa, 0.4090789763356509926, 1e-12, "Pn06a", "epsa")

	vvd(t, rb[0][0], 0.9999999999999942497, 1e-12, "Pn06a", "rb11")
	vvd(t, rb[0][1], -0.7078368960971557145e-7, 1e-14, "Pn06a", "rb12")
	vvd(t, rb[0][2], 0.8056213977613185606e-7, 1e-14, "Pn06a", "rb13")

	vvd(t, rb[1][0], 0.7078368694637674333e-7, 1e-14, "Pn06a", "rb21")
	vvd(t, rb[1][1], 0.9999999999999969484, 1e-12, "Pn06a", "rb22")
	vvd(t, rb[1][2], 0.3305943742989134124e-7, 1e-14, "Pn06a", "rb23")

	vvd(t, rb[2][0], -0.8056214211620056792e-7, 1e-14, "Pn06a", "rb31")
	vvd(t, rb[2][1], -0.3305943172740586950e-7, 1e-14, "Pn06a", "rb32")
	vvd(t, rb[2][2], 0.9999999999999962084, 1e-12, "Pn06a", "rb33")

	vvd(t, rp[0][0], 0.9999989300536854831, 1e-12, "Pn06a", "rp11")
	vvd(t, rp[0][1], -0.1341646886204443795e-2, 1e-14, "Pn06a", "rp12")
	vvd(t, rp[0][2], -0.5829880933488627759e-3, 1e-14, "Pn06a", "rp13")

	vvd(t, rp[1][0], 0.1341646890569782183e-2, 1e-14, "Pn06a", "rp21")
	vvd(t, rp[1][1], 0.9999990999913319321, 1e-12, "Pn06a", "rp22")
	vvd(t, rp[1][2], -0.3835944216374477457e-6, 1e-14, "Pn06a", "rp23")

	vvd(t, rp[2][0], 0.5829880833027867368e-3, 1e-14, "Pn06a", "rp31")
	vvd(t, rp[2][1], -0.3985701514686976112e-6, 1e-14, "Pn06a", "rp32")
	vvd(t, rp[2][2], 0.9999998300623534950, 1e-12, "Pn06a", "rp33")

	vvd(t, rbp[0][0], 0.9999989300056797893, 1e-12, "Pn06a", "rbp11")
	vvd(t, rbp[0][1], -0.1341717650545059598e-2, 1e-14, "Pn06a", "rbp12")
	vvd(t, rbp[0][2], -0.5829075756493728856e-3, 1e-14, "Pn06a", "rbp13")

	vvd(t, rbp[1][0], 0.1341717674223918101e-2, 1e-14, "Pn06a", "rbp21")
	vvd(t, rbp[1][1], 0.9999990998963748448, 1e-12, "Pn06a", "rbp22")
	vvd(t, rbp[1][2], -0.3504269280170069029e-6, 1e-14, "Pn06a", "rbp23")

	vvd(t, rbp[2][0], 0.5829075211461454599e-3, 1e-14, "Pn06a", "rbp31")
	vvd(t, rbp[2][1], -0.4316708436255949093e-6, 1e-14, "Pn06a", "rbp32")
	vvd(t, rbp[2][2], 0.9999998301093032943, 1e-12, "Pn06a", "rbp33")

	vvd(t, rn[0][0], 0.9999999999536227668, 1e-12, "Pn06a", "rn11")
	vvd(t, rn[0][1], 0.8836241998111535233e-5, 1e-14, "Pn06a", "rn12")
	vvd(t, rn[0][2], 0.3830834608415287707e-5, 1e-14, "Pn06a", "rn13")

	vvd(t, rn[1][0], -0.8836086334870740138e-5, 1e-14, "Pn06a", "rn21")
	vvd(t, rn[1][1], 0.9999999991354657474, 1e-12, "Pn06a", "rn22")
	vvd(t, rn[1][2], -0.4063240188248455065e-4, 1e-14, "Pn06a", "rn23")

	vvd(t, rn[2][0], -0.3831193642839398128e-5, 1e-14, "Pn06a", "rn31")
	vvd(t, rn[2][1], 0.4063236803101479770e-4, 1e-14, "Pn06a", "rn32")
	vvd(t, rn[2][2], 0.9999999991671663114, 1e-12, "Pn06a", "rn33")

	vvd(t, rbpn[0][0], 0.9999989440480669738, 1e-12, "Pn06a", "rbpn11")
	vvd(t, rbpn[0][1], -0.1332881418091915973e-2, 1e-14, "Pn06a", "rbpn12")
	vvd(t, rbpn[0][2], -0.5790767447612042565e-3, 1e-14, "Pn06a", "rbpn13")

	vvd(t, rbpn[1][0], 0.1332857911250989133e-2, 1e-14, "Pn06a", "rbpn21")
	vvd(t, rbpn[1][1], 0.9999991109049141908, 1e-12, "Pn06a", "rbpn22")
	vvd(t, rbpn[1][2], -0.4097767128546784878e-4, 1e-14, "Pn06a", "rbpn23")

	vvd(t, rbpn[2][0], 0.5791308482835292617e-3, 1e-14, "Pn06a", "rbpn31")
	vvd(t, rbpn[2][1], 0.4020580099454020310e-4, 1e-14, "Pn06a", "rbpn32")
	vvd(t, rbpn[2][2], 0.9999998314954628695, 1e-12, "Pn06a", "rbpn33")
}

func TestPnm00a(t *testing.T) {
	var rbpn [3][3]float64

	gofa.Pnm00a(2400000.5, 50123.9999, &rbpn)

	vvd(t, rbpn[0][0], 0.9999995832793134257, 1e-12, "Pnm00a", "11")
	vvd(t, rbpn[0][1], 0.8372384254137809439e-3, 1e-14, "Pnm00a", "12")
	vvd(t, rbpn[0][2], 0.3639684306407150645e-3, 1e-14, "Pnm00a", "13")

	vvd(t, rbpn[1][0], -0.8372535226570394543e-3, 1e-14, "Pnm00a", "21")
	vvd(t, rbpn[1][1], 0.9999996486491582471, 1e-12, "Pnm00a", "22")
	vvd(t, rbpn[1][2], 0.4132915262664072381e-4, 1e-14, "Pnm00a", "23")

	vvd(t, rbpn[2][0], -0.3639337004054317729e-3, 1e-14, "Pnm00a", "31")
	vvd(t, rbpn[2][1], -0.4163386925461775873e-4, 1e-14, "Pnm00a", "32")
	vvd(t, rbpn[2][2], 0.9999999329094390695, 1e-12, "Pnm00a", "33")

}

func TestPnm00b(t *testing.T) {
	var rbpn [3][3]float64

	gofa.Pnm00b(2400000.5, 50123.9999, &rbpn)

	vvd(t, rbpn[0][0], 0.9999995832776208280, 1e-12, "Pnm00b", "11")
	vvd(t, rbpn[0][1], 0.8372401264429654837e-3, 1e-14, "Pnm00b", "12")
	vvd(t, rbpn[0][2], 0.3639691681450271771e-3, 1e-14, "Pnm00b", "13")

	vvd(t, rbpn[1][0], -0.8372552234147137424e-3, 1e-14, "Pnm00b", "21")
	vvd(t, rbpn[1][1], 0.9999996486477686123, 1e-12, "Pnm00b", "22")
	vvd(t, rbpn[1][2], 0.4132832190946052890e-4, 1e-14, "Pnm00b", "23")

	vvd(t, rbpn[2][0], -0.3639344385341866407e-3, 1e-14, "Pnm00b", "31")
	vvd(t, rbpn[2][1], -0.4163303977421522785e-4, 1e-14, "Pnm00b", "32")
	vvd(t, rbpn[2][2], 0.9999999329092049734, 1e-12, "Pnm00b", "33")
}

func TestPnm06a(t *testing.T) {
	var rbpn [3][3]float64

	gofa.Pnm06a(2400000.5, 50123.9999, &rbpn)
	r := [3][3]float64{
		{0.9999995832794205484, 0.8372382772630962111e-3, 0.3639684771140623099e-3},
		{-0.8372533744743683605e-3, 0.9999996486492861646, 0.4132905944611019498e-4},
		{-0.3639337469629464969e-3, -0.4163377605910663999e-4, 0.9999999329094260057},
	}
	vrv(t, rbpn, r, 1e-14, "Pnm06a")
}
func TestPnm80(t *testing.T) {

	var rmatpn [3][3]float64

	gofa.Pnm80(2400000.5, 50123.9999, &rmatpn)

	vvd(t, rmatpn[0][0], 0.9999995831934611169, 1e-12, "Pnm80", "11")
	vvd(t, rmatpn[0][1], 0.8373654045728124011e-3, 1e-14, "Pnm80", "12")
	vvd(t, rmatpn[0][2], 0.3639121916933106191e-3, 1e-14, "Pnm80", "13")

	vvd(t, rmatpn[1][0], -0.8373804896118301316e-3, 1e-14, "Pnm80", "21")
	vvd(t, rmatpn[1][1], 0.9999996485439674092, 1e-12, "Pnm80", "22")
	vvd(t, rmatpn[1][2], 0.4130202510421549752e-4, 1e-14, "Pnm80", "23")

	vvd(t, rmatpn[2][0], -0.3638774789072144473e-3, 1e-14, "Pnm80", "31")
	vvd(t, rmatpn[2][1], -0.4160674085851722359e-4, 1e-14, "Pnm80", "32")
	vvd(t, rmatpn[2][2], 0.9999999329310274805, 1e-12, "Pnm80", "33")
}

func TestP06e(t *testing.T) {
	var eps0, psia, oma, bpa, bqa, pia, bpia,
		epsa, chia, za, zetaa, thetaa, pa, gam, phi, psi float64

	gofa.P06e(2400000.5, 52541.0, &eps0, &psia, &oma, &bpa,
		&bqa, &pia, &bpia, &epsa, &chia, &za,
		&zetaa, &thetaa, &pa, &gam, &phi, &psi)

	vvd(t, eps0, 0.4090926006005828715, 1e-14, "P06e", "eps0")
	vvd(t, psia, 0.6664369630191613431e-3, 1e-14, "P06e", "psia")
	vvd(t, oma, 0.4090925973783255982, 1e-14, "P06e", "oma")
	vvd(t, bpa, 0.5561149371265209445e-6, 1e-14, "P06e", "bpa")
	vvd(t, bqa, -0.6191517193290621270e-5, 1e-14, "P06e", "bqa")
	vvd(t, pia, 0.6216441751884382923e-5, 1e-14, "P06e", "pia")
	vvd(t, bpia, 3.052014180023779882, 1e-14, "P06e", "bpia")
	vvd(t, epsa, 0.4090864054922431688, 1e-14, "P06e", "epsa")
	vvd(t, chia, 0.1387703379530915364e-5, 1e-14, "P06e", "chia")
	vvd(t, za, 0.2921789846651790546e-3, 1e-14, "P06e", "za")
	vvd(t, zetaa, 0.3178773290332009310e-3, 1e-14, "P06e", "zetaa")
	vvd(t, thetaa, 0.2650932701657497181e-3, 1e-14, "P06e", "thetaa")
	vvd(t, pa, 0.6651637681381016288e-3, 1e-14, "P06e", "pa")
	vvd(t, gam, 0.1398077115963754987e-5, 1e-14, "P06e", "gam")
	vvd(t, phi, 0.4090864090837462602, 1e-14, "P06e", "phi")
	vvd(t, psi, 0.6664464807480920325e-3, 1e-14, "P06e", "psi")
}

func TestPom00(t *testing.T) {
	var xp, yp, sp float64
	var rpom [3][3]float64

	xp = 2.55060238e-7
	yp = 1.860359247e-6
	sp = -0.1367174580728891460e-10

	gofa.Pom00(xp, yp, sp, &rpom)

	vvd(t, rpom[0][0], 0.9999999999999674721, 1e-12, "Pom00", "11")
	vvd(t, rpom[0][1], -0.1367174580728846989e-10, 1e-16, "Pom00", "12")
	vvd(t, rpom[0][2], 0.2550602379999972345e-6, 1e-16, "Pom00", "13")

	vvd(t, rpom[1][0], 0.1414624947957029801e-10, 1e-16, "Pom00", "21")
	vvd(t, rpom[1][1], 0.9999999999982695317, 1e-12, "Pom00", "22")
	vvd(t, rpom[1][2], -0.1860359246998866389e-5, 1e-16, "Pom00", "23")

	vvd(t, rpom[2][0], -0.2550602379741215021e-6, 1e-16, "Pom00", "31")
	vvd(t, rpom[2][1], 0.1860359247002414021e-5, 1e-16, "Pom00", "32")
	vvd(t, rpom[2][2], 0.9999999999982370039, 1e-12, "Pom00", "33")
}

func TestPr00(t *testing.T) {
	var dpsipr, depspr float64

	gofa.Pr00(2400000.5, 53736, &dpsipr, &depspr)

	vvd(t, dpsipr, -0.8716465172668347629e-7, 1e-22,
		"Pr00", "dpsipr")
	vvd(t, depspr, -0.7342018386722813087e-8, 1e-22,
		"Pr00", "depspr")

}

func TestPrec76(t *testing.T) {
	var ep01, ep02, ep11, ep12, zeta, z, theta float64

	ep01 = 2400000.5
	ep02 = 33282.0
	ep11 = 2400000.5
	ep12 = 51544.0

	gofa.Prec76(ep01, ep02, ep11, ep12, &zeta, &z, &theta)

	vvd(t, zeta, 0.5588961642000161243e-2, 1e-12, "Prec76", "zeta")
	vvd(t, z, 0.5589922365870680624e-2, 1e-12, "Prec76", "z")
	vvd(t, theta, 0.4858945471687296760e-2, 1e-12, "Prec76", "theta")

}
func TestS00(t *testing.T) {

	var x, y, s float64

	x = 0.5791308486706011000e-3
	y = 0.4020579816732961219e-4

	s = gofa.S00(2400000.5, 53736.0, x, y)

	vvd(t, s, -0.1220036263270905693e-7, 1e-18, "S00", "")

}
func TestS00a(t *testing.T) {
	s := gofa.S00a(2400000.5, 52541.0)

	vvd(t, s, -0.1340684448919163584e-7, 1e-18, "S00a", "")
}

func TestS00b(t *testing.T) {
	s := gofa.S00b(2400000.5, 52541.0)

	vvd(t, s, -0.1340695782951026584e-7, 1e-18, "S00b", "")
}

func TestS06(t *testing.T) {
	var x, y, s float64

	x = 0.5791308486706011000e-3
	y = 0.4020579816732961219e-4

	s = gofa.S06(2400000.5, 53736.0, x, y)

	vvd(t, s, -0.1220032213076463117e-7, 1e-18, "S06", "")
}

func TestS06a(t *testing.T) {

	s := gofa.S06a(2400000.5, 52541.0)

	vvd(t, s, -0.1340680437291812383e-7, 1e-18, "S06a", "")
}

func TestSp00(t *testing.T) {
	vvd(t, gofa.Sp00(2400000.5, 52541.0),
		-0.6216698469981019309e-11, 1e-12, "Sp00", "")

}

func TestXy06(t *testing.T) {
	var x, y float64

	gofa.Xy06(2400000.5, 53736.0, &x, &y)

	vvd(t, x, 0.5791308486706010975e-3, 1e-15, "Xy06", "x")
	vvd(t, y, 0.4020579816732958141e-4, 1e-16, "Xy06", "y")
}

func TestXys00a(t *testing.T) {
	var x, y, s float64

	gofa.Xys00a(2400000.5, 53736.0, &x, &y, &s)

	vvd(t, x, 0.5791308472168152904e-3, 1e-14, "Xys00a", "x")
	vvd(t, y, 0.4020595661591500259e-4, 1e-15, "Xys00a", "y")
	vvd(t, s, -0.1220040848471549623e-7, 1e-18, "Xys00a", "s")
}

func TestXys00b(t *testing.T) {
	var x, y, s float64

	gofa.Xys00b(2400000.5, 53736.0, &x, &y, &s)

	vvd(t, x, 0.5791301929950208873e-3, 1e-14, "Xys00b", "x")
	vvd(t, y, 0.4020553681373720832e-4, 1e-15, "Xys00b", "y")
	vvd(t, s, -0.1220027377285083189e-7, 1e-18, "Xys00b", "s")
}

func TestXys06a(t *testing.T) {
	var x, y, s float64

	gofa.Xys06a(2400000.5, 53736.0, &x, &y, &s)

	vvd(t, x, 0.5791308482835292617e-3, 1e-14, "Xys06a", "x")
	vvd(t, y, 0.4020580099454020310e-4, 1e-15, "Xys06a", "y")
	vvd(t, s, -0.1220032294164579896e-7, 1e-18, "Xys06a", "s")

}
