import math as ma
import numpy as np

def pymAnp(A):
    '''
    Normalize angle into the range 0 <= a < 2pi.

    Parameters
    ----------
    a : float
        angle (radians)

    Returns
    -------
    function value : float
        angle in range 0-2pi

    '''
    W=np.abs(A)%(2*ma.pi)
    if A<0:
        W=-W
    if (W<0.0):
        W=W+2*ma.pi
    
    return(W)

def pymEra00(DJ1,DJ2):
    '''
    Earth rotation angle (IAU 2000 model).

    Parameters
    ----------
    dj1 : float
        UT1 as a 2-part Julian Date    
    dj2 : float
        UT1 as a 2-part Julian Date

    Returns
    -------
    function value : float
        Earth rotation angle (radians), range 0-2pi

    '''
    #参考历元(J2000.0), JD
    DJ00=2451545.0
    
    #自参考历元经过的天数
    if (DJ1<=DJ2):
        D1=DJ1
        D2=DJ2
    else:
        D1=DJ2
        D2=DJ1
    T=D1+(D2-DJ00)

    #时间的小数部分(days).
    F=D1%1.0+D2%1.0
    
    #在这个UT1时刻的地球自转角.
    A=2*ma.pi*(F+0.7790572732640+0.00273781191135448*T)
    ERA=pymAnp(A)
    
    return(ERA)

def pymGmst00(UTA,UTB,TTA,TTB):
    '''
    Greenwich mean sidereal time (model consistent with IAU 2000
    resolutions).

    Parameters
    ----------
    uta : float
        UT1 as a 2-part Julian Date    
    utb : float
        UT1 as a 2-part Julian Date    
    tta : float
        TT as a 2-part Julian Date    
    ttb : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        Greenwich mean sidereal time (radians)

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6

    #参考历元(J2000.0), JD
    DJ00=2451545.0

    #1儒略世纪对应的天数
    DJC=36525.0
    
    #自参考历元起的世界时间隔
    T=((TTA-DJ00)+TTB)/DJC
    
    #格林尼治平恒星时, IAU 2000.
    A=pymEra00(UTA,UTB)
    B=A+(0.014506+(4612.15739966+(+1.39667721+(-0.00009344+(+0.00001882)
                                               *T)*T)*T)*T)*DAS2R
    GMST=pymAnp(B)  
    
    return(GMST)

def pymGmst06(UTA,UTB,TTA,TTB):
    '''
    Greenwich mean sidereal time (consistent with IAU 2006 precession).

    Parameters
    ----------
    uta : float
        UT1 as a 2-part Julian Date    
    utb : float
        UT1 as a 2-part Julian Date    
    tta : float
        TT as a 2-part Julian Date    
    ttb : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        Greenwich mean sidereal time (radians)

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6

    #参考历元(J2000.0), JD
    DJ00=2451545.0

    #1儒略世纪对应的天数
    DJC=36525.0
    
    #自参考历元起的世界时间隔
    T=((TTA-DJ00)+TTB)/DJC
    
    #格林尼治平恒星时, IAU 2006.
    A=pymEra00(UTA,UTB)
    GMST=pymAnp(A+(0.014506+(4612.156534+(1.3915817+(-0.00000044+
                    (-0.000029956+(-0.0000000368)*T)*T)*T)*T)*T)*DAS2R)
    
    return(GMST)

def pymGmst82(DJ1,DJ2):
    '''
    Universal Time to Greenwich mean sidereal time (IAU 1982 model).

    Parameters
    ----------
    dj1 : float
        UT1 Julian Date    
    dj2 : float
        UT1 Julian Date

    Returns
    -------
    function value : float
        Greenwich mean sidereal time (radians)

    '''
    #1秒对应的弧度
    DS2R=7.272205216643039903848712e-5
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0
    
    #1天的秒数，1世纪的天数
    D2S=86400.0
    DJC=36525.0
    
    #IAU 1982 GMST-UT1模型的系数
    A=24110.54841-D2S/2.0
    B=8640184.812866
    C=0.093104
    D=-6.2e-6
    #UT1的一天的起始点为当天正午，因此第一个系数A要调整12个小时 
    #给定时间与参考历元之间的时间间隔，儒略世纪数
    if (DJ1<DJ2):
        D1=DJ1
        D2=DJ2
    else:
        D1=DJ2
        D2=DJ1
    T=(D1+(D2-DJ00))/DJC
    
    #儒略日的小数部分(UT1), 单位：秒.
    C1=np.abs(D1)%1.0
    C2=np.abs(D2)%1.0
    if D1<0:
        B1=-C1
    else:
        B1=C1
    if D2<0:
        B2=-C2
    else:
        B2=C2
    F=D2S*(B1+B2)
    
    #给定时刻的格林尼治平恒星时
    GMST=pymAnp(DS2R*((A+(B+(C+D*T)*T)*T)+F))
    
    return(GMST)

def pymPr00(DATE1,DATE2):
    '''
    Precession-rate part of the IAU 2000 precession-nutation models
    (part of MHB2000).

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    dpsipr : float 
        precession corrections    
    depspr : float
        precession corrections

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #参考历元(J2000.0), JD
    DJ00=2451545

    #1儒略世纪的天数
    DJC=36525.0
    
    # 进动和倾角的改正，单位：弧度/世纪
    PRECOR=-0.29965*DAS2R
    OBLCOR=-0.02524*DAS2R
    
    #与参考历元之间的时间间隔，单位：儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #进动速率
    DPSIPR=PRECOR*T
    DEPSPR=OBLCOR*T
    
    return(DPSIPR,DEPSPR)

def pymObl80(DATE1,DATE2):
    '''
    Mean obliquity of the ecliptic, IAU 1980 model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        obliquity of the ecliptic (radians)

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0

    #1儒略世纪对应的天数
    DJC=36525.0
    
    #与参考历元之间的时间间隔(JC).
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #平均倾角
    OBL=DAS2R*(84381.448+(-46.8150+(-0.00059+0.001813*T)*T)*T)
    
    return(OBL)

def pymNut00a(DATE1,DATE2):
    '''
    Nutation, IAU 2000A model (MHB2000 luni-solar and planetary nutation
    with free core nutation omitted).

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    dpsi : float
        nutation, luni-solar + planetary    
    deps : float
        nutation, luni-solar + planetary

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #一个圆内的总角秒数
    TURNAS=1296000.0
    
    #2Pi
    D2PI = 6.283185307179586476925287
    
    #0.1微角秒对应的弧度
    U2R=DAS2R/1e7
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0
    
    #1儒略世纪对应的天数
    DJC=36525.0
    
    #日月章动模型
    #模型中的项数
    NLS=678
    
    #模型中的项数
    #行星章动模型
    NPL=687
    
    #日月章动的参数和项系数的表格
    #  L     L'    F     D     Om
    NALS=[[0,0,0,0,1],
            [0,0,2,-2,2],
            [0,0,2,0,2],
            [0,0,0,0,2],
            [0,1,0,0,0],
            [0,1,2,-2,2],
            [1,0,0,0,0],
            [0,0,2,0,1],
            [1,0,2,0,2],
            [0,-1,2,-2,2],
            [0,0,2,-2,1],
            [-1,0,2,0,2],
            [-1,0,0,2,0],
            [1,0,0,0,1],
            [-1,0,0,0,1],
            [-1,0,2,2,2],
            [1,0,2,0,1],
            [-2,0,2,0,1],
            [0,0,0,2,0],
            [0,0,2,2,2],
            [0,-2,2,-2,2],
            [-2,0,0,2,0],
            [2,0,2,0,2],
            [1,0,2,-2,2],
            [-1,0,2,0,1],
            [2,0,0,0,0],
            [0,0,2,0,0],
            [0,1,0,0,1],
            [-1,0,0,2,1],
            [0,2,2,-2,2],
            [0,0,-2,2,0],
            [1,0,0,-2,1],
            [0,-1,0,0,1],
            [-1,0,2,2,1],
            [0,2,0,0,0],
            [1,0,2,2,2],
            [-2,0,2,0,0],
            [0,1,2,0,2],
            [0,0,2,2,1],
            [0,-1,2,0,2],
            [0,0,0,2,1],
            [1,0,2,-2,1],
            [2,0,2,-2,2],
            [-2,0,0,2,1],
            [2,0,2,0,1],
            [0,-1,2,-2,1],
            [0,0,0,-2,1],
            [-1,-1,0,2,0],
            [2,0,0,-2,1],
            [1,0,0,2,0],
            [0,1,2,-2,1],
            [1,-1,0,0,0],
            [-2,0,2,0,2],
            [3,0,2,0,2],
            [0,-1,0,2,0],
            [1,-1,2,0,2],
            [0,0,0,1,0],
            [-1,-1,2,2,2],
            [-1,0,2,0,0],
            [0,-1,2,2,2],
            [-2,0,0,0,1],
            [1,1,2,0,2],
            [2,0,0,0,1],
            [-1,1,0,1,0],
            [1,1,0,0,0],
            [1,0,2,0,0],
            [-1,0,2,-2,1],
            [1,0,0,0,2],
            [-1,0,0,1,0],
            [0,0,2,1,2],
            [-1,0,2,4,2],
            [-1,1,0,1,1],
            [0,-2,2,-2,1],
            [1,0,2,2,1],
            [-2,0,2,2,2],
            [-1,0,0,0,2],
            [1,1,2,-2,2],
            [-2,0,2,4,2],
            [-1,0,4,0,2],
            [2,0,2,-2,1],
            [2,0,2,2,2],
            [1,0,0,2,1],
            [3,0,0,0,0],
            [3,0,2,-2,2],
            [0,0,4,-2,2],
            [0,1,2,0,1],
            [0,0,-2,2,1],
            [0,0,2,-2,3],
            [-1,0,0,4,0],
            [2,0,-2,0,1],
            [-2,0,0,4,0],
            [-1,-1,0,2,1],
            [-1,0,0,1,1],
            [0,1,0,0,2],
            [0,0,-2,0,1],
            [0,-1,2,0,1],
            [0,0,2,-1,2],
            [0,0,2,4,2],
            [-2,-1,0,2,0],
            [1,1,0,-2,1],
            [-1,1,0,2,0],
            [-1,1,0,1,2],
            [1,-1,0,0,1],
            [1,-1,2,2,2],
            [-1,1,2,2,2],
            [3,0,2,0,1],
            [0,1,-2,2,0],
            [-1,0,0,-2,1],
            [0,1,2,2,2],
            [-1,-1,2,2,1],
            [0,-1,0,0,2],
            [1,0,2,-4,1],
            [-1,0,-2,2,0],
            [0,-1,2,2,1],
            [2,-1,2,0,2],
            [0,0,0,2,2],
            [1,-1,2,0,1],
            [-1,1,2,0,2],
            [0,1,0,2,0],
            [0,-1,-2,2,0],
            [0,3,2,-2,2],
            [0,0,0,1,1],
            [-1,0,2,2,0],
            [2,1,2,0,2],
            [1,1,0,0,1],
            [1,1,2,0,1],
            [2,0,0,2,0],
            [1,0,-2,2,0],
            [-1,0,0,2,2],
            [0,1,0,1,0],
            [0,1,0,-2,1],
            [-1,0,2,-2,2],
            [0,0,0,-1,1],
            [-1,1,0,0,1],
            [1,0,2,-1,2],
            [1,-1,0,2,0],
            [0,0,0,4,0],
            [1,0,2,1,2],
            [0,0,2,1,1],
            [1,0,0,-2,2],
            [-1,0,2,4,1],
            [1,0,-2,0,1],
            [1,1,2,-2,1],
            [0,0,2,2,0],
            [-1,0,2,-1,1],
            [-2,0,2,2,1],
            [4,0,2,0,2],
            [2,-1,0,0,0],
            [2,1,2,-2,2],
            [0,1,2,1,2],
            [1,0,4,-2,2],
            [-1,-1,0,0,1],
            [0,1,0,2,1],
            [-2,0,2,4,1],
            [2,0,2,0,0],
            [1,0,0,1,0],
            [-1,0,0,4,1],
            [-1,0,4,0,1],
            [2,0,2,2,1],
            [0,0,2,-3,2],
            [-1,-2,0,2,0],
            [2,1,0,0,0],
            [0,0,4,0,2],
            [0,0,0,0,3],
            [0,3,0,0,0],
            [0,0,2,-4,1],
            [0,-1,0,2,1],
            [0,0,0,4,1],
            [-1,-1,2,4,2],
            [1,0,2,4,2],
            [-2,2,0,2,0],
            [-2,-1,2,0,1],
            [-2,0,0,2,2],
            [-1,-1,2,0,2],
            [0,0,4,-2,1],
            [3,0,2,-2,1],
            [-2,-1,0,2,1],
            [1,0,0,-1,1],
            [0,-2,0,2,0],
            [-2,0,0,4,1],
            [-3,0,0,0,1],
            [1,1,2,2,2],
            [0,0,2,4,1],
            [3,0,2,2,2],
            [-1,1,2,-2,1],
            [2,0,0,-4,1],
            [0,0,0,-2,2],
            [2,0,2,-4,1],
            [-1,1,0,2,1],
            [0,0,2,-1,1],
            [0,-2,2,2,2],
            [2,0,0,2,1],
            [4,0,2,-2,2],
            [2,0,0,-2,2],
            [0,2,0,0,1],
            [1,0,0,-4,1],
            [0,2,2,-2,1],
            [-3,0,0,4,0],
            [-1,1,2,0,1],
            [-1,-1,0,4,0],
            [-1,-2,2,2,2],
            [-2,-1,2,4,2],
            [1,-1,2,2,1],
            [-2,1,0,2,0],
            [-2,1,2,0,1],
            [2,1,0,-2,1],
            [-3,0,2,0,1],
            [-2,0,2,-2,1],
            [-1,1,0,2,2],
            [0,-1,2,-1,2],
            [-1,0,4,-2,2],
            [0,-2,2,0,2],
            [-1,0,2,1,2],
            [2,0,0,0,2],
            [0,0,2,0,3],
            [-2,0,4,0,2],
            [-1,0,-2,0,1],
            [-1,1,2,2,1],
            [3,0,0,0,1],
            [-1,0,2,3,2],
            [2,-1,2,0,1],
            [0,1,2,2,1],
            [0,-1,2,4,2],
            [2,-1,2,2,2],
            [0,2,-2,2,0],
            [-1,-1,2,-1,1],
            [0,-2,0,0,1],
            [1,0,2,-4,2],
            [1,-1,0,-2,1],
            [-1,-1,2,0,1],
            [1,-1,2,-2,2],
            [-2,-1,0,4,0],
            [-1,0,0,3,0],
            [-2,-1,2,2,2],
            [0,2,2,0,2],
            [1,1,0,2,0],
            [2,0,2,-1,2],
            [1,0,2,1,1],
            [4,0,0,0,0],
            [2,1,2,0,1],
            [3,-1,2,0,2],
            [-2,2,0,2,1],
            [1,0,2,-3,1],
            [1,1,2,-4,1],
            [-1,-1,2,-2,1],
            [0,-1,0,-1,1],
            [0,-1,0,-2,1],
            [-2,0,0,0,2],
            [-2,0,-2,2,0],
            [-1,0,-2,4,0],
            [1,-2,0,0,0],
            [0,1,0,1,1],
            [-1,2,0,2,0],
            [1,-1,2,-2,1],
            [1,2,2,-2,2],
            [2,-1,2,-2,2],
            [1,0,2,-1,1],
            [2,1,2,-2,1],
            [-2,0,0,-2,1],
            [1,-2,2,0,2],
            [0,1,2,1,1],
            [1,0,4,-2,1],
            [-2,0,4,2,2],
            [1,1,2,1,2],
            [1,0,0,4,0],
            [1,0,2,2,0],
            [2,0,2,1,2],
            [3,1,2,0,2],
            [4,0,2,0,1],
            [-2,-1,2,0,0],
            [0,1,-2,2,1],
            [1,0,-2,1,0],
            [0,-1,-2,2,1],
            [2,-1,0,-2,1],
            [-1,0,2,-1,2],
            [1,0,2,-3,2],
            [0,1,2,-2,3],
            [0,0,2,-3,1],
            [-1,0,-2,2,1],
            [0,0,2,-4,2],
            [-2,1,0,0,1],
            [-1,0,0,-1,1],
            [2,0,2,-4,2],
            [0,0,4,-4,4],
            [0,0,4,-4,2],
            [-1,-2,0,2,1],
            [-2,0,0,3,0],
            [1,0,-2,2,1],
            [-3,0,2,2,2],
            [-3,0,2,2,1],
            [-2,0,2,2,0],
            [2,-1,0,0,1],
            [-2,1,2,2,2],
            [1,1,0,1,0],
            [0,1,4,-2,2],
            [-1,1,0,-2,1],
            [0,0,0,-4,1],
            [1,-1,0,2,1],
            [1,1,0,2,1],
            [-1,2,2,2,2],
            [3,1,2,-2,2],
            [0,-1,0,4,0],
            [2,-1,0,2,0],
            [0,0,4,0,1],
            [2,0,4,-2,2],
            [-1,-1,2,4,1],
            [1,0,0,4,1],
            [1,-2,2,2,2],
            [0,0,2,3,2],
            [-1,1,2,4,2],
            [3,0,0,2,0],
            [-1,0,4,2,2],
            [1,1,2,2,1],
            [-2,0,2,6,2],
            [2,1,2,2,2],
            [-1,0,2,6,2],
            [1,0,2,4,1],
            [2,0,2,4,2],
            [1,1,-2,1,0],
            [-3,1,2,1,2],
            [2,0,-2,0,2],
            [-1,0,0,1,2],
            [-4,0,2,2,1],
            [-1,-1,0,1,0],
            [0,0,-2,2,2],
            [1,0,0,-1,2],
            [0,-1,2,-2,3],
            [-2,1,2,0,0],
            [0,0,2,-2,4],
            [-2,-2,0,2,0],
            [-2,0,-2,4,0],
            [0,-2,-2,2,0],
            [1,2,0,-2,1],
            [3,0,0,-4,1],
            [-1,1,2,-2,2],
            [1,-1,2,-4,1],
            [1,1,0,-2,2],
            [-3,0,2,0,0],
            [-3,0,2,0,2],
            [-2,0,0,1,0],
            [0,0,-2,1,0],
            [-3,0,0,2,1],
            [-1,-1,-2,2,0],
            [0,1,2,-4,1],
            [2,1,0,-4,1],
            [0,2,0,-2,1],
            [1,0,0,-3,1],
            [-2,0,2,-2,2],
            [-2,-1,0,0,1],
            [-4,0,0,2,0],
            [1,1,0,-4,1],
            [-1,0,2,-4,1],
            [0,0,4,-4,1],
            [0,3,2,-2,2],
            [-3,-1,0,4,0],
            [-3,0,0,4,1],
            [1,-1,-2,2,0],
            [-1,-1,0,2,2],
            [1,-2,0,0,1],
            [1,-1,0,0,2],
            [0,0,0,1,2],
            [-1,-1,2,0,0],
            [1,-2,2,-2,2],
            [0,-1,2,-1,1],
            [-1,0,2,0,3],
            [1,1,0,0,2],
            [-1,1,2,0,0],
            [1,2,0,0,0],
            [-1,2,2,0,2],
            [-1,0,4,-2,1],
            [3,0,2,-4,2],
            [1,2,2,-2,1],
            [1,0,4,-4,2],
            [-2,-1,0,4,1],
            [0,-1,0,2,2],
            [-2,1,0,4,0],
            [-2,-1,2,2,1],
            [2,0,-2,2,0],
            [1,0,0,1,1],
            [0,1,0,2,2],
            [1,-1,2,-1,2],
            [-2,0,4,0,1],
            [2,1,0,0,1],
            [0,1,2,0,0],
            [0,-1,4,-2,2],
            [0,0,4,-2,4],
            [0,2,2,0,1],
            [-3,0,0,6,0],
            [-1,-1,0,4,1],
            [1,-2,0,2,0],
            [-1,0,0,4,2],
            [-1,-2,2,2,1],
            [-1,0,0,-2,2],
            [1,0,-2,-2,1],
            [0,0,-2,-2,1],
            [-2,0,-2,0,1],
            [0,0,0,3,1],
            [0,0,0,3,0],
            [-1,1,0,4,0],
            [-1,-1,2,2,0],
            [-2,0,2,3,2],
            [1,0,0,2,2],
            [0,-1,2,1,2],
            [3,-1,0,0,0],
            [2,0,0,1,0],
            [1,-1,2,0,0],
            [0,0,2,1,0],
            [1,0,2,0,3],
            [3,1,0,0,0],
            [3,-1,2,-2,2],
            [2,0,2,-1,1],
            [1,1,2,0,0],
            [0,0,4,-1,2],
            [1,2,2,0,2],
            [-2,0,0,6,0],
            [0,-1,0,4,1],
            [-2,-1,2,4,1],
            [0,-2,2,2,1],
            [0,-1,2,2,0],
            [-1,0,2,3,1],
            [-2,1,2,4,2],
            [2,0,0,2,2],
            [2,-2,2,0,2],
            [-1,1,2,3,2],
            [3,0,2,-1,2],
            [4,0,2,-2,1],
            [-1,0,0,6,0],
            [-1,-2,2,4,2],
            [-3,0,2,6,2],
            [-1,0,2,4,0],
            [3,0,0,2,1],
            [3,-1,2,0,1],
            [3,0,2,0,0],
            [1,0,4,0,2],
            [5,0,2,-2,2],
            [0,-1,2,4,1],
            [2,-1,2,2,1],
            [0,1,2,4,2],
            [1,-1,2,4,2],
            [3,-1,2,2,2],
            [3,0,2,2,1],
            [5,0,2,0,2],
            [0,0,2,6,2],
            [4,0,2,2,2],
            [0,-1,1,-1,1],
            [-1,0,1,0,3],
            [0,-2,2,-2,3],
            [1,0,-1,0,1],
            [2,-2,0,-2,1],
            [-1,0,1,0,2],
            [-1,0,1,0,1],
            [-1,-1,2,-1,2],
            [-2,2,0,2,2],
            [-1,0,1,0,0],
            [-4,1,2,2,2],
            [-3,0,2,1,1],
            [-2,-1,2,0,2],
            [1,0,-2,1,1],
            [2,-1,-2,0,1],
            [-4,0,2,2,0],
            [-3,1,0,3,0],
            [-1,0,-1,2,0],
            [0,-2,0,0,2],
            [0,-2,0,0,2],
            [-3,0,0,3,0],
            [-2,-1,0,2,2],
            [-1,0,-2,3,0],
            [-4,0,0,4,0],
            [2,1,-2,0,1],
            [2,-1,0,-2,2],
            [0,0,1,-1,0],
            [-1,2,0,1,0],
            [-2,1,2,0,2],
            [1,1,0,-1,1],
            [1,0,1,-2,1],
            [0,2,0,0,2],
            [1,-1,2,-3,1],
            [-1,1,2,-1,1],
            [-2,0,4,-2,2],
            [-2,0,4,-2,1],
            [-2,-2,0,2,1],
            [-2,0,-2,4,0],
            [1,2,2,-4,1],
            [1,1,2,-4,2],
            [-1,2,2,-2,1],
            [2,0,0,-3,1],
            [-1,2,0,0,1],
            [0,0,0,-2,0],
            [-1,-1,2,-2,2],
            [-1,1,0,0,2],
            [0,0,0,-1,2],
            [-2,1,0,1,0],
            [1,-2,0,-2,1],
            [1,0,-2,0,2],
            [-3,1,0,2,0],
            [-1,1,-2,2,0],
            [-1,-1,0,0,2],
            [-3,0,0,2,0],
            [-3,-1,0,2,0],
            [2,0,2,-6,1],
            [0,1,2,-4,2],
            [2,0,0,-4,2],
            [-2,1,2,-2,1],
            [0,-1,2,-4,1],
            [0,1,0,-2,2],
            [-1,0,0,-2,0],
            [2,0,-2,-2,1],
            [-4,0,2,0,1],
            [-1,-1,0,-1,1],
            [0,0,-2,0,2],
            [-3,0,0,1,0],
            [-1,0,-2,1,0],
            [-2,0,-2,2,1],
            [0,0,-4,2,0],
            [-2,-1,-2,2,0],
            [1,0,2,-6,1],
            [-1,0,2,-4,2],
            [1,0,0,-4,2],
            [2,1,2,-4,2],
            [2,1,2,-4,1],
            [0,1,4,-4,4],
            [0,1,4,-4,2],
            [-1,-1,-2,4,0],
            [-1,-3,0,2,0],
            [-1,0,-2,4,1],
            [-2,-1,0,3,0],
            [0,0,-2,3,0],
            [-2,0,0,3,1],
            [0,-1,0,1,0],
            [-3,0,2,2,0],
            [1,1,-2,2,0],
            [-1,1,0,2,2],
            [1,-2,2,-2,1],
            [0,0,1,0,2],
            [0,0,1,0,1],
            [0,0,1,0,0],
            [-1,2,0,2,1],
            [0,0,2,0,2],
            [-2,0,2,0,2],
            [2,0,0,-1,1],
            [3,0,0,-2,1],
            [1,0,2,-2,3],
            [1,2,0,0,1],
            [2,0,2,-3,2],
            [-1,1,4,-2,2],
            [-2,-2,0,4,0],
            [0,-3,0,2,0],
            [0,0,-2,4,0],
            [-1,-1,0,3,0],
            [-2,0,0,4,2],
            [-1,0,0,3,1],
            [2,-2,0,0,0],
            [1,-1,0,1,0],
            [-1,0,0,2,0],
            [0,-2,2,0,1],
            [-1,0,1,2,1],
            [-1,1,0,3,0],
            [-1,-1,2,1,2],
            [0,-1,2,0,0],
            [-2,1,2,2,1],
            [2,-2,2,-2,2],
            [1,1,0,1,1],
            [1,0,1,0,1],
            [1,0,1,0,0],
            [0,2,0,2,0],
            [2,-1,2,-2,1],
            [0,-1,4,-2,1],
            [0,0,4,-2,3],
            [0,1,4,-2,1],
            [4,0,2,-4,2],
            [2,2,2,-2,2],
            [2,0,4,-4,2],
            [-1,-2,0,4,0],
            [-1,-3,2,2,2],
            [-3,0,2,4,2],
            [-3,0,2,-2,1],
            [-1,-1,0,-2,1],
            [-3,0,0,0,2],
            [-3,0,-2,2,0],
            [0,1,0,-4,1],
            [-2,1,0,-2,1],
            [-4,0,0,0,1],
            [-1,0,0,-4,1],
            [-3,0,0,-2,1],
            [0,0,0,3,2],
            [-1,1,0,4,1],
            [1,-2,2,0,1],
            [0,1,0,3,0],
            [-1,0,2,2,3],
            [0,0,2,2,2],
            [-2,0,2,2,2],
            [-1,1,2,2,0],
            [3,0,0,0,2],
            [2,1,0,1,0],
            [2,-1,2,-1,2],
            [0,0,2,0,1],
            [0,0,3,0,3],
            [0,0,3,0,2],
            [-1,2,2,2,1],
            [-1,0,4,0,0],
            [1,2,2,0,1],
            [3,1,2,-2,1],
            [1,1,4,-2,2],
            [-2,-1,0,6,0],
            [0,-2,0,4,0],
            [-2,0,0,6,1],
            [-2,-2,2,4,2],
            [0,-3,2,2,2],
            [0,0,0,4,2],
            [-1,-1,2,3,2],
            [-2,0,2,4,0],
            [2,-1,0,2,1],
            [1,0,0,3,0],
            [0,1,0,4,1],
            [0,1,0,4,0],
            [1,-1,2,1,2],
            [0,0,2,2,3],
            [1,0,2,2,2],
            [-1,0,2,2,2],
            [-2,0,4,2,1],
            [2,1,0,2,1],
            [2,1,0,2,0],
            [2,-1,2,0,0],
            [1,0,2,1,0],
            [0,1,2,2,0],
            [2,0,2,0,3],
            [3,0,2,0,2],
            [1,0,2,0,2],
            [1,0,3,0,3],
            [1,1,2,1,1],
            [0,2,2,2,2],
            [2,1,2,0,0],
            [2,0,4,-2,1],
            [4,1,2,-2,2],
            [-1,-1,0,6,0],
            [-3,-1,2,6,2],
            [-1,0,0,6,1],
            [-3,0,2,6,1],
            [1,-1,0,4,1],
            [1,-1,0,4,0],
            [-2,0,2,5,2],
            [1,-2,2,2,1],
            [3,-1,0,2,0],
            [1,-1,2,2,0],
            [0,0,2,3,1],
            [-1,1,2,4,1],
            [0,1,2,3,2],
            [-1,0,4,2,1],
            [2,0,2,1,1],
            [5,0,0,0,0],
            [2,1,2,1,2],
            [1,0,4,0,1],
            [3,1,2,0,1],
            [3,0,4,-2,2],
            [-2,-1,2,6,2],
            [0,0,0,6,0],
            [0,-2,2,4,2],
            [-2,0,2,6,1],
            [2,0,0,4,1],
            [2,0,0,4,0],
            [2,-2,2,2,2],
            [0,0,2,4,0],
            [1,0,2,3,2],
            [4,0,0,2,0],
            [2,0,2,2,0],
            [0,0,4,2,2],
            [4,-1,2,0,2],
            [3,0,2,1,2],
            [2,1,2,2,1],
            [4,1,2,0,2],
            [-1,-1,2,6,2],
            [-1,0,2,6,1],
            [1,-1,2,4,1],
            [1,1,2,4,2],
            [3,1,2,2,2],
            [5,0,2,0,1],
            [2,-1,2,4,2],
            [2,0,2,4,1]]
    
    #日月章动系数，单位：0.1微角秒
    #*  longitude (sin, t*sin, cos), obliquity (cos, t*cos, sin)
    CLS=[[-172064161.0,-174666.0,33386.0,92052331.0,9086.0,15377.0],
            [-13170906.0,-1675.0,-13696.0,5730336.0,-3015.0,-4587.0],
            [-2276413.0,-234.0,2796.0,978459.0,-485.0,1374.0],
            [2074554.0,207.0,-698.0,-897492.0,470.0,-291.0],
            [1475877.0,-3633.0,11817.0,73871.0,-184.0,-1924.0],
            [-516821.0,1226.0,-524.0,224386.0,-677.0,-174.0],
            [711159.0,73.0,-872.0,-6750.0,0.0,358.0],
            [-387298.0,-367.0,380.0,200728.0,18.0,318.0],
            [-301461.0,-36.0,816.0,129025.0,-63.0,367.0],
            [215829.0,-494.0,111.0,-95929.0,299.0,132.0],
            [128227.0,137.0,181.0,-68982.0,-9.0,39.0],
            [123457.0,11.0,19.0,-53311.0,32.0,-4.0],
            [156994.0,10.0,-168.0,-1235.0,0.0,82.0],
            [63110.0,63.0,27.0,-33228.0,0.0,-9.0],
            [-57976.0,-63.0,-189.0,31429.0,0.0,-75.0],
            [-59641.0,-11.0,149.0,25543.0,-11.0,66.0],
            [-51613.0,-42.0,129.0,26366.0,0.0,78.0],
            [45893.0,50.0,31.0,-24236.0,-10.0,20.0],
            [63384.0,11.0,-150.0,-1220.0,0.0,29.0],
            [-38571.0,-1.0,158.0,16452.0,-11.0,68.0],
            [32481.0,0.0,0.0,-13870.0,0.0,0.0],
            [-47722.0,0.0,-18.0,477.0,0.0,-25.0],
            [-31046.0,-1.0,131.0,13238.0,-11.0,59.0],
            [28593.0,0.0,-1.0,-12338.0,10.0,-3.0],
            [20441.0,21.0,10.0,-10758.0,0.0,-3.0],
            [29243.0,0.0,-74.0,-609.0,0.0,13.0],
            [25887.0,0.0,-66.0,-550.0,0.0,11.0],
            [-14053.0,-25.0,79.0,8551.0,-2.0,-45.0],
            [15164.0,10.0,11.0,-8001.0,0.0,-1.0],
            [-15794.0,72.0,-16.0,6850.0,-42.0,-5.0],
            [21783.0,0.0,13.0,-167.0,0.0,13.0],
            [-12873.0,-10.0,-37.0,6953.0,0.0,-14.0],
            [-12654.0,11.0,63.0,6415.0,0.0,26.0],
            [-10204.0,0.0,25.0,5222.0,0.0,15.0],
            [16707.0,-85.0,-10.0,168.0,-1.0,10.0],
            [-7691.0,0.0,44.0,3268.0,0.0,19.0],
            [-11024.0,0.0,-14.0,104.0,0.0,2.0],
            [7566.0,-21.0,-11.0,-3250.0,0.0,-5.0],
            [-6637.0,-11.0,25.0,3353.0,0.0,14.0],
            [-7141.0,21.0,8.0,3070.0,0.0,4.0],
            [-6302.0,-11.0,2.0,3272.0,0.0,4.0],
            [5800.0,10.0,2.0,-3045.0,0.0,-1.0],
            [6443.0,0.0,-7.0,-2768.0,0.0,-4.0],
            [-5774.0,-11.0,-15.0,3041.0,0.0,-5.0],
            [-5350.0,0.0,21.0,2695.0,0.0,12.0],
            [-4752.0,-11.0,-3.0,2719.0,0.0,-3.0],
            [-4940.0,-11.0,-21.0,2720.0,0.0,-9.0],
            [7350.0,0.0,-8.0,-51.0,0.0,4.0],
            [4065.0,0.0,6.0,-2206.0,0.0,1.0],
            [6579.0,0.0,-24.0,-199.0,0.0,2.0],
            [3579.0,0.0,5.0,-1900.0,0.0,1.0],
            [4725.0,0.0,-6.0,-41.0,0.0,3.0],
            [-3075.0,0.0,-2.0,1313.0,0.0,-1.0],
            [-2904.0,0.0,15.0,1233.0,0.0,7.0],
            [4348.0,0.0,-10.0,-81.0,0.0,2.0],
            [-2878.0,0.0,8.0,1232.0,0.0,4.0],
            [-4230.0,0.0,5.0,-20.0,0.0,-2.0],
            [-2819.0,0.0,7.0,1207.0,0.0,3.0],
            [-4056.0,0.0,5.0,40.0,0.0,-2.0],
            [-2647.0,0.0,11.0,1129.0,0.0,5.0],
            [-2294.0,0.0,-10.0,1266.0,0.0,-4.0],
            [2481.0,0.0,-7.0,-1062.0,0.0,-3.0],
            [2179.0,0.0,-2.0,-1129.0,0.0,-2.0],
            [3276.0,0.0,1.0,-9.0,0.0,0.0],
            [-3389.0,0.0,5.0,35.0,0.0,-2.0],
            [3339.0,0.0,-13.0,-107.0,0.0,1.0],
            [-1987.0,0.0,-6.0,1073.0,0.0,-2.0],
            [-1981.0,0.0,0.0,854.0,0.0,0.0],
            [4026.0,0.0,-353.0,-553.0,0.0,-139.0],
            [1660.0,0.0,-5.0,-710.0,0.0,-2.0],
            [-1521.0,0.0,9.0,647.0,0.0,4.0],
            [1314.0,0.0,0.0,-700.0,0.0,0.0],
            [-1283.0,0.0,0.0,672.0,0.0,0.0],
            [-1331.0,0.0,8.0,663.0,0.0,4.0],
            [1383.0,0.0,-2.0,-594.0,0.0,-2.0],
            [1405.0,0.0,4.0,-610.0,0.0,2.0],
            [1290.0,0.0,0.0,-556.0,0.0,0.0],
            [-1214.0,0.0,5.0,518.0,0.0,2.0],
            [1146.0,0.0,-3.0,-490.0,0.0,-1.0],
            [1019.0,0.0,-1.0,-527.0,0.0,-1.0],
            [-1100.0,0.0,9.0,465.0,0.0,4.0],
            [-970.0,0.0,2.0,496.0,0.0,1.0],
            [1575.0,0.0,-6.0,-50.0,0.0,0.0],
            [934.0,0.0,-3.0,-399.0,0.0,-1.0],
            [922.0,0.0,-1.0,-395.0,0.0,-1.0],
            [815.0,0.0,-1.0,-422.0,0.0,-1.0],
            [834.0,0.0,2.0,-440.0,0.0,1.0],
            [1248.0,0.0,0.0,-170.0,0.0,1.0],
            [1338.0,0.0,-5.0,-39.0,0.0,0.0],
            [716.0,0.0,-2.0,-389.0,0.0,-1.0],
            [1282.0,0.0,-3.0,-23.0,0.0,1.0],
            [742.0,0.0,1.0,-391.0,0.0,0.0],
            [1020.0,0.0,-25.0,-495.0,0.0,-10.0],
            [715.0,0.0,-4.0,-326.0,0.0,2.0],
            [-666.0,0.0,-3.0,369.0,0.0,-1.0],
            [-667.0,0.0,1.0,346.0,0.0,1.0],
            [-704.0,0.0,0.0,304.0,0.0,0.0],
            [-694.0,0.0,5.0,294.0,0.0,2.0],
            [-1014.0,0.0,-1.0,4.0,0.0,-1.0],
            [-585.0,0.0,-2.0,316.0,0.0,-1.0],
            [-949.0,0.0,1.0,8.0,0.0,-1.0],
            [-595.0,0.0,0.0,258.0,0.0,0.0],
            [528.0,0.0,0.0,-279.0,0.0,0.0],
            [-590.0,0.0,4.0,252.0,0.0,2.0],
            [570.0,0.0,-2.0,-244.0,0.0,-1.0],
            [-502.0,0.0,3.0,250.0,0.0,2.0],
            [-875.0,0.0,1.0,29.0,0.0,0.0],
            [-492.0,0.0,-3.0,275.0,0.0,-1.0],
            [535.0,0.0,-2.0,-228.0,0.0,-1.0],
            [-467.0,0.0,1.0,240.0,0.0,1.0],
            [591.0,0.0,0.0,-253.0,0.0,0.0],
            [-453.0,0.0,-1.0,244.0,0.0,-1.0],
            [766.0,0.0,1.0,9.0,0.0,0.0],
            [-446.0,0.0,2.0,225.0,0.0,1.0],
            [-488.0,0.0,2.0,207.0,0.0,1.0],
            [-468.0,0.0,0.0,201.0,0.0,0.0],
            [-421.0,0.0,1.0,216.0,0.0,1.0],
            [463.0,0.0,0.0,-200.0,0.0,0.0],
            [-673.0,0.0,2.0,14.0,0.0,0.0],
            [658.0,0.0,0.0,-2.0,0.0,0.0],
            [-438.0,0.0,0.0,188.0,0.0,0.0],
            [-390.0,0.0,0.0,205.0,0.0,0.0],
            [639.0,-11.0,-2.0,-19.0,0.0,0.0],
            [412.0,0.0,-2.0,-176.0,0.0,-1.0],
            [-361.0,0.0,0.0,189.0,0.0,0.0],
            [360.0,0.0,-1.0,-185.0,0.0,-1.0],
            [588.0,0.0,-3.0,-24.0,0.0,0.0],
            [-578.0,0.0,1.0,5.0,0.0,0.0],
            [-396.0,0.0,0.0,171.0,0.0,0.0],
            [565.0,0.0,-1.0,-6.0,0.0,0.0],
            [-335.0,0.0,-1.0,184.0,0.0,-1.0],
            [357.0,0.0,1.0,-154.0,0.0,0.0],
            [321.0,0.0,1.0,-174.0,0.0,0.0],
            [-301.0,0.0,-1.0,162.0,0.0,0.0],
            [-334.0,0.0,0.0,144.0,0.0,0.0],
            [493.0,0.0,-2.0,-15.0,0.0,0.0],
            [494.0,0.0,-2.0,-19.0,0.0,0.0],
            [337.0,0.0,-1.0,-143.0,0.0,-1.0],
            [280.0,0.0,-1.0,-144.0,0.0,0.0],
            [309.0,0.0,1.0,-134.0,0.0,0.0],
            [-263.0,0.0,2.0,131.0,0.0,1.0],
            [253.0,0.0,1.0,-138.0,0.0,0.0],
            [245.0,0.0,0.0,-128.0,0.0,0.0],
            [416.0,0.0,-2.0,-17.0,0.0,0.0],
            [-229.0,0.0,0.0,128.0,0.0,0.0],
            [231.0,0.0,0.0,-120.0,0.0,0.0],
            [-259.0,0.0,2.0,109.0,0.0,1.0],
            [375.0,0.0,-1.0,-8.0,0.0,0.0],
            [252.0,0.0,0.0,-108.0,0.0,0.0],
            [-245.0,0.0,1.0,104.0,0.0,0.0],
            [243.0,0.0,-1.0,-104.0,0.0,0.0],
            [208.0,0.0,1.0,-112.0,0.0,0.0],
            [199.0,0.0,0.0,-102.0,0.0,0.0],
            [-208.0,0.0,1.0,105.0,0.0,0.0],
            [335.0,0.0,-2.0,-14.0,0.0,0.0],
            [-325.0,0.0,1.0,7.0,0.0,0.0],
            [-187.0,0.0,0.0,96.0,0.0,0.0],
            [197.0,0.0,-1.0,-100.0,0.0,0.0],
            [-192.0,0.0,2.0,94.0,0.0,1.0],
            [-188.0,0.0,0.0,83.0,0.0,0.0],
            [276.0,0.0,0.0,-2.0,0.0,0.0],
            [-286.0,0.0,1.0,6.0,0.0,0.0],
            [186.0,0.0,-1.0,-79.0,0.0,0.0],
            [-219.0,0.0,0.0,43.0,0.0,0.0],
            [276.0,0.0,0.0,2.0,0.0,0.0],
            [-153.0,0.0,-1.0,84.0,0.0,0.0],
            [-156.0,0.0,0.0,81.0,0.0,0.0],
            [-154.0,0.0,1.0,78.0,0.0,0.0],
            [-174.0,0.0,1.0,75.0,0.0,0.0],
            [-163.0,0.0,2.0,69.0,0.0,1.0],
            [-228.0,0.0,0.0,1.0,0.0,0.0],
            [91.0,0.0,-4.0,-54.0,0.0,-2.0],
            [175.0,0.0,0.0,-75.0,0.0,0.0],
            [-159.0,0.0,0.0,69.0,0.0,0.0],
            [141.0,0.0,0.0,-72.0,0.0,0.0],
            [147.0,0.0,0.0,-75.0,0.0,0.0],
            [-132.0,0.0,0.0,69.0,0.0,0.0],
            [159.0,0.0,-28.0,-54.0,0.0,11.0],
            [213.0,0.0,0.0,-4.0,0.0,0.0],
            [123.0,0.0,0.0,-64.0,0.0,0.0],
            [-118.0,0.0,-1.0,66.0,0.0,0.0],
            [144.0,0.0,-1.0,-61.0,0.0,0.0],
            [-121.0,0.0,1.0,60.0,0.0,0.0],
            [-134.0,0.0,1.0,56.0,0.0,1.0],
            [-105.0,0.0,0.0,57.0,0.0,0.0],
            [-102.0,0.0,0.0,56.0,0.0,0.0],
            [120.0,0.0,0.0,-52.0,0.0,0.0],
            [101.0,0.0,0.0,-54.0,0.0,0.0],
            [-113.0,0.0,0.0,59.0,0.0,0.0],
            [-106.0,0.0,0.0,61.0,0.0,0.0],
            [-129.0,0.0,1.0,55.0,0.0,0.0],
            [-114.0,0.0,0.0,57.0,0.0,0.0],
            [113.0,0.0,-1.0,-49.0,0.0,0.0],
            [-102.0,0.0,0.0,44.0,0.0,0.0],
            [-94.0,0.0,0.0,51.0,0.0,0.0],
            [-100.0,0.0,-1.0,56.0,0.0,0.0],
            [87.0,0.0,0.0,-47.0,0.0,0.0],
            [161.0,0.0,0.0,-1.0,0.0,0.0],
            [96.0,0.0,0.0,-50.0,0.0,0.0],
            [151.0,0.0,-1.0,-5.0,0.0,0.0],
            [-104.0,0.0,0.0,44.0,0.0,0.0],
            [-110.0,0.0,0.0,48.0,0.0,0.0],
            [-100.0,0.0,1.0,50.0,0.0,0.0],
            [92.0,0.0,-5.0,12.0,0.0,-2.0],
            [82.0,0.0,0.0,-45.0,0.0,0.0],
            [82.0,0.0,0.0,-45.0,0.0,0.0],
            [-78.0,0.0,0.0,41.0,0.0,0.0],
            [-77.0,0.0,0.0,43.0,0.0,0.0],
            [2.0,0.0,0.0,54.0,0.0,0.0],
            [94.0,0.0,0.0,-40.0,0.0,0.0],
            [-93.0,0.0,0.0,40.0,0.0,0.0],
            [-83.0,0.0,10.0,40.0,0.0,-2.0],
            [83.0,0.0,0.0,-36.0,0.0,0.0],
            [-91.0,0.0,0.0,39.0,0.0,0.0],
            [128.0,0.0,0.0,-1.0,0.0,0.0],
            [-79.0,0.0,0.0,34.0,0.0,0.0],
            [-83.0,0.0,0.0,47.0,0.0,0.0],
            [84.0,0.0,0.0,-44.0,0.0,0.0],
            [83.0,0.0,0.0,-43.0,0.0,0.0],
            [91.0,0.0,0.0,-39.0,0.0,0.0],
            [-77.0,0.0,0.0,39.0,0.0,0.0],
            [84.0,0.0,0.0,-43.0,0.0,0.0],
            [-92.0,0.0,1.0,39.0,0.0,0.0],
            [-92.0,0.0,1.0,39.0,0.0,0.0],
            [-94.0,0.0,0.0,0.0,0.0,0.0],
            [68.0,0.0,0.0,-36.0,0.0,0.0],
            [-61.0,0.0,0.0,32.0,0.0,0.0],
            [71.0,0.0,0.0,-31.0,0.0,0.0],
            [62.0,0.0,0.0,-34.0,0.0,0.0],
            [-63.0,0.0,0.0,33.0,0.0,0.0],
            [-73.0,0.0,0.0,32.0,0.0,0.0],
            [115.0,0.0,0.0,-2.0,0.0,0.0],
            [-103.0,0.0,0.0,2.0,0.0,0.0],
            [63.0,0.0,0.0,-28.0,0.0,0.0],
            [74.0,0.0,0.0,-32.0,0.0,0.0],
            [-103.0,0.0,-3.0,3.0,0.0,-1.0],
            [-69.0,0.0,0.0,30.0,0.0,0.0],
            [57.0,0.0,0.0,-29.0,0.0,0.0],
            [94.0,0.0,0.0,-4.0,0.0,0.0],
            [64.0,0.0,0.0,-33.0,0.0,0.0],
            [-63.0,0.0,0.0,26.0,0.0,0.0],
            [-38.0,0.0,0.0,20.0,0.0,0.0],
            [-43.0,0.0,0.0,24.0,0.0,0.0],
            [-45.0,0.0,0.0,23.0,0.0,0.0],
            [47.0,0.0,0.0,-24.0,0.0,0.0],
            [-48.0,0.0,0.0,25.0,0.0,0.0],
            [45.0,0.0,0.0,-26.0,0.0,0.0],
            [56.0,0.0,0.0,-25.0,0.0,0.0],
            [88.0,0.0,0.0,2.0,0.0,0.0],
            [-75.0,0.0,0.0,0.0,0.0,0.0],
            [85.0,0.0,0.0,0.0,0.0,0.0],
            [49.0,0.0,0.0,-26.0,0.0,0.0],
            [-74.0,0.0,-3.0,-1.0,0.0,-1.0],
            [-39.0,0.0,0.0,21.0,0.0,0.0],
            [45.0,0.0,0.0,-20.0,0.0,0.0],
            [51.0,0.0,0.0,-22.0,0.0,0.0],
            [-40.0,0.0,0.0,21.0,0.0,0.0],
            [41.0,0.0,0.0,-21.0,0.0,0.0],
            [-42.0,0.0,0.0,24.0,0.0,0.0],
            [-51.0,0.0,0.0,22.0,0.0,0.0],
            [-42.0,0.0,0.0,22.0,0.0,0.0],
            [39.0,0.0,0.0,-21.0,0.0,0.0],
            [46.0,0.0,0.0,-18.0,0.0,0.0],
            [-53.0,0.0,0.0,22.0,0.0,0.0],
            [82.0,0.0,0.0,-4.0,0.0,0.0],
            [81.0,0.0,-1.0,-4.0,0.0,0.0],
            [47.0,0.0,0.0,-19.0,0.0,0.0],
            [53.0,0.0,0.0,-23.0,0.0,0.0],
            [-45.0,0.0,0.0,22.0,0.0,0.0],
            [-44.0,0.0,0.0,-2.0,0.0,0.0],
            [-33.0,0.0,0.0,16.0,0.0,0.0],
            [-61.0,0.0,0.0,1.0,0.0,0.0],
            [28.0,0.0,0.0,-15.0,0.0,0.0],
            [-38.0,0.0,0.0,19.0,0.0,0.0],
            [-33.0,0.0,0.0,21.0,0.0,0.0],
            [-60.0,0.0,0.0,0.0,0.0,0.0],
            [48.0,0.0,0.0,-10.0,0.0,0.0],
            [27.0,0.0,0.0,-14.0,0.0,0.0],
            [38.0,0.0,0.0,-20.0,0.0,0.0],
            [31.0,0.0,0.0,-13.0,0.0,0.0],
            [-29.0,0.0,0.0,15.0,0.0,0.0],
            [28.0,0.0,0.0,-15.0,0.0,0.0],
            [-32.0,0.0,0.0,15.0,0.0,0.0],
            [45.0,0.0,0.0,-8.0,0.0,0.0],
            [-44.0,0.0,0.0,19.0,0.0,0.0],
            [28.0,0.0,0.0,-15.0,0.0,0.0],
            [-51.0,0.0,0.0,0.0,0.0,0.0],
            [-36.0,0.0,0.0,20.0,0.0,0.0],
            [44.0,0.0,0.0,-19.0,0.0,0.0],
            [26.0,0.0,0.0,-14.0,0.0,0.0],
            [-60.0,0.0,0.0,2.0,0.0,0.0],
            [35.0,0.0,0.0,-18.0,0.0,0.0],
            [-27.0,0.0,0.0,11.0,0.0,0.0],
            [47.0,0.0,0.0,-1.0,0.0,0.0],
            [36.0,0.0,0.0,-15.0,0.0,0.0],
            [-36.0,0.0,0.0,20.0,0.0,0.0],
            [-35.0,0.0,0.0,19.0,0.0,0.0],
            [-37.0,0.0,0.0,19.0,0.0,0.0],
            [32.0,0.0,0.0,-16.0,0.0,0.0],
            [35.0,0.0,0.0,-14.0,0.0,0.0],
            [32.0,0.0,0.0,-13.0,0.0,0.0],
            [65.0,0.0,0.0,-2.0,0.0,0.0],
            [47.0,0.0,0.0,-1.0,0.0,0.0],
            [32.0,0.0,0.0,-16.0,0.0,0.0],
            [37.0,0.0,0.0,-16.0,0.0,0.0],
            [-30.0,0.0,0.0,15.0,0.0,0.0],
            [-32.0,0.0,0.0,16.0,0.0,0.0],
            [-31.0,0.0,0.0,13.0,0.0,0.0],
            [37.0,0.0,0.0,-16.0,0.0,0.0],
            [31.0,0.0,0.0,-13.0,0.0,0.0],
            [49.0,0.0,0.0,-2.0,0.0,0.0],
            [32.0,0.0,0.0,-13.0,0.0,0.0],
            [23.0,0.0,0.0,-12.0,0.0,0.0],
            [-43.0,0.0,0.0,18.0,0.0,0.0],
            [26.0,0.0,0.0,-11.0,0.0,0.0],
            [-32.0,0.0,0.0,14.0,0.0,0.0],
            [-29.0,0.0,0.0,14.0,0.0,0.0],
            [-27.0,0.0,0.0,12.0,0.0,0.0],
            [30.0,0.0,0.0,0.0,0.0,0.0],
            [-11.0,0.0,0.0,5.0,0.0,0.0],
            [-21.0,0.0,0.0,10.0,0.0,0.0],
            [-34.0,0.0,0.0,15.0,0.0,0.0],
            [-10.0,0.0,0.0,6.0,0.0,0.0],
            [-36.0,0.0,0.0,0.0,0.0,0.0],
            [-9.0,0.0,0.0,4.0,0.0,0.0],
            [-12.0,0.0,0.0,5.0,0.0,0.0],
            [-21.0,0.0,0.0,5.0,0.0,0.0],
            [-29.0,0.0,0.0,-1.0,0.0,0.0],
            [-15.0,0.0,0.0,3.0,0.0,0.0],
            [-20.0,0.0,0.0,0.0,0.0,0.0],
            [28.0,0.0,0.0,0.0,0.0,-2.0],
            [17.0,0.0,0.0,0.0,0.0,0.0],
            [-22.0,0.0,0.0,12.0,0.0,0.0],
            [-14.0,0.0,0.0,7.0,0.0,0.0],
            [24.0,0.0,0.0,-11.0,0.0,0.0],
            [11.0,0.0,0.0,-6.0,0.0,0.0],
            [14.0,0.0,0.0,-6.0,0.0,0.0],
            [24.0,0.0,0.0,0.0,0.0,0.0],
            [18.0,0.0,0.0,-8.0,0.0,0.0],
            [-38.0,0.0,0.0,0.0,0.0,0.0],
            [-31.0,0.0,0.0,0.0,0.0,0.0],
            [-16.0,0.0,0.0,8.0,0.0,0.0],
            [29.0,0.0,0.0,0.0,0.0,0.0],
            [-18.0,0.0,0.0,10.0,0.0,0.0],
            [-10.0,0.0,0.0,5.0,0.0,0.0],
            [-17.0,0.0,0.0,10.0,0.0,0.0],
            [9.0,0.0,0.0,-4.0,0.0,0.0],
            [16.0,0.0,0.0,-6.0,0.0,0.0],
            [22.0,0.0,0.0,-12.0,0.0,0.0],
            [20.0,0.0,0.0,0.0,0.0,0.0],
            [-13.0,0.0,0.0,6.0,0.0,0.0],
            [-17.0,0.0,0.0,9.0,0.0,0.0],
            [-14.0,0.0,0.0,8.0,0.0,0.0],
            [0.0,0.0,0.0,-7.0,0.0,0.0],
            [14.0,0.0,0.0,0.0,0.0,0.0],
            [19.0,0.0,0.0,-10.0,0.0,0.0],
            [-34.0,0.0,0.0,0.0,0.0,0.0],
            [-20.0,0.0,0.0,8.0,0.0,0.0],
            [9.0,0.0,0.0,-5.0,0.0,0.0],
            [-18.0,0.0,0.0,7.0,0.0,0.0],
            [13.0,0.0,0.0,-6.0,0.0,0.0],
            [17.0,0.0,0.0,0.0,0.0,0.0],
            [-12.0,0.0,0.0,5.0,0.0,0.0],
            [15.0,0.0,0.0,-8.0,0.0,0.0],
            [-11.0,0.0,0.0,3.0,0.0,0.0],
            [13.0,0.0,0.0,-5.0,0.0,0.0],
            [-18.0,0.0,0.0,0.0,0.0,0.0],
            [-35.0,0.0,0.0,0.0,0.0,0.0],
            [9.0,0.0,0.0,-4.0,0.0,0.0],
            [-19.0,0.0,0.0,10.0,0.0,0.0],
            [-26.0,0.0,0.0,11.0,0.0,0.0],
            [8.0,0.0,0.0,-4.0,0.0,0.0],
            [-10.0,0.0,0.0,4.0,0.0,0.0],
            [10.0,0.0,0.0,-6.0,0.0,0.0],
            [-21.0,0.0,0.0,9.0,0.0,0.0],
            [-15.0,0.0,0.0,0.0,0.0,0.0],
            [9.0,0.0,0.0,-5.0,0.0,0.0],
            [-29.0,0.0,0.0,0.0,0.0,0.0],
            [-19.0,0.0,0.0,10.0,0.0,0.0],
            [12.0,0.0,0.0,-5.0,0.0,0.0],
            [22.0,0.0,0.0,-9.0,0.0,0.0],
            [-10.0,0.0,0.0,5.0,0.0,0.0],
            [-20.0,0.0,0.0,11.0,0.0,0.0],
            [-20.0,0.0,0.0,0.0,0.0,0.0],
            [-17.0,0.0,0.0,7.0,0.0,0.0],
            [15.0,0.0,0.0,-3.0,0.0,0.0],
            [8.0,0.0,0.0,-4.0,0.0,0.0],
            [14.0,0.0,0.0,0.0,0.0,0.0],
            [-12.0,0.0,0.0,6.0,0.0,0.0],
            [25.0,0.0,0.0,0.0,0.0,0.0],
            [-13.0,0.0,0.0,6.0,0.0,0.0],
            [-14.0,0.0,0.0,8.0,0.0,0.0],
            [13.0,0.0,0.0,-5.0,0.0,0.0],
            [-17.0,0.0,0.0,9.0,0.0,0.0],
            [-12.0,0.0,0.0,6.0,0.0,0.0],
            [-10.0,0.0,0.0,5.0,0.0,0.0],
            [10.0,0.0,0.0,-6.0,0.0,0.0],
            [-15.0,0.0,0.0,0.0,0.0,0.0],
            [-22.0,0.0,0.0,0.0,0.0,0.0],
            [28.0,0.0,0.0,-1.0,0.0,0.0],
            [15.0,0.0,0.0,-7.0,0.0,0.0],
            [23.0,0.0,0.0,-10.0,0.0,0.0],
            [12.0,0.0,0.0,-5.0,0.0,0.0],
            [29.0,0.0,0.0,-1.0,0.0,0.0],
            [-25.0,0.0,0.0,1.0,0.0,0.0],
            [22.0,0.0,0.0,0.0,0.0,0.0],
            [-18.0,0.0,0.0,0.0,0.0,0.0],
            [15.0,0.0,0.0,3.0,0.0,0.0],
            [-23.0,0.0,0.0,0.0,0.0,0.0],
            [12.0,0.0,0.0,-5.0,0.0,0.0],
            [-8.0,0.0,0.0,4.0,0.0,0.0],
            [-19.0,0.0,0.0,0.0,0.0,0.0],
            [-10.0,0.0,0.0,4.0,0.0,0.0],
            [21.0,0.0,0.0,-9.0,0.0,0.0],
            [23.0,0.0,0.0,-1.0,0.0,0.0],
            [-16.0,0.0,0.0,8.0,0.0,0.0],
            [-19.0,0.0,0.0,9.0,0.0,0.0],
            [-22.0,0.0,0.0,10.0,0.0,0.0],
            [27.0,0.0,0.0,-1.0,0.0,0.0],
            [16.0,0.0,0.0,-8.0,0.0,0.0],
            [19.0,0.0,0.0,-8.0,0.0,0.0],
            [9.0,0.0,0.0,-4.0,0.0,0.0],
            [-9.0,0.0,0.0,4.0,0.0,0.0],
            [-9.0,0.0,0.0,4.0,0.0,0.0],
            [-8.0,0.0,0.0,4.0,0.0,0.0],
            [18.0,0.0,0.0,-9.0,0.0,0.0],
            [16.0,0.0,0.0,-1.0,0.0,0.0],
            [-10.0,0.0,0.0,4.0,0.0,0.0],
            [-23.0,0.0,0.0,9.0,0.0,0.0],
            [16.0,0.0,0.0,-1.0,0.0,0.0],
            [-12.0,0.0,0.0,6.0,0.0,0.0],
            [-8.0,0.0,0.0,4.0,0.0,0.0],
            [30.0,0.0,0.0,-2.0,0.0,0.0],
            [24.0,0.0,0.0,-10.0,0.0,0.0],
            [10.0,0.0,0.0,-4.0,0.0,0.0],
            [-16.0,0.0,0.0,7.0,0.0,0.0],
            [-16.0,0.0,0.0,7.0,0.0,0.0],
            [17.0,0.0,0.0,-7.0,0.0,0.0],
            [-24.0,0.0,0.0,10.0,0.0,0.0],
            [-12.0,0.0,0.0,5.0,0.0,0.0],
            [-24.0,0.0,0.0,11.0,0.0,0.0],
            [-23.0,0.0,0.0,9.0,0.0,0.0],
            [-13.0,0.0,0.0,5.0,0.0,0.0],
            [-15.0,0.0,0.0,7.0,0.0,0.0],
            [0.0,0.0,-1988.0,0.0,0.0,-1679.0],
            [0.0,0.0,-63.0,0.0,0.0,-27.0],
            [-4.0,0.0,0.0,0.0,0.0,0.0],
            [0.0,0.0,5.0,0.0,0.0,4.0],
            [5.0,0.0,0.0,-3.0,0.0,0.0],
            [0.0,0.0,364.0,0.0,0.0,176.0],
            [0.0,0.0,-1044.0,0.0,0.0,-891.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [0.0,0.0,330.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [3.0,0.0,0.0,-2.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [-5.0,0.0,0.0,2.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [0.0,0.0,5.0,0.0,0.0,0.0],
            [0.0,0.0,0.0,1.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [6.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [-7.0,0.0,0.0,0.0,0.0,0.0],
            [-12.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-3.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [-5.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [-7.0,0.0,0.0,3.0,0.0,0.0],
            [7.0,0.0,0.0,-4.0,0.0,0.0],
            [0.0,0.0,-12.0,0.0,0.0,-10.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [3.0,0.0,0.0,-2.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [-7.0,0.0,0.0,3.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [0.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [7.0,0.0,0.0,-3.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [-5.0,0.0,0.0,3.0,0.0,0.0],
            [5.0,0.0,0.0,0.0,0.0,0.0],
            [-5.0,0.0,0.0,2.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [-8.0,0.0,0.0,3.0,0.0,0.0],
            [9.0,0.0,0.0,0.0,0.0,0.0],
            [6.0,0.0,0.0,-3.0,0.0,0.0],
            [-5.0,0.0,0.0,2.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [-7.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [5.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [-5.0,0.0,0.0,2.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [9.0,0.0,0.0,-3.0,0.0,0.0],
            [4.0,0.0,0.0,0.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [9.0,0.0,0.0,-3.0,0.0,0.0],
            [-4.0,0.0,0.0,0.0,0.0,0.0],
            [-4.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,-2.0,0.0,0.0],
            [8.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [6.0,0.0,0.0,-3.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [-7.0,0.0,0.0,0.0,0.0,0.0],
            [9.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [-3.0,0.0,0.0,0.0,0.0,0.0],
            [-4.0,0.0,0.0,0.0,0.0,0.0],
            [-5.0,0.0,0.0,3.0,0.0,0.0],
            [-13.0,0.0,0.0,0.0,0.0,0.0],
            [-7.0,0.0,0.0,0.0,0.0,0.0],
            [10.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [10.0,0.0,13.0,6.0,0.0,-5.0],
            [0.0,0.0,30.0,0.0,0.0,14.0],
            [0.0,0.0,-162.0,0.0,0.0,-138.0],
            [0.0,0.0,75.0,0.0,0.0,0.0],
            [-7.0,0.0,0.0,4.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [5.0,0.0,0.0,-3.0,0.0,0.0],
            [-3.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [-5.0,0.0,0.0,2.0,0.0,0.0],
            [6.0,0.0,0.0,0.0,0.0,0.0],
            [9.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,0.0,0.0,0.0],
            [-7.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [7.0,0.0,0.0,0.0,0.0,0.0],
            [-4.0,0.0,0.0,0.0,0.0,0.0],
            [4.0,0.0,0.0,0.0,0.0,0.0],
            [-6.0,0.0,-3.0,3.0,0.0,1.0],
            [0.0,0.0,-3.0,0.0,0.0,-2.0],
            [11.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [11.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [-1.0,0.0,3.0,3.0,0.0,-1.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [0.0,0.0,-13.0,0.0,0.0,-11.0],
            [3.0,0.0,6.0,0.0,0.0,0.0],
            [-7.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-3.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-3.0,0.0,0.0],
            [-7.0,0.0,0.0,3.0,0.0,0.0],
            [8.0,0.0,0.0,-3.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [11.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [8.0,0.0,0.0,-4.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [11.0,0.0,0.0,0.0,0.0,0.0],
            [-6.0,0.0,0.0,3.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [-8.0,0.0,0.0,4.0,0.0,0.0],
            [-7.0,0.0,0.0,3.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [6.0,0.0,0.0,-3.0,0.0,0.0],
            [-6.0,0.0,0.0,3.0,0.0,0.0],
            [6.0,0.0,0.0,0.0,0.0,0.0],
            [6.0,0.0,0.0,-1.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [-5.0,0.0,0.0,2.0,0.0,0.0],
            [-4.0,0.0,0.0,0.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [4.0,0.0,0.0,0.0,0.0,0.0],
            [6.0,0.0,0.0,-3.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [0.0,0.0,-26.0,0.0,0.0,-11.0],
            [0.0,0.0,-10.0,0.0,0.0,-5.0],
            [5.0,0.0,0.0,-3.0,0.0,0.0],
            [-13.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,-2.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [7.0,0.0,0.0,-3.0,0.0,0.0],
            [4.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [-6.0,0.0,0.0,2.0,0.0,0.0],
            [-5.0,0.0,0.0,2.0,0.0,0.0],
            [-7.0,0.0,0.0,3.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [13.0,0.0,0.0,0.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [-3.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [-11.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [4.0,0.0,0.0,0.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [6.0,0.0,0.0,-3.0,0.0,0.0],
            [3.0,0.0,0.0,-2.0,0.0,0.0],
            [-12.0,0.0,0.0,0.0,0.0,0.0],
            [4.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,0.0,0.0,0.0],
            [-4.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [0.0,0.0,-5.0,0.0,0.0,-2.0],
            [-7.0,0.0,0.0,4.0,0.0,0.0],
            [6.0,0.0,0.0,-3.0,0.0,0.0],
            [-3.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-3.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [-5.0,0.0,0.0,3.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [12.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [4.0,0.0,0.0,0.0,0.0,0.0],
            [6.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-3.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [-6.0,0.0,0.0,3.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [6.0,0.0,0.0,-3.0,0.0,0.0],
            [6.0,0.0,0.0,0.0,0.0,0.0],
            [-6.0,0.0,0.0,3.0,0.0,0.0],
            [3.0,0.0,0.0,-2.0,0.0,0.0],
            [7.0,0.0,0.0,-4.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [-5.0,0.0,0.0,2.0,0.0,0.0],
            [5.0,0.0,0.0,0.0,0.0,0.0],
            [-6.0,0.0,0.0,3.0,0.0,0.0],
            [-6.0,0.0,0.0,3.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [10.0,0.0,0.0,0.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [7.0,0.0,0.0,0.0,0.0,0.0],
            [7.0,0.0,0.0,-3.0,0.0,0.0],
            [4.0,0.0,0.0,0.0,0.0,0.0],
            [11.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [-6.0,0.0,0.0,2.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [3.0,0.0,0.0,-2.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0]]
    
    #行星章动参数
    #*   L   L'  F   D   Om  Me  Ve  E  Ma  Ju  Sa  Ur  Ne  pre
    NAPL=[[0,0,0,0,0,0,0,8,-16,4,5,0,0,0],
            [0,0,0,0,0,0,0,-8,16,-4,-5,0,0,2],
            [0,0,0,0,0,0,0,8,-16,4,5,0,0,2],
            [0,0,0,0,0,0,0,0,0,0,0,-1,2,2],
            [0,0,0,0,0,0,0,-4,8,-1,-5,0,0,2],
            [0,0,0,0,0,0,0,4,-8,3,0,0,0,1],
            [0,0,1,-1,1,0,0,3,-8,3,0,0,0,0],
            [-1,0,0,0,0,0,10,-3,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,-2,6,-3,0,2],
            [0,0,0,0,0,0,0,4,-8,3,0,0,0,0],
            [0,0,1,-1,1,0,0,-5,8,-3,0,0,0,0],
            [0,0,0,0,0,0,0,-4,8,-3,0,0,0,1],
            [0,0,0,0,0,0,0,4,-8,1,5,0,0,2],
            [0,0,0,0,0,0,-5,6,4,0,0,0,0,2],
            [0,0,0,0,0,0,0,0,0,2,-5,0,0,2],
            [0,0,0,0,0,0,0,0,0,2,-5,0,0,1],
            [0,0,1,-1,1,0,0,-1,0,2,-5,0,0,0],
            [0,0,0,0,0,0,0,0,0,2,-5,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,-2,5,0,0,0],
            [0,0,0,0,0,0,0,0,0,-2,5,0,0,1],
            [0,0,0,0,0,0,0,0,0,-2,5,0,0,2],
            [2,0,-1,-1,0,0,0,3,-7,0,0,0,0,0],
            [1,0,0,-2,0,0,19,-21,3,0,0,0,0,0],
            [0,0,1,-1,1,0,2,-4,0,-3,0,0,0,0],
            [1,0,0,-1,1,0,0,-1,0,2,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,-4,10,0,0,0],
            [-2,0,0,2,1,0,0,2,0,0,-5,0,0,0],
            [0,0,0,0,0,0,3,-7,4,0,0,0,0,0],
            [0,0,-1,1,0,0,0,1,0,1,-1,0,0,0],
            [-2,0,0,2,1,0,0,2,0,-2,0,0,0,0],
            [-1,0,0,0,0,0,18,-16,0,0,0,0,0,0],
            [-2,0,1,1,2,0,0,1,0,-2,0,0,0,0],
            [-1,0,1,-1,1,0,18,-17,0,0,0,0,0,0],
            [-1,0,0,1,1,0,0,2,-2,0,0,0,0,0],
            [0,0,0,0,0,0,-8,13,0,0,0,0,0,2],
            [0,0,2,-2,2,0,-8,11,0,0,0,0,0,0],
            [0,0,0,0,0,0,-8,13,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-8,12,0,0,0,0,0,0],
            [0,0,0,0,0,0,8,-13,0,0,0,0,0,0],
            [0,0,1,-1,1,0,8,-14,0,0,0,0,0,0],
            [0,0,0,0,0,0,8,-13,0,0,0,0,0,1],
            [-2,0,0,2,1,0,0,2,0,-4,5,0,0,0],
            [-2,0,0,2,2,0,3,-3,0,0,0,0,0,0],
            [-2,0,0,2,0,0,0,2,0,-3,1,0,0,0],
            [0,0,0,0,1,0,3,-5,0,2,0,0,0,0],
            [-2,0,0,2,0,0,0,2,0,-4,3,0,0,0],
            [0,0,-1,1,0,0,0,0,2,0,0,0,0,0],
            [0,0,0,0,1,0,0,-1,2,0,0,0,0,0],
            [0,0,1,-1,2,0,0,-2,2,0,0,0,0,0],
            [-1,0,1,0,1,0,3,-5,0,0,0,0,0,0],
            [-1,0,0,1,0,0,3,-4,0,0,0,0,0,0],
            [-2,0,0,2,0,0,0,2,0,-2,-2,0,0,0],
            [-2,0,2,0,2,0,0,-5,9,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,0,-1,0,0],
            [0,0,0,0,0,0,0,0,0,0,0,1,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,0,0,2,0],
            [0,0,0,0,0,0,0,0,0,0,0,0,2,1],
            [0,0,0,0,0,0,0,0,0,0,0,0,2,2],
            [-1,0,0,1,0,0,0,3,-4,0,0,0,0,0],
            [0,0,-1,1,0,0,0,1,0,0,2,0,0,0],
            [0,0,1,-1,2,0,0,-1,0,0,2,0,0,0],
            [0,0,0,0,1,0,0,-9,17,0,0,0,0,0],
            [0,0,0,0,2,0,-3,5,0,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,-1,2,0,0,0],
            [0,0,0,0,0,0,0,0,0,1,-2,0,0,0],
            [1,0,0,-2,0,0,17,-16,0,-2,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,1,-3,0,0,0],
            [-2,0,0,2,1,0,0,5,-6,0,0,0,0,0],
            [0,0,-2,2,0,0,0,9,-13,0,0,0,0,0],
            [0,0,1,-1,2,0,0,-1,0,0,1,0,0,0],
            [0,0,0,0,1,0,0,0,0,0,1,0,0,0],
            [0,0,-1,1,0,0,0,1,0,0,1,0,0,0],
            [0,0,-2,2,0,0,5,-6,0,0,0,0,0,0],
            [0,0,-1,1,1,0,5,-7,0,0,0,0,0,0],
            [-2,0,0,2,0,0,6,-8,0,0,0,0,0,0],
            [2,0,1,-3,1,0,-6,7,0,0,0,0,0,0],
            [0,0,0,0,2,0,0,0,0,1,0,0,0,0],
            [0,0,-1,1,1,0,0,1,0,1,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,0,2,0,0],
            [0,0,0,0,0,0,0,0,0,0,0,2,0,1],
            [0,0,0,0,0,0,0,0,0,0,0,2,0,2],
            [0,0,0,0,0,0,0,-8,15,0,0,0,0,2],
            [0,0,0,0,0,0,0,-8,15,0,0,0,0,1],
            [0,0,1,-1,1,0,0,-9,15,0,0,0,0,0],
            [0,0,0,0,0,0,0,8,-15,0,0,0,0,0],
            [1,0,-1,-1,0,0,0,8,-15,0,0,0,0,0],
            [2,0,0,-2,0,0,2,-5,0,0,0,0,0,0],
            [-2,0,0,2,0,0,0,2,0,-5,5,0,0,0],
            [2,0,0,-2,1,0,0,-6,8,0,0,0,0,0],
            [2,0,0,-2,1,0,0,-2,0,3,0,0,0,0],
            [-2,0,1,1,0,0,0,1,0,-3,0,0,0,0],
            [-2,0,1,1,1,0,0,1,0,-3,0,0,0,0],
            [-2,0,0,2,0,0,0,2,0,-3,0,0,0,0],
            [-2,0,0,2,0,0,0,6,-8,0,0,0,0,0],
            [-2,0,0,2,0,0,0,2,0,-1,-5,0,0,0],
            [-1,0,0,1,0,0,0,1,0,-1,0,0,0,0],
            [-1,0,1,1,1,0,-20,20,0,0,0,0,0,0],
            [1,0,0,-2,0,0,20,-21,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,8,-15,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-10,15,0,0,0,0,0],
            [0,0,-1,1,0,0,0,1,0,1,0,0,0,0],
            [0,0,0,0,1,0,0,0,0,1,0,0,0,0],
            [0,0,1,-1,2,0,0,-1,0,1,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,-2,4,0,0,0],
            [2,0,0,-2,1,0,-6,8,0,0,0,0,0,0],
            [0,0,-2,2,1,0,5,-6,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,0,-1,0,0,1],
            [0,0,1,-1,1,0,0,-1,0,0,-1,0,0,0],
            [0,0,0,0,0,0,0,0,0,0,1,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,1,0,0,0],
            [0,0,0,0,0,0,0,0,0,0,1,0,0,1],
            [0,0,0,0,0,0,0,0,0,0,1,0,0,2],
            [0,0,2,-2,1,0,0,-9,13,0,0,0,0,0],
            [0,0,0,0,1,0,0,7,-13,0,0,0,0,0],
            [-2,0,0,2,0,0,0,5,-6,0,0,0,0,0],
            [0,0,0,0,0,0,0,9,-17,0,0,0,0,0],
            [0,0,0,0,0,0,0,-9,17,0,0,0,0,2],
            [1,0,0,-1,1,0,0,-3,4,0,0,0,0,0],
            [1,0,0,-1,1,0,-3,4,0,0,0,0,0,0],
            [0,0,0,0,2,0,0,-1,2,0,0,0,0,0],
            [0,0,-1,1,1,0,0,0,2,0,0,0,0,0],
            [0,0,-2,2,0,1,0,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,3,-5,0,2,0,0,0,0],
            [-2,0,0,2,1,0,0,2,0,-3,1,0,0,0],
            [-2,0,0,2,1,0,3,-3,0,0,0,0,0,0],
            [0,0,0,0,1,0,8,-13,0,0,0,0,0,0],
            [0,0,-1,1,0,0,8,-12,0,0,0,0,0,0],
            [0,0,2,-2,1,0,-8,11,0,0,0,0,0,0],
            [-1,0,0,1,0,0,0,2,-2,0,0,0,0,0],
            [-1,0,0,0,1,0,18,-16,0,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,-1,1,0,0,0],
            [0,0,0,0,1,0,3,-7,4,0,0,0,0,0],
            [-2,0,1,1,1,0,0,-3,7,0,0,0,0,0],
            [0,0,1,-1,2,0,0,-1,0,-2,5,0,0,0],
            [0,0,0,0,1,0,0,0,0,-2,5,0,0,0],
            [0,0,0,0,1,0,0,-4,8,-3,0,0,0,0],
            [1,0,0,0,1,0,-10,3,0,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-2,0,0,0,0,0,0],
            [-1,0,0,0,1,0,10,-3,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,4,-8,3,0,0,0,0],
            [0,0,0,0,1,0,0,0,0,2,-5,0,0,0],
            [0,0,-1,1,0,0,0,1,0,2,-5,0,0,0],
            [2,0,-1,-1,1,0,0,3,-7,0,0,0,0,0],
            [-2,0,0,2,0,0,0,2,0,0,-5,0,0,0],
            [0,0,0,0,1,0,-3,7,-4,0,0,0,0,0],
            [-2,0,0,2,0,0,0,2,0,-2,0,0,0,0],
            [1,0,0,0,1,0,-18,16,0,0,0,0,0,0],
            [-2,0,1,1,1,0,0,1,0,-2,0,0,0,0],
            [0,0,1,-1,2,0,-8,12,0,0,0,0,0,0],
            [0,0,0,0,1,0,-8,13,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,-2,0,0,0,0,1],
            [0,0,1,-1,1,0,0,0,-2,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,-2,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-2,2,0,0,0,0,0],
            [0,0,0,0,0,0,0,-1,2,0,0,0,0,1],
            [-1,0,0,1,1,0,3,-4,0,0,0,0,0,0],
            [-1,0,0,1,1,0,0,3,-4,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,-2,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,2,0,0,0],
            [0,0,0,0,0,0,0,0,0,0,2,0,0,1],
            [0,0,0,0,0,0,0,0,0,0,2,0,0,2],
            [0,0,1,-1,0,0,3,-6,0,0,0,0,0,0],
            [0,0,0,0,1,0,-3,5,0,0,0,0,0,0],
            [0,0,1,-1,2,0,-3,4,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,-2,4,0,0,0,0,0],
            [0,0,2,-2,1,0,-5,6,0,0,0,0,0,0],
            [0,0,-1,1,0,0,5,-7,0,0,0,0,0,0],
            [0,0,0,0,1,0,5,-8,0,0,0,0,0,0],
            [-2,0,0,2,1,0,6,-8,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,-8,15,0,0,0,0,0],
            [-2,0,0,2,1,0,0,2,0,-3,0,0,0,0],
            [-2,0,0,2,1,0,0,6,-8,0,0,0,0,0],
            [1,0,0,-1,1,0,0,-1,0,1,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,3,-5,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,-1,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,-1,0,0,0,1],
            [0,0,0,0,0,0,0,0,0,1,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,1,0,0,0,1],
            [0,0,1,-1,1,0,0,-1,0,1,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,1,0,0,0,1],
            [0,0,0,0,0,0,0,0,0,1,0,0,0,2],
            [0,0,1,-1,2,0,0,-1,0,0,-1,0,0,0],
            [0,0,0,0,1,0,0,0,0,0,-1,0,0,0],
            [0,0,-1,1,0,0,0,1,0,0,-1,0,0,0],
            [0,0,0,0,0,0,0,-7,13,0,0,0,0,2],
            [0,0,0,0,0,0,0,7,-13,0,0,0,0,0],
            [2,0,0,-2,1,0,0,-5,6,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-8,11,0,0,0,0,0],
            [0,0,2,-2,1,-1,0,2,0,0,0,0,0,0],
            [-2,0,0,2,0,0,0,4,-4,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,2,-2,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,3,0,0,0],
            [0,0,0,0,0,0,0,0,0,0,3,0,0,1],
            [0,0,0,0,0,0,0,0,0,0,3,0,0,2],
            [-2,0,0,2,0,0,3,-3,0,0,0,0,0,0],
            [0,0,0,0,2,0,0,-4,8,-3,0,0,0,0],
            [0,0,0,0,2,0,0,4,-8,3,0,0,0,0],
            [2,0,0,-2,1,0,0,-2,0,2,0,0,0,0],
            [0,0,1,-1,2,0,0,-1,0,2,0,0,0,0],
            [0,0,1,-1,2,0,0,0,-2,0,0,0,0,0],
            [0,0,0,0,1,0,0,1,-2,0,0,0,0,0],
            [0,0,-1,1,0,0,0,2,-2,0,0,0,0,0],
            [0,0,-1,1,0,0,0,1,0,0,-2,0,0,0],
            [0,0,2,-2,1,0,0,-2,0,0,2,0,0,0],
            [0,0,1,-1,1,0,3,-6,0,0,0,0,0,0],
            [0,0,0,0,0,0,3,-5,0,0,0,0,0,1],
            [0,0,0,0,0,0,3,-5,0,0,0,0,0,0],
            [0,0,1,-1,1,0,-3,4,0,0,0,0,0,0],
            [0,0,0,0,0,0,-3,5,0,0,0,0,0,1],
            [0,0,0,0,0,0,-3,5,0,0,0,0,0,2],
            [0,0,2,-2,2,0,-3,3,0,0,0,0,0,0],
            [0,0,0,0,0,0,-3,5,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,-4,0,0,0,0,1],
            [0,0,1,-1,1,0,0,1,-4,0,0,0,0,0],
            [0,0,0,0,0,0,0,2,-4,0,0,0,0,0],
            [0,0,0,0,0,0,0,-2,4,0,0,0,0,1],
            [0,0,1,-1,1,0,0,-3,4,0,0,0,0,0],
            [0,0,0,0,0,0,0,-2,4,0,0,0,0,1],
            [0,0,0,0,0,0,0,-2,4,0,0,0,0,2],
            [0,0,0,0,0,0,-5,8,0,0,0,0,0,2],
            [0,0,2,-2,2,0,-5,6,0,0,0,0,0,0],
            [0,0,0,0,0,0,-5,8,0,0,0,0,0,2],
            [0,0,0,0,0,0,-5,8,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-5,7,0,0,0,0,0,0],
            [0,0,0,0,0,0,-5,8,0,0,0,0,0,1],
            [0,0,0,0,0,0,5,-8,0,0,0,0,0,0],
            [0,0,1,-1,2,0,0,-1,0,-1,0,0,0,0],
            [0,0,0,0,1,0,0,0,0,-1,0,0,0,0],
            [0,0,-1,1,0,0,0,1,0,-1,0,0,0,0],
            [0,0,2,-2,1,0,0,-2,0,1,0,0,0,0],
            [0,0,0,0,0,0,0,-6,11,0,0,0,0,2],
            [0,0,0,0,0,0,0,6,-11,0,0,0,0,0],
            [0,0,0,0,0,-1,0,4,0,0,0,0,0,2],
            [0,0,0,0,0,1,0,-4,0,0,0,0,0,0],
            [2,0,0,-2,1,0,-3,3,0,0,0,0,0,0],
            [-2,0,0,2,0,0,0,2,0,0,-2,0,0,0],
            [0,0,2,-2,1,0,0,-7,9,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,4,-5,0,0,2],
            [0,0,0,0,0,0,0,0,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,2,0,0,0,1],
            [0,0,1,-1,1,0,0,-1,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,2,0,0,0,1],
            [0,0,0,0,0,0,0,0,0,2,0,0,0,2],
            [0,0,2,-2,2,0,0,-2,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,0,5,0,0,2],
            [0,0,0,0,1,0,3,-5,0,0,0,0,0,0],
            [0,0,-1,1,0,0,3,-4,0,0,0,0,0,0],
            [0,0,2,-2,1,0,-3,3,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,2,-4,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-4,4,0,0,0,0,0],
            [0,0,1,-1,2,0,-5,7,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,3,-6,0,0,0,0,0],
            [0,0,0,0,0,0,0,-3,6,0,0,0,0,1],
            [0,0,1,-1,1,0,0,-4,6,0,0,0,0,0],
            [0,0,0,0,0,0,0,-3,6,0,0,0,0,1],
            [0,0,0,0,0,0,0,-3,6,0,0,0,0,2],
            [0,0,-1,1,0,0,2,-2,0,0,0,0,0,0],
            [0,0,0,0,1,0,2,-3,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,-5,9,0,0,0,0,2],
            [0,0,0,0,0,0,0,-5,9,0,0,0,0,1],
            [0,0,0,0,0,0,0,5,-9,0,0,0,0,0],
            [0,0,-1,1,0,0,0,1,0,-2,0,0,0,0],
            [0,0,2,-2,1,0,0,-2,0,2,0,0,0,0],
            [-2,0,1,1,1,0,0,1,0,0,0,0,0,0],
            [0,0,-2,2,0,0,3,-3,0,0,0,0,0,0],
            [0,0,0,0,0,0,-6,10,0,0,0,0,0,1],
            [0,0,0,0,0,0,-6,10,0,0,0,0,0,2],
            [0,0,0,0,0,0,-2,3,0,0,0,0,0,2],
            [0,0,0,0,0,0,-2,3,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-2,2,0,0,0,0,0,0],
            [0,0,0,0,0,0,2,-3,0,0,0,0,0,0],
            [0,0,0,0,0,0,2,-3,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,0,0,3,0,0,0,1],
            [0,0,1,-1,1,0,0,-1,0,3,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,3,0,0,0,1],
            [0,0,0,0,0,0,0,0,0,3,0,0,0,2],
            [0,0,0,0,0,0,0,4,-8,0,0,0,0,0],
            [0,0,0,0,0,0,0,-4,8,0,0,0,0,2],
            [0,0,-2,2,0,0,0,2,0,-2,0,0,0,0],
            [0,0,0,0,0,0,0,-4,7,0,0,0,0,2],
            [0,0,0,0,0,0,0,-4,7,0,0,0,0,1],
            [0,0,0,0,0,0,0,4,-7,0,0,0,0,0],
            [0,0,0,0,1,0,-2,3,0,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-2,0,3,0,0,0,0],
            [0,0,0,0,0,0,0,-5,10,0,0,0,0,2],
            [0,0,0,0,1,0,-1,2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,4,0,0,0,2],
            [0,0,0,0,0,0,0,-3,5,0,0,0,0,2],
            [0,0,0,0,0,0,0,-3,5,0,0,0,0,1],
            [0,0,0,0,0,0,0,3,-5,0,0,0,0,0],
            [0,0,0,0,0,0,1,-2,0,0,0,0,0,1],
            [0,0,1,-1,1,0,1,-3,0,0,0,0,0,0],
            [0,0,0,0,0,0,1,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,-1,2,0,0,0,0,0,1],
            [0,0,0,0,0,0,-1,2,0,0,0,0,0,2],
            [0,0,0,0,0,0,-7,11,0,0,0,0,0,2],
            [0,0,0,0,0,0,-7,11,0,0,0,0,0,1],
            [0,0,-2,2,0,0,4,-4,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,2,-3,0,0,0,0,0],
            [0,0,2,-2,1,0,-4,4,0,0,0,0,0,0],
            [0,0,-1,1,0,0,4,-5,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,-1,0,0,0,0,0],
            [0,0,0,0,0,0,-4,7,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-4,6,0,0,0,0,0,0],
            [0,0,0,0,0,0,-4,7,0,0,0,0,0,2],
            [0,0,0,0,0,0,-4,6,0,0,0,0,0,2],
            [0,0,0,0,0,0,-4,6,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-4,5,0,0,0,0,0,0],
            [0,0,0,0,0,0,-4,6,0,0,0,0,0,1],
            [0,0,0,0,0,0,4,-6,0,0,0,0,0,0],
            [-2,0,0,2,0,0,2,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,1,0,0,0,0,0],
            [0,0,-1,1,0,0,1,0,0,0,0,0,0,0],
            [0,0,0,0,1,0,1,-1,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,-1,0,5,0,0,0,2],
            [0,0,0,0,0,0,0,1,-3,0,0,0,0,0],
            [0,0,0,0,0,0,0,-1,3,0,0,0,0,2],
            [0,0,0,0,0,0,0,-7,12,0,0,0,0,2],
            [0,0,0,0,0,0,-1,1,0,0,0,0,0,2],
            [0,0,0,0,0,0,-1,1,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-1,0,0,0,0,0,0,0],
            [0,0,0,0,0,0,1,-1,0,0,0,0,0,0],
            [0,0,0,0,0,0,1,-1,0,0,0,0,0,1],
            [0,0,1,-1,1,0,1,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,-2,5,0,0,0,0,2],
            [0,0,0,0,0,0,0,-1,0,4,0,0,0,2],
            [0,0,0,0,0,0,0,1,0,-4,0,0,0,0],
            [0,0,0,0,1,0,-1,1,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,-6,10,0,0,0,0,2],
            [0,0,0,0,0,0,0,-6,10,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-3,0,3,0,0,0,0],
            [0,0,0,0,0,0,0,-3,7,0,0,0,0,2],
            [-2,0,0,2,0,0,4,-4,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,-5,8,0,0,0,0,2],
            [0,0,0,0,0,0,0,5,-8,0,0,0,0,0],
            [0,0,0,0,0,0,0,-1,0,3,0,0,0,2],
            [0,0,0,0,0,0,0,-1,0,3,0,0,0,1],
            [0,0,0,0,0,0,0,1,0,-3,0,0,0,0],
            [0,0,0,0,0,0,2,-4,0,0,0,0,0,0],
            [0,0,0,0,0,0,-2,4,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-2,3,0,0,0,0,0,0],
            [0,0,0,0,0,0,-2,4,0,0,0,0,0,2],
            [0,0,0,0,0,0,-6,9,0,0,0,0,0,2],
            [0,0,0,0,0,0,-6,9,0,0,0,0,0,1],
            [0,0,0,0,0,0,6,-9,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,1,0,-2,0,0,0,0],
            [0,0,2,-2,1,0,-2,2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,-4,6,0,0,0,0,2],
            [0,0,0,0,0,0,0,4,-6,0,0,0,0,0],
            [0,0,0,0,1,0,3,-4,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,-1,0,2,0,0,0,2],
            [0,0,0,0,0,0,0,1,0,-2,0,0,0,0],
            [0,0,0,0,1,0,0,1,0,-1,0,0,0,0],
            [0,0,0,0,0,0,-5,9,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,-4,0,0,0,0,0],
            [0,0,0,0,0,0,-3,4,0,0,0,0,0,2],
            [0,0,0,0,0,0,-3,4,0,0,0,0,0,1],
            [0,0,0,0,0,0,3,-4,0,0,0,0,0,0],
            [0,0,0,0,0,0,3,-4,0,0,0,0,0,1],
            [0,0,0,0,1,0,0,2,-2,0,0,0,0,0],
            [0,0,0,0,1,0,0,-1,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,0,-3,0,0,0],
            [0,0,0,0,0,0,0,1,0,1,-5,0,0,0],
            [0,0,0,0,0,0,0,-1,0,1,0,0,0,1],
            [0,0,0,0,0,0,0,1,0,-1,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,-1,0,0,0,1],
            [0,0,0,0,0,0,0,1,0,-3,5,0,0,0],
            [0,0,0,0,1,0,-3,4,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,0,-2,0,0,0],
            [0,0,0,0,0,0,0,2,-2,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,0,-1,0,0,0],
            [0,0,0,0,1,0,0,-1,0,1,0,0,0,0],
            [0,0,0,0,1,0,0,-2,2,0,0,0,0,0],
            [0,0,0,0,0,0,-8,14,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,1,0,2,-5,0,0,0],
            [0,0,0,0,0,0,0,5,-8,3,0,0,0,0],
            [0,0,0,0,0,0,0,5,-8,3,0,0,0,2],
            [0,0,0,0,0,0,0,-1,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,1,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,3,-8,3,0,0,0,0],
            [0,0,0,0,0,0,0,-3,8,-3,0,0,0,2],
            [0,0,0,0,0,0,0,1,0,-2,5,0,0,2],
            [0,0,0,0,0,0,-8,12,0,0,0,0,0,2],
            [0,0,0,0,0,0,-8,12,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,1,-2,0,0,0],
            [0,0,0,0,0,0,0,1,0,0,1,0,0,2],
            [0,0,0,0,0,0,0,0,2,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,2,0,0,0,0,2],
            [0,0,0,0,0,0,0,1,0,0,2,0,0,2],
            [0,0,2,-2,1,0,-5,5,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,1,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,1,0,0,0,1],
            [0,0,0,0,0,0,0,1,0,1,0,0,0,2],
            [0,0,0,0,0,0,3,-6,0,0,0,0,0,0],
            [0,0,0,0,0,0,-3,6,0,0,0,0,0,1],
            [0,0,0,0,0,0,-3,6,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,-1,4,0,0,0,0,2],
            [0,0,0,0,0,0,-5,7,0,0,0,0,0,2],
            [0,0,0,0,0,0,-5,7,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-5,6,0,0,0,0,0,0],
            [0,0,0,0,0,0,5,-7,0,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-1,0,1,0,0,0,0],
            [0,0,0,0,0,0,0,-1,0,1,0,0,0,0],
            [0,0,0,0,0,-1,0,3,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,1,0,2,0,0,0,2],
            [0,0,0,0,0,0,0,-2,6,0,0,0,0,2],
            [0,0,0,0,1,0,2,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,-6,9,0,0,0,0,2],
            [0,0,0,0,0,0,0,6,-9,0,0,0,0,0],
            [0,0,0,0,0,0,-2,2,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-2,1,0,0,0,0,0,0],
            [0,0,0,0,0,0,2,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,2,-2,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,1,0,3,0,0,0,2],
            [0,0,0,0,0,0,0,-5,7,0,0,0,0,2],
            [0,0,0,0,0,0,0,5,-7,0,0,0,0,0],
            [0,0,0,0,1,0,-2,2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,4,-5,0,0,0,0,0],
            [0,0,0,0,0,0,1,-3,0,0,0,0,0,0],
            [0,0,0,0,0,0,-1,3,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-1,2,0,0,0,0,0,0],
            [0,0,0,0,0,0,-1,3,0,0,0,0,0,2],
            [0,0,0,0,0,0,-7,10,0,0,0,0,0,2],
            [0,0,0,0,0,0,-7,10,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,3,-3,0,0,0,0,0],
            [0,0,0,0,0,0,-4,8,0,0,0,0,0,2],
            [0,0,0,0,0,0,-4,5,0,0,0,0,0,2],
            [0,0,0,0,0,0,-4,5,0,0,0,0,0,1],
            [0,0,0,0,0,0,4,-5,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,1,0,0,0,0,2],
            [0,0,0,0,0,0,0,-2,0,5,0,0,0,2],
            [0,0,0,0,0,0,0,0,3,0,0,0,0,2],
            [0,0,0,0,0,0,1,0,0,0,0,0,0,0],
            [0,0,0,0,0,0,1,0,0,0,0,0,0,2],
            [0,0,0,0,0,0,-9,13,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,-1,5,0,0,0,0,2],
            [0,0,0,0,0,0,0,-2,0,4,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,-4,0,0,0,0],
            [0,0,0,0,0,0,0,-2,7,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,-3,0,0,0,0],
            [0,0,0,0,0,0,-2,5,0,0,0,0,0,1],
            [0,0,0,0,0,0,-2,5,0,0,0,0,0,2],
            [0,0,0,0,0,0,-6,8,0,0,0,0,0,2],
            [0,0,0,0,0,0,-6,8,0,0,0,0,0,1],
            [0,0,0,0,0,0,6,-8,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,2,0,-2,0,0,0,0],
            [0,0,0,0,0,0,0,-3,9,0,0,0,0,2],
            [0,0,0,0,0,0,0,5,-6,0,0,0,0,0],
            [0,0,0,0,0,0,0,5,-6,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,-2,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,-2,0,0,0,1],
            [0,0,0,0,0,0,0,2,0,-2,0,0,0,2],
            [0,0,0,0,0,0,-5,10,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,4,-4,0,0,0,0,0],
            [0,0,0,0,0,0,0,4,-4,0,0,0,0,2],
            [0,0,0,0,0,0,-3,3,0,0,0,0,0,1],
            [0,0,0,0,0,0,3,-3,0,0,0,0,0,0],
            [0,0,0,0,0,0,3,-3,0,0,0,0,0,1],
            [0,0,0,0,0,0,3,-3,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,0,-3,0,0,0],
            [0,0,0,0,0,0,0,-5,13,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,-1,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,-1,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,0,-2,0,0,0],
            [0,0,0,0,0,0,0,2,0,0,-2,0,0,1],
            [0,0,0,0,0,0,0,3,-2,0,0,0,0,0],
            [0,0,0,0,0,0,0,3,-2,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,0,-1,0,0,2],
            [0,0,0,0,0,0,0,-6,15,0,0,0,0,2],
            [0,0,0,0,0,0,-8,15,0,0,0,0,0,2],
            [0,0,0,0,0,0,-3,9,-4,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,2,-5,0,0,2],
            [0,0,0,0,0,0,0,-2,8,-1,-5,0,0,2],
            [0,0,0,0,0,0,0,6,-8,3,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,0,0,0,0,1],
            [0,0,1,-1,1,0,0,1,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,2,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,-6,16,-4,-5,0,0,2],
            [0,0,0,0,0,0,0,-2,8,-3,0,0,0,2],
            [0,0,0,0,0,0,0,-2,8,-3,0,0,0,2],
            [0,0,0,0,0,0,0,6,-8,1,5,0,0,2],
            [0,0,0,0,0,0,0,2,0,-2,5,0,0,2],
            [0,0,0,0,0,0,3,-5,4,0,0,0,0,2],
            [0,0,0,0,0,0,-8,11,0,0,0,0,0,2],
            [0,0,0,0,0,0,-8,11,0,0,0,0,0,1],
            [0,0,0,0,0,0,-8,11,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,11,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,0,1,0,0,2],
            [0,0,0,0,0,0,3,-3,0,2,0,0,0,2],
            [0,0,2,-2,1,0,0,4,-8,3,0,0,0,0],
            [0,0,1,-1,0,0,0,1,0,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-4,8,-3,0,0,0,0],
            [0,0,0,0,0,0,0,1,2,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,1,0,0,0,2],
            [0,0,0,0,0,0,-3,7,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,0,4,0,0,0,0,2],
            [0,0,0,0,0,0,-5,6,0,0,0,0,0,2],
            [0,0,0,0,0,0,-5,6,0,0,0,0,0,1],
            [0,0,0,0,0,0,5,-6,0,0,0,0,0,0],
            [0,0,0,0,0,0,5,-6,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,2,0,0,0,2],
            [0,0,0,0,0,0,0,-1,6,0,0,0,0,2],
            [0,0,0,0,0,0,0,7,-9,0,0,0,0,2],
            [0,0,0,0,0,0,2,-1,0,0,0,0,0,0],
            [0,0,0,0,0,0,2,-1,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,6,-7,0,0,0,0,2],
            [0,0,0,0,0,0,0,5,-5,0,0,0,0,2],
            [0,0,0,0,0,0,-1,4,0,0,0,0,0,1],
            [0,0,0,0,0,0,-1,4,0,0,0,0,0,2],
            [0,0,0,0,0,0,-7,9,0,0,0,0,0,2],
            [0,0,0,0,0,0,-7,9,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,4,-3,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,-1,0,0,0,0,2],
            [0,0,0,0,0,0,-4,4,0,0,0,0,0,1],
            [0,0,0,0,0,0,4,-4,0,0,0,0,0,0],
            [0,0,0,0,0,0,4,-4,0,0,0,0,0,1],
            [0,0,0,0,0,0,4,-4,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,1,0,0,0,0,2],
            [0,0,0,0,0,0,0,-3,0,5,0,0,0,2],
            [0,0,0,0,0,0,1,1,0,0,0,0,0,0],
            [0,0,0,0,0,0,1,1,0,0,0,0,0,1],
            [0,0,0,0,0,0,1,1,0,0,0,0,0,2],
            [0,0,0,0,0,0,-9,12,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,-4,0,0,0,0],
            [0,0,2,-2,1,0,1,-1,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,7,-8,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,-3,0,0,0,0],
            [0,0,0,0,0,0,0,3,0,-3,0,0,0,2],
            [0,0,0,0,0,0,-2,6,0,0,0,0,0,2],
            [0,0,0,0,0,0,-6,7,0,0,0,0,0,1],
            [0,0,0,0,0,0,6,-7,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,6,-6,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,-2,0,0,0,0],
            [0,0,0,0,0,0,0,3,0,-2,0,0,0,2],
            [0,0,0,0,0,0,0,5,-4,0,0,0,0,2],
            [0,0,0,0,0,0,3,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,3,-2,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,-1,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,-1,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,0,-2,0,0,2],
            [0,0,0,0,0,0,0,4,-2,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,0,-1,0,0,2],
            [0,0,2,-2,1,0,0,1,0,-1,0,0,0,0],
            [0,0,0,0,0,0,-8,16,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,2,-5,0,0,2],
            [0,0,0,0,0,0,0,7,-8,3,0,0,0,2],
            [0,0,0,0,0,0,0,-5,16,-4,-5,0,0,2],
            [0,0,0,0,0,0,0,3,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,-1,8,-3,0,0,0,2],
            [0,0,0,0,0,0,-8,10,0,0,0,0,0,2],
            [0,0,0,0,0,0,-8,10,0,0,0,0,0,1],
            [0,0,0,0,0,0,-8,10,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,2,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,1,0,0,0,2],
            [0,0,0,0,0,0,-3,8,0,0,0,0,0,2],
            [0,0,0,0,0,0,-5,5,0,0,0,0,0,1],
            [0,0,0,0,0,0,5,-5,0,0,0,0,0,0],
            [0,0,0,0,0,0,5,-5,0,0,0,0,0,1],
            [0,0,0,0,0,0,5,-5,0,0,0,0,0,2],
            [0,0,0,0,0,0,2,0,0,0,0,0,0,0],
            [0,0,0,0,0,0,2,0,0,0,0,0,0,1],
            [0,0,0,0,0,0,2,0,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,7,-7,0,0,0,0,2],
            [0,0,0,0,0,0,0,7,-7,0,0,0,0,2],
            [0,0,0,0,0,0,0,6,-5,0,0,0,0,2],
            [0,0,0,0,0,0,7,-8,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,5,-3,0,0,0,0,2],
            [0,0,0,0,0,0,4,-3,0,0,0,0,0,2],
            [0,0,0,0,0,0,1,2,0,0,0,0,0,2],
            [0,0,0,0,0,0,-9,11,0,0,0,0,0,2],
            [0,0,0,0,0,0,-9,11,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,4,0,-4,0,0,0,2],
            [0,0,0,0,0,0,0,4,0,-3,0,0,0,2],
            [0,0,0,0,0,0,-6,6,0,0,0,0,0,1],
            [0,0,0,0,0,0,6,-6,0,0,0,0,0,0],
            [0,0,0,0,0,0,6,-6,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,4,0,-2,0,0,0,2],
            [0,0,0,0,0,0,0,6,-4,0,0,0,0,2],
            [0,0,0,0,0,0,3,-1,0,0,0,0,0,0],
            [0,0,0,0,0,0,3,-1,0,0,0,0,0,1],
            [0,0,0,0,0,0,3,-1,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,4,0,-1,0,0,0,2],
            [0,0,0,0,0,0,0,4,0,0,-2,0,0,2],
            [0,0,0,0,0,0,0,5,-2,0,0,0,0,2],
            [0,0,0,0,0,0,0,4,0,0,0,0,0,0],
            [0,0,0,0,0,0,8,-9,0,0,0,0,0,0],
            [0,0,0,0,0,0,5,-4,0,0,0,0,0,2],
            [0,0,0,0,0,0,2,1,0,0,0,0,0,2],
            [0,0,0,0,0,0,2,1,0,0,0,0,0,1],
            [0,0,0,0,0,0,2,1,0,0,0,0,0,1],
            [0,0,0,0,0,0,-7,7,0,0,0,0,0,1],
            [0,0,0,0,0,0,7,-7,0,0,0,0,0,0],
            [0,0,0,0,0,0,4,-2,0,0,0,0,0,1],
            [0,0,0,0,0,0,4,-2,0,0,0,0,0,2],
            [0,0,0,0,0,0,4,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,4,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,5,0,-4,0,0,0,2],
            [0,0,0,0,0,0,0,5,0,-3,0,0,0,2],
            [0,0,0,0,0,0,0,5,0,-2,0,0,0,2],
            [0,0,0,0,0,0,3,0,0,0,0,0,0,2],
            [0,0,0,0,0,0,-8,8,0,0,0,0,0,1],
            [0,0,0,0,0,0,8,-8,0,0,0,0,0,0],
            [0,0,0,0,0,0,5,-3,0,0,0,0,0,1],
            [0,0,0,0,0,0,5,-3,0,0,0,0,0,2],
            [0,0,0,0,0,0,-9,9,0,0,0,0,0,1],
            [0,0,0,0,0,0,-9,9,0,0,0,0,0,1],
            [0,0,0,0,0,0,-9,9,0,0,0,0,0,1],
            [0,0,0,0,0,0,9,-9,0,0,0,0,0,0],
            [0,0,0,0,0,0,6,-4,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,0,0,0,0,0,0,2],
            [1,0,0,-2,0,0,0,2,0,-2,0,0,0,0],
            [1,0,0,-2,0,0,2,-2,0,0,0,0,0,0],
            [1,0,0,-2,0,0,0,1,0,-1,0,0,0,0],
            [1,0,0,-2,0,0,1,-1,0,0,0,0,0,0],
            [-1,0,0,0,0,0,3,-3,0,0,0,0,0,0],
            [-1,0,0,0,0,0,0,2,0,-2,0,0,0,0],
            [-1,0,0,2,0,0,0,4,-8,3,0,0,0,0],
            [1,0,0,-2,0,0,0,4,-8,3,0,0,0,0],
            [-2,0,0,2,0,0,0,4,-8,3,0,0,0,0],
            [-1,0,0,0,0,0,0,2,0,-3,0,0,0,0],
            [-1,0,0,0,0,0,0,1,0,-1,0,0,0,0],
            [-1,0,0,0,0,0,1,-1,0,0,0,0,0,0],
            [-1,0,0,2,0,0,2,-2,0,0,0,0,0,0],
            [1,0,-1,1,0,0,0,1,0,0,0,0,0,0],
            [-1,0,0,2,0,0,0,2,0,-3,0,0,0,0],
            [-2,0,0,0,0,0,0,2,0,-3,0,0,0,0],
            [1,0,0,0,0,0,0,4,-8,3,0,0,0,0],
            [-1,0,1,-1,1,0,0,-1,0,0,0,0,0,0],
            [1,0,1,-1,1,0,0,-1,0,0,0,0,0,0],
            [-1,0,0,0,0,0,0,4,-8,3,0,0,0,0],
            [-1,0,0,2,1,0,0,2,0,-2,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,-2,0,0,0,0],
            [-1,0,0,2,0,0,0,2,0,-2,0,0,0,0],
            [-1,0,0,2,0,0,3,-3,0,0,0,0,0,0],
            [1,0,0,-2,1,0,0,-2,0,2,0,0,0,0],
            [1,0,2,-2,2,0,-3,3,0,0,0,0,0,0],
            [1,0,2,-2,2,0,0,-2,0,2,0,0,0,0],
            [1,0,0,0,0,0,1,-1,0,0,0,0,0,0],
            [1,0,0,0,0,0,0,1,0,-1,0,0,0,0],
            [0,0,0,-2,0,0,2,-2,0,0,0,0,0,0],
            [0,0,0,-2,0,0,0,1,0,-1,0,0,0,0],
            [0,0,2,0,2,0,-2,2,0,0,0,0,0,0],
            [0,0,2,0,2,0,0,-1,0,1,0,0,0,0],
            [0,0,2,0,2,0,-1,1,0,0,0,0,0,0],
            [0,0,2,0,2,0,-2,3,0,0,0,0,0,0],
            [0,0,0,2,0,0,0,2,0,-2,0,0,0,0],
            [0,0,1,1,2,0,0,1,0,0,0,0,0,0],
            [1,0,2,0,2,0,0,1,0,0,0,0,0,0],
            [-1,0,2,0,2,0,10,-3,0,0,0,0,0,0],
            [0,0,1,1,1,0,0,1,0,0,0,0,0,0],
            [1,0,2,0,2,0,0,1,0,0,0,0,0,0],
            [0,0,2,0,2,0,0,4,-8,3,0,0,0,0],
            [0,0,2,0,2,0,0,-4,8,-3,0,0,0,0],
            [-1,0,2,0,2,0,0,-4,8,-3,0,0,0,0],
            [2,0,2,-2,2,0,0,-2,0,3,0,0,0,0],
            [1,0,2,0,1,0,0,-2,0,3,0,0,0,0],
            [0,0,1,1,0,0,0,1,0,0,0,0,0,0],
            [-1,0,2,0,1,0,0,1,0,0,0,0,0,0],
            [-2,0,2,2,2,0,0,2,0,-2,0,0,0,0],
            [0,0,2,0,2,0,2,-3,0,0,0,0,0,0],
            [0,0,2,0,2,0,1,-1,0,0,0,0,0,0],
            [0,0,2,0,2,0,0,1,0,-1,0,0,0,0],
            [0,0,2,0,2,0,2,-2,0,0,0,0,0,0],
            [-1,0,2,2,2,0,0,-1,0,1,0,0,0,0],
            [1,0,2,0,2,0,-1,1,0,0,0,0,0,0],
            [-1,0,2,2,2,0,0,2,0,-3,0,0,0,0],
            [2,0,2,0,2,0,0,2,0,-3,0,0,0,0],
            [1,0,2,0,2,0,0,-4,8,-3,0,0,0,0],
            [1,0,2,0,2,0,0,4,-8,3,0,0,0,0],
            [1,0,1,1,1,0,0,1,0,0,0,0,0,0],
            [0,0,2,0,2,0,0,1,0,0,0,0,0,0],
            [2,0,2,0,1,0,0,1,0,0,0,0,0,0],
            [-1,0,2,2,2,0,0,2,0,-2,0,0,0,0],
            [-1,0,2,2,2,0,3,-3,0,0,0,0,0,0],
            [1,0,2,0,2,0,1,-1,0,0,0,0,0,0],
            [0,0,2,2,2,0,0,2,0,-2,0,0,0,0]]
    
    #行星章动系数，单位：0.1微角秒
    #*  longitude (sin, cos), obliquity (sin, cos)
    ICPL=[[1440,0,0,0],
            [56,-117,-42,-40],
            [125,-43,0,-54],
            [0,5,0,0],
            [3,-7,-3,0],
            [3,0,0,-2],
            [-114,0,0,61],
            [-219,89,0,0],
            [-3,0,0,0],
            [-462,1604,0,0],
            [99,0,0,-53],
            [-3,0,0,2],
            [0,6,2,0],
            [3,0,0,0],
            [-12,0,0,0],
            [14,-218,117,8],
            [31,-481,-257,-17],
            [-491,128,0,0],
            [-3084,5123,2735,1647],
            [-1444,2409,-1286,-771],
            [11,-24,-11,-9],
            [26,-9,0,0],
            [103,-60,0,0],
            [0,-13,-7,0],
            [-26,-29,-16,14],
            [9,-27,-14,-5],
            [12,0,0,-6],
            [-7,0,0,0],
            [0,24,0,0],
            [284,0,0,-151],
            [226,101,0,0],
            [0,-8,-2,0],
            [0,-6,-3,0],
            [5,0,0,-3],
            [-41,175,76,17],
            [0,15,6,0],
            [425,212,-133,269],
            [1200,598,319,-641],
            [235,334,0,0],
            [11,-12,-7,-6],
            [5,-6,3,3],
            [-5,0,0,3],
            [6,0,0,-3],
            [15,0,0,0],
            [13,0,0,-7],
            [-6,-9,0,0],
            [266,-78,0,0],
            [-460,-435,-232,246],
            [0,15,7,0],
            [-3,0,0,2],
            [0,131,0,0],
            [4,0,0,0],
            [0,3,0,0],
            [0,4,2,0],
            [0,3,0,0],
            [-17,-19,-10,9],
            [-9,-11,6,-5],
            [-6,0,0,3],
            [-16,8,0,0],
            [0,3,0,0],
            [11,24,11,-5],
            [-3,-4,-2,1],
            [3,0,0,-1],
            [0,-8,-4,0],
            [0,3,0,0],
            [0,5,0,0],
            [0,3,2,0],
            [-6,4,2,3],
            [-3,-5,0,0],
            [-5,0,0,2],
            [4,24,13,-2],
            [-42,20,0,0],
            [-10,233,0,0],
            [-3,0,0,1],
            [78,-18,0,0],
            [0,3,1,0],
            [0,-3,-1,0],
            [0,-4,-2,1],
            [0,-8,-4,-1],
            [0,-5,3,0],
            [-7,0,0,3],
            [-14,8,3,6],
            [0,8,-4,0],
            [0,19,10,0],
            [45,-22,0,0],
            [-3,0,0,0],
            [0,-3,0,0],
            [0,3,0,0],
            [3,5,3,-2],
            [89,-16,-9,-48],
            [0,3,0,0],
            [-3,7,4,2],
            [-349,-62,0,0],
            [-15,22,0,0],
            [-3,0,0,0],
            [-53,0,0,0],
            [5,0,0,-3],
            [0,-8,0,0],
            [15,-7,-4,-8],
            [-3,0,0,1],
            [-21,-78,0,0],
            [20,-70,-37,-11],
            [0,6,3,0],
            [5,3,2,-2],
            [-17,-4,-2,9],
            [0,6,3,0],
            [32,15,-8,17],
            [174,84,45,-93],
            [11,56,0,0],
            [-66,-12,-6,35],
            [47,8,4,-25],
            [0,8,4,0],
            [10,-22,-12,-5],
            [-3,0,0,2],
            [-24,12,0,0],
            [5,-6,0,0],
            [3,0,0,-2],
            [4,3,1,-2],
            [0,29,15,0],
            [-5,-4,-2,2],
            [8,-3,-1,-5],
            [0,-3,0,0],
            [10,0,0,0],
            [3,0,0,-2],
            [-5,0,0,3],
            [46,66,35,-25],
            [-14,7,0,0],
            [0,3,2,0],
            [-5,0,0,0],
            [-68,-34,-18,36],
            [0,14,7,0],
            [10,-6,-3,-5],
            [-5,-4,-2,3],
            [-3,5,2,1],
            [76,17,9,-41],
            [84,298,159,-45],
            [3,0,0,-1],
            [-3,0,0,2],
            [-3,0,0,1],
            [-82,292,156,44],
            [-73,17,9,39],
            [-9,-16,0,0],
            [3,0,-1,-2],
            [-3,0,0,0],
            [-9,-5,-3,5],
            [-439,0,0,0],
            [57,-28,-15,-30],
            [0,-6,-3,0],
            [-4,0,0,2],
            [-40,57,30,21],
            [23,7,3,-13],
            [273,80,43,-146],
            [-449,430,0,0],
            [-8,-47,-25,4],
            [6,47,25,-3],
            [0,23,13,0],
            [-3,0,0,2],
            [3,-4,-2,-2],
            [-48,-110,-59,26],
            [51,114,61,-27],
            [-133,0,0,57],
            [0,4,0,0],
            [-21,-6,-3,11],
            [0,-3,-1,0],
            [-11,-21,-11,6],
            [-18,-436,-233,9],
            [35,-7,0,0],
            [0,5,3,0],
            [11,-3,-1,-6],
            [-5,-3,-1,3],
            [-53,-9,-5,28],
            [0,3,2,1],
            [4,0,0,-2],
            [0,-4,0,0],
            [-50,194,103,27],
            [-13,52,28,7],
            [-91,248,0,0],
            [6,49,26,-3],
            [-6,-47,-25,3],
            [0,5,3,0],
            [52,23,10,-23],
            [-3,0,0,1],
            [0,5,3,0],
            [-4,0,0,0],
            [-4,8,3,2],
            [10,0,0,0],
            [3,0,0,-2],
            [0,8,4,0],
            [0,8,4,1],
            [-4,0,0,0],
            [-4,0,0,0],
            [-8,4,2,4],
            [8,-4,-2,-4],
            [0,15,7,0],
            [-138,0,0,0],
            [0,-7,-3,0],
            [0,-7,-3,0],
            [54,0,0,-29],
            [0,10,4,0],
            [-7,0,0,3],
            [-37,35,19,20],
            [0,4,0,0],
            [-4,9,0,0],
            [8,0,0,-4],
            [-9,-14,-8,5],
            [-3,-9,-5,3],
            [-145,47,0,0],
            [-10,40,21,5],
            [11,-49,-26,-7],
            [-2150,0,0,932],
            [-12,0,0,5],
            [85,0,0,-37],
            [4,0,0,-2],
            [3,0,0,-2],
            [-86,153,0,0],
            [-6,9,5,3],
            [9,-13,-7,-5],
            [-8,12,6,4],
            [-51,0,0,22],
            [-11,-268,-116,5],
            [0,12,5,0],
            [0,7,3,0],
            [31,6,3,-17],
            [140,27,14,-75],
            [57,11,6,-30],
            [-14,-39,0,0],
            [0,-6,-2,0],
            [4,15,8,-2],
            [0,4,0,0],
            [-3,0,0,1],
            [0,11,5,0],
            [9,6,0,0],
            [-4,10,4,2],
            [5,3,0,0],
            [16,0,0,-9],
            [-3,0,0,0],
            [0,3,2,-1],
            [7,0,0,-3],
            [-25,22,0,0],
            [42,223,119,-22],
            [-27,-143,-77,14],
            [9,49,26,-5],
            [-1166,0,0,505],
            [-5,0,0,2],
            [-6,0,0,3],
            [-8,0,1,4],
            [0,-4,0,0],
            [117,0,0,-63],
            [-4,8,4,2],
            [3,0,0,-2],
            [-5,0,0,2],
            [0,31,0,0],
            [-5,0,1,3],
            [4,0,0,-2],
            [-4,0,0,2],
            [-24,-13,-6,10],
            [3,0,0,0],
            [0,-32,-17,0],
            [8,12,5,-3],
            [3,0,0,-1],
            [7,13,0,0],
            [-3,16,0,0],
            [50,0,0,-27],
            [0,-5,-3,0],
            [13,0,0,0],
            [0,5,3,1],
            [24,5,2,-11],
            [5,-11,-5,-2],
            [30,-3,-2,-16],
            [18,0,0,-9],
            [8,614,0,0],
            [3,-3,-1,-2],
            [6,17,9,-3],
            [-3,-9,-5,2],
            [0,6,3,-1],
            [-127,21,9,55],
            [3,5,0,0],
            [-6,-10,-4,3],
            [5,0,0,0],
            [16,9,4,-7],
            [3,0,0,-2],
            [0,22,0,0],
            [0,19,10,0],
            [7,0,0,-4],
            [0,-5,-2,0],
            [0,3,1,0],
            [-9,3,1,4],
            [17,0,0,-7],
            [0,-3,-2,-1],
            [-20,34,0,0],
            [-10,0,1,5],
            [-4,0,0,2],
            [22,-87,0,0],
            [-4,0,0,2],
            [-3,-6,-2,1],
            [-16,-3,-1,7],
            [0,-3,-2,0],
            [4,0,0,0],
            [-68,39,0,0],
            [27,0,0,-14],
            [0,-4,0,0],
            [-25,0,0,0],
            [-12,-3,-2,6],
            [3,0,0,-1],
            [3,66,29,-1],
            [490,0,0,-213],
            [-22,93,49,12],
            [-7,28,15,4],
            [-3,13,7,2],
            [-46,14,0,0],
            [-5,0,0,0],
            [2,1,0,0],
            [0,-3,0,0],
            [-28,0,0,15],
            [5,0,0,-2],
            [0,3,0,0],
            [-11,0,0,5],
            [0,3,1,0],
            [-3,0,0,1],
            [25,106,57,-13],
            [5,21,11,-3],
            [1485,0,0,0],
            [-7,-32,-17,4],
            [0,5,3,0],
            [-6,-3,-2,3],
            [30,-6,-2,-13],
            [-4,4,0,0],
            [-19,0,0,10],
            [0,4,2,-1],
            [0,3,0,0],
            [4,0,0,-2],
            [0,-3,-1,0],
            [-3,0,0,0],
            [5,3,1,-2],
            [0,11,0,0],
            [118,0,0,-52],
            [0,-5,-3,0],
            [-28,36,0,0],
            [5,-5,0,0],
            [14,-59,-31,-8],
            [0,9,5,1],
            [-458,0,0,198],
            [0,-45,-20,0],
            [9,0,0,-5],
            [0,-3,0,0],
            [0,-4,-2,-1],
            [11,0,0,-6],
            [6,0,0,-2],
            [-16,23,0,0],
            [0,-4,-2,0],
            [-5,0,0,2],
            [-166,269,0,0],
            [15,0,0,-8],
            [10,0,0,-4],
            [-78,45,0,0],
            [0,-5,-2,0],
            [7,0,0,-4],
            [-5,328,0,0],
            [3,0,0,-2],
            [5,0,0,-2],
            [0,3,1,0],
            [-3,0,0,0],
            [-3,0,0,0],
            [0,-4,-2,0],
            [-1223,-26,0,0],
            [0,7,3,0],
            [3,0,0,0],
            [0,3,2,0],
            [-6,20,0,0],
            [-368,0,0,0],
            [-75,0,0,0],
            [11,0,0,-6],
            [3,0,0,-2],
            [-3,0,0,1],
            [-13,-30,0,0],
            [21,3,0,0],
            [-3,0,0,1],
            [-4,0,0,2],
            [8,-27,0,0],
            [-19,-11,0,0],
            [-4,0,0,2],
            [0,5,2,0],
            [-6,0,0,2],
            [-8,0,0,0],
            [-1,0,0,0],
            [-14,0,0,6],
            [6,0,0,0],
            [-74,0,0,32],
            [0,-3,-1,0],
            [4,0,0,-2],
            [8,11,0,0],
            [0,3,2,0],
            [-262,0,0,114],
            [0,-4,0,0],
            [-7,0,0,4],
            [0,-27,-12,0],
            [-19,-8,-4,8],
            [202,0,0,-87],
            [-8,35,19,5],
            [0,4,2,0],
            [16,-5,0,0],
            [5,0,0,-3],
            [0,-3,0,0],
            [1,0,0,0],
            [-35,-48,-21,15],
            [-3,-5,-2,1],
            [6,0,0,-3],
            [3,0,0,-1],
            [0,-5,0,0],
            [12,55,29,-6],
            [0,5,3,0],
            [-598,0,0,0],
            [-3,-13,-7,1],
            [-5,-7,-3,2],
            [3,0,0,-1],
            [5,-7,0,0],
            [4,0,0,-2],
            [16,-6,0,0],
            [8,-3,0,0],
            [8,-31,-16,-4],
            [0,3,1,0],
            [113,0,0,-49],
            [0,-24,-10,0],
            [4,0,0,-2],
            [27,0,0,0],
            [-3,0,0,1],
            [0,-4,-2,0],
            [5,0,0,-2],
            [0,-3,0,0],
            [-13,0,0,6],
            [5,0,0,-2],
            [-18,-10,-4,8],
            [-4,-28,0,0],
            [-5,6,3,2],
            [-3,0,0,1],
            [-5,-9,-4,2],
            [17,0,0,-7],
            [11,4,0,0],
            [0,-6,-2,0],
            [83,15,0,0],
            [-4,0,0,2],
            [0,-114,-49,0],
            [117,0,0,-51],
            [-5,19,10,2],
            [-3,0,0,0],
            [-3,0,0,2],
            [0,-3,-1,0],
            [3,0,0,0],
            [0,-6,-2,0],
            [393,3,0,0],
            [-4,21,11,2],
            [-6,0,-1,3],
            [-3,8,4,1],
            [8,0,0,0],
            [18,-29,-13,-8],
            [8,34,18,-4],
            [89,0,0,0],
            [3,12,6,-1],
            [54,-15,-7,-24],
            [0,3,0,0],
            [3,0,0,-1],
            [0,35,0,0],
            [-154,-30,-13,67],
            [15,0,0,0],
            [0,4,2,0],
            [0,9,0,0],
            [80,-71,-31,-35],
            [0,-20,-9,0],
            [11,5,2,-5],
            [61,-96,-42,-27],
            [14,9,4,-6],
            [-11,-6,-3,5],
            [0,-3,-1,0],
            [123,-415,-180,-53],
            [0,0,0,-35],
            [-5,0,0,0],
            [7,-32,-17,-4],
            [0,-9,-5,0],
            [0,-4,2,0],
            [-89,0,0,38],
            [0,-86,-19,-6],
            [0,0,-19,6],
            [-123,-416,-180,53],
            [0,-3,-1,0],
            [12,-6,-3,-5],
            [-13,9,4,6],
            [0,-15,-7,0],
            [3,0,0,-1],
            [-62,-97,-42,27],
            [-11,5,2,5],
            [0,-19,-8,0],
            [-3,0,0,1],
            [0,4,2,0],
            [0,3,0,0],
            [0,4,2,0],
            [-85,-70,-31,37],
            [163,-12,-5,-72],
            [-63,-16,-7,28],
            [-21,-32,-14,9],
            [0,-3,-1,0],
            [3,0,0,-2],
            [0,8,0,0],
            [3,10,4,-1],
            [3,0,0,-1],
            [0,-7,-3,0],
            [0,-4,-2,0],
            [6,19,0,0],
            [5,-173,-75,-2],
            [0,-7,-3,0],
            [7,-12,-5,-3],
            [-3,0,0,2],
            [3,-4,-2,-1],
            [74,0,0,-32],
            [-3,12,6,2],
            [26,-14,-6,-11],
            [19,0,0,-8],
            [6,24,13,-3],
            [83,0,0,0],
            [0,-10,-5,0],
            [11,-3,-1,-5],
            [3,0,1,-1],
            [3,0,0,-1],
            [-4,0,0,0],
            [5,-23,-12,-3],
            [-339,0,0,147],
            [0,-10,-5,0],
            [5,0,0,0],
            [3,0,0,-1],
            [0,-4,-2,0],
            [18,-3,0,0],
            [9,-11,-5,-4],
            [-8,0,0,4],
            [3,0,0,-1],
            [0,9,0,0],
            [6,-9,-4,-2],
            [-4,-12,0,0],
            [67,-91,-39,-29],
            [30,-18,-8,-13],
            [0,0,0,0],
            [0,-114,-50,0],
            [0,0,0,23],
            [517,16,7,-224],
            [0,-7,-3,0],
            [143,-3,-1,-62],
            [29,0,0,-13],
            [-4,0,0,2],
            [-6,0,0,3],
            [5,12,5,-2],
            [-25,0,0,11],
            [-3,0,0,1],
            [0,4,2,0],
            [-22,12,5,10],
            [50,0,0,-22],
            [0,7,4,0],
            [0,3,1,0],
            [-4,4,2,2],
            [-5,-11,-5,2],
            [0,4,2,0],
            [4,17,9,-2],
            [59,0,0,0],
            [0,-4,-2,0],
            [-8,0,0,4],
            [-3,0,0,0],
            [4,-15,-8,-2],
            [370,-8,0,-160],
            [0,0,-3,0],
            [0,3,1,0],
            [-6,3,1,3],
            [0,6,0,0],
            [-10,0,0,4],
            [0,9,4,0],
            [4,17,7,-2],
            [34,0,0,-15],
            [0,5,3,0],
            [-5,0,0,2],
            [-37,-7,-3,16],
            [3,13,7,-2],
            [40,0,0,0],
            [0,-3,-2,0],
            [-184,-3,-1,80],
            [-3,0,0,1],
            [-3,0,0,0],
            [0,-10,-6,-1],
            [31,-6,0,-13],
            [-3,-32,-14,1],
            [-7,0,0,3],
            [0,-8,-4,0],
            [3,-4,0,0],
            [0,4,0,0],
            [0,3,1,0],
            [19,-23,-10,2],
            [0,0,0,-10],
            [0,3,2,0],
            [0,9,5,-1],
            [28,0,0,0],
            [0,-7,-4,0],
            [8,-4,0,-4],
            [0,0,-2,0],
            [0,3,0,0],
            [-3,0,0,1],
            [-9,0,1,4],
            [3,12,5,-1],
            [17,-3,-1,0],
            [0,7,4,0],
            [19,0,0,0],
            [0,-5,-3,0],
            [14,-3,0,-1],
            [0,0,-1,0],
            [0,0,0,-5],
            [0,5,3,0],
            [13,0,0,0],
            [0,-3,-2,0],
            [2,9,4,3],
            [0,0,0,-4],
            [8,0,0,0],
            [0,4,2,0],
            [6,0,0,-3],
            [6,0,0,0],
            [0,3,1,0],
            [5,0,0,-2],
            [3,0,0,-1],
            [-3,0,0,0],
            [6,0,0,0],
            [7,0,0,0],
            [-4,0,0,0],
            [4,0,0,0],
            [6,0,0,0],
            [0,-4,0,0],
            [0,-4,0,0],
            [5,0,0,0],
            [-3,0,0,0],
            [4,0,0,0],
            [-5,0,0,0],
            [4,0,0,0],
            [0,3,0,0],
            [13,0,0,0],
            [21,11,0,0],
            [0,-5,0,0],
            [0,-5,-2,0],
            [0,5,3,0],
            [0,-5,0,0],
            [-3,0,0,2],
            [20,10,0,0],
            [-34,0,0,0],
            [-19,0,0,0],
            [3,0,0,-2],
            [-3,0,0,1],
            [-6,0,0,3],
            [-4,0,0,0],
            [3,0,0,0],
            [3,0,0,0],
            [4,0,0,0],
            [3,0,0,-1],
            [6,0,0,-3],
            [-8,0,0,3],
            [0,3,1,0],
            [-3,0,0,0],
            [0,-3,-2,0],
            [126,-63,-27,-55],
            [-5,0,1,2],
            [-3,28,15,2],
            [5,0,1,-2],
            [0,9,4,1],
            [0,9,4,-1],
            [-126,-63,-27,55],
            [3,0,0,-1],
            [21,-11,-6,-11],
            [0,-4,0,0],
            [-21,-11,-6,11],
            [-3,0,0,1],
            [0,3,1,0],
            [8,0,0,-4],
            [-6,0,0,3],
            [-3,0,0,1],
            [3,0,0,-1],
            [-3,0,0,1],
            [-5,0,0,2],
            [24,-12,-5,-11],
            [0,3,1,0],
            [0,3,1,0],
            [0,3,2,0],
            [-24,-12,-5,10],
            [4,0,-1,-2],
            [13,0,0,-6],
            [7,0,0,-3],
            [3,0,0,-1],
            [3,0,0,-1]]
    
    #给定日期与参考日期之间的时间间隔，单位：儒略世纪数
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #日月章动
    #基本参数（Delaunay）

    #月亮的平均近点角
    EL=pymFal03(T)
    
    #太阳的平均近点角
    A=1287104.79305+T*(129596581.0481+T*(-0.5532+\
        T*(0.000136+T*(-0.00001149))))
    ELP=(A%TURNAS)*DAS2R
    
    #月亮的平黄经减去升交点的平黄经
    F=pymFaf03(T)
    
    #月亮相对于太阳的平均距角
    B=1072260.70369+T*(1602961601.2090+T*(-6.3706+\
        T*(0.006593+T*(-0.00003169))))
    D=(B%TURNAS)*DAS2R
    
    #月亮升交点的平黄经
    OM=pymFaom03(T)
    
    #初始化章动量
    DP=0.0
    DE=0.0
    
    #从后到前对日月章动的各项求和
    for I in range(NLS-1,-1,-1):
        C=float(NALS[I][0])*EL+float(NALS[I][1])*ELP+\
            float(NALS[I][2])*F+float(NALS[I][3])*D+float(NALS[I][4])*OM
        ARG=C%D2PI
        SARG=ma.sin(ARG)
        CARG=ma.cos(ARG)
    
        #Term.
        DP=DP+(CLS[I][0]+CLS[I][1]*T)*SARG+CLS[I][2]*CARG
        DE=DE+(CLS[I][3]+CLS[I][4]*T)*CARG+CLS[I][5]*SARG
    
    #从0.1微弧秒单位转换为弧度。
    DPSILS=DP*U2R
    DEPSLS=DE*U2R
    
    #行星章动

    #月亮的平近点角(MHB2000).
    AL=(2.35555598+8328.6914269554*T)%D2PI
    
    #太阳的平近点角(MHB2000).
    ALSU=(6.24006013+628.301955*T)%D2PI
    
    #月亮的平黄经减去升交点的平黄经(MHB2000).
    AF=(1.627905234+8433.466158131*T)%D2PI

    #月亮相对于太阳的平均距角(MHB2000).
    AD=(5.198466741+7771.3771468121*T)%D2PI

    #月亮升交点的平黄经(MHB2000).
    AOM=(2.18243920-33.757045*T)%D2PI

    #黄经的一般累计进动(IERS 2003).
    APA=pymFapa03(T)
    
    #行星的平黄经，水星到天王星(IERS 2003).
    ALME=pymFame03(T)
    ALVE=pymFave03(T)
    ALEA=pymFae03(T)
    ALMA=pymFama03(T)
    ALJU=pymFaju03(T)
    ALSA=pymFasa03(T)
    ALUR=pymFaur03(T)
    
    #海王星的平黄经(MHB2000).
    ALNE=(5.321159000+3.8127774000*T)%D2PI
    
    #初始化章动量
    DP=0.0
    DE=0.0
    
    #从后到前对行星章动的各项求和
    for I in range(NPL-1,-1,-1):
        D=float(NAPL[I][0])*AL+float(NAPL[I][1])*ALSU+float(NAPL[I][2])*AF+\
            float(NAPL[I][3])*AD+float(NAPL[I][4])*AOM+float(NAPL[I][5])*ALME+\
            float(NAPL[I][6])*ALVE+float(NAPL[I][7])*ALEA+\
            float(NAPL[I][8])*ALMA+float(NAPL[I][9])*ALJU+\
            float(NAPL[I][10])*ALSA+float(NAPL[I][11])*ALUR+\
            float(NAPL[I][12])*ALNE+float(NAPL[I][13])*APA
        ARG=D%D2PI
        SARG=ma.sin(ARG)
        CARG=ma.cos(ARG)
    
        #Term.
        DP=DP+float(ICPL[I][0])*SARG+float(ICPL[I][1])*CARG
        DE=DE+float(ICPL[I][2])*SARG+float(ICPL[I][3])*CARG
    
    #从0.1微弧秒单位转换为弧度。
    DPSIPL=DP*U2R
    DEPSPL=DE*U2R

    #日月章动和行星章动相加
    DPSI=DPSILS+DPSIPL
    DEPS=DEPSLS+DEPSPL
    
    return(DPSI,DEPS)

def pymFaur03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Uranus.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0

    Returns
    -------
    function value : float
        mean longitude of Uranus, radians

    '''
    FAUR=(5.481293872+7.4781598567*T)%(2*ma.pi)
    
    return(FAUR)

def pymFasa03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Saturn.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0

    Returns
    -------
    function value : float
        mean longitude of Saturn, radians

    '''
    FASA=(0.874016757+21.3299104960*T)%(2*ma.pi)
    
    return(FASA)

def pymFaju03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Jupiter.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0
 
    Returns
    -------
    function value : float
        mean longitude of Jupiter, radians

    '''
    FAJU=(0.599546497+52.9690962641*T)%(2*ma.pi)

    return(FAJU)

def pymFama03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Mars.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0
 
    Returns
    -------
    function value : float
        mean longitude of Mars, radians

    '''
    FAMA=(6.203480913+334.0612426700*T)%(2*ma.pi)
    
    return(FAMA)

def pymFae03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Earth.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0
 
    Returns
    -------
    function value : float
        mean longitude of Earth, radians

    '''
    FAE=(1.753470314+628.3075849991*T)%(2*ma.pi)    
    
    return(FAE)

def pymFave03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Venus.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0

    Returns
    -------
    function value : float
        mean longitude of Venus, radians

    '''
    FAVE=(3.176146697+1021.3285546211*T)%(2*ma.pi)
    
    return(FAVE)

def pymFame03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Mercury.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0
 
    Returns
    -------
    function value : float
        mean longitude of Mercury, radians

    '''
    
    FAME=(4.402608842+2608.7903141574*T)%(2*ma.pi)
    
    return(FAME)

def pymFapa03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    general accumulated precession in longitude.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0

    Returns
    -------
    function value : float
        general precession in longitude, radians

    '''
    FAPA=(0.024381750+0.00000538691*T)*T
    
    return(FAPA)

def pymFaom03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of the Moon's ascending node.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0

    Returns
    -------
    function value : float
        Omega, radians 

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #一个圆内的总角秒数
    TURNAS=1296000.0
    
    #月亮升交点的平黄经
    A=450160.398036+T*(-6962890.5431+T*(7.4722+\
        T*(0.007702+T*(-0.00005939))))
    FAOM=(A%TURNAS)*DAS2R
    
    return(FAOM)

def pymFaf03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of the Moon minus mean longitude of the ascending
    node.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0
 
    Returns
    -------
    function value : float
        F, radians

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #一个圆内的总角秒数
    TURNAS=1296000.0
    
    #月球的平黄经减去其上升点的平黄经 
    A=335779.526232+T*(1739527262.8478+T*(-12.7512+\
        T*(-0.001037+T*(0.00000417))))
    FAF=(A%TURNAS)*DAS2R
    
    return(FAF)

def pymFal03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean anomaly of the Moon.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0
 
    Returns
    -------
    function value : float
        l, radians

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #一个圆内的总角秒数
    TURNAS=1296000.0
    
    #月亮的平近点角
    A=485868.249036+T*(1717915923.2178+T*(31.8792+\
        T*(0.051635+T*(-0.00024470))))
    FAL=(A%TURNAS)*DAS2R
    
    return(FAL)

def pymEect00(DATE1,DATE2):
    '''
    Equation of the equinoxes complementary terms, consistent with
    IAU 2000 resolutions.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        complementary terms

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6

    #参考历元(J2000.0), JD
    DJ00=2451545.0

    #1儒略世纪的天数
    DJC=36525.0
    
    #基本参数
    FA=[0 for i in range(14)]
    
    # EE补充项部分
    
    #项数
    NE0=33
    NE1=1
    
    #Argument coefficients for t^0
    KE0=[[0,0,0,0,1,0,0,0],
            [0,0,0,0,2,0,0,0],
            [0,0,2,-2,3,0,0,0],
            [0,0,2,-2,1,0,0,0],
            [0,0,2,-2,2,0,0,0],
            [0,0,2,0,3,0,0,0],
            [0,0,2,0,1,0,0,0],
            [0,0,0,0,3,0,0,0],
            [0,1,0,0,1,0,0,0],
            [0,1,0,0,-1,0,0,0],
            [1,0,0,0,-1,0,0,0],
            [1,0,0,0,1,0,0,0],
            [0,1,2,-2,3,0,0,0],
            [0,1,2,-2,1,0,0,0],
            [0,0,4,-4,4,0,0,0],
            [0,0,1,-1,1,-8,12,0],
            [0,0,2,0,0,0,0,0],
            [0,0,2,0,2,0,0,0],
            [1,0,2,0,3,0,0,0],
            [1,0,2,0,1,0,0,0],
            [0,0,2,-2,0,0,0,0],
            [0,1,-2,2,-3,0,0,0],
            [0,1,-2,2,-1,0,0,0],
            [0,0,0,0,0,8,-13,-1],
            [0,0,0,2,0,0,0,0],
            [2,0,-2,0,-1,0,0,0],
            [1,0,0,-2,1,0,0,0],
            [0,1,2,-2,2,0,0,0],
            [1,0,0,-2,-1,0,0,0],
            [0,0,4,-2,4,0,0,0],
            [0,0,2,-2,4,0,0,0],
            [1,0,-2,0,-3,0,0,0],
            [1,0,-2,0,-1,0,0,0]]
    
    #Argument coefficients for t^1
    KE1=[[0,0,0,0,1,0,0,0]]
    
    #Sine and cosine coefficients for t^0
    SE0=[[+2640.96e-6,-0.39e-6],
            [+63.52e-6,-0.02e-6],
            [+11.75e-6,+0.01e-6],
            [+11.21e-6,+0.01e-6],
            [-4.55e-6,+0.00e-6],
            [+2.02e-6,+0.00e-6],
            [+1.98e-6,+0.00e-6],
            [-1.72e-6,+0.00e-6],
            [-1.41e-6,-0.01e-6],
            [-1.26e-6,-0.01e-6],
            [-0.63e-6,+0.00e-6],
            [-0.63e-6,+0.00e-6],
            [+0.46e-6,+0.00e-6],
            [+0.45e-6,+0.00e-6],
            [+0.36e-6,+0.00e-6],
            [-0.24e-6,-0.12e-6],
            [+0.32e-6,+0.00e-6],
            [+0.28e-6,+0.00e-6],
            [+0.27e-6,+0.00e-6],
            [+0.26e-6,+0.00e-6],
            [-0.21e-6,+0.00e-6],
            [+0.19e-6,+0.00e-6],
            [+0.18e-6,+0.00e-6],
            [-0.10e-6,+0.05e-6],
            [+0.15e-6,+0.00e-6],
            [-0.14e-6,+0.00e-6],
            [+0.14e-6,+0.00e-6],
            [-0.14e-6,+0.00e-6],
            [+0.14e-6,+0.00e-6],
            [+0.13e-6,+0.00e-6],
            [-0.11e-6,+0.00e-6],
            [+0.11e-6,+0.00e-6],
            [+0.11e-6,+0.00e-6]]
    
    #Sine and cosine coefficients for t^1
    SE1=[[-0.87e-6,+0.00e-6]]
    
    #给定日期和参考历元之间的时间间隔儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #基本参数(依据IERS Conventions 2003)
    #月亮的平近点角
    FA[0]=pymFal03(T)
    
    #太阳的平近点角
    FA[1]=pymFalp03(T)

    #月亮的平黄经减去升交点黄经
    FA[2]=pymFaf03(T)

    #月亮到太阳的平均距角
    FA[3]=pymFad03(T)

    #月亮的升交点平黄经
    FA[4]=pymFaom03(T)

    #金星的平黄经
    FA[5]=pymFave03(T)

    #地球的平黄经
    FA[6]=pymFae03(T)

    #黄经上的累计岁差
    FA[7]=pymFapa03(T)
    
    #估计EE补充项
    S0=0.0
    S1=0.0
    
    for i in range(NE0-1,-1,-1):
        A=0.0
        for j in range(8):
            A=A+float(KE0[i][j])*FA[j]
        S0=S0+(SE0[i][0]*ma.sin(A)+SE0[i][1]*ma.cos(A))
    
    for i in range(NE1-1,-1,-1):
        A=0.0
        for j in range(8):
            A=A+float(KE1[i][j])*FA[j]
        S1=S1+(SE1[i][0]*ma.sin(A)+SE1[i][1]*ma.cos(A))
    
    EECT=(S0+S1*T)*DAS2R    
    
    return(EECT)

def pymFad03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean elongation of the Moon from the Sun.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0
 
    Returns
    -------
    function value : float
        D, radians

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #一个圆内的总角秒数
    TURNAS=1296000.0
    
    #月亮到太阳的平均距角 
    A=1072260.703692+T*(1602961601.2090+T*(-6.3706+\
            T*(0.006593+T*(-0.00003169))))
    FAD=(A%TURNAS)*DAS2R
    
    return(FAD)

def pymFalp03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean anomaly of the Sun.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0
 
    Returns
    -------
    function value : float
        l', radians

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #一个圆内的总角秒数
    TURNAS=1296000.0
    
    #太阳的平近点角
    A=1287104.793048+T*(129596581.0481+T*(-0.5532+\
        T*(0.000136+T*(-0.00001149))))
    FALP=(A%TURNAS)*DAS2R

    return(FALP)

def pymEe00(DATE1,DATE2,EPSA,DPSI):
    '''
    The equation of the equinoxes, compatible with IAU 2000 resolutions,
    given the nutation in longitude and the mean obliquity.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date    
    epsa : float
        mean obliquity    
    dpsi : float
        nutation in longitude

    Returns
    -------
    function value : float
        equation of the equinoxes

    '''
    #赤经章动
    EE=DPSI*ma.cos(EPSA)+pymEect00(DATE1,DATE2)
    
    return(EE)

def pymEe00a(DATE1,DATE2):
    '''
    Equation of the equinoxes, compatible with IAU 2000 resolutions.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        equation of the equinoxes

    '''
    #IAU 2000岁差速率改正.
    DPSIPR,DEPSPR=pymPr00(DATE1,DATE2)
    
    #平均倾角，与IAU 2000岁差章动模型一致.
    EPSA=pymObl80(DATE1,DATE2)+DEPSPR
    
    #经度上的章动
    DPSI,DEPS=pymNut00a(DATE1,DATE2)
    
    #赤经章动
    EE=pymEe00(DATE1,DATE2,EPSA,DPSI)
    
    return(EE)

def pymGst00a(UTA,UTB,TTA,TTB):
    '''
    Greenwich apparent sidereal time (consistent with IAU 2000
    resolutions).

    Parameters
    ----------
    uta : float
        UT1 as a 2-part Julian Date    
    utb : float
        UT1 as a 2-part Julian Date    
    tta : float
        TT as a 2-part Julian Date    
    ttb : float
        TT as a 2-part Julian Date

    Returns
    -------
    GST : float
        Greenwich apparent sidereal time (radians)

    '''
    A=pymGmst00(UTA,UTB,TTA,TTB)
    B=pymEe00a(TTA,TTB)
    GST=pymAnp(A+B)
    
    return(GST)

def pymNut00b(DATE1,DATE2):
    '''
    Nutation, IAU 2000B model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    dpsi : float
        nutation, luni-solar + planetary    
    deps : float
        nutation, luni-solar + planetary

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6

    #1毫角秒对应的弧度
    DMAS2R=DAS2R/1e3
    
    #360度对应的角秒
    TURNAS=1296000.0
    
    #2Pi
    D2PI=6.283185307179586476925287
    
    #0.1微角秒对应的弧度
    U2R=DAS2R/1e7
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0

    #1儒略世纪对应的天数
    DJC=36525.0
    
    #日月章动模型
    #模型中的项数
    NLS=77
    
    #基本参数的系数
    NALS=[]
    
    #经度和倾角系数
    CLS=[]
    
    #修正的偏移代替行星项，弧度制
    DPPLAN=-0.135*DMAS2R
    DEPLAN=+0.388*DMAS2R
    
    #各项参数的系数表
    
    # *  Luni-Solar argument multipliers:
    # *
    # *               L     L'    F     D     Om
    NALS=[[0,0,0,0,1],
            [0,0,2,-2,2],
            [0,0,2,0,2],
            [0,0,0,0,2],
            [0,1,0,0,0],
            [0,1,2,-2,2],
            [1,0,0,0,0],
            [0,0,2,0,1],
            [1,0,2,0,2],
            [0,-1,2,-2,2],
            [0,0,2,-2,1],
            [-1,0,2,0,2],
            [-1,0,0,2,0],
            [1,0,0,0,1],
            [-1,0,0,0,1],
            [-1,0,2,2,2],
            [1,0,2,0,1],
            [-2,0,2,0,1],
            [0,0,0,2,0],
            [0,0,2,2,2],
            [0,-2,2,-2,2],
            [-2,0,0,2,0],
            [2,0,2,0,2],
            [1,0,2,-2,2],
            [-1,0,2,0,1],
            [2,0,0,0,0],
            [0,0,2,0,0],
            [0,1,0,0,1],
            [-1,0,0,2,1],
            [0,2,2,-2,2],
            [0,0,-2,2,0],
            [1,0,0,-2,1],
            [0,-1,0,0,1],
            [-1,0,2,2,1],
            [0,2,0,0,0],
            [1,0,2,2,2],
            [-2,0,2,0,0],
            [0,1,2,0,2],
            [0,0,2,2,1],
            [0,-1,2,0,2],
            [0,0,0,2,1],
            [1,0,2,-2,1],
            [2,0,2,-2,2],
            [-2,0,0,2,1],
            [2,0,2,0,1],
            [0,-1,2,-2,1],
            [0,0,0,-2,1],
            [-1,-1,0,2,0],
            [2,0,0,-2,1],
            [1,0,0,2,0],
            [0,1,2,-2,1],
            [1,-1,0,0,0],
            [-2,0,2,0,2],
            [3,0,2,0,2],
            [0,-1,0,2,0],
            [1,-1,2,0,2],
            [0,0,0,1,0],
            [-1,-1,2,2,2],
            [-1,0,2,0,0],
            [0,-1,2,2,2],
            [-2,0,0,0,1],
            [1,1,2,0,2],
            [2,0,0,0,1],
            [-1,1,0,1,0],
            [1,1,0,0,0],
            [1,0,2,0,0],
            [-1,0,2,-2,1],
            [1,0,0,0,2],
            [-1,0,0,1,0],
            [0,0,2,1,2],
            [-1,0,2,4,2],
            [-1,1,0,1,1],
            [0,-2,2,-2,1],
            [1,0,2,2,1],
            [-2,0,2,2,2],
            [-1,0,0,0,2],
            [1,1,2,-2,2]]
    
    # *  Luni-Solar nutation coefficients, unit 1e-7 arcsec:
    # *  longitude (sin, t*sin, cos), obliquity (cos, t*cos, sin)
    CLS=[[-172064161e0,-174666e0,33386e0,92052331e0,9086e0,15377e0],
            [-13170906e0,-1675e0,-13696e0,5730336e0,-3015e0,-4587e0],
            [-2276413e0,-234e0,2796e0,978459e0,-485e0,1374e0],
            [2074554e0,207e0,-698e0,-897492e0,470e0,-291e0],
            [1475877e0,-3633e0,11817e0,73871e0,-184e0,-1924e0],
            [-516821e0,1226e0,-524e0,224386e0,-677e0,-174e0],
            [711159e0,73e0,-872e0,-6750e0,0e0,358e0],
            [-387298e0,-367e0,380e0,200728e0,18e0,318e0],
            [-301461e0,-36e0,816e0,129025e0,-63e0,367e0],
            [215829e0,-494e0,111e0,-95929e0,299e0,132e0],
            [128227e0,137e0,181e0,-68982e0,-9e0,39e0],
            [123457e0,11e0,19e0,-53311e0,32e0,-4e0],
            [156994e0,10e0,-168e0,-1235e0,0e0,82e0],
            [63110e0,63e0,27e0,-33228e0,0e0,-9e0],
            [-57976e0,-63e0,-189e0,31429e0,0e0,-75e0],
            [-59641e0,-11e0,149e0,25543e0,-11e0,66e0],
            [-51613e0,-42e0,129e0,26366e0,0e0,78e0],
            [45893e0,50e0,31e0,-24236e0,-10e0,20e0],
            [63384e0,11e0,-150e0,-1220e0,0e0,29e0],
            [-38571e0,-1e0,158e0,16452e0,-11e0,68e0],
            [32481e0,0e0,0e0,-13870e0,0e0,0e0],
            [-47722e0,0e0,-18e0,477e0,0e0,-25e0],
            [-31046e0,-1e0,131e0,13238e0,-11e0,59e0],
            [28593e0,0e0,-1e0,-12338e0,10e0,-3e0],
            [20441e0,21e0,10e0,-10758e0,0e0,-3e0],
            [29243e0,0e0,-74e0,-609e0,0e0,13e0],
            [25887e0,0e0,-66e0,-550e0,0e0,11e0],
            [-14053e0,-25e0,79e0,8551e0,-2e0,-45e0],
            [15164e0,10e0,11e0,-8001e0,0e0,-1e0],
            [-15794e0,72e0,-16e0,6850e0,-42e0,-5e0],
            [21783e0,0e0,13e0,-167e0,0e0,13e0],
            [-12873e0,-10e0,-37e0,6953e0,0e0,-14e0],
            [-12654e0,11e0,63e0,6415e0,0e0,26e0],
            [-10204e0,0e0,25e0,5222e0,0e0,15e0],
            [16707e0,-85e0,-10e0,168e0,-1e0,10e0],
            [-7691e0,0e0,44e0,3268e0,0e0,19e0],
            [-11024e0,0e0,-14e0,104e0,0e0,2e0],
            [7566e0,-21e0,-11e0,-3250e0,0e0,-5e0],
            [-6637e0,-11e0,25e0,3353e0,0e0,14e0],
            [-7141e0,21e0,8e0,3070e0,0e0,4e0],
            [-6302e0,-11e0,2e0,3272e0,0e0,4e0],
            [5800e0,10e0,2e0,-3045e0,0e0,-1e0],
            [6443e0,0e0,-7e0,-2768e0,0e0,-4e0],
            [-5774e0,-11e0,-15e0,3041e0,0e0,-5e0],
            [-5350e0,0e0,21e0,2695e0,0e0,12e0],
            [-4752e0,-11e0,-3e0,2719e0,0e0,-3e0],
            [-4940e0,-11e0,-21e0,2720e0,0e0,-9e0],
            [7350e0,0e0,-8e0,-51e0,0e0,4e0],
            [4065e0,0e0,6e0,-2206e0,0e0,1e0],
            [6579e0,0e0,-24e0,-199e0,0e0,2e0],
            [3579e0,0e0,5e0,-1900e0,0e0,1e0],
            [4725e0,0e0,-6e0,-41e0,0e0,3e0],
            [-3075e0,0e0,-2e0,1313e0,0e0,-1e0],
            [-2904e0,0e0,15e0,1233e0,0e0,7e0],
            [4348e0,0e0,-10e0,-81e0,0e0,2e0],
            [-2878e0,0e0,8e0,1232e0,0e0,4e0],
            [-4230e0,0e0,5e0,-20e0,0e0,-2e0],
            [-2819e0,0e0,7e0,1207e0,0e0,3e0],
            [-4056e0,0e0,5e0,40e0,0e0,-2e0],
            [-2647e0,0e0,11e0,1129e0,0e0,5e0],
            [-2294e0,0e0,-10e0,1266e0,0e0,-4e0],
            [2481e0,0e0,-7e0,-1062e0,0e0,-3e0],
            [2179e0,0e0,-2e0,-1129e0,0e0,-2e0],
            [3276e0,0e0,1e0,-9e0,0e0,0e0],
            [-3389e0,0e0,5e0,35e0,0e0,-2e0],
            [3339e0,0e0,-13e0,-107e0,0e0,1e0],
            [-1987e0,0e0,-6e0,1073e0,0e0,-2e0],
            [-1981e0,0e0,0e0,854e0,0e0,0e0],
            [4026e0,0e0,-353e0,-553e0,0e0,-139e0],
            [1660e0,0e0,-5e0,-710e0,0e0,-2e0],
            [-1521e0,0e0,9e0,647e0,0e0,4e0],
            [1314e0,0e0,0e0,-700e0,0e0,0e0],
            [-1283e0,0e0,0e0,672e0,0e0,0e0],
            [-1331e0,0e0,8e0,663e0,0e0,4e0],
            [1383e0,0e0,-2e0,-594e0,0e0,-2e0],
            [1405e0,0e0,4e0,-610e0,0e0,2e0],
            [1290e0,0e0,0e0,-556e0,0e0,0e0]]
    
    #给定日期与参考历元之间的时间间隔，儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #日月章动
    
    #Fundamental (Delaunay) arguments from Simon et al. (1994)
    
    #月亮的平近点角
    A=(485868.249036+(+1717915923.2178)*T)%TURNAS
    EL=A*DAS2R

    #太阳的平近点角
    A=(1287104.79305+(+129596581.0481)*T)%TURNAS
    ELP=A*DAS2R

    #月亮的平均升交角距
    A=(335779.526232+(+1739527262.8478)*T)%TURNAS
    F=A*DAS2R

    #月亮到太阳的平均角距
    A=(1072260.70369+(+1602961601.2090)*T)%TURNAS
    D=A*DAS2R

    #月亮的升交点平黄经
    A=(450160.398036+(-6962890.5431)*T)%TURNAS
    OM=A*DAS2R
    
    #初始化章动量
    DP=0.0
    DE=0.0
    
    #日月章动级数求和(倒序)
    for i in range(NLS-1,-1,-1):
        B=float(NALS[i][0])*EL+float(NALS[i][1])*ELP+\
           float(NALS[i][2])*F+float(NALS[i][3])*D+\
           float(NALS[i][4])*OM
        ARG=B%D2PI
        SARG=ma.sin(ARG)
        CARG=ma.cos(ARG)
    
        #Term.
        DP=DP+(CLS[i][0]+CLS[i][1]*T)*SARG+CLS[i][2]*CARG
        DE=DE+(CLS[i][3]+CLS[i][4]*T)*CARG+CLS[i][5]*SARG
    
    #从0.1微弧秒单位转换为弧度。
    DPSILS=DP*U2R
    DEPSLS=DE*U2R
    
    #代替行星章动项
    
    #修正偏移量，以纠正截断系数中缺失的项
    DPSIPL=DPPLAN
    DEPSPL=DEPLAN
    
    #日月和行星章动相加
    DPSI=DPSILS+DPSIPL
    DEPS=DEPSLS+DEPSPL
    
    return(DPSI,DEPS)


def pymEe00b(DATE1,DATE2):
    '''
    Equation of the equinoxes, compatible with IAU 2000 resolutions but
    using the truncated nutation model IAU 2000B.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        equation of the equinoxes

    '''
    #IAU 2000岁差速率改正.
    DPSIPR,DEPSPR=pymPr00(DATE1,DATE2)
    
    #平均倾角，与IAU 2000岁差章动模型一致.
    EPSA=pymObl80(DATE1,DATE2)+DEPSPR
    
    #经度上的章动，采用删减后的章动模型IAU 2000B  
    DPSI,DEPS=pymNut00b(DATE1,DATE2)
    
    #赤经章动
    EE=pymEe00(DATE1,DATE2,EPSA,DPSI)
    
    return(EE)

def pymGst00b(UTA,UTB):
    '''
    Greenwich apparent sidereal time (consistent with IAU 2000
    resolutions but using the truncated nutation model IAU 2000B).

    Parameters
    ----------
    uta : float
        UT1 as a 2-part Julian Date    
    utb : float
        UT1 as a 2-part Julian Date

    Returns
    -------
    GST : float
        Greenwich apparent sidereal time (radians)

    '''
    A=pymGmst00(UTA,UTB,UTA,UTB)
    B=pymEe00b(UTA,UTB)
    GST=pymAnp(A+B)

    return(GST)

def pymBpn2xy(RBPN):
    '''
    Extract from the bias-precession-nutation matrix the X,Y coordinates
    of the Celestial Intermediate Pole.

    Parameters
    ----------
    rbpn : list(3,3)
        celestial-to-true matrix

    Returns
    -------
    x : float
        Celestial Intermediate Pole    
    y : float
        Celestial Intermediate Pole

    '''
    #分离出X,Y坐标
    X=RBPN[2][0]
    Y=RBPN[2][1]

    return(X,Y)

def pymS06(DATE1,DATE2,X,Y):
    '''
    The CIO locator s, positioning the Celestial Intermediate Origin on
    the equator of the Celestial Intermediate Pole, given the CIP's X,Y
    coordinates.  Compatible with IAU 2006/2000A precession-nutation.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date    
    x : float
        CIP coordinates    
    y : float
        CIP coordinates

    Returns
    -------
    function value : float
        the CIO locator s in radians

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0
    
    #1儒略世纪对应的天数
    DJC=36525.0
    
    #自J2000.0起经过的儒略世纪数，此处为定义
    T=0
    
    #基本的参数
    FA=[0 for i in range(8)]
    
    # *  ---------------------
    # *  The series for s+XY/2
    # *  ---------------------
    
    NSP=6
    NS0=33
    NS1=3
    NS2=25
    NS3=4
    NS4=1
    
    #多项式系数
    SP=[94e-6,3808.65e-6,-122.68e-6,-72574.11e-6,27.98e-6,15.62e-6]
    
    #Coefficients of l,l',F,D,Om,LVe,LE,pA
    #Argument coefficients for t^0
    KS0=[[0,0,0,0,1,0,0,0],
            [0,0,0,0,2,0,0,0],
            [0,0,2,-2,3,0,0,0],
            [0,0,2,-2,1,0,0,0],
            [0,0,2,-2,2,0,0,0],
            [0,0,2,0,3,0,0,0],
            [0,0,2,0,1,0,0,0],
            [0,0,0,0,3,0,0,0],
            [0,1,0,0,1,0,0,0],
            [0,1,0,0,-1,0,0,0],
            [1,0,0,0,-1,0,0,0],
            [1,0,0,0,1,0,0,0],
            [0,1,2,-2,3,0,0,0],
            [0,1,2,-2,1,0,0,0],
            [0,0,4,-4,4,0,0,0],
            [0,0,1,-1,1,-8,12,0],
            [0,0,2,0,0,0,0,0],
            [0,0,2,0,2,0,0,0],
            [1,0,2,0,3,0,0,0],
            [1,0,2,0,1,0,0,0],
            [0,0,2,-2,0,0,0,0],
            [0,1,-2,2,-3,0,0,0],
            [0,1,-2,2,-1,0,0,0],
            [0,0,0,0,0,8,-13,-1],
            [0,0,0,2,0,0,0,0],
            [2,0,-2,0,-1,0,0,0],
            [0,1,2,-2,2,0,0,0],
            [1,0,0,-2,1,0,0,0],
            [1,0,0,-2,-1,0,0,0],
            [0,0,4,-2,4,0,0,0],
            [0,0,2,-2,4,0,0,0],
            [1,0,-2,0,-3,0,0,0],
            [1,0,-2,0,-1,0,0,0]]
    
    #Argument coefficients for t^1
    KS1=[[0,0,0,0,2,0,0,0],
            [0,0,0,0,1,0,0,0],
            [0,0,2,-2,3,0,0,0]]
    
    #Argument coefficients for t^2
    KS2=[[0,0,0,0,1,0,0,0],
            [0,0,2,-2,2,0,0,0],
            [0,0,2,0,2,0,0,0],
            [0,0,0,0,2,0,0,0],
            [0,1,0,0,0,0,0,0],
            [1,0,0,0,0,0,0,0],
            [0,1,2,-2,2,0,0,0],
            [0,0,2,0,1,0,0,0],
            [1,0,2,0,2,0,0,0],
            [0,1,-2,2,-2,0,0,0],
            [1,0,0,-2,0,0,0,0],
            [0,0,2,-2,1,0,0,0],
            [1,0,-2,0,-2,0,0,0],
            [0,0,0,2,0,0,0,0],
            [1,0,0,0,1,0,0,0],
            [1,0,-2,-2,-2,0,0,0],
            [1,0,0,0,-1,0,0,0],
            [1,0,2,0,1,0,0,0],
            [2,0,0,-2,0,0,0,0],
            [2,0,-2,0,-1,0,0,0],
            [0,0,2,2,2,0,0,0],
            [2,0,2,0,2,0,0,0],
            [2,0,0,0,0,0,0,0],
            [1,0,2,-2,2,0,0,0],
            [0,0,2,0,0,0,0,0]]
    
    #Argument coefficients for t^3
    KS3=[[0,0,0,0,1,0,0,0],
            [0,0,2,-2,2,0,0,0],
            [0,0,2,0,2,0,0,0],
            [0,0,0,0,2,0,0,0]]
    
    #Argument coefficients for t^4
    KS4=[[0,0,0,0,1,0,0,0]]
 
    #Sine and cosine coefficients
    #Sine and cosine coefficients for t^0
    SS0=[[-2640.73e-6,+0.39e-6],
            [-63.53e-6,+0.02e-6],
            [-11.75e-6,-0.01e-6],
            [-11.21e-6,-0.01e-6],
            [+4.57e-6,0.00e-6],
            [-2.02e-6,0.00e-6],
            [-1.98e-6,0.00e-6],
            [+1.72e-6,0.00e-6],
            [+1.41e-6,+0.01e-6],
            [+1.26e-6,+0.01e-6],
            [+0.63e-6,0.00e-6],
            [+0.63e-6,0.00e-6],
            [-0.46e-6,0.00e-6],
            [-0.45e-6,0.00e-6],
            [-0.36e-6,0.00e-6],
            [+0.24e-6,+0.12e-6],
            [-0.32e-6,0.00e-6],
            [-0.28e-6,0.00e-6],
            [-0.27e-6,0.00e-6],
            [-0.26e-6,0.00e-6],
            [+0.21e-6,0.00e-6],
            [-0.19e-6,0.00e-6],
            [-0.18e-6,0.00e-6],
            [+0.10e-6,-0.05e-6],
            [-0.15e-6,0.00e-6],
            [+0.14e-6,0.00e-6],
            [+0.14e-6,0.00e-6],
            [-0.14e-6,0.00e-6],
            [-0.14e-6,0.00e-6],
            [-0.13e-6,0.00e-6],
            [+0.11e-6,0.00e-6],
            [-0.11e-6,0.00e-6],
            [-0.11e-6,0.00e-6]]
    
    #Sine and cosine coefficients for t^1
    SS1=[[-0.07e-6,+3.57e-6],
            [+1.73e-6,-0.03e-6],
            [0.00e-6,+0.48e-6]]
    
    #Sine and cosine coefficients for t^2
    SS2=[[+743.52e-6,-0.17e-6],
            [+56.91e-6,+0.06e-6],
            [+9.84e-6,-0.01e-6],
            [-8.85e-6,+0.01e-6],
            [-6.38e-6,-0.05e-6],
            [-3.07e-6,0.00e-6],
            [+2.23e-6,0.00e-6],
            [+1.67e-6,0.00e-6],
            [+1.30e-6,0.00e-6],
            [+0.93e-6,0.00e-6],
            [+0.68e-6,0.00e-6],
            [-0.55e-6,0.00e-6],
            [+0.53e-6,0.00e-6],
            [-0.27e-6,0.00e-6],
            [-0.27e-6,0.00e-6],
            [-0.26e-6,0.00e-6],
            [-0.25e-6,0.00e-6],
            [+0.22e-6,0.00e-6],
            [-0.21e-6,0.00e-6],
            [+0.20e-6,0.00e-6],
            [+0.17e-6,0.00e-6],
            [+0.13e-6,0.00e-6],
            [-0.13e-6,0.00e-6],
            [-0.12e-6,0.00e-6],
            [-0.11e-6,0.00e-6]]
    
    #Sine and cosine coefficients for t^3
    SS3=[[+0.30e-6,-23.42e-6],
            [-0.03e-6,-1.46e-6],
            [-0.01e-6,-0.25e-6],
            [0.00e-6,+0.23e-6]]
    
    #Sine and cosine coefficients for t^4
    SS4=[[-0.26e-6,-0.01e-6]]
    
    #参考历元J2000.0和当前日期之间的间隔，儒略世纪数
    T=((DATE1-DJ00)+DATE2)/DJC

    #Fundamental Arguments (from IERS Conventions 2003)

    #月亮的平近点角
    FA[0]=pymFal03(T)
    
    #太阳的平近点角
    FA[1]=pymFalp03(T)

    #月亮的平黄经减去升交点黄经
    FA[2]=pymFaf03(T)

    #月亮到太阳的平均距角
    FA[3]=pymFad03(T)

    #月亮的升交点平黄经
    FA[4]=pymFaom03(T)

    #金星的平黄经
    FA[5]=pymFave03(T)

    #地球的平黄经
    FA[6]=pymFae03(T)

    #黄经上的累计进动
    FA[7]=pymFapa03(T)
    
    #估值 s.
    S0 = SP[0]
    S1 = SP[1]
    S2 = SP[2]
    S3 = SP[3]
    S4 = SP[4]
    S5 = SP[5]
    
    for I in range(NS0-1,-1,-1):
        A=0.0
        for J in range(8):
            A=A+float(KS0[I][J])*FA[J]
        S0=S0+(SS0[I][0]*ma.sin(A)+SS0[I][1]*ma.cos(A))
    
    for I in range(NS1-1,-1,-1):
        A=0.0    
        for J in range(8):
            A=A+float(KS1[I][J])*FA[J]
        S1=S1+(SS1[I][0]*ma.sin(A)+SS1[I][1]*ma.cos(A))
    
    for I in range(NS2-1,-1,-1):
        A=0.0
        for J in range(8):
            A=A+float(KS2[I][J])*FA[J]
        S2=S2+(SS2[I][0]*ma.sin(A)+SS2[I][1]*ma.cos(A))
    
    for I in range(NS3-1,-1,-1):
        A=0.0    
        for J in range(8):
            A=A+float(KS3[I][J])*FA[J]
        S3=S3+(SS3[I][0]*ma.sin(A)+SS3[I][1]*ma.cos(A))
    
    for I in range(NS4-1,-1,-1):
        A=0.0    
        for J in range(8):
            A=A+float(KS4[I][J])*FA[J]
        S4=S4+(SS4[I][0]*ma.sin(A)+SS4[I][1]*ma.cos(A))
        
    S06=(S0+(S1+(S2+(S3+(S4+S5*T)*T)*T)*T)*T)*DAS2R-X*Y/2.0
        
    return(S06)

def pymEors(RNPB,S):
    '''
    Equation of the origins, given the classical NPB matrix and the
    quantity s.

    Parameters
    ----------
    rnpb : list(3,3) 
        classical nutation x precession x bias matrix    
    s : float
        the quantity s (the CIO locator) in radians

    Returns
    -------
    function value : float
        the equation of the origins in radians

    '''
    #由 Wallace & Capitaine (2006) expression (16)估计.
    X=RNPB[2][0]
    AX=X/(1.0+RNPB[2][2])
    XS=1.0-AX*X
    YS=-AX*RNPB[2][1]
    ZS=-X
    P=RNPB[0][0]*XS+RNPB[0][1]*YS+RNPB[0][2]*ZS
    Q=RNPB[1][0]*XS+RNPB[1][1]*YS+RNPB[1][2]*ZS
    
    if (P!=0.0)|(Q!=0.0):
        EORS=S-ma.atan2(Q,P)
    else:
        EORS=S
    
    return(EORS)

def pymGst06(UTA,UTB,TTA,TTB,RNPB):
    '''
    Greenwich apparent sidereal time, IAU 2006, given the NPB matrix.

    Parameters
    ----------
    uta : float
        UT1 as a 2-part Julian Date    
    utb : float
        UT1 as a 2-part Julian Date    
    tta : float
        TT as a 2-part Julian Date     
    ttb : float
        TT as a 2-part Julian Date    
    rnpb : float
        nutation x precession x bias matrix

    Returns
    -------
    function value : float
        Greenwich apparent sidereal time (radians)

    '''
    #从CIP坐标中分离出X,Y
    X,Y=pymBpn2xy(RNPB)
    
    #CIO定位角, s.
    S=pymS06(TTA,TTB,X,Y)
    
    #格林尼治视恒星时
    GST=pymAnp(pymEra00(UTA,UTB)-pymEors(RNPB,S))
    
    return(GST)

def pymObl06(DATE1,DATE2):
    '''
    Mean obliquity of the ecliptic, IAU 2006 precession model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        obliquity of the ecliptic (radians)

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0
    
    #1儒略世纪的天数
    DJC=36525.0
    
    #相对于参考历元的时间间隔，单位：儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #黄道平均倾角.
    OBL=(84381.406+(-46.836769+(-0.0001831+(0.00200340+(-0.000000576+\
            (-0.0000000434)*T)*T)*T)*T)*T)*DAS2R

    return(OBL)

def pymPfw06(DATE1,DATE2):
    '''
    Precession angles, IAU 2006 (Fukushima-Williams 4-angle formulation).

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    gamb : flaot
        F-W angle gamma_bar (radians)    
    phib : float
        F-W angle phi_bar (radians)    
    psib : float
        F-W angle psi_bar (radians)    
    epsa : float
        F-W angle epsilon_A (radians)

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0
    
    #1儒略世纪的天数
    DJC=36525.0
    
    #相对于参考历元的时间间隔，单位：儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #P03极移+进动角.
    GAMB=(-0.052928+(10.556378+(0.4932044+(-0.00031238+\
           (-0.000002788+(0.0000000260)*T)*T)*T)*T)*T)*DAS2R
    PHIB=(84381.412819+(-46.811016+(0.0511268+(0.00053289+\
           (-0.000000440+(-0.0000000176)*T)*T)*T)*T)*T)*DAS2R
    PSIB=(-0.041775+(5038.481484+(1.5584175+(-0.00018522+\
           (-0.000026452+(-0.0000000148)*T)*T)*T)*T)*T)*DAS2R      
    EPSA=pymObl06(DATE1,DATE2)
    
    return(GAMB,PHIB,PSIB,EPSA)

def pymNut06a(DATE1,DATE2):
    '''
    IAU 2000A nutation with adjustments to match the IAU 2006
    precession.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    dpsi : float
        nutation, luni-solar + planetary    
    deps : float
        nutation, luni-solar + planetary

    '''
    #参考历元(J2000.0), JD
    DJ00=2451545.0
    
    #1儒略世纪的天数
    DJC=36525.0
    
    #相对于参考历元的时间间隔，单位：儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #根据IAU 2006，对于J2的长期变化进行改正
    FJ2=-2.7774e-6*T
    
    #调用pymNut00a 获得IAU 2000A 的章动.
    DP,DE=pymNut00a(DATE1, DATE2)
    
    #应用改正项(Wallace & Capitaine, 2006, Eqs.5).
    DPSI=DP+DP*(0.4697e-6+FJ2)
    DEPS=DE+DE*FJ2
    
    return(DPSI,DEPS)

def pymIr():
    '''
    Initialize an r-matrix to the identity matrix.

    Returns
    -------
    r : list(3,3)
        r-matrix

    '''

    R=[[1,0,0],[0,1,0],[0,0,1]]
    
    return(R)

def pymRz(PSI,R):
    '''
    Rotate an r-matrix about the z-axis.

    Parameters
    ----------
    psi : float
        angle (radians)    
    r : list(3,3)
        r-matrix

    Returns
    -------
    r : list(3,3)
        r-matrix, rotated

    '''
    S=ma.sin(PSI)
    C=ma.cos(PSI)

    A11=C*R[0][0]+S*R[1][0]
    A12=C*R[0][1]+S*R[1][1]
    A13=C*R[0][2]+S*R[1][2]
    A21=-S*R[0][0]+C*R[1][0]
    A22=-S*R[0][1]+C*R[1][1]
    A23=-S*R[0][2]+C*R[1][2]

    R[0][0]=A11
    R[0][1]=A12
    R[0][2]=A13
    R[1][0]=A21
    R[1][1]=A22
    R[1][2]=A23
 
    return(R)

def pymRx(PHI,R):
    '''
    Rotate an r-matrix about the x-axis.

    Parameters
    ----------
    phi : float    
        angle (radians)    
    r : list(3,3)    
        r-matrix    

    Returns
    -------
    r : list(3,3)    
        r-matrix, rotated

    '''
    S=ma.sin(PHI)
    C=ma.cos(PHI)
    
    A21=C*R[1][0]+S*R[2][0]
    A22=C*R[1][1]+S*R[2][1]
    A23=C*R[1][2]+S*R[2][2]
    A31=-S*R[1][0]+C*R[2][0]
    A32=-S*R[1][1]+C*R[2][1]
    A33=-S*R[1][2]+C*R[2][2]

    R[1][0]=A21
    R[1][1]=A22
    R[1][2]=A23
    R[2][0]=A31
    R[2][1]=A32
    R[2][2]=A33
    
    return(R)

def pymFw2m(GAMB,PHIB,PSI,EPS):
    '''
    Form rotation matrix given the Fukushima-Williams angles.

    Parameters
    ----------
    gamb : float
        F-W angle gamma_bar (radians)    
    phib : float
        F-W angle phi_bar (radians)    
    psi : float
        F-W angle psi (radians)    
    eps : float
        F-W angle epsilon (radians)

    Returns
    -------
    r : list(3,3)
        rotation matrix

    '''
    R=pymIr()
    R=pymRz(GAMB, R)
    R=pymRx(PHIB, R)
    R=pymRz(-PSI, R)
    R=pymRx(-EPS, R)
   
    return(R)

def pymPnm06a(DATE1,DATE2):
    '''
    Form the matrix of precession-nutation for a given date (including
    frame bias), equinox based, IAU 2006 precession and IAU 2000A
    nutation models.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    rbpn : list(3,3)
        bias-precession-nutation matrix

    '''
    #调用pymPfw06获得四个进动角
    GAMB,PHIB,PSIB,EPSA=pymPfw06(DATE1, DATE2)
    
    #调用pymNut06a获得章动参数
    DP,DE=pymNut06a(DATE1,DATE2)
    
    #结合进动角，章动参数，获得用于计算的极移进动章动矩阵
    RBPN=pymFw2m(GAMB,PHIB,PSIB+DP,EPSA+DE)
    
    return(RBPN)

def pymGst06a(UTA,UTB,TTA,TTB):
    '''
    Greenwich apparent sidereal time (consistent with IAU 2000 and 2006
    resolutions).

    Parameters
    ----------
    uta : float
        UT1 as a 2-part Julian Date    
    utb : float
        UT1 as a 2-part Julian Date     
    tta : float
        TT as a 2-part Julian Date    
    ttb : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        Greenwich apparent sidereal time (radians)

    '''
    #调用pymPnm06a获得NPB（参考架偏差-岁差-章动）矩阵, IAU 2000A/2006.
    RNPB=pymPnm06a(TTA,TTB)
    
    #格林尼治视恒星时
    GST=pymGst06(UTA,UTB,TTA,TTB,RNPB)
    
    return(GST)

def pymAnpm(A):
    '''
    Normalize angle into the range -pi <= a < +pi.

    Parameters
    ----------
    a : float
        angle (radians)

    Returns
    -------
    function value : float
        angle in range +/-pi

    '''
    W=np.abs(A)%(2*ma.pi)
    if A<0:
        W=-W
    if (np.abs(W)>=ma.pi):
        if A>=0:
            W=W-2*ma.pi
        else:
            W=W+2*ma.pi
        
    return(W)

def pymNut80(DATE1,DATE2):
    '''
    Nutation, IAU 1980 model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    dpsi : float
        nutation in longitude (radians)    
    deps : float
        nutation in obliquity (radians)

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6

    #2Pi
    D2PI=6.283185307179586476925287
    
    #0.1微角秒对应的弧度
    U2R=DAS2R/1e4
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0

    #1儒略世纪对应的天数
    DJC=36525.0
    
    # 参数和多项式系数表
    #系数单位为0.1微角秒，变化速率为角秒/千儒略年
    # *                Multiple of            Longitude        Obliquity
    # *           L    L'   F    D  Omega   coeff. of sin    coeff. of cos
    # *                                         1       t        1     t
    
    X=[[0.,0.,0.,0.,1.,-171996.,-1742.,92025.,89.],
            [0.,0.,0.,0.,2.,2062.,2.,-895.,5.],
            [-2.,0.,2.,0.,1.,46.,0.,-24.,0.],
            [2.,0.,-2.,0.,0.,11.,0.,0.,0.],
            [-2.,0.,2.,0.,2.,-3.,0.,1.,0.],
            [1.,-1.,0.,-1.,0.,-3.,0.,0.,0.],
            [0.,-2.,2.,-2.,1.,-2.,0.,1.,0.],
            [2.,0.,-2.,0.,1.,1.,0.,0.,0.],
            [0.,0.,2.,-2.,2.,-13187.,-16.,5736.,-31.],
            [0.,1.,0.,0.,0.,1426.,-34.,54.,-1.],
            [0.,1.,2.,-2.,2.,-517.,12.,224.,-6.],
            [0.,-1.,2.,-2.,2.,217.,-5.,-95.,3.],
            [0.,0.,2.,-2.,1.,129.,1.,-70.,0.],
            [2.,0.,0.,-2.,0.,48.,0.,1.,0.],
            [0.,0.,2.,-2.,0.,-22.,0.,0.,0.],
            [0.,2.,0.,0.,0.,17.,-1.,0.,0.],
            [0.,1.,0.,0.,1.,-15.,0.,9.,0.],
            [0.,2.,2.,-2.,2.,-16.,1.,7.,0.],
            [0.,-1.,0.,0.,1.,-12.,0.,6.,0.],
            [-2.,0.,0.,2.,1.,-6.,0.,3.,0.],
            [0.,-1.,2.,-2.,1.,-5.,0.,3.,0.],
            [2.,0.,0.,-2.,1.,4.,0.,-2.,0.],
            [0.,1.,2.,-2.,1.,4.,0.,-2.,0.],
            [1.,0.,0.,-1.,0.,-4.,0.,0.,0.],
            [2.,1.,0.,-2.,0.,1.,0.,0.,0.],
            [0.,0.,-2.,2.,1.,1.,0.,0.,0.],
            [0.,1.,-2.,2.,0.,-1.,0.,0.,0.],
            [0.,1.,0.,0.,2.,1.,0.,0.,0.],
            [-1.,0.,0.,1.,1.,1.,0.,0.,0.],
            [0.,1.,2.,-2.,0.,-1.,0.,0.,0.],
            [0.,0.,2.,0.,2.,-2274.,-2.,977.,-5.],
            [1.,0.,0.,0.,0.,712.,1.,-7.,0.],
            [0.,0.,2.,0.,1.,-386.,-4.,200.,0.],
            [1.,0.,2.,0.,2.,-301.,0.,129.,-1.],
            [1.,0.,0.,-2.,0.,-158.,0.,-1.,0.],
            [-1.,0.,2.,0.,2.,123.,0.,-53.,0.],
            [0.,0.,0.,2.,0.,63.,0.,-2.,0.],
            [1.,0.,0.,0.,1.,63.,1.,-33.,0.],
            [-1.,0.,0.,0.,1.,-58.,-1.,32.,0.],
            [-1.,0.,2.,2.,2.,-59.,0.,26.,0.],
            [1.,0.,2.,0.,1.,-51.,0.,27.,0.],
            [0.,0.,2.,2.,2.,-38.,0.,16.,0.],
            [2.,0.,0.,0.,0.,29.,0.,-1.,0.],
            [1.,0.,2.,-2.,2.,29.,0.,-12.,0.],
            [2.,0.,2.,0.,2.,-31.,0.,13.,0.],
            [0.,0.,2.,0.,0.,26.,0.,-1.,0.],
            [-1.,0.,2.,0.,1.,21.,0.,-10.,0.],
            [-1.,0.,0.,2.,1.,16.,0.,-8.,0.],
            [1.,0.,0.,-2.,1.,-13.,0.,7.,0.],
            [-1.,0.,2.,2.,1.,-10.,0.,5.,0.],
            [1.,1.,0.,-2.,0.,-7.,0.,0.,0.],
            [0.,1.,2.,0.,2.,7.,0.,-3.,0.],
            [0.,-1.,2.,0.,2.,-7.,0.,3.,0.],
            [1.,0.,2.,2.,2.,-8.,0.,3.,0.],
            [1.,0.,0.,2.,0.,6.,0.,0.,0.],
            [2.,0.,2.,-2.,2.,6.,0.,-3.,0.],
            [0.,0.,0.,2.,1.,-6.,0.,3.,0.],
            [0.,0.,2.,2.,1.,-7.,0.,3.,0.],
            [1.,0.,2.,-2.,1.,6.,0.,-3.,0.],
            [0.,0.,0.,-2.,1.,-5.,0.,3.,0.],
            [1.,-1.,0.,0.,0.,5.,0.,0.,0.],
            [2.,0.,2.,0.,1.,-5.,0.,3.,0.],
            [0.,1.,0.,-2.,0.,-4.,0.,0.,0.],
            [1.,0.,-2.,0.,0.,4.,0.,0.,0.],
            [0.,0.,0.,1.,0.,-4.,0.,0.,0.],
            [1.,1.,0.,0.,0.,-3.,0.,0.,0.],
            [1.,0.,2.,0.,0.,3.,0.,0.,0.],
            [1.,-1.,2.,0.,2.,-3.,0.,1.,0.],
            [-1.,-1.,2.,2.,2.,-3.,0.,1.,0.],
            [-2.,0.,0.,0.,1.,-2.,0.,1.,0.],
            [3.,0.,2.,0.,2.,-3.,0.,1.,0.],
            [0.,-1.,2.,2.,2.,-3.,0.,1.,0.],
            [1.,1.,2.,0.,2.,2.,0.,-1.,0.],
            [-1.,0.,2.,-2.,1.,-2.,0.,1.,0.],
            [2.,0.,0.,0.,1.,2.,0.,-1.,0.],
            [1.,0.,0.,0.,2.,-2.,0.,1.,0.],
            [3.,0.,0.,0.,0.,2.,0.,0.,0.],
            [0.,0.,2.,1.,2.,2.,0.,-1.,0.],
            [-1.,0.,0.,0.,2.,1.,0.,-1.,0.],
            [1.,0.,0.,-4.,0.,-1.,0.,0.,0.],
            [-2.,0.,2.,2.,2.,1.,0.,-1.,0.],
            [-1.,0.,2.,4.,2.,-2.,0.,1.,0.],
            [2.,0.,0.,-4.,0.,-1.,0.,0.,0.],
            [1.,1.,2.,-2.,2.,1.,0.,-1.,0.],
            [1.,0.,2.,2.,1.,-1.,0.,1.,0.],
            [-2.,0.,2.,4.,2.,-1.,0.,1.,0.],
            [-1.,0.,4.,0.,2.,1.,0.,0.,0.],
            [1.,-1.,0.,-2.,0.,1.,0.,0.,0.],
            [2.,0.,2.,-2.,1.,1.,0.,-1.,0.],
            [2.,0.,2.,2.,2.,-1.,0.,0.,0.],
            [1.,0.,0.,2.,1.,-1.,0.,0.,0.],
            [0.,0.,4.,-2.,2.,1.,0.,0.,0.],
            [3.,0.,2.,-2.,2.,1.,0.,0.,0.],
            [1.,0.,2.,-2.,0.,-1.,0.,0.,0.],
            [0.,1.,2.,0.,1.,1.,0.,0.,0.],
            [-1.,-1.,0.,2.,1.,1.,0.,0.,0.],
            [0.,0.,-2.,0.,1.,-1.,0.,0.,0.],
            [0.,0.,2.,-1.,2.,-1.,0.,0.,0.],
            [0.,1.,0.,2.,0.,-1.,0.,0.,0.],
            [1.,0.,-2.,-2.,0.,-1.,0.,0.,0.],
            [0.,-1.,2.,0.,1.,-1.,0.,0.,0.],
            [1.,1.,0.,-2.,1.,-1.,0.,0.,0.],
            [1.,0.,-2.,2.,0.,-1.,0.,0.,0.],
            [2.,0.,0.,2.,0.,1.,0.,0.,0.],
            [0.,0.,2.,4.,2.,-1.,0.,0.,0.],
            [0.,1.,0.,1.,0.,1.,0.,0.,0.]]

    #给定日期与参考历元之间的时间间隔，儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #FK5参考系中的基本参数
    #月球的平黄经减去月球近地点的平黄经。
    C=np.abs(1325.0*T)%1.0
    if (1325.0*T)<0:
        B=-C
    else:
        B=C
    A=(485866.733+(715922.633+(31.310+0.064*T)*T)*T)*DAS2R+B*D2PI
    EL=pymAnpm(A)
    
    #太阳的平黄经减去近地点的平黄经。
    C=np.abs(99.0*T)%1.0
    if (99.0*T)<0:
        B=-C
    else:
        B=C
    A=(1287099.804+(1292581.224+(-0.577-0.012*T)*T)*T)*DAS2R+B*D2PI
    ELP=pymAnpm(A)
    
    #月球的平黄经减去月球交点的平黄经。
    C=np.abs(1342.0*T)%1.0
    if (1342.0*T)<0:
        B=-C
    else:
        B=C
    A=(335778.877+(295263.137+(-13.257+0.011*T)*T)*T)*DAS2R+B*D2PI
    F=pymAnpm(A)
    
    #月亮与太阳之间的平角距
    C=np.abs(1236.0*T)%1.0
    if (1236.0*T)<0:
        B=-C
    else:
        B=C
    A=(1072261.307+(1105601.328+(-6.891+0.019*T)*T)*T)*DAS2R+B*D2PI
    D=pymAnpm(A)
    
    #月球轨道在黄道上的升交点平黄经，起量点为春分点
    C=np.abs(-5.0*T)%1.0
    if (-5.0*T)<0:
        B=-C
    else:
        B=C
    A=(450160.280+(-482890.539+(7.455+0.008*T)*T)*T)*DAS2R+B*D2PI
    OM=pymAnpm(A)
    
    # 章动部分
    
    #将时间单位由世纪转换为千年
    T=T/10.0
    
    #初始化章动参数
    DP=0.0
    DE=0.0
    
    #对章动项求和，从小到大
    for j in range(105,-1,-1):
        
        #构建当前项的参数
        ARG=float(X[j][0])*EL+float(X[j][1])*ELP+float(X[j][2])*F+\
            float(X[j][3])*D+float(X[j][4])*OM
            
        #累加当前章动项。
        S=float(X[j][5])+float(X[j][6])*T
        C=float(X[j][7])+float(X[j][8])*T
        if (S!=0.0):
            DP=DP+S*ma.sin(ARG)
        if (C!=0.0):
            DE=DE+C*ma.cos(ARG)
        
    #转换单位到弧度制
    DPSI=DP*U2R
    DEPS=DE*U2R
    
    return(DPSI,DEPS)


def pymEqeq94(DATE1,DATE2):
    '''
    Equation of the equinoxes, IAU 1994 model.

    Parameters
    ----------
    date1 : float
        TDB date    
    date2 : float
        TDB date

    Returns
    -------
    function value : float
        equation of the equinoxes

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6

    #2Pi
    D2PI=6.283185307179586476925287
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0

    #1儒略世纪的天数
    DJC=36525.0
    
    #给定日期和参考历元之间的时间间隔，儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #月球轨道在黄道上的升交点平黄经，起量点为春分点
    C=np.abs(-5.0*T)%1.0
    if (-5.0*T)<0:
        B=-C
    else:
        B=C
    A=(450160.280+(-482890.539+(7.455+0.008*T)*T)*T)*DAS2R+B*D2PI
    OM=pymAnpm(A)
    
    #章动参数和平均倾角
    DPSI,DEPS=pymNut80(DATE1,DATE2)
    EPS0=pymObl80(DATE1,DATE2)
    
    #赤经章动
    EQEQ=DPSI*ma.cos(EPS0)+DAS2R*(0.00264*ma.sin(OM)+0.000063*ma.sin(OM+OM))
    
    return(EQEQ)

def pymGst94(UTA,UTB):
    '''
    Greenwich apparent sidereal time (consistent with IAU 1982/94
    resolutions).

    Parameters
    ----------
    uta : float
        UT1 as a 2-part Julian Date    
    utb : float
        UT1 as a 2-part Julian Date

    Returns
    -------
    GST : float
        Greenwich apparent sidereal time (radians)

    '''
    A=pymGmst82(UTA,UTB)
    B=pymEqeq94(UTA,UTB)
    GST=pymAnp(A+B)

    return(GST)

def pymCal2jd(IY,IM,ID):
    '''
    Gregorian Calendar to Julian Date.

    Parameters
    ----------
    iyear : int
        year in Gregorian calendar
    imon : int
        month in Gregorian calendar
    iday : int
        day in Gregorian calendar

    Returns
    -------
    djm0 : float
        MJD zero-point: always 2400000.5
    djm : float
        Modified Julian Date for 0 hrs
    J : ValueError
        -1: 'bad year,  the year is simply valid from -4800 March 1',
        -2: 'bad month, the month is not from 1 to 12',
        -3: 'bad day,   the day is not related to the month'
    '''

    #函数允许的最早日期
    IYMIN=-4799
    #每个月的天长
    MTAB=[31,28,31,30,31,30,31,31,30,31,30,31]
    
    J=0
    
    if (IY<IYMIN):
        J=-1
    else:
        if (IM>=1)&(IM<=12):
            #当月的天数
            NDAYS=MTAB[int(IM-1)]
            #判断是否闰年
            if (IM==2):
                if (IY%4==0):
                    NDAYS=29
                if (IY%100==0)&(IY%400!=0):
                    NDAYS=28
            if (ID<1)|(ID>NDAYS):
                J=-3
                
            MY=int((IM-14)/12)
            IYPMY=int(IY+MY)
            DJM0=2400000.5
            #儒略日自-4800年3月1日起
            DJM=float(int((1461*(IYPMY+4800))/4)+int((367*(IM-2-12*MY))/12)-\
                      int(3*(int((IYPMY+4900)/100))/4)+ID-2432076)
        else:
            J=-2
    return(DJM0,DJM,J)

def pymJd2cal(DJ1,DJ2):
    '''
    Julian Date to Gregorian year, month, day, and fraction of a day.

    Parameters
    ----------
    dj1 : float
        Julian Date    
    dj2 : float
        Julian Date

    Returns
    -------
    iy : int
        yaer    
    im : int
        month
    id : int
        day
    fd : float
        fraction of day
    J : ValueError
        -1: 'The valid date is -68569.5 (-4900 March 1) up to 1e9'
    '''
    #极小值，使1.0+EPS!=1.0
    EPS=2.2204460492503131e-16
    #最小，最大允许的输入儒略日
    DJMIN=-68569.5
    DJMAX=1e9
    
    DJ=DJ1+DJ2
    if (DJ<DJMIN)|(DJ>DJMAX):
        J=-1
    else:
        J=0
    
    #将天数和小数分离开(其中小数部分绝对值小于0.5).
        D=float(int(DJ1+0.5))
        F1=DJ1-D
        JD=int(D)
        D=float(int(DJ2+0.5))
        F2=DJ2-D
        JD=JD+int(D)
    
    #使用补偿求和计算F1+F2+0.5(Klein 2006)
        S=0.5
        CS=0.0
        V=[F1,F2]
        for i in range(2):
            X=V[i]
            T=S+X
            if (np.abs(S)>=np.abs(X)):
                C=(S-T)+X
            else:
                C=(X-T)+S
            CS=CS+C
            S=T
            if (S>=1.0):
                JD+=1
                S=S-1.0
        F=S+CS
        CS=F-S
            
    #当F为负数时.
        if (F<0.0):
        #补偿求和，假设 |S| <= 1.
            F=S+1.0
            CS=CS+((1.0-F)+S)
            S=F
            F=S+CS
            CS=F-S
            JD=JD-1
        
    #当F=1.0或者更大时.
        if ((F-1.0)>=(-EPS/4.0)):
        #补偿求和，假设 |S| <= 1. */
            T=S-1.0
            CS=CS+((S-T)-1.0)
            S=T
            F=S+CS
            if ((-EPS/2.0)<F):
                JD=JD+1
                F=max(F,0.0)
            
    #在公历下表示日期.
        L=JD+68569
        N=int((4*L)/146097)
        L=int(L-int((146097*N+3)/4))
        I=int((4000*(L+1))/1461001)
        L=int(L-int((1461*I)/4)+31)
        K=int((80*L)/2447)
        ID=int(L-int((2447*K)/80))
        L=int(K/11)
        IM=int(K+2-12*L)
        IY=int(100*(N-49)+I+L)
        FD=F      
        
    return(IY,IM,ID,FD,J)

def pymJdcalf(NDP,DJ1,DJ2):
    '''
    Julian Date to Gregorian Calendar, expressed in a form convenient 
    for formatting messages:  rounded to a specified precision.

    Parameters
    ----------
    ndp : int
        number of decimal places of days in fraction    
    dj1 : float
        Julian Date    
    dj2 : float
        Julian Date    

    Returns
    -------
    iymdf : list(4)
        year, month, day, fraction in Gregorian calendar

    '''
    #小数的位数 (e.g. NDP取2时，DENOM取100)
    if (NDP>=0)&(NDP<=9):
        J=0
        DENOM=float(10**NDP)
    else:
        J=1
        DENOM=1.0
    
    #D1为大值，D2为小值
    if (np.abs(DJ1)>=np.abs(DJ2)):
        D1=DJ1
        D2=DJ2
    else:
        D1=DJ2
        D2=DJ1            
    
    #调整到午夜(没有舍入误差)。
    D1=D1-0.5
    
    #尽可能精确地分离开日和小数.
    D=float(int(D1+0.5))
    F1=D1-D
    DJD=D
    D=float(int(D2+0.5))
    F2=D2-D
    DJD=DJD+D;
    D=float(int(F1+F2+0.5))
    F=(F1-D)+F2
    if (F<0.0):
        F=F+1.0
        D=D-1.0
    DJD+=D
    
    #在指定的精度要求下四舍五入.
    RF=float(int(F*DENOM+0.5))/DENOM
    
    #重新调整到中午
    DJD+=0.5
    
    #转换到公历.
    IYMDF=[0,0,0,0]
    IYMDF[0],IYMDF[1],IYMDF[2],F,JS=pymJd2cal(DJD, RF)
    if (JS==0):
        IYMDF[3]=int(F*DENOM+0.5)
    else:
        J=JS    
    
    return(IYMDF, J)

def pymD2tf(NDP,DAYS):
    '''
    Decompose days to hours, minutes, seconds, fraction.

    Parameters
    ----------
    ndp : int
        resolution    
    days : float
        interval in days

    Returns
    -------
    sign : str
        '+' or '-'    
    ihmsf : list(4)    
        hours, minutes, seconds, fraction

         NDP         resolution
          :      ...0000 00 00
         -7         1000 00 00
         -6          100 00 00
         -5           10 00 00
         -4            1 00 00
         -3            0 10 00
         -2            0 01 00
         -1            0 00 10
          0            0 00 01
          1            0 00 00.1
          2            0 00 00.01
          3            0 00 00.001
          :            0 00 00.000...

    '''
    if DAYS>=0:
        SIGN='+'
    else:
        SIGN='-'
    
    #一天的秒数
    D2S=86400.0
    A=D2S*np.abs(DAYS)
    
    #根据精度要求处理
    if NDP<0:
        NRS=1.0
        N=1
        while N<=-NDP:
           if (N==2)|(N==4):
               NRS=NRS*6
           else:
               NRS=NRS*10
           N+=1
        RS=float(NRS)
        A=RS*float(int(A/RS+0.5))
    
    NRS=1.0
    N=1
    while N<=NDP:
        NRS=NRS*10
        N+=1
    RS=float(NRS)
    RM=RS*60.0
    RH=RM*60.0
    
    A=float(int(RS*A+0.5))
    
    AH=float(int(A/RH))
    A=A-AH*RH
    AM=float(int(A/RM))
    A=A-AM*RM
    AS=float(int(A/RS))
    AF=A-AS*RS
    
    IHMSF=[int(AH+0.5),int(AM+0.5),int(AS+0.5),int(AF+0.5)]
    return(SIGN,IHMSF)

def pymD2dtf(SCALE,NDP,D1,D2):
    '''
    Format for output a 2-part Julian Date (or in the case of UTC a 
    quasi-JD form that includes special provision for leap seconds).

    Parameters
    ----------
    scale : str
        time scale ID(Only the value "UTC" is significant)       
    ndp : int
        resolution    
    d1 : float
        time as a 2-part Julian Date    
    d2 : float
        time as a 2-part Julian Date

    Returns
    -------
    iy,im,id : int
        year, month, day in Gregorian calendar    
    ihmsf : list(4)
        hours, minutes, seconds, fraction
    J : ValueError
        1: 'dubious year',
       -1: 'unacceptable date',
       
         NDP         resolution
          :      ...0000 00 00
         -7         1000 00 00
         -6          100 00 00
         -5           10 00 00
         -4            1 00 00
         -3            0 10 00
         -2            0 01 00
         -1            0 00 10
          0            0 00 01
          1            0 00 00.1
          2            0 00 00.01
          3            0 00 00.001
          :            0 00 00.000...
    '''
    
    #一天的秒长
    D2S = 86400.0
    IY,IM,ID=0,0,0
    IHMSF=[0,0,0,0]
    i=1
    while i<2:
    
        A1 = D1
        B1 = D2
        
        #临时日历日期.
        IY1,IM1,ID1,FD,JS=pymJd2cal(A1,B1)
        if (JS!=0):
            print('ERROR',JS)
            break
        
        #判断是否是闰秒当天
        LEAP=False
        if (SCALE=='UTC'):
            
            #当天0时的TAI-UTC时间差.
            DAT0,JS=pymDat(IY1,IM1,ID1,0.0)
            if (JS<0):
                print('ERROR',JS)
                break
            
            #当天12h的TAI-UTC时间差（判断是否存在drift——1972年之前情况）
            DAT12,JS=pymDat(IY1,IM1,ID1,0.5)
            if (JS<0):
                print('ERROR',JS)
                break
            
            #第二天0h的TAI-UTC时间差（判断是否存在跳秒）
            IY2,IM2,ID2,W,JS=pymJd2cal(A1+1.5,B1-FD)
            if (JS!=0):
                print('ERROR',JS)
                break
            DAT24,JS=pymDat(IY2,IM2,ID2,0.0)
            if (JS<0):
                print('ERROR',JS)
                break
    
            #当天和第二天之间TAI-UTC是否有突然变化。
            DLEAP=DAT24-(2.0*DAT12-DAT0)
            
            #如果是闰秒日，则修改小数部分.
            LEAP=np.abs(DLEAP)>0.5
            if (LEAP):
                FD=FD+FD*DLEAP/D2S
        
        #临时时间.
        S,IHMSF1=pymD2tf(NDP,FD)
            
        #判断四舍五入的时间是否超过24点
        if (IHMSF1[0]>23):
            
            #是，则日期为第二天
            IY2,IM2,ID2,W,JS=pymJd2cal(A1+1.5,B1-FD)
            if (JS<0):
                print('ERROR',JS)
                break
            
            #判断是否当天有闰秒
            if (not LEAP):
                
                #否，则用第二天的零时
                IY1=IY2
                IM1=IM2
                ID1=ID2
                IHMSF1[0]=0
                IHMSF1[1]=0
                IHMSF1[2]=0
            
            else:
                
                #是，则判断是否已经过了闰秒的时间
                if (IHMSF1[2]>0):
                    
                    #是，则第二天要考虑到闰秒。
                    IY1=IY2
                    IM1=IM2
                    ID1=ID2
                    IHMSF1[0]=0
                    IHMSF1[1]=0
                    IHMSF1[2]=0
                
                else:
                    
                    #否，则当天采用23：59：60
                    IHMSF1[0]=23
                    IHMSF1[1]=59
                    IHMSF1[2]=60
            
                #如果四舍五入到10s，且大致上是要到第二天
                if (NDP<0)&(IHMSF1[2]==60):
                    IY1=IY2
                    IM1=IM2
                    ID1=ID2
                    IHMSF1[0]=0
                    IHMSF1[1]=0
                    IHMSF1[2]=0
        
        IY=IY1
        IM=IM1
        ID=ID1
        for i in range(4):
           IHMSF[i]=IHMSF1[i]
        
        i+=1
    J=JS
    
    return(IY,IM,ID,IHMSF,J)

def pymDat(IY,IM,ID,FD):
    
    '''
    For a given UTC date, calculate Delta(AT) = TAI-UTC.

    Parameters
    ----------
    iy : int
        year    
    im : int
        month    
    iday : int
        day    
    fday : float    
        fraction of day

    Returns
    -------
    deltat : float
        TAI minus UTC, seconds
    J : ValueError
        1: 'dubious year', 
       -1: 'bad year,  the year is simply valid from -4800 March 1',
       -2: 'bad month, the month is not from 1 to 12',
       -3: 'bad day,   the day is not related to the month',
       -4: 'bad fraction of day',
       -5: 'internal error', 
    '''

    #此函数版本发布日期
    IYV=2023
    '闰秒需要调整的参数，修改时年份'
    
    #闰秒的次数（当有新的闰秒出现时，+1）
    NDAT = 42
    '闰秒需要调整的参数，+1'
    
    #在闰秒被引入之前的时间差变化数目
    NERA1 = 14
    '为1972年前的次数，无需调整'
    IDAT=[[1960,1],[1961,1],[1961,8],[1962,1],[1963,11],[1964,1],[1964,4],
          [1964,9],[1965,1],[1965,3],[1965,7],[1965,9],[1966,1],[1968,2],
          [1972,1],[1972,7],[1973,1],[1974,1],[1975,1],[1976,1],[1977,1],
          [1978,1],[1979,1],[1980,1],[1981,7],[1982,7],[1983,7],[1985,7],
          [1988,1],[1990,1],[1991,1],[1992,7],[1993,7],[1994,7],[1996,1],
          [1997,7],[1999,1],[2006,1],[2009,1],[2012,7],[2015,7],[2017,1]]
    '闰秒需要调整的参数，最后增加一项闰秒时间'
    DATS=[1.4178180,1.4228180,1.3728180,1.8458580,1.9458580,3.2401300,
          3.3401300,3.4401300,3.5401300,3.6401300,3.7401300,3.8401300,
          4.3131700,4.2131700,10.0,11.0,12.0,13.0,14.0,15.0,16.0,17.0,
          18.0,19.0,20.0,21.0,22.0,23.0,24.0,25.0,26.0,27.0,28.0,29.0,
          30.0,31.0,32.0,33.0,34.0,35.0,36.0,37.0]
    '闰秒需要调整的参数，最后增加一项总闰秒时间，通常+-1'
    DRIFT=[[37300, 0.001296],[37300, 0.001296],[37300, 0.001296],
           [37665, 0.0011232],[37665, 0.0011232],[38761, 0.001296],
           [38761, 0.001296],[38761, 0.001296],[38761, 0.001296],
           [38761, 0.001296],[38761, 0.001296],[38761, 0.001296],
           [39126, 0.002592],[39126, 0.002592]]
    '为1972年前数据，无需调整'
    
    #初始化
    DA=0.0
    JS=0
    
    #用循环替代goto函数
    i=1
    while i<2:

        if (FD<0.0)|(FD>1.0):
            JS=-4
            print('ERROR1',JS)
            break
        
        #将日期转换为儒略日.
        DJM0,DJM,JS=pymCal2jd(IY,IM,ID)
        if (JS<0):
            print('ERROR2',JS)
            break
        
        if (IY<IDAT[0][0]):
            #早于1960年，报错
            JS=1
            print('ERROR3',JS)
            break
        #晚于函数版本5年后，报错
        if (IY>(IYV+5)):
            JS=1
        
        #年月结合
        M=12*IY+IM
        
        #找到最接近的项目.
        IS=0
        MORE=True
        N=NDAT
        while N>0:
            if MORE:
                IS=N
                MORE=M<(12*IDAT[N-1][0]+IDAT[N-1][1])    
            N-=1
        
        if (IS<1):
            JS=-5
            print('ERROR4',JS)
            break
        
        DA=DATS[IS-1]
        
        #早于1972年时，需要一个额外的调整
        if (IS<=NERA1):
            DA=DA+(DJM+FD-DRIFT[IS-1][0])*DRIFT[IS-1][1]
        
        i+=1
    DELTAT=DA
    J=JS
    
    return(DELTAT,J)

def pymDtdb(DATE1,DATE2,UT,ELONG,U,V):
    '''
    An approximation to TDB-TT, the difference between barycentric
    dynamical time and terrestrial time, for an observer on the Earth.

    Parameters
    ----------
    date1 : float
        date, TDB    
    date2 : float
        date, TDB    
    ut : float
        universal time (UT1, fraction of one day)    
    elong : float
        longitude (east positive, radians)     
    u : float
        distance from Earth spin axis (km)    
    v : float
        distance north of equatorial plane (km)

    Returns
    -------
    DTDB : float
        TDB-TT (seconds)

    '''

    D2PI=2*ma.pi
    #1度对应的弧度
    DD2R=1.745329251994329576923691e-2
    #参考时期 (J2000.0), 儒略日期
    DJ00=2451545.0
    #每千儒略年的天数
    DJM=365250.0
# =============================================================================
#     *  =====================
#     *  Fairhead et al. model
#     *  =====================
#     *
#     *  787 sets of three coefficients.
#     *
#     *  Each set is amplitude (microseconds)
#     *              frequency (radians per Julian millennium since J2000.0),
#     *              phase (radians).
#     *
#     *  Sets   1-474 are the T**0 terms,
#     *   "   475-679  "   "  T**1   "
#     *   "   680-764  "   "  T**2   "
#     *   "   765-784  "   "  T**3   "
#     *   "   785-787  "   "  T**4   "  .
# =============================================================================
    #模型中的参数
    #FAIRHD(787,3)
    FAIRHD=[[1656.674564e-6,6283.075849991,6.240054195],
            [22.417471e-6,5753.384884897,4.296977442],
            [13.839792e-6,12566.151699983,6.196904410],
            [4.770086e-6,529.690965095,0.444401603],
            [4.676740e-6,6069.776754553,4.021195093],
            [2.256707e-6,213.299095438,5.543113262],
            [1.694205e-6,-3.523118349,5.025132748],
            [1.554905e-6,77713.771467920,5.198467090],
            [1.276839e-6,7860.419392439,5.988822341],
            [1.193379e-6,5223.693919802,3.649823730],
            [1.115322e-6,3930.209696220,1.422745069],
            [0.794185e-6,11506.769769794,2.322313077],
            [0.447061e-6,26.298319800,3.615796498],
            [0.435206e-6,-398.149003408,4.349338347],
            [0.600309e-6,1577.343542448,2.678271909],
            [0.496817e-6,6208.294251424,5.696701824],
            [0.486306e-6,5884.926846583,0.520007179],
            [0.432392e-6,74.781598567,2.435898309],
            [0.468597e-6,6244.942814354,5.866398759],
            [0.375510e-6,5507.553238667,4.103476804],
            [0.243085e-6,-775.522611324,3.651837925],
            [0.173435e-6,18849.227549974,6.153743485],
            [0.230685e-6,5856.477659115,4.773852582],
            [0.203747e-6,12036.460734888,4.333987818],
            [0.143935e-6,-796.298006816,5.957517795],
            [0.159080e-6,10977.078804699,1.890075226],
            [0.119979e-6,38.133035638,4.551585768],
            [0.118971e-6,5486.777843175,1.914547226],
            [0.116120e-6,1059.381930189,0.873504123],
            [0.137927e-6,11790.629088659,1.135934669],
            [0.098358e-6,2544.314419883,0.092793886],
            [0.101868e-6,-5573.142801634,5.984503847],
            [0.080164e-6,206.185548437,2.095377709],
            [0.079645e-6,4694.002954708,2.949233637],
            [0.062617e-6,20.775395492,2.654394814],
            [0.075019e-6,2942.463423292,4.980931759],
            [0.064397e-6,5746.271337896,1.280308748],
            [0.063814e-6,5760.498431898,4.167901731],
            [0.048042e-6,2146.165416475,1.495846011],
            [0.048373e-6,155.420399434,2.251573730],
            [0.058844e-6,426.598190876,4.839650148],
            [0.046551e-6,-0.980321068,0.921573539],
            [0.054139e-6,17260.154654690,3.411091093],
            [0.042411e-6,6275.962302991,2.869567043],
            [0.040184e-6,-7.113547001,3.565975565],
            [0.036564e-6,5088.628839767,3.324679049],
            [0.040759e-6,12352.852604545,3.981496998],
            [0.036507e-6,801.820931124,6.248866009],
            [0.036955e-6,3154.687084896,5.071801441],
            [0.042732e-6,632.783739313,5.720622217],
            [0.042560e-6,161000.685737473,1.270837679],
            [0.040480e-6,15720.838784878,2.546610123],
            [0.028244e-6,-6286.598968340,5.069663519],
            [0.033477e-6,6062.663207553,4.144987272],
            [0.034867e-6,522.577418094,5.210064075],
            [0.032438e-6,6076.890301554,0.749317412],
            [0.030215e-6,7084.896781115,3.389610345],
            [0.029247e-6,-71430.695617928,4.183178762],
            [0.033529e-6,9437.762934887,2.404714239],
            [0.032423e-6,8827.390269875,5.541473556],
            [0.027567e-6,6279.552731642,5.040846034],
            [0.029862e-6,12139.553509107,1.770181024],
            [0.022509e-6,10447.387839604,1.460726241],
            [0.020937e-6,8429.241266467,0.652303414],
            [0.020322e-6,419.484643875,3.735430632],
            [0.024816e-6,-1194.447010225,1.087136918],
            [0.025196e-6,1748.016413067,2.901883301],
            [0.021691e-6,14143.495242431,5.952658009],
            [0.017673e-6,6812.766815086,3.186129845],
            [0.022567e-6,6133.512652857,3.307984806],
            [0.016155e-6,10213.285546211,1.331103168],
            [0.014751e-6,1349.867409659,4.308933301],
            [0.015949e-6,-220.412642439,4.005298270],
            [0.015974e-6,-2352.866153772,6.145309371],
            [0.014223e-6,17789.845619785,2.104551349],
            [0.017806e-6,73.297125859,3.475975097],
            [0.013671e-6,-536.804512095,5.971672571],
            [0.011942e-6,8031.092263058,2.053414715],
            [0.014318e-6,16730.463689596,3.016058075],
            [0.012462e-6,103.092774219,1.737438797],
            [0.010962e-6,3.590428652,2.196567739],
            [0.015078e-6,19651.048481098,3.969480770],
            [0.010396e-6,951.718406251,5.717799605],
            [0.011707e-6,-4705.732307544,2.654125618],
            [0.010453e-6,5863.591206116,1.913704550],
            [0.012420e-6,4690.479836359,4.734090399],
            [0.011847e-6,5643.178563677,5.489005403],
            [0.008610e-6,3340.612426700,3.661698944],
            [0.011622e-6,5120.601145584,4.863931876],
            [0.010825e-6,553.569402842,0.842715011],
            [0.008666e-6,-135.065080035,3.293406547],
            [0.009963e-6,149.563197135,4.870690598],
            [0.009858e-6,6309.374169791,1.061816410],
            [0.007959e-6,316.391869657,2.465042647],
            [0.010099e-6,283.859318865,1.942176992],
            [0.007147e-6,-242.728603974,3.661486981],
            [0.007505e-6,5230.807466803,4.920937029],
            [0.008323e-6,11769.853693166,1.229392026],
            [0.007490e-6,-6256.777530192,3.658444681],
            [0.009370e-6,149854.400134205,0.673880395],
            [0.007117e-6,38.027672636,5.294249518],
            [0.007857e-6,12168.002696575,0.525733528],
            [0.007019e-6,6206.809778716,0.837688810],
            [0.006056e-6,955.599741609,4.194535082],
            [0.008107e-6,13367.972631107,3.793235253],
            [0.006731e-6,5650.292110678,5.639906583],
            [0.007332e-6,36.648562930,0.114858677],
            [0.006366e-6,4164.311989613,2.262081818],
            [0.006858e-6,5216.580372801,0.642063318],
            [0.006919e-6,6681.224853400,6.018501522],
            [0.006826e-6,7632.943259650,3.458654112],
            [0.005308e-6,-1592.596013633,2.500382359],
            [0.005096e-6,11371.704689758,2.547107806],
            [0.004841e-6,5333.900241022,0.437078094],
            [0.005582e-6,5966.683980335,2.246174308],
            [0.006304e-6,11926.254413669,2.512929171],
            [0.006603e-6,23581.258177318,5.393136889],
            [0.005123e-6,-1.484472708,2.999641028],
            [0.004648e-6,1589.072895284,1.275847090],
            [0.005119e-6,6438.496249426,1.486539246],
            [0.004521e-6,4292.330832950,6.140635794],
            [0.005680e-6,23013.539539587,4.557814849],
            [0.005488e-6,-3.455808046,0.090675389],
            [0.004193e-6,7234.794256242,4.869091389],
            [0.003742e-6,7238.675591600,4.691976180],
            [0.004148e-6,-110.206321219,3.016173439],
            [0.004553e-6,11499.656222793,5.554998314],
            [0.004892e-6,5436.993015240,1.475415597],
            [0.004044e-6,4732.030627343,1.398784824],
            [0.004164e-6,12491.370101415,5.650931916],
            [0.004349e-6,11513.883316794,2.181745369],
            [0.003919e-6,12528.018664345,5.823319737],
            [0.003129e-6,6836.645252834,0.003844094],
            [0.004080e-6,-7058.598461315,3.690360123],
            [0.003270e-6,76.266071276,1.517189902],
            [0.002954e-6,6283.143160294,4.447203799],
            [0.002872e-6,28.449187468,1.158692983],
            [0.002881e-6,735.876513532,0.349250250],
            [0.003279e-6,5849.364112115,4.893384368],
            [0.003625e-6,6209.778724132,1.473760578],
            [0.003074e-6,949.175608970,5.185878737],
            [0.002775e-6,9917.696874510,1.030026325],
            [0.002646e-6,10973.555686350,3.918259169],
            [0.002575e-6,25132.303399966,6.109659023],
            [0.003500e-6,263.083923373,1.892100742],
            [0.002740e-6,18319.536584880,4.320519510],
            [0.002464e-6,202.253395174,4.698203059],
            [0.002409e-6,2.542797281,5.325009315],
            [0.003354e-6,-90955.551694697,1.942656623],
            [0.002296e-6,6496.374945429,5.061810696],
            [0.003002e-6,6172.869528772,2.797822767],
            [0.003202e-6,27511.467873537,0.531673101],
            [0.002954e-6,-6283.008539689,4.533471191],
            [0.002353e-6,639.897286314,3.734548088],
            [0.002401e-6,16200.772724501,2.605547070],
            [0.003053e-6,233141.314403759,3.029030662],
            [0.003024e-6,83286.914269554,2.355556099],
            [0.002863e-6,17298.182327326,5.240963796],
            [0.002103e-6,-7079.373856808,5.756641637],
            [0.002303e-6,83996.847317911,2.013686814],
            [0.002303e-6,18073.704938650,1.089100410],
            [0.002381e-6,63.735898303,0.759188178],
            [0.002493e-6,6386.168624210,0.645026535],
            [0.002366e-6,3.932153263,6.215885448],
            [0.002169e-6,11015.106477335,4.845297676],
            [0.002397e-6,6243.458341645,3.809290043],
            [0.002183e-6,1162.474704408,6.179611691],
            [0.002353e-6,6246.427287062,4.781719760],
            [0.002199e-6,-245.831646229,5.956152284],
            [0.001729e-6,3894.181829542,1.264976635],
            [0.001896e-6,-3128.388765096,4.914231596],
            [0.002085e-6,35.164090221,1.405158503],
            [0.002024e-6,14712.317116458,2.752035928],
            [0.001737e-6,6290.189396992,5.280820144],
            [0.002229e-6,491.557929457,1.571007057],
            [0.001602e-6,14314.168113050,4.203664806],
            [0.002186e-6,454.909366527,1.402101526],
            [0.001897e-6,22483.848574493,4.167932508],
            [0.001825e-6,-3738.761430108,0.545828785],
            [0.001894e-6,1052.268383188,5.817167450],
            [0.001421e-6,20.355319399,2.419886601],
            [0.001408e-6,10984.192351700,2.732084787],
            [0.001847e-6,10873.986030480,2.903477885],
            [0.001391e-6,-8635.942003763,0.593891500],
            [0.001388e-6,-7.046236698,1.166145902],
            [0.001810e-6,-88860.057071188,0.487355242],
            [0.001288e-6,-1990.745017041,3.913022880],
            [0.001297e-6,23543.230504682,3.063805171],
            [0.001335e-6,-266.607041722,3.995764039],
            [0.001376e-6,10969.965257698,5.152914309],
            [0.001745e-6,244287.600007027,3.626395673],
            [0.001649e-6,31441.677569757,1.952049260],
            [0.001416e-6,9225.539273283,4.996408389],
            [0.001238e-6,4804.209275927,5.503379738],
            [0.001472e-6,4590.910180489,4.164913291],
            [0.001169e-6,6040.347246017,5.841719038],
            [0.001039e-6,5540.085789459,2.769753519],
            [0.001004e-6,-170.672870619,0.755008103],
            [0.001284e-6,10575.406682942,5.306538209],
            [0.001278e-6,71.812653151,4.713486491],
            [0.001321e-6,18209.330263660,2.624866359],
            [0.001297e-6,21228.392023546,0.382603541],
            [0.000954e-6,6282.095528923,0.882213514],
            [0.001145e-6,6058.731054289,1.169483931],
            [0.000979e-6,5547.199336460,5.448375984],
            [0.000987e-6,-6262.300454499,2.656486959],
            [0.001070e-6,-154717.609887482,1.827624012],
            [0.000991e-6,4701.116501708,4.387001801],
            [0.001155e-6,-14.227094002,3.042700750],
            [0.001176e-6,277.034993741,3.335519004],
            [0.000890e-6,13916.019109642,5.601498297],
            [0.000884e-6,-1551.045222648,1.088831705],
            [0.000876e-6,5017.508371365,3.969902609],
            [0.000806e-6,15110.466119866,5.142876744],
            [0.000773e-6,-4136.910433516,0.022067765],
            [0.001077e-6,175.166059800,1.844913056],
            [0.000954e-6,-6284.056171060,0.968480906],
            [0.000737e-6,5326.786694021,4.923831588],
            [0.000845e-6,-433.711737877,4.749245231],
            [0.000819e-6,8662.240323563,5.991247817],
            [0.000852e-6,199.072001436,2.189604979],
            [0.000723e-6,17256.631536341,6.068719637],
            [0.000940e-6,6037.244203762,6.197428148],
            [0.000885e-6,11712.955318231,3.280414875],
            [0.000706e-6,12559.038152982,2.824848947],
            [0.000732e-6,2379.164473572,2.501813417],
            [0.000764e-6,-6127.655450557,2.236346329],
            [0.000908e-6,131.541961686,2.521257490],
            [0.000907e-6,35371.887265976,3.370195967],
            [0.000673e-6,1066.495477190,3.876512374],
            [0.000814e-6,17654.780539750,4.627122566],
            [0.000630e-6,36.027866677,0.156368499],
            [0.000798e-6,515.463871093,5.151962502],
            [0.000798e-6,148.078724426,5.909225055],
            [0.000806e-6,309.278322656,6.054064447],
            [0.000607e-6,-39.617508346,2.839021623],
            [0.000601e-6,412.371096874,3.984225404],
            [0.000646e-6,11403.676995575,3.852959484],
            [0.000704e-6,13521.751441591,2.300991267],
            [0.000603e-6,-65147.619767937,4.140083146],
            [0.000609e-6,10177.257679534,0.437122327],
            [0.000631e-6,5767.611978898,4.026532329],
            [0.000576e-6,11087.285125918,4.760293101],
            [0.000674e-6,14945.316173554,6.270510511],
            [0.000726e-6,5429.879468239,6.039606892],
            [0.000710e-6,28766.924424484,5.672617711],
            [0.000647e-6,11856.218651625,3.397132627],
            [0.000678e-6,-5481.254918868,6.249666675],
            [0.000618e-6,22003.914634870,2.466427018],
            [0.000738e-6,6134.997125565,2.242668890],
            [0.000660e-6,625.670192312,5.864091907],
            [0.000694e-6,3496.032826134,2.668309141],
            [0.000531e-6,6489.261398429,1.681888780],
            [0.000611e-6,-143571.324284214,2.424978312],
            [0.000575e-6,12043.574281889,4.216492400],
            [0.000553e-6,12416.588502848,4.772158039],
            [0.000689e-6,4686.889407707,6.224271088],
            [0.000495e-6,7342.457780181,3.817285811],
            [0.000567e-6,3634.621024518,1.649264690],
            [0.000515e-6,18635.928454536,3.945345892],
            [0.000486e-6,-323.505416657,4.061673868],
            [0.000662e-6,25158.601719765,1.794058369],
            [0.000509e-6,846.082834751,3.053874588],
            [0.000472e-6,-12569.674818332,5.112133338],
            [0.000461e-6,6179.983075773,0.513669325],
            [0.000641e-6,83467.156352816,3.210727723],
            [0.000520e-6,10344.295065386,2.445597761],
            [0.000493e-6,18422.629359098,1.676939306],
            [0.000478e-6,1265.567478626,5.487314569],
            [0.000472e-6,-18.159247265,1.999707589],
            [0.000559e-6,11190.377900137,5.783236356],
            [0.000494e-6,9623.688276691,3.022645053],
            [0.000463e-6,5739.157790895,1.411223013],
            [0.000432e-6,16858.482532933,1.179256434],
            [0.000574e-6,72140.628666286,1.758191830],
            [0.000484e-6,17267.268201691,3.290589143],
            [0.000550e-6,4907.302050146,0.864024298],
            [0.000399e-6,14.977853527,2.094441910],
            [0.000491e-6,224.344795702,0.878372791],
            [0.000432e-6,20426.571092422,6.003829241],
            [0.000481e-6,5749.452731634,4.309591964],
            [0.000480e-6,5757.317038160,1.142348571],
            [0.000485e-6,6702.560493867,0.210580917],
            [0.000426e-6,6055.549660552,4.274476529],
            [0.000480e-6,5959.570433334,5.031351030],
            [0.000466e-6,12562.628581634,4.959581597],
            [0.000520e-6,39302.096962196,4.788002889],
            [0.000458e-6,12132.439962106,1.880103788],
            [0.000470e-6,12029.347187887,1.405611197],
            [0.000416e-6,-7477.522860216,1.082356330],
            [0.000449e-6,11609.862544012,4.179989585],
            [0.000465e-6,17253.041107690,0.353496295],
            [0.000362e-6,-4535.059436924,1.583849576],
            [0.000383e-6,21954.157609398,3.747376371],
            [0.000389e-6,17.252277143,1.395753179],
            [0.000331e-6,18052.929543158,0.566790582],
            [0.000430e-6,13517.870106233,0.685827538],
            [0.000368e-6,-5756.908003246,0.731374317],
            [0.000330e-6,10557.594160824,3.710043680],
            [0.000332e-6,20199.094959633,1.652901407],
            [0.000384e-6,11933.367960670,5.827781531],
            [0.000387e-6,10454.501386605,2.541182564],
            [0.000325e-6,15671.081759407,2.178850542],
            [0.000318e-6,138.517496871,2.253253037],
            [0.000305e-6,9388.005909415,0.578340206],
            [0.000352e-6,5749.861766548,3.000297967],
            [0.000311e-6,6915.859589305,1.693574249],
            [0.000297e-6,24072.921469776,1.997249392],
            [0.000363e-6,-640.877607382,5.071820966],
            [0.000323e-6,12592.450019783,1.072262823],
            [0.000341e-6,12146.667056108,4.700657997],
            [0.000290e-6,9779.108676125,1.812320441],
            [0.000342e-6,6132.028180148,4.322238614],
            [0.000329e-6,6268.848755990,3.033827743],
            [0.000374e-6,17996.031168222,3.388716544],
            [0.000285e-6,-533.214083444,4.687313233],
            [0.000338e-6,6065.844601290,0.877776108],
            [0.000276e-6,24.298513841,0.770299429],
            [0.000336e-6,-2388.894020449,5.353796034],
            [0.000290e-6,3097.883822726,4.075291557],
            [0.000318e-6,709.933048357,5.941207518],
            [0.000271e-6,13095.842665077,3.208912203],
            [0.000331e-6,6073.708907816,4.007881169],
            [0.000292e-6,742.990060533,2.714333592],
            [0.000362e-6,29088.811415985,3.215977013],
            [0.000280e-6,12359.966151546,0.710872502],
            [0.000267e-6,10440.274292604,4.730108488],
            [0.000262e-6,838.969287750,1.327720272],
            [0.000250e-6,16496.361396202,0.898769761],
            [0.000325e-6,20597.243963041,0.180044365],
            [0.000268e-6,6148.010769956,5.152666276],
            [0.000284e-6,5636.065016677,5.655385808],
            [0.000301e-6,6080.822454817,2.135396205],
            [0.000294e-6,-377.373607916,3.708784168],
            [0.000236e-6,2118.763860378,1.733578756],
            [0.000234e-6,5867.523359379,5.575209112],
            [0.000268e-6,-226858.238553767,0.069432392],
            [0.000265e-6,167283.761587465,4.369302826],
            [0.000280e-6,28237.233459389,5.304829118],
            [0.000292e-6,12345.739057544,4.096094132],
            [0.000223e-6,19800.945956225,3.069327406],
            [0.000301e-6,43232.306658416,6.205311188],
            [0.000264e-6,18875.525869774,1.417263408],
            [0.000304e-6,-1823.175188677,3.409035232],
            [0.000301e-6,109.945688789,0.510922054],
            [0.000260e-6,813.550283960,2.389438934],
            [0.000299e-6,316428.228673312,5.384595078],
            [0.000211e-6,5756.566278634,3.789392838],
            [0.000209e-6,5750.203491159,1.661943545],
            [0.000240e-6,12489.885628707,5.684549045],
            [0.000216e-6,6303.851245484,3.862942261],
            [0.000203e-6,1581.959348283,5.549853589],
            [0.000200e-6,5642.198242609,1.016115785],
            [0.000197e-6,-70.849445304,4.690702525],
            [0.000227e-6,6287.008003254,2.911891613],
            [0.000197e-6,533.623118358,1.048982898],
            [0.000205e-6,-6279.485421340,1.829362730],
            [0.000209e-6,-10988.808157535,2.636140084],
            [0.000208e-6,-227.526189440,4.127883842],
            [0.000191e-6,415.552490612,4.401165650],
            [0.000190e-6,29296.615389579,4.175658539],
            [0.000264e-6,66567.485864652,4.601102551],
            [0.000256e-6,-3646.350377354,0.506364778],
            [0.000188e-6,13119.721102825,2.032195842],
            [0.000185e-6,-209.366942175,4.694756586],
            [0.000198e-6,25934.124331089,3.832703118],
            [0.000195e-6,4061.219215394,3.308463427],
            [0.000234e-6,5113.487598583,1.716090661],
            [0.000188e-6,1478.866574064,5.686865780],
            [0.000222e-6,11823.161639450,1.942386641],
            [0.000181e-6,10770.893256262,1.999482059],
            [0.000171e-6,6546.159773364,1.182807992],
            [0.000206e-6,70.328180442,5.934076062],
            [0.000169e-6,20995.392966449,2.169080622],
            [0.000191e-6,10660.686935042,5.405515999],
            [0.000228e-6,33019.021112205,4.656985514],
            [0.000184e-6,-4933.208440333,3.327476868],
            [0.000220e-6,-135.625325010,1.765430262],
            [0.000166e-6,23141.558382925,3.454132746],
            [0.000191e-6,6144.558353121,5.020393445],
            [0.000180e-6,6084.003848555,0.602182191],
            [0.000163e-6,17782.732072784,4.960593133],
            [0.000225e-6,16460.333529525,2.596451817],
            [0.000222e-6,5905.702242076,3.731990323],
            [0.000204e-6,227.476132789,5.636192701],
            [0.000159e-6,16737.577236597,3.600691544],
            [0.000200e-6,6805.653268085,0.868220961],
            [0.000187e-6,11919.140866668,2.629456641],
            [0.000161e-6,127.471796607,2.862574720],
            [0.000205e-6,6286.666278643,1.742882331],
            [0.000189e-6,153.778810485,4.812372643],
            [0.000168e-6,16723.350142595,0.027860588],
            [0.000149e-6,11720.068865232,0.659721876],
            [0.000189e-6,5237.921013804,5.245313000],
            [0.000143e-6,6709.674040867,4.317625647],
            [0.000146e-6,4487.817406270,4.815297007],
            [0.000144e-6,-664.756045130,5.381366880],
            [0.000175e-6,5127.714692584,4.728443327],
            [0.000162e-6,6254.626662524,1.435132069],
            [0.000187e-6,47162.516354635,1.354371923],
            [0.000146e-6,11080.171578918,3.369695406],
            [0.000180e-6,-348.924420448,2.490902145],
            [0.000148e-6,151.047669843,3.799109588],
            [0.000157e-6,6197.248551160,1.284375887],
            [0.000167e-6,146.594251718,0.759969109],
            [0.000133e-6,-5331.357443741,5.409701889],
            [0.000154e-6,95.979227218,3.366890614],
            [0.000148e-6,-6418.140930027,3.384104996],
            [0.000128e-6,-6525.804453965,3.803419985],
            [0.000130e-6,11293.470674356,0.939039445],
            [0.000152e-6,-5729.506447149,0.734117523],
            [0.000138e-6,210.117701700,2.564216078],
            [0.000123e-6,6066.595360816,4.517099537],
            [0.000140e-6,18451.078546566,0.642049130],
            [0.000126e-6,11300.584221356,3.485280663],
            [0.000119e-6,10027.903195729,3.217431161],
            [0.000151e-6,4274.518310832,4.404359108],
            [0.000117e-6,6072.958148291,0.366324650],
            [0.000165e-6,-7668.637425143,4.298212528],
            [0.000117e-6,-6245.048177356,5.379518958],
            [0.000130e-6,-5888.449964932,4.527681115],
            [0.000121e-6,-543.918059096,6.109429504],
            [0.000162e-6,9683.594581116,5.720092446],
            [0.000141e-6,6219.339951688,0.679068671],
            [0.000118e-6,22743.409379516,4.881123092],
            [0.000129e-6,1692.165669502,0.351407289],
            [0.000126e-6,5657.405657679,5.146592349],
            [0.000114e-6,728.762966531,0.520791814],
            [0.000120e-6,52.596639600,0.948516300],
            [0.000115e-6,65.220371012,3.504914846],
            [0.000126e-6,5881.403728234,5.577502482],
            [0.000158e-6,163096.180360983,2.957128968],
            [0.000134e-6,12341.806904281,2.598576764],
            [0.000151e-6,16627.370915377,3.985702050],
            [0.000109e-6,1368.660252845,0.014730471],
            [0.000131e-6,6211.263196841,0.085077024],
            [0.000146e-6,5792.741760812,0.708426604],
            [0.000146e-6,-77.750543984,3.121576600],
            [0.000107e-6,5341.013788022,0.288231904],
            [0.000138e-6,6281.591377283,2.797450317],
            [0.000113e-6,-6277.552925684,2.788904128],
            [0.000115e-6,-525.758811831,5.895222200],
            [0.000138e-6,6016.468808270,6.096188999],
            [0.000139e-6,23539.707386333,2.028195445],
            [0.000146e-6,-4176.041342449,4.660008502],
            [0.000107e-6,16062.184526117,4.066520001],
            [0.000142e-6,83783.548222473,2.936315115],
            [0.000128e-6,9380.959672717,3.223844306],
            [0.000135e-6,6205.325306007,1.638054048],
            [0.000101e-6,2699.734819318,5.481603249],
            [0.000104e-6,-568.821874027,2.205734493],
            [0.000103e-6,6321.103522627,2.440421099],
            [0.000119e-6,6321.208885629,2.547496264],
            [0.000138e-6,1975.492545856,2.314608466],
            [0.000121e-6,137.033024162,4.539108237],
            [0.000123e-6,19402.796952817,4.538074405],
            [0.000119e-6,22805.735565994,2.869040566],
            [0.000133e-6,64471.991241142,6.056405489],
            [0.000129e-6,-85.827298831,2.540635083],
            [0.000131e-6,13613.804277336,4.005732868],
            [0.000104e-6,9814.604100291,1.959967212],
            [0.000112e-6,16097.679950283,3.589026260],
            [0.000123e-6,2107.034507542,1.728627253],
            [0.000121e-6,36949.230808424,6.072332087],
            [0.000108e-6,-12539.853380183,3.716133846],
            [0.000113e-6,-7875.671863624,2.725771122],
            [0.000109e-6,4171.425536614,4.033338079],
            [0.000101e-6,6247.911759770,3.441347021],
            [0.000113e-6,7330.728427345,0.656372122],
            [0.000113e-6,51092.726050855,2.791483066],
            [0.000106e-6,5621.842923210,1.815323326],
            [0.000101e-6,111.430161497,5.711033677],
            [0.000103e-6,909.818733055,2.812745443],
            [0.000101e-6,1790.642637886,1.965746028],
            #T
            [102.156724e-6,6283.075849991,4.249032005],
            [1.706807e-6,12566.151699983,4.205904248],
            [0.269668e-6,213.299095438,3.400290479],
            [0.265919e-6,529.690965095,5.836047367],
            [0.210568e-6,-3.523118349,6.262738348],
            [0.077996e-6,5223.693919802,4.670344204],
            [0.054764e-6,1577.343542448,4.534800170],
            [0.059146e-6,26.298319800,1.083044735],
            [0.034420e-6,-398.149003408,5.980077351],
            [0.032088e-6,18849.227549974,4.162913471],
            [0.033595e-6,5507.553238667,5.980162321],
            [0.029198e-6,5856.477659115,0.623811863],
            [0.027764e-6,155.420399434,3.745318113],
            [0.025190e-6,5746.271337896,2.980330535],
            [0.022997e-6,-796.298006816,1.174411803],
            [0.024976e-6,5760.498431898,2.467913690],
            [0.021774e-6,206.185548437,3.854787540],
            [0.017925e-6,-775.522611324,1.092065955],
            [0.013794e-6,426.598190876,2.699831988],
            [0.013276e-6,6062.663207553,5.845801920],
            [0.011774e-6,12036.460734888,2.292832062],
            [0.012869e-6,6076.890301554,5.333425680],
            [0.012152e-6,1059.381930189,6.222874454],
            [0.011081e-6,-7.113547001,5.154724984],
            [0.010143e-6,4694.002954708,4.044013795],
            [0.009357e-6,5486.777843175,3.416081409],
            [0.010084e-6,522.577418094,0.749320262],
            [0.008587e-6,10977.078804699,2.777152598],
            [0.008628e-6,6275.962302991,4.562060226],
            [0.008158e-6,-220.412642439,5.806891533],
            [0.007746e-6,2544.314419883,1.603197066],
            [0.007670e-6,2146.165416475,3.000200440],
            [0.007098e-6,74.781598567,0.443725817],
            [0.006180e-6,-536.804512095,1.302642751],
            [0.005818e-6,5088.628839767,4.827723531],
            [0.004945e-6,-6286.598968340,0.268305170],
            [0.004774e-6,1349.867409659,5.808636673],
            [0.004687e-6,-242.728603974,5.154890570],
            [0.006089e-6,1748.016413067,4.403765209],
            [0.005975e-6,-1194.447010225,2.583472591],
            [0.004229e-6,951.718406251,0.931172179],
            [0.005264e-6,553.569402842,2.336107252],
            [0.003049e-6,5643.178563677,1.362634430],
            [0.002974e-6,6812.766815086,1.583012668],
            [0.003403e-6,-2352.866153772,2.552189886],
            [0.003030e-6,419.484643875,5.286473844],
            [0.003210e-6,-7.046236698,1.863796539],
            [0.003058e-6,9437.762934887,4.226420633],
            [0.002589e-6,12352.852604545,1.991935820],
            [0.002927e-6,5216.580372801,2.319951253],
            [0.002425e-6,5230.807466803,3.084752833],
            [0.002656e-6,3154.687084896,2.487447866],
            [0.002445e-6,10447.387839604,2.347139160],
            [0.002990e-6,4690.479836359,6.235872050],
            [0.002890e-6,5863.591206116,0.095197563],
            [0.002498e-6,6438.496249426,2.994779800],
            [0.001889e-6,8031.092263058,3.569003717],
            [0.002567e-6,801.820931124,3.425611498],
            [0.001803e-6,-71430.695617928,2.192295512],
            [0.001782e-6,3.932153263,5.180433689],
            [0.001694e-6,-4705.732307544,4.641779174],
            [0.001704e-6,-1592.596013633,3.997097652],
            [0.001735e-6,5849.364112115,0.417558428],
            [0.001643e-6,8429.241266467,2.180619584],
            [0.001680e-6,38.133035638,4.164529426],
            [0.002045e-6,7084.896781115,0.526323854],
            [0.001458e-6,4292.330832950,1.356098141],
            [0.001437e-6,20.355319399,3.895439360],
            [0.001738e-6,6279.552731642,0.087484036],
            [0.001367e-6,14143.495242431,3.987576591],
            [0.001344e-6,7234.794256242,0.090454338],
            [0.001438e-6,11499.656222793,0.974387904],
            [0.001257e-6,6836.645252834,1.509069366],
            [0.001358e-6,11513.883316794,0.495572260],
            [0.001628e-6,7632.943259650,4.968445721],
            [0.001169e-6,103.092774219,2.838496795],
            [0.001162e-6,4164.311989613,3.408387778],
            [0.001092e-6,6069.776754553,3.617942651],
            [0.001008e-6,17789.845619785,0.286350174],
            [0.001008e-6,639.897286314,1.610762073],
            [0.000918e-6,10213.285546211,5.532798067],
            [0.001011e-6,-6256.777530192,0.661826484],
            [0.000753e-6,16730.463689596,3.905030235],
            [0.000737e-6,11926.254413669,4.641956361],
            [0.000694e-6,3340.612426700,2.111120332],
            [0.000701e-6,3894.181829542,2.760823491],
            [0.000689e-6,-135.065080035,4.768800780],
            [0.000700e-6,13367.972631107,5.760439898],
            [0.000664e-6,6040.347246017,1.051215840],
            [0.000654e-6,5650.292110678,4.911332503],
            [0.000788e-6,6681.224853400,4.699648011],
            [0.000628e-6,5333.900241022,5.024608847],
            [0.000755e-6,-110.206321219,4.370971253],
            [0.000628e-6,6290.189396992,3.660478857],
            [0.000635e-6,25132.303399966,4.121051532],
            [0.000534e-6,5966.683980335,1.173284524],
            [0.000543e-6,-433.711737877,0.345585464],
            [0.000517e-6,-1990.745017041,5.414571768],
            [0.000504e-6,5767.611978898,2.328281115],
            [0.000485e-6,5753.384884897,1.685874771],
            [0.000463e-6,7860.419392439,5.297703006],
            [0.000604e-6,515.463871093,0.591998446],
            [0.000443e-6,12168.002696575,4.830881244],
            [0.000570e-6,199.072001436,3.899190272],
            [0.000465e-6,10969.965257698,0.476681802],
            [0.000424e-6,-7079.373856808,1.112242763],
            [0.000427e-6,735.876513532,1.994214480],
            [0.000478e-6,-6127.655450557,3.778025483],
            [0.000414e-6,10973.555686350,5.441088327],
            [0.000512e-6,1589.072895284,0.107123853],
            [0.000378e-6,10984.192351700,0.915087231],
            [0.000402e-6,11371.704689758,4.107281715],
            [0.000453e-6,9917.696874510,1.917490952],
            [0.000395e-6,149.563197135,2.763124165],
            [0.000371e-6,5739.157790895,3.112111866],
            [0.000350e-6,11790.629088659,0.440639857],
            [0.000356e-6,6133.512652857,5.444568842],
            [0.000344e-6,412.371096874,5.676832684],
            [0.000383e-6,955.599741609,5.559734846],
            [0.000333e-6,6496.374945429,0.261537984],
            [0.000340e-6,6055.549660552,5.975534987],
            [0.000334e-6,1066.495477190,2.335063907],
            [0.000399e-6,11506.769769794,5.321230910],
            [0.000314e-6,18319.536584880,2.313312404],
            [0.000424e-6,1052.268383188,1.211961766],
            [0.000307e-6,63.735898303,3.169551388],
            [0.000329e-6,29.821438149,6.106912080],
            [0.000357e-6,6309.374169791,4.223760346],
            [0.000312e-6,-3738.761430108,2.180556645],
            [0.000301e-6,309.278322656,1.499984572],
            [0.000268e-6,12043.574281889,2.447520648],
            [0.000257e-6,12491.370101415,3.662331761],
            [0.000290e-6,625.670192312,1.272834584],
            [0.000256e-6,5429.879468239,1.913426912],
            [0.000339e-6,3496.032826134,4.165930011],
            [0.000283e-6,3930.209696220,4.325565754],
            [0.000241e-6,12528.018664345,3.832324536],
            [0.000304e-6,4686.889407707,1.612348468],
            [0.000259e-6,16200.772724501,3.470173146],
            [0.000238e-6,12139.553509107,1.147977842],
            [0.000236e-6,6172.869528772,3.776271728],
            [0.000296e-6,-7058.598461315,0.460368852],
            [0.000306e-6,10575.406682942,0.554749016],
            [0.000251e-6,17298.182327326,0.834332510],
            [0.000290e-6,4732.030627343,4.759564091],
            [0.000261e-6,5884.926846583,0.298259862],
            [0.000249e-6,5547.199336460,3.749366406],
            [0.000213e-6,11712.955318231,5.415666119],
            [0.000223e-6,4701.116501708,2.703203558],
            [0.000268e-6,-640.877607382,0.283670793],
            [0.000209e-6,5636.065016677,1.238477199],
            [0.000193e-6,10177.257679534,1.943251340],
            [0.000182e-6,6283.143160294,2.456157599],
            [0.000184e-6,-227.526189440,5.888038582],
            [0.000182e-6,-6283.008539689,0.241332086],
            [0.000228e-6,-6284.056171060,2.657323816],
            [0.000166e-6,7238.675591600,5.930629110],
            [0.000167e-6,3097.883822726,5.570955333],
            [0.000159e-6,-323.505416657,5.786670700],
            [0.000154e-6,-4136.910433516,1.517805532],
            [0.000176e-6,12029.347187887,3.139266834],
            [0.000167e-6,12132.439962106,3.556352289],
            [0.000153e-6,202.253395174,1.463313961],
            [0.000157e-6,17267.268201691,1.586837396],
            [0.000142e-6,83996.847317911,0.022670115],
            [0.000152e-6,17260.154654690,0.708528947],
            [0.000144e-6,6084.003848555,5.187075177],
            [0.000135e-6,5756.566278634,1.993229262],
            [0.000134e-6,5750.203491159,3.457197134],
            [0.000144e-6,5326.786694021,6.066193291],
            [0.000160e-6,11015.106477335,1.710431974],
            [0.000133e-6,3634.621024518,2.836451652],
            [0.000134e-6,18073.704938650,5.453106665],
            [0.000134e-6,1162.474704408,5.326898811],
            [0.000128e-6,5642.198242609,2.511652591],
            [0.000160e-6,632.783739313,5.628785365],
            [0.000132e-6,13916.019109642,0.819294053],
            [0.000122e-6,14314.168113050,5.677408071],
            [0.000125e-6,12359.966151546,5.251984735],
            [0.000121e-6,5749.452731634,2.210924603],
            [0.000136e-6,-245.831646229,1.646502367],
            [0.000120e-6,5757.317038160,3.240883049],
            [0.000134e-6,12146.667056108,3.059480037],
            [0.000137e-6,6206.809778716,1.867105418],
            [0.000141e-6,17253.041107690,2.069217456],
            [0.000129e-6,-7477.522860216,2.781469314],
            [0.000116e-6,5540.085789459,4.281176991],
            [0.000116e-6,9779.108676125,3.320925381],
            [0.000129e-6,5237.921013804,3.497704076],
            [0.000113e-6,5959.570433334,0.983210840],
            [0.000122e-6,6282.095528923,2.674938860],
            [0.000140e-6,-11.045700264,4.957936982],
            [0.000108e-6,23543.230504682,1.390113589],
            [0.000106e-6,-12569.674818332,0.429631317],
            [0.000110e-6,-266.607041722,5.501340197],
            [0.000115e-6,12559.038152982,4.691456618],
            [0.000134e-6,-2388.894020449,0.577313584],
            [0.000109e-6,10440.274292604,6.218148717],
            [0.000102e-6,-543.918059096,1.477842615],
            [0.000108e-6,21228.392023546,2.237753948],
            [0.000101e-6,-4535.059436924,3.100492232],
            [0.000103e-6,76.266071276,5.594294322],
            [0.000104e-6,949.175608970,5.674287810],
            [0.000101e-6,13517.870106233,2.196632348],
            [0.000100e-6,11933.367960670,4.056084160],
            #T^2
            [4.322990e-6,6283.075849991,2.642893748],
            [0.406495e-6,0.000000000,4.712388980],
            [0.122605e-6,12566.151699983,2.438140634],
            [0.019476e-6,213.299095438,1.642186981],
            [0.016916e-6,529.690965095,4.510959344],
            [0.013374e-6,-3.523118349,1.502210314],
            [0.008042e-6,26.298319800,0.478549024],
            [0.007824e-6,155.420399434,5.254710405],
            [0.004894e-6,5746.271337896,4.683210850],
            [0.004875e-6,5760.498431898,0.759507698],
            [0.004416e-6,5223.693919802,6.028853166],
            [0.004088e-6,-7.113547001,0.060926389],
            [0.004433e-6,77713.771467920,3.627734103],
            [0.003277e-6,18849.227549974,2.327912542],
            [0.002703e-6,6062.663207553,1.271941729],
            [0.003435e-6,-775.522611324,0.747446224],
            [0.002618e-6,6076.890301554,3.633715689],
            [0.003146e-6,206.185548437,5.647874613],
            [0.002544e-6,1577.343542448,6.232904270],
            [0.002218e-6,-220.412642439,1.309509946],
            [0.002197e-6,5856.477659115,2.407212349],
            [0.002897e-6,5753.384884897,5.863842246],
            [0.001766e-6,426.598190876,0.754113147],
            [0.001738e-6,-796.298006816,2.714942671],
            [0.001695e-6,522.577418094,2.629369842],
            [0.001584e-6,5507.553238667,1.341138229],
            [0.001503e-6,-242.728603974,0.377699736],
            [0.001552e-6,-536.804512095,2.904684667],
            [0.001370e-6,-398.149003408,1.265599125],
            [0.001889e-6,-5573.142801634,4.413514859],
            [0.001722e-6,6069.776754553,2.445966339],
            [0.001124e-6,1059.381930189,5.041799657],
            [0.001258e-6,553.569402842,3.849557278],
            [0.000831e-6,951.718406251,2.471094709],
            [0.000767e-6,4694.002954708,5.363125422],
            [0.000756e-6,1349.867409659,1.046195744],
            [0.000775e-6,-11.045700264,0.245548001],
            [0.000597e-6,2146.165416475,4.543268798],
            [0.000568e-6,5216.580372801,4.178853144],
            [0.000711e-6,1748.016413067,5.934271972],
            [0.000499e-6,12036.460734888,0.624434410],
            [0.000671e-6,-1194.447010225,4.136047594],
            [0.000488e-6,5849.364112115,2.209679987],
            [0.000621e-6,6438.496249426,4.518860804],
            [0.000495e-6,-6286.598968340,1.868201275],
            [0.000456e-6,5230.807466803,1.271231591],
            [0.000451e-6,5088.628839767,0.084060889],
            [0.000435e-6,5643.178563677,3.324456609],
            [0.000387e-6,10977.078804699,4.052488477],
            [0.000547e-6,161000.685737473,2.841633844],
            [0.000522e-6,3154.687084896,2.171979966],
            [0.000375e-6,5486.777843175,4.983027306],
            [0.000421e-6,5863.591206116,4.546432249],
            [0.000439e-6,7084.896781115,0.522967921],
            [0.000309e-6,2544.314419883,3.172606705],
            [0.000347e-6,4690.479836359,1.479586566],
            [0.000317e-6,801.820931124,3.553088096],
            [0.000262e-6,419.484643875,0.606635550],
            [0.000248e-6,6836.645252834,3.014082064],
            [0.000245e-6,-1592.596013633,5.519526220],
            [0.000225e-6,4292.330832950,2.877956536],
            [0.000214e-6,7234.794256242,1.605227587],
            [0.000205e-6,5767.611978898,0.625804796],
            [0.000180e-6,10447.387839604,3.499954526],
            [0.000229e-6,199.072001436,5.632304604],
            [0.000214e-6,639.897286314,5.960227667],
            [0.000175e-6,-433.711737877,2.162417992],
            [0.000209e-6,515.463871093,2.322150893],
            [0.000173e-6,6040.347246017,2.556183691],
            [0.000184e-6,6309.374169791,4.732296790],
            [0.000227e-6,149854.400134205,5.385812217],
            [0.000154e-6,8031.092263058,5.120720920],
            [0.000151e-6,5739.157790895,4.815000443],
            [0.000197e-6,7632.943259650,0.222827271],
            [0.000197e-6,74.781598567,3.910456770],
            [0.000138e-6,6055.549660552,1.397484253],
            [0.000149e-6,-6127.655450557,5.333727496],
            [0.000137e-6,3894.181829542,4.281749907],
            [0.000135e-6,9437.762934887,5.979971885],
            [0.000139e-6,-2352.866153772,4.715630782],
            [0.000142e-6,6812.766815086,0.513330157],
            [0.000120e-6,-4705.732307544,0.194160689],
            [0.000131e-6,-71430.695617928,0.000379226],
            [0.000124e-6,6279.552731642,2.122264908],
            [0.000108e-6,-6256.777530192,0.883445696],
            #T^3
            [0.143388e-6,6283.075849991,1.131453581],
            [0.006671e-6,12566.151699983,0.775148887],
            [0.001480e-6,155.420399434,0.480016880],
            [0.000934e-6,213.299095438,6.144453084],
            [0.000795e-6,529.690965095,2.941595619],
            [0.000673e-6,5746.271337896,0.120415406],
            [0.000672e-6,5760.498431898,5.317009738],
            [0.000389e-6,-220.412642439,3.090323467],
            [0.000373e-6,6062.663207553,3.003551964],
            [0.000360e-6,6076.890301554,1.918913041],
            [0.000316e-6,-21.340641002,5.545798121],
            [0.000315e-6,-242.728603974,1.884932563],
            [0.000278e-6,206.185548437,1.266254859],
            [0.000238e-6,-536.804512095,4.532664830],
            [0.000185e-6,522.577418094,4.578313856],
            [0.000245e-6,18849.227549974,0.587467082],
            [0.000180e-6,426.598190876,5.151178553],
            [0.000200e-6,553.569402842,5.355983739],
            [0.000141e-6,5223.693919802,1.336556009],
            [0.000104e-6,5856.477659115,4.239842759],
            #T^4
            [0.003826e-6,6283.075849991,5.705257275],
            [0.000303e-6,12566.151699983,5.407132842],
            [0.000209e-6,155.420399434,1.989815753]]

    #自J200.0经过的时间，单位：千儒略年
    T=((DATE1-DJ00)+DATE2)/DJM
    
    #=================
    #Topocentric terms（以地面上的某点为中心的）
    #=================

    #将UT转换为以弧度为单位的当地太阳时间
    TSOL=(UT%1.0)*D2PI+ELONG
    
    #基于以下理论:  Simon et al. 1994.

    #将时间与度和角秒之间的关系联系起来.
    W=T/3600.0

    #太阳的平赤经.
    ELSUN=(280.46645683+1296027711.03429*W)%360.0*DD2R

    #太阳的平近点角.
    EMSUN=(357.52910918+1295965810.481*W)%360.0*DD2R

    #日月之间平均距离
    D=(297.85019547+16029616012.090*W)%360.0*DD2R

    #木星的平赤经.
    ELJ=(34.35151874+109306899.89453*W)%360.0*DD2R

    #土星的平赤经.
    ELS=(50.07744430+44046398.47038*W)%360.0*DD2R
    
    #TOPOCENTRIC TERMS:  Moyer 1981 and Murray 1983.
    WT=+0.00029e-10*U*ma.sin(TSOL+ELSUN-ELS)\
        +0.00100e-10*U*ma.sin(TSOL-2.0*EMSUN)\
        +0.00133e-10*U*ma.sin(TSOL-D)\
        +0.00133e-10*U*ma.sin(TSOL+ELSUN-ELJ)\
        -0.00229e-10*U*ma.sin(TSOL+2.0*ELSUN+EMSUN)\
        -0.02200e-10*V*ma.cos(ELSUN+EMSUN)\
        +0.05312e-10*U*ma.sin(TSOL-EMSUN)\
        -0.13677e-10*U*ma.sin(TSOL + 2.0*ELSUN)\
        -1.31840e-10*V*ma.cos(ELSUN)\
        +3.17679e-10*U*ma.sin(TSOL)
    
    #=====================
    #Fairhead et al. model
    #=====================
    
    #T**0
    W0=0.0
    for J in range(473,-1,-1):
        W0=W0+FAIRHD[J][0]*ma.sin(FAIRHD[J][1]*T + FAIRHD[J][2])
    
    #T**1
    W1=0.0
    for J in range(678,473,-1):
        W1=W1+FAIRHD[J][0]*ma.sin(FAIRHD[J][1]*T + FAIRHD[J][2])
    
    #T**2
    W2=0.0
    for J in range(763,678,-1):
        W2=W2+FAIRHD[J][0]*ma.sin(FAIRHD[J][1]*T + FAIRHD[J][2])
    
    #T**3
    W3=0.0
    for J in range(783,763,-1):
        W3=W3+FAIRHD[J][0]*ma.sin(FAIRHD[J][1]*T + FAIRHD[J][2])
    
    #T**4
    W4=0.0
    for J in range(786,783,-1):
        W4=W4+FAIRHD[J][0]*ma.sin(FAIRHD[J][1]*T + FAIRHD[J][2])
    
    #T的不同次项结合
    WF=T*(T*(T*(T*W4+W3)+W2)+W1)+W0
    
    #调整使用JPL的行星质量而不是IAU.
    WJ=0.00065e-6*ma.sin(6069.776754*T+4.021194)+\
        0.00033e-6*ma.sin(213.299095*T+5.543132)+\
            (-0.00196e-6*ma.sin(6208.294251*T+5.696701))+\
                (-0.00173e-6*ma.sin(74.781599*T+2.435900))+\
                    0.03638e-6*T*T
    #以秒为单位的TDB-TT时间差
    DTDB=WT+WF+WJ
    
    return(DTDB)

def pymDtf2d(SCALE,IY,IM,ID,IHR,IMN,SEC):
    '''
    Encode date and time fields into 2-part Julian Date (or in the case
    of UTC a quasi-JD form that includes special provision for leap seconds).

    Parameters
    ----------
    scale : str
        time scale ID(Only the value "UTC" is significant)
    iy : int
        year in Gregorian calendar
    im : int
        month in Gregorian calendar
    id : int
        day in Gregorian calendar
    ihr : int
        hour
    imn : int
        minute
    sec : float
        seconds

    Returns
    -------
    d1,d2 : float
        2-part Julian Date
    J : ValueError
        3: 'both of next two',
        2: 'time is after end of day',
        1: 'dubious year', 
       -1: 'bad year,  the year is simply valid from -4800 March 1',
       -2: 'bad month, the month is not from 1 to 12',
       -3: 'bad day,   the day is not related to the month',
       -4: 'bad hour',
       -5: 'bad minute',
       -6: 'bad second', 
    '''
    #一天的秒数
    D2S = 86400.0
    D1,D2=0,0
    i=1
    while i<2:
    
        #当日的儒略日数
        DJ,W,JS=pymCal2jd(IY,IM,ID)
    
        if (JS!=0):
            print('ERROR',JS)
            break
        DJ=DJ+W
        
        #一天的长度以及暂时认为的最后一分钟的分长（用于闰秒情况）
        DAY=D2S
        SECLIM=60.0
        
        #处理在UTC中出现闰秒时的情况
        if (SCALE=='UTC'):
            
            #当天0时的TAI-UTC时间差.
            DAT0,JS=pymDat(IY,IM,ID,0.0)
            if (JS<0):
                print('ERROR',JS)
                break
            
            #当天12h的TAI-UTC时间差（判断是否存在drift——1972年之前情况）
            DAT12,JS=pymDat(IY,IM,ID,0.5)
            if (JS<0):
                print('ERROR',JS)
                break
           
            #第二天0h的TAI-UTC时间差（判断是否存在跳秒）
            IY2,IM2,ID2,W,JS=pymJd2cal(DJ,1.5)
            if (JS!=0):
                print('ERROR',JS)
                break
            DAT24,JS=pymDat(IY2,IM2,ID2,0.0)
            if (JS<0):
                print('ERROR',JS)
                break
            
            #当天和第二天之间TAI-UTC是否有突然变化。
            DLEAP=DAT24-(2.0*DAT12-DAT0)
            
            #如果是闰秒日，改正日常以及当天最后一分钟的分长
            DAY=DAY+DLEAP
            if (IHR==23)&(IMN==59):
                SECLIM+=DLEAP
        
        #验证时间.
        if (IHR>=0)&(IHR<=23):
            if (IMN>=0)&(IMN<=59):
                if (SEC>=0.0):
                    if (SEC>=SECLIM):
                        JS=JS+2
                else:
                    JS=-6
            else:
                JS=-5
        else:
            JS=-4
        
        if (JS<0):
            print('ERROR',JS)
            break
        
        #以天为单位的时间.
        TIME=(60.0*float(60*IHR+IMN)+SEC)/DAY
        
        D1 = DJ
        D2 = TIME
  
        i+=1
    J=JS
    
    return(D1,D2,J)

def pymEpb(DJ1,DJ2):
    '''
    Julian Date to Besselian Epoch.

    Parameters
    ----------
    dj1 : float
        Julian Date    
    dj2 : float
        Julian Date

    Returns
    -------
    function value : float
        Besselian Epoch

    '''

    #J2000.0
    DJ00=2451545.0
    #J2000.0 减去 B1900.0 (2415019.81352)的天数
    D1900=36524.68648
    #B1900的回归年长度(天)
    TY=365.242198781

    EPB=1900.0+((DJ1-DJ00)+(DJ2+D1900))/TY

    return(EPB)

def pymEpb2jd(EPB):
    '''
    Besselian Epoch to Julian Date

    Parameters
    ----------
    epb : float
        Besselian Epoch

    Returns
    -------
    djm0 : float
        MJD zero-point: always 2400000.5    
    djm : float
        Modified Julian Date

    '''

    #B1900的回归年长度(天)
    TY=365.242198781
    
    DJM0=2400000.5
    DJM=15019.81352+(EPB-1900.0)*TY
        
    return(DJM0,DJM)

def pymEpj(DJ1,DJ2):
    '''
    Julian Date to Julian Epoch

    Parameters
    ----------
    dj1 : float
        Julian Date    
    dj2 : float
        Julian Date

    Returns
    -------
    function value : float
        Julian Epoch

    '''

    #参考日期(J2000.0), JD
    DJ00=2451545.0
    #儒略年的天数
    DJY=365.25
    
    EPJ=2000.0+((DJ1-DJ00)+DJ2)/DJY
    
    return(EPJ)

def pymEpj2jd (EPJ):
    '''
    Julian Epoch to Julian Date

    Parameters
    ----------
    epj : float
        Julian Epoch (e.g. 1996.8)

    Returns
    -------
    djm0 : float
        MJD zero-point: always 2400000.5     
    djm : float
        Modified Julian Date

    '''
    DJM0=2400000.5
    DJM=51544.5+(EPJ-2000.0)*365.25
    
    return(DJM0,DJM)

def pymTaitt(TAI1,TAI2):
    '''
    Time scale transformation:  International Atomic Time, TAI, to
    Terrestrial Time, TT.

    Parameters
    ----------
    tai1 : float
        TAI as a 2-part Julian Date    
    tai2 : float
        TAI as a 2-part Julian Date

    Returns
    -------
    tt1,tt2 : float
        TT as a 2-part Julian Date
    J : ValueError
        0 : OK
    '''
    #地球时与国际原子时之间的时差（单位：天）
    DTAT=32.184/86400.0
    
    #尽可能保留精度的结果
    if (np.abs(TAI1)>np.abs(TAI2)):
        TT1=TAI1
        TT2=TAI2+DTAT
    else:
        TT1=TAI1+DTAT
        TT2=TAI2
    
    J=0
    return(TT1,TT2,J)

def pymTaiut1(TAI1,TAI2,DTA):
    '''
    Time scale transformation:  International Atomic Time, TAI, to
    Universal Time, UT1.

    Parameters
    ----------
    tai1 : float
        TAI as a 2-part Julian Date    
    tai2 : float
        TAI as a 2-part Julian Date    
    dta : float
        UT1-TAI in seconds

    Returns
    -------
    ut11,ut12 : float
        UT1 as a 2-part Julian Date
    J : ValueError
        0 : OK
    '''
    #一天的秒长
    D2S=86400.0
    
    #维持精度
    DTAD=DTA/D2S
    if (np.abs(TAI1)>np.abs(TAI2)):
        UT11=TAI1
        UT12=TAI2+DTAD
    else:
        UT11=TAI1+DTAD
        UT12=TAI2
    
    J=0
    return(UT11,UT12,J)

def pymTaiutc(TAI1,TAI2):
    '''
    Time scale transformation:  International Atomic Time, TAI, to
    Coordinated Universal Time, UTC.

    Parameters
    ----------
    tai1 : float
        TAI as a 2-part Julian Date
    tai2 : float
        TAI as a 2-part Julian Date

    Returns
    -------
    utc1 : float
        UTC as a 2-part quasi Julian Date
    utc2 : float
        UTC as a 2-part quasi Julian Date
    J : ValueError
        1: 'dubious year',
       -1: 'unacceptable date',
    '''

    UTC1,UTC2=0,0
    
    #将两参数日期按先大后小排列
    BIG1=np.abs(TAI1)>=np.abs(TAI2)
    if (BIG1):
        A1=TAI1
        A2=TAI2
    else:
        A1=TAI2
        A2=TAI1
    
    #初始假设日期
    U1=A1
    U2=A2
    
    i=1
    while i<2:
        
        #迭代
        for j in range(3):
            
            #假设UTC
            G1,G2,JS=pymUtctai(U1, U2)
            if (JS<0):
                print('ERROR',JS)
                break
        
            #调整UTC
            U2=U2+(A1-G1)
            U2=U2+(A2-G2)
    
        #得到UTC的结果
        if (BIG1):
            UTC1=U1
            UTC2=U2
        else:
            UTC1=U2
            UTC2=U1
    
        i+=1
    J=JS
    return(UTC1,UTC2,J)

def pymTcbtdb(TCB1,TCB2):
    '''
    Time scale transformation:  Barycentric Coordinate Time, TCB, to
    Barycentric Dynamical Time, TDB.

    Parameters
    ----------
    tcb1 : float
        TCB as a 2-part Julian Date
    tcb2 : float
        TCB as a 2-part Julian Date

    Returns
    -------
    tdb1 : float
        TDB as a 2-part Julian Date
    tdb2 :float
        TDB as a 2-part Julian Date
    J : ValueError
        0 : OK
    '''
    #1977 Jan 1.0 TAI = 1977/1/1 00:00:32.184 TCB, 两参数的儒略日
    T77TD=2443144.0
    T77TF=0.5003725
    
    #L_B, and TDB0 (d)，TCB-TDB之间的关系
    ELB=1.550519768e-8
    TDB0=-6.55e-5/86400.0
    
    if (np.abs(TCB1)>np.abs(TCB2)):
        D=TCB1-T77TD
        TDB1=TCB1
        TDB2=TCB2+TDB0-(D+(TCB2-T77TF))*ELB
    else:
        D=TCB2-T77TD
        TDB1=TCB1+TDB0-(D+(TCB1-T77TF))*ELB
        TDB2=TCB2
    
    J=0
    return(TDB1,TDB2,J)

def pymTcgtt(TCG1,TCG2):
    '''
    Time scale transformation:  Geocentric Coordinate Time, TCG, to
    Terrestrial Time, TT.

    Parameters
    ----------
    tcg1 : float
        TCG as a 2-part Julian Date
    tcg2 : float
        TCG as a 2-part Julian Date

    Returns
    -------
    tt1 : float
        TT as a 2-part Julian Date
    tt2 : float
        TT as a 2-part Julian Date
    J : ValueError
        0 : OK
    '''  
    #约简儒略日零点
    DJM0=2400000.5
    
    #1977 Jan 1 00:00:32.184 TT, 儒略日期
    T77T=43144.0003725
    
    #L_G = 1 - dTT/dTCG，TT和TCG之间的关系
    ELG=6.969290134e-10
    
    #尽可能保留精度
    if (np.abs(TCG1)>(np.abs(TCG2))):
        TT1=TCG1
        TT2=TCG2-((TCG1-DJM0)+(TCG2-T77T))*ELG
    else:
        TT1=TCG1-((TCG2-DJM0)+(TCG1-T77T))*ELG
        TT2=TCG2
    
    J=0
    return(TT1,TT2,J)

def pymTdbtcb(TDB1,TDB2):
    '''
    Time scale transformation:  Barycentric Dynamical Time, TDB, to
    Barycentric Coordinate Time, TCB.

    Parameters
    ----------
    tdb1 : float
        TDB as a 2-part Julian Date
    tdb2 : float
        TDB as a 2-part Julian Date

    Returns
    -------
    tcb1 : float
        TCB as a 2-part Julian Date
    tcb2 : float
        TCB as a 2-part Julian Date
    J : ValueError
        0 : OK
    '''    
    #1977 Jan 1.0 TAI = 1977/1/1 00:00:32.184 TCB, 作为两参数的儒略日期
    T77TD=2443144.0
    T77TF=0.5003725
    
    #L_B， TDB0 (d)，TDB-TCB之间的关系
    ELB=1.550519768e-8
    TDB0=-6.55e-5/86400.0
    ELBB=ELB/(1.0-ELB)
    
    if (np.abs(TDB1)>np.abs(TDB2)):
        D=T77TD-TDB1
        F=TDB2-TDB0
        TCB1=TDB1
        TCB2=F-(D-(F-T77TF))*ELBB
    else:
        D=T77TD-TDB2
        F=TDB1-TDB0
        TCB1=F-(D-(F-T77TF))*ELBB
        TCB2=TDB2
    
    J=0
    return(TCB1,TCB2,J)

def pymTdbtt(TDB1,TDB2,DTR):
    '''
    Time scale transformation:  Barycentric Dynamical Time, TDB, to
    Terrestrial Time, TT.

    Parameters
    ----------
    tdb1 : float
        TDB as a 2-part Julian Date    
    tdb2 : float
        TDB as a 2-part Julian Date    
    dtr : float
        TDB-TT in seconds

    Returns
    -------
    tt1 : float
        TT as a 2-part Julian Date    
    tt2 : float
        TT as a 2-part Julian Date    
    J : ValueError
        0 : OK
    '''
    #一天的秒长
    D2S=86400.0
    
    DTRD=DTR/D2S
    if (np.abs(TDB1)>np.abs(TDB2)):
        TT1=TDB1
        TT2=TDB2-DTRD
    else:
        TT1=TDB1-DTRD
        TT2=TDB2
    
    J=0  
    return(TT1,TT2,J)

def pymTf2d(S,IHOUR,IMIN,SEC):
    '''
    Convert hours, minutes, seconds to days

    Parameters
    ----------
    s : bytes
        sign:  '-' = negative, otherwise positive
    ihour : int
        hours
    imin : int
        minutes
    sec : float
        seconds
    
    Returns
    -------
    days : float
        interval in days
    J : ValueError
        1:'ihour outside range 0-23',
        2:'imin outside range 0-59',
        3:'sec outside range 0-59.999...'
    '''

    #一天的秒数
    D2S=86400.0
    
    #判断输入数据是否有误
    J=0
    if (SEC<0)|(SEC>=60):
        J=3
    if (IMIN<0)|(IMIN>59):
        J=2
    if (IHOUR<0)|(IHOUR>23):
        J=1
        
    W=(60.0*(60.0*float(np.abs(IHOUR))+float(np.abs(IMIN)))+np.abs(SEC))/D2S
    
    if (S=='-'):
        W=-W
    
    DAYS=W

    return(DAYS,J)

def pymTttai(TT1,TT2):
    '''
    Time scale transformation:  Terrestrial Time, TT, to International
    Atomic Time, TAI.

    Parameters
    ----------
    tt1 : float
        TT as a 2-part Julian Date
    tt2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    tai1 : float
        TAI as a 2-part Julian Date
    tai2 : float
        TAI as a 2-part Julian Date
    J : ValueError
        0 : OK
    '''
    #TT-TAI的时差，单位：天.
    DTAT = 32.184/86400.0
    
    if (np.abs(TT1)>np.abs(TT2)):
        TAI1=TT1
        TAI2=TT2-DTAT
    else:
        TAI1=TT1-DTAT
        TAI2=TT2
    
    J=0
    return(TAI1,TAI2,J)

def pymTttcg(TT1,TT2):
    '''
    Time scale transformation:  Terrestrial Time, TT, to Geocentric
    Coordinate Time, TCG.

    Parameters
    ----------
    tt1 : float
        TT as a 2-part Julian Date
    tt2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    tcg1 : float
        TCG as a 2-part Julian Date
    tcg2 : float
        TCG as a 2-part Julian Date
    J : ValueError
        0 : OK
    '''    
    #约简儒略日零点
    DJM0=2400000.5
    
    #1977 Jan 1 00:00:32.184 TT, 约简儒略日下的表达
    T77T=43144.0003725
    
    #L_G = 1 - dTT/dTCG，TT和TCG之间关系的定义
    ELG=6.969290134e-10
    
    #TT 与 TCG 之间的关系
    ELGG=ELG/(1.0-ELG)
    
    #尽可能保留精度的结果
    if (np.abs(TT1)>np.abs(TT2)):
        TCG1=TT1
        TCG2=TT2+((TT1-DJM0)+(TT2-T77T))*ELGG
    else:
        TCG1=TT1+((TT2-DJM0)+(TT1-T77T))*ELGG
        TCG2=TT2
    
    J=0    
    return(TCG1,TCG2,J)

def pymTttdb(TT1,TT2,DTR):
    '''
    Time scale transformation:  Terrestrial Time, TT, to Barycentric
    Dynamical Time, TDB.

    Parameters
    ----------
    tt1 : float
        TT as a 2-part Julian Date
    tt2 : float
        TT as a 2-part Julian Date
    dtr : float
        TDB-TT in seconds

    Returns
    -------
    tdb1 : float
        TDB as a 2-part Julian Date
    tdb2 : float
        TDB as a 2-part Julian Date
    J : ValueError
        0 : OK
    '''
    #一天的秒长
    D2S=86400.0
    
    DTRD=DTR/D2S
    if (np.abs(TT1)>np.abs(TT2)):
        TDB1=TT1
        TDB2=TT2+DTRD
    else:
        TDB1=TT1+DTRD
        TDB2=TT2
    
    J=0   
    return(TDB1,TDB2,J)

def pymTtut1(TT1,TT2,DT):
    '''
    Time scale transformation:  Terrestrial Time, TT, to Universal Time,
    UT1.

    Parameters
    ----------
    tt1 : float
        TT as a 2-part Julian Date
    tt2 : float
        TT as a 2-part Julian Date
    dtr : float
        TT-UT1 in seconds

    Returns
    -------
    ut11 : float
        UT1 as a 2-part Julian Date
    ut12 : float
        UT1 as a 2-part Julian Date
    J : ValueError
        0 : OK
    '''
    #一天的秒长
    D2S=86400.0
    DTD=DT/D2S
    if (np.abs(TT1)>np.abs(TT2)):
        UT11=TT1
        UT12=TT2-DTD
    else:
        UT11=TT1-DTD
        UT12=TT2
    
    J=0    
    return(UT11,UT12,J)

def pymUt1tai(UT11,UT12,DTA):
    '''
    Time scale transformation:  Universal Time, UT1, to International
    Atomic Time, TAI.

    Parameters
    ----------
    ut11 : float
        UT1 as a 2-part Julian Date
    ut12 : float
        UT1 as a 2-part Julian Date
    dta : float
        UT1-TAI in seconds

    Returns
    -------
    tai1 : float
        TAI as a 2-part Julian Date
    tai2 : float
        TAI as a 2-part Julian Date
    J : ValueError
        0 : OK
    '''
    #一天的秒长
    D2S=86400.0
       
    DTAD=DTA/D2S
    if (np.abs(UT11)>np.abs(UT12)):
        TAI1=UT11
        TAI2=UT12-DTAD
    else:
        TAI1=UT11-DTAD
        TAI2=UT12
    
    J=0
    return(TAI1,TAI2,J)

def pymUt1tt(UT11,UT12,DT):
    '''
    Time scale transformation:  Universal Time, UT1, to Terrestrial
    Time, TT.

    Parameters
    ----------
    ut11 : float
        UT1 as a 2-part Julian Date
    ut12 : float
        UT1 as a 2-part Julian Date
    dt : float
        TT-UT1 in seconds

    Returns
    -------
    tt1 : float
        TT as a 2-part Julian Date
    tt2 : float
        TT as a 2-part Julian Date
    J : ValueError
        0 : OK
    '''
    #一天的秒长
    D2S = 86400.0
    
    DTD=DT/D2S
    if (np.abs(UT11)>np.abs(UT12)):
        TT1=UT11
        TT2=UT12+DTD
    else:
        TT1=UT11+DTD
        TT2=UT12
    
    J=0
    return(TT1,TT2,J)

def pymUt1utc(UT11,UT12,DUT1):
    '''
    Time scale transformation:  Universal Time, UT1, to Coordinated
    Universal Time, UTC.

    Parameters
    ----------
    ut11 : float
        UT1 as a 2-part Julian Date
    ut12 : float
        UT1 as a 2-part Julian Date
    dut1 : float
        Delta UT1: UT1-UTC in seconds

    Returns
    -------
    utc1 : float
        UTC as a 2-part quasi Julian Date
    utc2 : float
        UTC as a 2-part quasi Julian Date
    J : ValueError
        1: 'dubious year',
       -1: 'unacceptable date',    
    '''
    #一天的秒长
    D2S=86400.0
    
    #UT1-UTC的时差，单位：秒
    DUTS=DUT1
    #大的在前.
    BIG1=np.abs(UT11)>=np.abs(UT12)
    if (BIG1):
        U1=UT11
        U2=UT12
    else:
        U1=UT12
        U2=UT11
        
    #判断是否存在闰秒   
    D1=U1
    DATS1=0.0
    
    for j in range(-1,4,1):
        D2=U2+float(j)
        IY,IM,ID,FD,JS=pymJd2cal(D1, D2)
        if (JS!=0):
            break
        DATS2,JS=pymDat(IY,IM,ID,0.0)
        if (JS<0):
            break
        if (j==-1):
            DATS1=DATS2
        DDATS=DATS2-DATS1
        if (np.abs(DDATS)>=0.5):
            
            #是的，闰秒附近:确保UT1-UTC是“前”值
            if ((DDATS*DUTS)>=0.0):
                DUTS=DUTS-DDATS
            
            #将UT1作为一个UTC日的起始，并将闰秒作为结尾
            D1,D2,JS=pymCal2jd(IY,IM,ID)
            US1=D1
            US2=D2-1.0+DUTS/D2S
            
            #UT1是在这一个时间点之后么？
            DU=U1-US1
            DU=DU+(U2-US2)
            
            if (DU>0.0):
                
                #是，则这个时间点是过去的
                FD=DU*D2S/(D2S+DDATS)
                
                #在UT1-UTC中引入SOFA's JD(UTC)公约.
                DUTS=DUTS+DDATS*min(FD,1.0)
            
            break
        DATS1=DATS2
        
    #从UT1中减去UT1-UTC(可能已调整)得到UTC.
    U2 = U2 - DUTS/D2S
    
    #尽可能保留结果的精度
    if (BIG1):
        UTC1=U1
        UTC2=U2
    else:
        UTC1=U2
        UTC2=U1
 
    J=JS 
    return(UTC1,UTC2,J)

def pymUtctai(UTC1,UTC2):
    '''
    Time scale transformation:  Coordinated Universal Time, UTC, to
    International Atomic Time, TAI.

    Parameters
    ----------
    utc1 : float
        UTC as a 2-part quasi Julian Date
    utc2 : float
        UTC as a 2-part quasi Julian Date

    Returns
    -------
    tai1 : float
        TAI as a 2-part Julian Date
    tai2 : float
        TAI as a 2-part Julian Date
    J : ValueError
        1: 'dubious year',
       -1: 'unacceptable date',
    '''
    TAI1,TAI2=0,0
    
    #一天的秒长
    D2S=86400.0
    
    #将两参数日期按先大后小排列
    BIG1=(np.abs(UTC1)>=np.abs(UTC2))
    if (BIG1):
        U1=UTC1
        U2=UTC2
    else:
        U1=UTC2
        U2=UTC1
    
    i=1
    while i<2:
        
        #当天0时的TAI-UTC时间差.
        IY,IM,ID,FD,JS=pymJd2cal(U1,U2)
        if (JS!=0):
            print('ERROR1',JS)
            break
        DAT0,JS=pymDat(IY,IM,ID,0.0)
        if (JS<0):
            print('ERROR2',JS)
            break
    
        #当天12h的TAI-UTC时间差（判断是否存在drift——1972年之前情况）
        DAT12,JS=pymDat(IY,IM,ID,0.5)
        if (JS<0):
            print('ERROR3',JS)
            break
    
        #第二天0h的TAI-UTC时间差（判断是否存在跳秒）
        IYT,IMT,IDT,W,JS=pymJd2cal(U1+1.5,U2-FD)
        if (JS!=0):
            print('ERROR4',JS)
            break
        DAT24,JS=pymDat(IYT,IMT,IDT,0.0)
        if (JS<0):
            print('ERROR5',JS)
            break

        #将TAI和UTC之间的时间差分解到两个部分，前一天的时差和跳秒的
        DLOD=2.0*(DAT12-DAT0)
        DLEAP=DAT24-(DAT0+DLOD)
        
        #在前一天中去除掉跳秒
        FD=FD*(D2S+DLEAP)/D2S
        
        #从(1972年以前)UTC秒到SI秒
        FD=FD*(D2S+DLOD)/D2S

        #将当天的日期转换到儒略日.
        Z1,Z2,JS=pymCal2jd(IY,IM,ID)
        if (JS!=0):
            print('ERROR6',JS)
            break

        #构建TAI的时间表达
        A2=Z1-U1
        A2=(A2+Z2)+(FD+DAT0/D2S)
        if (BIG1):
            TAI1=U1
            TAI2=A2
        else:
            TAI1=A2
            TAI2=U1
       
        i+=1
    J=JS
    
    return(TAI1,TAI2,J)

def pymUtcut1(UTC1,UTC2,DUT1):
    '''
    Time scale transformation:  Coordinated Universal Time, UTC, to
    Universal Time, UT1.

    Parameters
    ----------
    utc1 : float
        UTC as a 2-part quasi Julian Date
    utc2 : float
        UTC as a 2-part quasi Julian Date
    dut1 : float
        Delta UT1 = UT1-UTC in seconds

    Returns
    -------
    ut11 : float
        UT1 as a 2-part Julian Date
    ut12 : float
        UT1 as a 2-part Julian Date
    J : ValueError
        1: 'dubious year',
       -1: 'unacceptable date',    
    '''   
    i=1
    while i<2:
            
        #调用TAI-UTC之间的关系.
        IY,IM,ID,W,JS=pymJd2cal(UTC1,UTC2)
        if (JS!=0):
            print('ERROR',JS)
            break
        DAT,JS=pymDat(IY,IM,ID,0.0)
        if (JS<0):
            print('ERROR',JS)
            break
    
        #构建UT1-TAI
        DTA=DUT1-DAT
        
        #由UTC到TAI，再到UT1
        TAI1,TAI2,JW=pymUtctai(UTC1,UTC2)
        if (JW<0):
            JS=JW
            print('ERROR',JS)
            break
        UT11,UT12,JW=pymTaiut1(TAI1,TAI2,DTA)
        
        i+=1
    J=JS
    return(UT11,UT12,J)

def pymA2af(NDP,ANGLE):
    '''
    Decompose radians into degrees, arcminutes, arcseconds, fraction.

    Parameters
    ----------
    ndp : int
        resolution      
    angle : float
        angle in radians

    Returns
    -------
    sign : str
        '+' or '-'    
    idmsf : list(4)
        degrees, arcminutes, arcseconds, fraction 

         NDP         resolution
          :      ...0000 00 00
         -7         1000 00 00
         -6          100 00 00
         -5           10 00 00
         -4            1 00 00
         -3            0 10 00
         -2            0 01 00
         -1            0 00 10
          0            0 00 01
          1            0 00 00.1
          2            0 00 00.01
          3            0 00 00.001
          :            0 00 00.000...
    '''
    #调用D2TF中，天转换为时分秒的函数，通过F将列表中第一项小时转换为角度
    F=15.0/(2*ma.pi)
    SIGN,IDMSF=pymD2tf(NDP,ANGLE*F)

    return(SIGN,IDMSF)

def pymA2tf (NDP,ANGLE):
    '''
    Decompose radians into hours, minutes, seconds, fraction.

    Parameters
    ----------
    ndp : int
        resolution    
    angle : float
        angle in radians

    Returns
    -------
    sign : str 
        '+' or '-'    
    ihmsf : list(4)
        
    NDP         resolution
     :      ...0000 00 00
    -7         1000 00 00
    -6          100 00 00
    -5           10 00 00
    -4            1 00 00
    -3            0 10 00
    -2            0 01 00
    -1            0 00 10
     0            0 00 01
     1            0 00 00.1
     2            0 00 00.01
     3            0 00 00.001
     :            0 00 00.000...
    '''
    #直接调用天到时分秒的函数
    SIGN,IHMSF=pymD2tf(NDP,ANGLE/(2*ma.pi))
    return(SIGN,IHMSF)

def pymAf2a(S,IDEG,IAMIN,ASEC):
    '''
    Convert degrees, arcminutes, arcseconds to radians.

    Parameters
    ----------
    s : str
        sign:  '-' = negative, otherwise positive    
    ideg : int
        degrees    
    iamin : int
        arcminutes    
    asec : float
        arcseconds    

    Returns
    -------
    rad : float
        angle in radians
    J : ValueError
        1: 'ideg outside range 0-359',
        2: 'iamin outside range 0-59',
        3: 'asec outside range 0-59.999...',
    '''

    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6

    #重置J
    J=0
    
    #判断输入合法性
    if (ASEC<0.0)|(ASEC>=60.0):
        J=3
    if (IAMIN<0)|(IAMIN>59):
        J=2
    if (IDEG<0)|(IDEG>359):
        J=1
    
    #计算角度
    W=(60.0*(60.0*float(np.abs(IDEG))+float(np.abs(IAMIN)))+np.abs(ASEC))*DAS2R

    #应用符号.
    if (S=='-'):
        W=-W
    RAD=W
    
    return(RAD,J)

def pymTf2a(S,IHOUR,IMIN,SEC):
    '''
    Convert hours, minutes, seconds to radians.

    Parameters
    ----------
    s : str  
        sign:  '-' = negative, otherwise positive      
    ihour : int    
        hours    
    imin : int    
        minutes    
    sec : float    
        seconds    
        
    Returns
    -------
    rad : float
        angle in radians
    J : ValueError
        1:'ihour outside range 0-23',
        2:'imin outside range 0-59',
        3:'sec outside range 0-59.999...'    
    '''
    #1秒对应的弧度
    DS2R=7.272205216643039903848712e-5
    
    J=0
    if (SEC<0.0)|(SEC>=60.0):
        J=3
    if (IMIN<0)|(IMIN>59):
        J=2
    if (IHOUR<0)|(IHOUR>23):
        J=1
        
    W=(60.0*(60.0*float(np.abs(IHOUR))+float(np.abs(IMIN)))+np.abs(SEC))*DS2R
    
    if (S=='-'):
        W=-W
        
    RAD=W
    
    return(RAD, J)


