import math as ma
import numpy as np

def pymIr():
    '''
    Initialize an r-matrix to the identity matrix.

    Returns
    -------
    r : list(3,3)
        r-matrix

    '''

    R=[[1,0,0],[0,1,0],[0,0,1]]
    
    return(R)

def pymRy(THETA,R):
    '''
    Rotate an r-matrix about the y-axis.

    Parameters
    ----------
    theta : float
        angle (radians)    
    r : list(3,3)
        r-matrix

    Returns
    -------
    r : list(3,3)
        r-matrix, rotated

    '''
    S=ma.sin(THETA)
    C=ma.cos(THETA)

    A11=C*R[0][0]-S*R[2][0]
    A12=C*R[0][1]-S*R[2][1]
    A13=C*R[0][2]-S*R[2][2]
    A31=S*R[0][0]+C*R[2][0]
    A32=S*R[0][1]+C*R[2][1]
    A33=S*R[0][2]+C*R[2][2]

    R[0][0]=A11
    R[0][1]=A12
    R[0][2]=A13
    R[2][0]=A31
    R[2][1]=A32
    R[2][2]=A33
    
    return(R)

def pymRz(PSI,R):
    '''
    Rotate an r-matrix about the z-axis.

    Parameters
    ----------
    psi : float
        angle (radians)    
    r : list(3,3)
        r-matrix

    Returns
    -------
    r : list(3,3)
        r-matrix, rotated

    '''
    S=ma.sin(PSI)
    C=ma.cos(PSI)

    A11=C*R[0][0]+S*R[1][0]
    A12=C*R[0][1]+S*R[1][1]
    A13=C*R[0][2]+S*R[1][2]
    A21=-S*R[0][0]+C*R[1][0]
    A22=-S*R[0][1]+C*R[1][1]
    A23=-S*R[0][2]+C*R[1][2]

    R[0][0]=A11
    R[0][1]=A12
    R[0][2]=A13
    R[1][0]=A21
    R[1][1]=A22
    R[1][2]=A23
 
    return(R)

def pymRx(PHI,R):
    '''
    Rotate an r-matrix about the x-axis.

    Parameters
    ----------
    phi : float    
        angle (radians)    
    r : list(3,3)    
        r-matrix    

    Returns
    -------
    r : list(3,3)    
        r-matrix, rotated

    '''
    S=ma.sin(PHI)
    C=ma.cos(PHI)
    
    A21=C*R[1][0]+S*R[2][0]
    A22=C*R[1][1]+S*R[2][1]
    A23=C*R[1][2]+S*R[2][2]
    A31=-S*R[1][0]+C*R[2][0]
    A32=-S*R[1][1]+C*R[2][1]
    A33=-S*R[1][2]+C*R[2][2]

    R[1][0]=A21
    R[1][1]=A22
    R[1][2]=A23
    R[2][0]=A31
    R[2][1]=A32
    R[2][2]=A33
    
    return(R)

def pymC2ixys(X,Y,S):
    '''
    Form the celestial to intermediate-frame-of-date matrix given the CIP
    X,Y and the CIO locator s.

    Parameters
    ----------
    x : float
        Celestial Intermediate Pole    
    y : float
        Celestial Intermediate Pole    
    s : float
        the CIO locator s 

    Returns
    -------
    rc2i : list(3,3)
        celestial-to-intermediate matrix

    '''
    #得到球面角E和d
    R2=X**2+Y**2
    if (R2>0.0):
        E=ma.atan2(Y,X)
    else:
        E=0.0
    D=ma.atan(ma.sqrt(R2/(1.0-R2)))
    
    #构建矩阵
    RC2I=pymIr()
    RC2I=pymRz(E,RC2I)
    RC2I=pymRy(D,RC2I)
    RC2I=pymRz(-(E+S),RC2I)    
    
    return(RC2I)

def pymFw2m(GAMB,PHIB,PSI,EPS):
    '''
    Form rotation matrix given the Fukushima-Williams angles.

    Parameters
    ----------
    gamb : float
        F-W angle gamma_bar (radians)    
    phib : float
        F-W angle phi_bar (radians)    
    psi : float
        F-W angle psi (radians)    
    eps : float
        F-W angle epsilon (radians)

    Returns
    -------
    r : list(3,3)
        rotation matrix

    '''
    R=pymIr()
    R=pymRz(GAMB, R)
    R=pymRx(PHIB, R)
    R=pymRz(-PSI, R)
    R=pymRx(-EPS, R)
   
    return(R)

def pymFad03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean elongation of the Moon from the Sun.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0
 
    Returns
    -------
    function value : float
        D, radians

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #一个圆内的总角秒数
    TURNAS=1296000.0
    
    #月亮到太阳的平均距角 
    A=1072260.703692+T*(1602961601.2090+T*(-6.3706+\
            T*(0.006593+T*(-0.00003169))))
    FAD=(A%TURNAS)*DAS2R
    
    return(FAD)

def pymFae03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Earth.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0
 
    Returns
    -------
    function value : float
        mean longitude of Earth, radians

    '''
    FAE=(1.753470314+628.3075849991*T)%(2*ma.pi)    
    
    return(FAE)

def pymFaf03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of the Moon minus mean longitude of the ascending
    node.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0
 
    Returns
    -------
    function value : float
        F, radians

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #一个圆内的总角秒数
    TURNAS=1296000.0
    
    #月球的平黄经减去其上升点的平黄经 
    A=335779.526232+T*(1739527262.8478+T*(-12.7512+\
        T*(-0.001037+T*(0.00000417))))
    FAF=(A%TURNAS)*DAS2R
    
    return(FAF)

def pymFaju03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Jupiter.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0
 
    Returns
    -------
    function value : float
        mean longitude of Jupiter, radians

    '''
    FAJU=(0.599546497+52.9690962641*T)%(2*ma.pi)

    return(FAJU)

def pymFal03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean anomaly of the Moon.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0
 
    Returns
    -------
    function value : float
        l, radians

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #一个圆内的总角秒数
    TURNAS=1296000.0
    
    #月亮的平近点角
    A=485868.249036+T*(1717915923.2178+T*(31.8792+\
        T*(0.051635+T*(-0.00024470))))
    FAL=(A%TURNAS)*DAS2R
    
    return(FAL)

def pymFalp03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean anomaly of the Sun.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0
 
    Returns
    -------
    function value : float
        l', radians

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #一个圆内的总角秒数
    TURNAS=1296000.0
    
    #太阳的平近点角
    A=1287104.793048+T*(129596581.0481+T*(-0.5532+\
        T*(0.000136+T*(-0.00001149))))
    FALP=(A%TURNAS)*DAS2R

    return(FALP)

def pymFama03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Mars.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0
 
    Returns
    -------
    function value : float
        mean longitude of Mars, radians

    '''
    FAMA=(6.203480913+334.0612426700*T)%(2*ma.pi)
    
    return(FAMA)

def pymFame03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Mercury.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0
 
    Returns
    -------
    function value : float
        mean longitude of Mercury, radians

    '''
    
    FAME=(4.402608842+2608.7903141574*T)%(2*ma.pi)
    
    return(FAME)

def pymFane03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Neptune.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0

    Returns
    -------
    function value : float
        mean longitude of Neptune, radians

    '''    
    FANE=(5.311886287+3.8133035638*T)%(2*ma.pi)
    
    return(FANE)

def pymFaom03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of the Moon's ascending node.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0

    Returns
    -------
    function value : float
        Omega, radians 

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #一个圆内的总角秒数
    TURNAS=1296000.0
    
    #月亮升交点的平黄经
    A=450160.398036+T*(-6962890.5431+T*(7.4722+\
        T*(0.007702+T*(-0.00005939))))
    FAOM=(A%TURNAS)*DAS2R
    
    return(FAOM)

def pymFapa03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    general accumulated precession in longitude.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0

    Returns
    -------
    function value : float
        general precession in longitude, radians

    '''
    FAPA=(0.024381750+0.00000538691*T)*T
    
    return(FAPA)

def pymFasa03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Saturn.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0

    Returns
    -------
    function value : float
        mean longitude of Saturn, radians

    '''
    FASA=(0.874016757+21.3299104960*T)%(2*ma.pi)
    
    return(FASA)

def pymFaur03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Uranus.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0

    Returns
    -------
    function value : float
        mean longitude of Uranus, radians

    '''
    FAUR=(5.481293872+7.4781598567*T)%(2*ma.pi)
    
    return(FAUR)

def pymFave03(T):
    '''
    Fundamental argument, IERS Conventions (2003):
    mean longitude of Venus.

    Parameters
    ----------
    t : float
        TDB, Julian centuries since J2000.0

    Returns
    -------
    function value : float
        mean longitude of Venus, radians

    '''
    FAVE=(3.176146697+1021.3285546211*T)%(2*ma.pi)
    
    return(FAVE)

def pymNut00a(DATE1,DATE2):
    '''
    Nutation, IAU 2000A model (MHB2000 luni-solar and planetary nutation
    with free core nutation omitted).

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    dpsi : float
        nutation, luni-solar + planetary    
    deps : float
        nutation, luni-solar + planetary

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #一个圆内的总角秒数
    TURNAS=1296000.0
    
    #2Pi
    D2PI = 6.283185307179586476925287
    
    #0.1微角秒对应的弧度
    U2R=DAS2R/1e7
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0
    
    #1儒略世纪对应的天数
    DJC=36525.0
    
    #日月章动模型
    #模型中的项数
    NLS=678
    
    #模型中的项数
    #行星章动模型
    NPL=687
    
    #日月章动的参数和项系数的表格
    #  L     L'    F     D     Om
    NALS=[[0,0,0,0,1],
            [0,0,2,-2,2],
            [0,0,2,0,2],
            [0,0,0,0,2],
            [0,1,0,0,0],
            [0,1,2,-2,2],
            [1,0,0,0,0],
            [0,0,2,0,1],
            [1,0,2,0,2],
            [0,-1,2,-2,2],
            [0,0,2,-2,1],
            [-1,0,2,0,2],
            [-1,0,0,2,0],
            [1,0,0,0,1],
            [-1,0,0,0,1],
            [-1,0,2,2,2],
            [1,0,2,0,1],
            [-2,0,2,0,1],
            [0,0,0,2,0],
            [0,0,2,2,2],
            [0,-2,2,-2,2],
            [-2,0,0,2,0],
            [2,0,2,0,2],
            [1,0,2,-2,2],
            [-1,0,2,0,1],
            [2,0,0,0,0],
            [0,0,2,0,0],
            [0,1,0,0,1],
            [-1,0,0,2,1],
            [0,2,2,-2,2],
            [0,0,-2,2,0],
            [1,0,0,-2,1],
            [0,-1,0,0,1],
            [-1,0,2,2,1],
            [0,2,0,0,0],
            [1,0,2,2,2],
            [-2,0,2,0,0],
            [0,1,2,0,2],
            [0,0,2,2,1],
            [0,-1,2,0,2],
            [0,0,0,2,1],
            [1,0,2,-2,1],
            [2,0,2,-2,2],
            [-2,0,0,2,1],
            [2,0,2,0,1],
            [0,-1,2,-2,1],
            [0,0,0,-2,1],
            [-1,-1,0,2,0],
            [2,0,0,-2,1],
            [1,0,0,2,0],
            [0,1,2,-2,1],
            [1,-1,0,0,0],
            [-2,0,2,0,2],
            [3,0,2,0,2],
            [0,-1,0,2,0],
            [1,-1,2,0,2],
            [0,0,0,1,0],
            [-1,-1,2,2,2],
            [-1,0,2,0,0],
            [0,-1,2,2,2],
            [-2,0,0,0,1],
            [1,1,2,0,2],
            [2,0,0,0,1],
            [-1,1,0,1,0],
            [1,1,0,0,0],
            [1,0,2,0,0],
            [-1,0,2,-2,1],
            [1,0,0,0,2],
            [-1,0,0,1,0],
            [0,0,2,1,2],
            [-1,0,2,4,2],
            [-1,1,0,1,1],
            [0,-2,2,-2,1],
            [1,0,2,2,1],
            [-2,0,2,2,2],
            [-1,0,0,0,2],
            [1,1,2,-2,2],
            [-2,0,2,4,2],
            [-1,0,4,0,2],
            [2,0,2,-2,1],
            [2,0,2,2,2],
            [1,0,0,2,1],
            [3,0,0,0,0],
            [3,0,2,-2,2],
            [0,0,4,-2,2],
            [0,1,2,0,1],
            [0,0,-2,2,1],
            [0,0,2,-2,3],
            [-1,0,0,4,0],
            [2,0,-2,0,1],
            [-2,0,0,4,0],
            [-1,-1,0,2,1],
            [-1,0,0,1,1],
            [0,1,0,0,2],
            [0,0,-2,0,1],
            [0,-1,2,0,1],
            [0,0,2,-1,2],
            [0,0,2,4,2],
            [-2,-1,0,2,0],
            [1,1,0,-2,1],
            [-1,1,0,2,0],
            [-1,1,0,1,2],
            [1,-1,0,0,1],
            [1,-1,2,2,2],
            [-1,1,2,2,2],
            [3,0,2,0,1],
            [0,1,-2,2,0],
            [-1,0,0,-2,1],
            [0,1,2,2,2],
            [-1,-1,2,2,1],
            [0,-1,0,0,2],
            [1,0,2,-4,1],
            [-1,0,-2,2,0],
            [0,-1,2,2,1],
            [2,-1,2,0,2],
            [0,0,0,2,2],
            [1,-1,2,0,1],
            [-1,1,2,0,2],
            [0,1,0,2,0],
            [0,-1,-2,2,0],
            [0,3,2,-2,2],
            [0,0,0,1,1],
            [-1,0,2,2,0],
            [2,1,2,0,2],
            [1,1,0,0,1],
            [1,1,2,0,1],
            [2,0,0,2,0],
            [1,0,-2,2,0],
            [-1,0,0,2,2],
            [0,1,0,1,0],
            [0,1,0,-2,1],
            [-1,0,2,-2,2],
            [0,0,0,-1,1],
            [-1,1,0,0,1],
            [1,0,2,-1,2],
            [1,-1,0,2,0],
            [0,0,0,4,0],
            [1,0,2,1,2],
            [0,0,2,1,1],
            [1,0,0,-2,2],
            [-1,0,2,4,1],
            [1,0,-2,0,1],
            [1,1,2,-2,1],
            [0,0,2,2,0],
            [-1,0,2,-1,1],
            [-2,0,2,2,1],
            [4,0,2,0,2],
            [2,-1,0,0,0],
            [2,1,2,-2,2],
            [0,1,2,1,2],
            [1,0,4,-2,2],
            [-1,-1,0,0,1],
            [0,1,0,2,1],
            [-2,0,2,4,1],
            [2,0,2,0,0],
            [1,0,0,1,0],
            [-1,0,0,4,1],
            [-1,0,4,0,1],
            [2,0,2,2,1],
            [0,0,2,-3,2],
            [-1,-2,0,2,0],
            [2,1,0,0,0],
            [0,0,4,0,2],
            [0,0,0,0,3],
            [0,3,0,0,0],
            [0,0,2,-4,1],
            [0,-1,0,2,1],
            [0,0,0,4,1],
            [-1,-1,2,4,2],
            [1,0,2,4,2],
            [-2,2,0,2,0],
            [-2,-1,2,0,1],
            [-2,0,0,2,2],
            [-1,-1,2,0,2],
            [0,0,4,-2,1],
            [3,0,2,-2,1],
            [-2,-1,0,2,1],
            [1,0,0,-1,1],
            [0,-2,0,2,0],
            [-2,0,0,4,1],
            [-3,0,0,0,1],
            [1,1,2,2,2],
            [0,0,2,4,1],
            [3,0,2,2,2],
            [-1,1,2,-2,1],
            [2,0,0,-4,1],
            [0,0,0,-2,2],
            [2,0,2,-4,1],
            [-1,1,0,2,1],
            [0,0,2,-1,1],
            [0,-2,2,2,2],
            [2,0,0,2,1],
            [4,0,2,-2,2],
            [2,0,0,-2,2],
            [0,2,0,0,1],
            [1,0,0,-4,1],
            [0,2,2,-2,1],
            [-3,0,0,4,0],
            [-1,1,2,0,1],
            [-1,-1,0,4,0],
            [-1,-2,2,2,2],
            [-2,-1,2,4,2],
            [1,-1,2,2,1],
            [-2,1,0,2,0],
            [-2,1,2,0,1],
            [2,1,0,-2,1],
            [-3,0,2,0,1],
            [-2,0,2,-2,1],
            [-1,1,0,2,2],
            [0,-1,2,-1,2],
            [-1,0,4,-2,2],
            [0,-2,2,0,2],
            [-1,0,2,1,2],
            [2,0,0,0,2],
            [0,0,2,0,3],
            [-2,0,4,0,2],
            [-1,0,-2,0,1],
            [-1,1,2,2,1],
            [3,0,0,0,1],
            [-1,0,2,3,2],
            [2,-1,2,0,1],
            [0,1,2,2,1],
            [0,-1,2,4,2],
            [2,-1,2,2,2],
            [0,2,-2,2,0],
            [-1,-1,2,-1,1],
            [0,-2,0,0,1],
            [1,0,2,-4,2],
            [1,-1,0,-2,1],
            [-1,-1,2,0,1],
            [1,-1,2,-2,2],
            [-2,-1,0,4,0],
            [-1,0,0,3,0],
            [-2,-1,2,2,2],
            [0,2,2,0,2],
            [1,1,0,2,0],
            [2,0,2,-1,2],
            [1,0,2,1,1],
            [4,0,0,0,0],
            [2,1,2,0,1],
            [3,-1,2,0,2],
            [-2,2,0,2,1],
            [1,0,2,-3,1],
            [1,1,2,-4,1],
            [-1,-1,2,-2,1],
            [0,-1,0,-1,1],
            [0,-1,0,-2,1],
            [-2,0,0,0,2],
            [-2,0,-2,2,0],
            [-1,0,-2,4,0],
            [1,-2,0,0,0],
            [0,1,0,1,1],
            [-1,2,0,2,0],
            [1,-1,2,-2,1],
            [1,2,2,-2,2],
            [2,-1,2,-2,2],
            [1,0,2,-1,1],
            [2,1,2,-2,1],
            [-2,0,0,-2,1],
            [1,-2,2,0,2],
            [0,1,2,1,1],
            [1,0,4,-2,1],
            [-2,0,4,2,2],
            [1,1,2,1,2],
            [1,0,0,4,0],
            [1,0,2,2,0],
            [2,0,2,1,2],
            [3,1,2,0,2],
            [4,0,2,0,1],
            [-2,-1,2,0,0],
            [0,1,-2,2,1],
            [1,0,-2,1,0],
            [0,-1,-2,2,1],
            [2,-1,0,-2,1],
            [-1,0,2,-1,2],
            [1,0,2,-3,2],
            [0,1,2,-2,3],
            [0,0,2,-3,1],
            [-1,0,-2,2,1],
            [0,0,2,-4,2],
            [-2,1,0,0,1],
            [-1,0,0,-1,1],
            [2,0,2,-4,2],
            [0,0,4,-4,4],
            [0,0,4,-4,2],
            [-1,-2,0,2,1],
            [-2,0,0,3,0],
            [1,0,-2,2,1],
            [-3,0,2,2,2],
            [-3,0,2,2,1],
            [-2,0,2,2,0],
            [2,-1,0,0,1],
            [-2,1,2,2,2],
            [1,1,0,1,0],
            [0,1,4,-2,2],
            [-1,1,0,-2,1],
            [0,0,0,-4,1],
            [1,-1,0,2,1],
            [1,1,0,2,1],
            [-1,2,2,2,2],
            [3,1,2,-2,2],
            [0,-1,0,4,0],
            [2,-1,0,2,0],
            [0,0,4,0,1],
            [2,0,4,-2,2],
            [-1,-1,2,4,1],
            [1,0,0,4,1],
            [1,-2,2,2,2],
            [0,0,2,3,2],
            [-1,1,2,4,2],
            [3,0,0,2,0],
            [-1,0,4,2,2],
            [1,1,2,2,1],
            [-2,0,2,6,2],
            [2,1,2,2,2],
            [-1,0,2,6,2],
            [1,0,2,4,1],
            [2,0,2,4,2],
            [1,1,-2,1,0],
            [-3,1,2,1,2],
            [2,0,-2,0,2],
            [-1,0,0,1,2],
            [-4,0,2,2,1],
            [-1,-1,0,1,0],
            [0,0,-2,2,2],
            [1,0,0,-1,2],
            [0,-1,2,-2,3],
            [-2,1,2,0,0],
            [0,0,2,-2,4],
            [-2,-2,0,2,0],
            [-2,0,-2,4,0],
            [0,-2,-2,2,0],
            [1,2,0,-2,1],
            [3,0,0,-4,1],
            [-1,1,2,-2,2],
            [1,-1,2,-4,1],
            [1,1,0,-2,2],
            [-3,0,2,0,0],
            [-3,0,2,0,2],
            [-2,0,0,1,0],
            [0,0,-2,1,0],
            [-3,0,0,2,1],
            [-1,-1,-2,2,0],
            [0,1,2,-4,1],
            [2,1,0,-4,1],
            [0,2,0,-2,1],
            [1,0,0,-3,1],
            [-2,0,2,-2,2],
            [-2,-1,0,0,1],
            [-4,0,0,2,0],
            [1,1,0,-4,1],
            [-1,0,2,-4,1],
            [0,0,4,-4,1],
            [0,3,2,-2,2],
            [-3,-1,0,4,0],
            [-3,0,0,4,1],
            [1,-1,-2,2,0],
            [-1,-1,0,2,2],
            [1,-2,0,0,1],
            [1,-1,0,0,2],
            [0,0,0,1,2],
            [-1,-1,2,0,0],
            [1,-2,2,-2,2],
            [0,-1,2,-1,1],
            [-1,0,2,0,3],
            [1,1,0,0,2],
            [-1,1,2,0,0],
            [1,2,0,0,0],
            [-1,2,2,0,2],
            [-1,0,4,-2,1],
            [3,0,2,-4,2],
            [1,2,2,-2,1],
            [1,0,4,-4,2],
            [-2,-1,0,4,1],
            [0,-1,0,2,2],
            [-2,1,0,4,0],
            [-2,-1,2,2,1],
            [2,0,-2,2,0],
            [1,0,0,1,1],
            [0,1,0,2,2],
            [1,-1,2,-1,2],
            [-2,0,4,0,1],
            [2,1,0,0,1],
            [0,1,2,0,0],
            [0,-1,4,-2,2],
            [0,0,4,-2,4],
            [0,2,2,0,1],
            [-3,0,0,6,0],
            [-1,-1,0,4,1],
            [1,-2,0,2,0],
            [-1,0,0,4,2],
            [-1,-2,2,2,1],
            [-1,0,0,-2,2],
            [1,0,-2,-2,1],
            [0,0,-2,-2,1],
            [-2,0,-2,0,1],
            [0,0,0,3,1],
            [0,0,0,3,0],
            [-1,1,0,4,0],
            [-1,-1,2,2,0],
            [-2,0,2,3,2],
            [1,0,0,2,2],
            [0,-1,2,1,2],
            [3,-1,0,0,0],
            [2,0,0,1,0],
            [1,-1,2,0,0],
            [0,0,2,1,0],
            [1,0,2,0,3],
            [3,1,0,0,0],
            [3,-1,2,-2,2],
            [2,0,2,-1,1],
            [1,1,2,0,0],
            [0,0,4,-1,2],
            [1,2,2,0,2],
            [-2,0,0,6,0],
            [0,-1,0,4,1],
            [-2,-1,2,4,1],
            [0,-2,2,2,1],
            [0,-1,2,2,0],
            [-1,0,2,3,1],
            [-2,1,2,4,2],
            [2,0,0,2,2],
            [2,-2,2,0,2],
            [-1,1,2,3,2],
            [3,0,2,-1,2],
            [4,0,2,-2,1],
            [-1,0,0,6,0],
            [-1,-2,2,4,2],
            [-3,0,2,6,2],
            [-1,0,2,4,0],
            [3,0,0,2,1],
            [3,-1,2,0,1],
            [3,0,2,0,0],
            [1,0,4,0,2],
            [5,0,2,-2,2],
            [0,-1,2,4,1],
            [2,-1,2,2,1],
            [0,1,2,4,2],
            [1,-1,2,4,2],
            [3,-1,2,2,2],
            [3,0,2,2,1],
            [5,0,2,0,2],
            [0,0,2,6,2],
            [4,0,2,2,2],
            [0,-1,1,-1,1],
            [-1,0,1,0,3],
            [0,-2,2,-2,3],
            [1,0,-1,0,1],
            [2,-2,0,-2,1],
            [-1,0,1,0,2],
            [-1,0,1,0,1],
            [-1,-1,2,-1,2],
            [-2,2,0,2,2],
            [-1,0,1,0,0],
            [-4,1,2,2,2],
            [-3,0,2,1,1],
            [-2,-1,2,0,2],
            [1,0,-2,1,1],
            [2,-1,-2,0,1],
            [-4,0,2,2,0],
            [-3,1,0,3,0],
            [-1,0,-1,2,0],
            [0,-2,0,0,2],
            [0,-2,0,0,2],
            [-3,0,0,3,0],
            [-2,-1,0,2,2],
            [-1,0,-2,3,0],
            [-4,0,0,4,0],
            [2,1,-2,0,1],
            [2,-1,0,-2,2],
            [0,0,1,-1,0],
            [-1,2,0,1,0],
            [-2,1,2,0,2],
            [1,1,0,-1,1],
            [1,0,1,-2,1],
            [0,2,0,0,2],
            [1,-1,2,-3,1],
            [-1,1,2,-1,1],
            [-2,0,4,-2,2],
            [-2,0,4,-2,1],
            [-2,-2,0,2,1],
            [-2,0,-2,4,0],
            [1,2,2,-4,1],
            [1,1,2,-4,2],
            [-1,2,2,-2,1],
            [2,0,0,-3,1],
            [-1,2,0,0,1],
            [0,0,0,-2,0],
            [-1,-1,2,-2,2],
            [-1,1,0,0,2],
            [0,0,0,-1,2],
            [-2,1,0,1,0],
            [1,-2,0,-2,1],
            [1,0,-2,0,2],
            [-3,1,0,2,0],
            [-1,1,-2,2,0],
            [-1,-1,0,0,2],
            [-3,0,0,2,0],
            [-3,-1,0,2,0],
            [2,0,2,-6,1],
            [0,1,2,-4,2],
            [2,0,0,-4,2],
            [-2,1,2,-2,1],
            [0,-1,2,-4,1],
            [0,1,0,-2,2],
            [-1,0,0,-2,0],
            [2,0,-2,-2,1],
            [-4,0,2,0,1],
            [-1,-1,0,-1,1],
            [0,0,-2,0,2],
            [-3,0,0,1,0],
            [-1,0,-2,1,0],
            [-2,0,-2,2,1],
            [0,0,-4,2,0],
            [-2,-1,-2,2,0],
            [1,0,2,-6,1],
            [-1,0,2,-4,2],
            [1,0,0,-4,2],
            [2,1,2,-4,2],
            [2,1,2,-4,1],
            [0,1,4,-4,4],
            [0,1,4,-4,2],
            [-1,-1,-2,4,0],
            [-1,-3,0,2,0],
            [-1,0,-2,4,1],
            [-2,-1,0,3,0],
            [0,0,-2,3,0],
            [-2,0,0,3,1],
            [0,-1,0,1,0],
            [-3,0,2,2,0],
            [1,1,-2,2,0],
            [-1,1,0,2,2],
            [1,-2,2,-2,1],
            [0,0,1,0,2],
            [0,0,1,0,1],
            [0,0,1,0,0],
            [-1,2,0,2,1],
            [0,0,2,0,2],
            [-2,0,2,0,2],
            [2,0,0,-1,1],
            [3,0,0,-2,1],
            [1,0,2,-2,3],
            [1,2,0,0,1],
            [2,0,2,-3,2],
            [-1,1,4,-2,2],
            [-2,-2,0,4,0],
            [0,-3,0,2,0],
            [0,0,-2,4,0],
            [-1,-1,0,3,0],
            [-2,0,0,4,2],
            [-1,0,0,3,1],
            [2,-2,0,0,0],
            [1,-1,0,1,0],
            [-1,0,0,2,0],
            [0,-2,2,0,1],
            [-1,0,1,2,1],
            [-1,1,0,3,0],
            [-1,-1,2,1,2],
            [0,-1,2,0,0],
            [-2,1,2,2,1],
            [2,-2,2,-2,2],
            [1,1,0,1,1],
            [1,0,1,0,1],
            [1,0,1,0,0],
            [0,2,0,2,0],
            [2,-1,2,-2,1],
            [0,-1,4,-2,1],
            [0,0,4,-2,3],
            [0,1,4,-2,1],
            [4,0,2,-4,2],
            [2,2,2,-2,2],
            [2,0,4,-4,2],
            [-1,-2,0,4,0],
            [-1,-3,2,2,2],
            [-3,0,2,4,2],
            [-3,0,2,-2,1],
            [-1,-1,0,-2,1],
            [-3,0,0,0,2],
            [-3,0,-2,2,0],
            [0,1,0,-4,1],
            [-2,1,0,-2,1],
            [-4,0,0,0,1],
            [-1,0,0,-4,1],
            [-3,0,0,-2,1],
            [0,0,0,3,2],
            [-1,1,0,4,1],
            [1,-2,2,0,1],
            [0,1,0,3,0],
            [-1,0,2,2,3],
            [0,0,2,2,2],
            [-2,0,2,2,2],
            [-1,1,2,2,0],
            [3,0,0,0,2],
            [2,1,0,1,0],
            [2,-1,2,-1,2],
            [0,0,2,0,1],
            [0,0,3,0,3],
            [0,0,3,0,2],
            [-1,2,2,2,1],
            [-1,0,4,0,0],
            [1,2,2,0,1],
            [3,1,2,-2,1],
            [1,1,4,-2,2],
            [-2,-1,0,6,0],
            [0,-2,0,4,0],
            [-2,0,0,6,1],
            [-2,-2,2,4,2],
            [0,-3,2,2,2],
            [0,0,0,4,2],
            [-1,-1,2,3,2],
            [-2,0,2,4,0],
            [2,-1,0,2,1],
            [1,0,0,3,0],
            [0,1,0,4,1],
            [0,1,0,4,0],
            [1,-1,2,1,2],
            [0,0,2,2,3],
            [1,0,2,2,2],
            [-1,0,2,2,2],
            [-2,0,4,2,1],
            [2,1,0,2,1],
            [2,1,0,2,0],
            [2,-1,2,0,0],
            [1,0,2,1,0],
            [0,1,2,2,0],
            [2,0,2,0,3],
            [3,0,2,0,2],
            [1,0,2,0,2],
            [1,0,3,0,3],
            [1,1,2,1,1],
            [0,2,2,2,2],
            [2,1,2,0,0],
            [2,0,4,-2,1],
            [4,1,2,-2,2],
            [-1,-1,0,6,0],
            [-3,-1,2,6,2],
            [-1,0,0,6,1],
            [-3,0,2,6,1],
            [1,-1,0,4,1],
            [1,-1,0,4,0],
            [-2,0,2,5,2],
            [1,-2,2,2,1],
            [3,-1,0,2,0],
            [1,-1,2,2,0],
            [0,0,2,3,1],
            [-1,1,2,4,1],
            [0,1,2,3,2],
            [-1,0,4,2,1],
            [2,0,2,1,1],
            [5,0,0,0,0],
            [2,1,2,1,2],
            [1,0,4,0,1],
            [3,1,2,0,1],
            [3,0,4,-2,2],
            [-2,-1,2,6,2],
            [0,0,0,6,0],
            [0,-2,2,4,2],
            [-2,0,2,6,1],
            [2,0,0,4,1],
            [2,0,0,4,0],
            [2,-2,2,2,2],
            [0,0,2,4,0],
            [1,0,2,3,2],
            [4,0,0,2,0],
            [2,0,2,2,0],
            [0,0,4,2,2],
            [4,-1,2,0,2],
            [3,0,2,1,2],
            [2,1,2,2,1],
            [4,1,2,0,2],
            [-1,-1,2,6,2],
            [-1,0,2,6,1],
            [1,-1,2,4,1],
            [1,1,2,4,2],
            [3,1,2,2,2],
            [5,0,2,0,1],
            [2,-1,2,4,2],
            [2,0,2,4,1]]
    
    #日月章动系数，单位：0.1微角秒
    #*  longitude (sin, t*sin, cos), obliquity (cos, t*cos, sin)
    CLS=[[-172064161.0,-174666.0,33386.0,92052331.0,9086.0,15377.0],
            [-13170906.0,-1675.0,-13696.0,5730336.0,-3015.0,-4587.0],
            [-2276413.0,-234.0,2796.0,978459.0,-485.0,1374.0],
            [2074554.0,207.0,-698.0,-897492.0,470.0,-291.0],
            [1475877.0,-3633.0,11817.0,73871.0,-184.0,-1924.0],
            [-516821.0,1226.0,-524.0,224386.0,-677.0,-174.0],
            [711159.0,73.0,-872.0,-6750.0,0.0,358.0],
            [-387298.0,-367.0,380.0,200728.0,18.0,318.0],
            [-301461.0,-36.0,816.0,129025.0,-63.0,367.0],
            [215829.0,-494.0,111.0,-95929.0,299.0,132.0],
            [128227.0,137.0,181.0,-68982.0,-9.0,39.0],
            [123457.0,11.0,19.0,-53311.0,32.0,-4.0],
            [156994.0,10.0,-168.0,-1235.0,0.0,82.0],
            [63110.0,63.0,27.0,-33228.0,0.0,-9.0],
            [-57976.0,-63.0,-189.0,31429.0,0.0,-75.0],
            [-59641.0,-11.0,149.0,25543.0,-11.0,66.0],
            [-51613.0,-42.0,129.0,26366.0,0.0,78.0],
            [45893.0,50.0,31.0,-24236.0,-10.0,20.0],
            [63384.0,11.0,-150.0,-1220.0,0.0,29.0],
            [-38571.0,-1.0,158.0,16452.0,-11.0,68.0],
            [32481.0,0.0,0.0,-13870.0,0.0,0.0],
            [-47722.0,0.0,-18.0,477.0,0.0,-25.0],
            [-31046.0,-1.0,131.0,13238.0,-11.0,59.0],
            [28593.0,0.0,-1.0,-12338.0,10.0,-3.0],
            [20441.0,21.0,10.0,-10758.0,0.0,-3.0],
            [29243.0,0.0,-74.0,-609.0,0.0,13.0],
            [25887.0,0.0,-66.0,-550.0,0.0,11.0],
            [-14053.0,-25.0,79.0,8551.0,-2.0,-45.0],
            [15164.0,10.0,11.0,-8001.0,0.0,-1.0],
            [-15794.0,72.0,-16.0,6850.0,-42.0,-5.0],
            [21783.0,0.0,13.0,-167.0,0.0,13.0],
            [-12873.0,-10.0,-37.0,6953.0,0.0,-14.0],
            [-12654.0,11.0,63.0,6415.0,0.0,26.0],
            [-10204.0,0.0,25.0,5222.0,0.0,15.0],
            [16707.0,-85.0,-10.0,168.0,-1.0,10.0],
            [-7691.0,0.0,44.0,3268.0,0.0,19.0],
            [-11024.0,0.0,-14.0,104.0,0.0,2.0],
            [7566.0,-21.0,-11.0,-3250.0,0.0,-5.0],
            [-6637.0,-11.0,25.0,3353.0,0.0,14.0],
            [-7141.0,21.0,8.0,3070.0,0.0,4.0],
            [-6302.0,-11.0,2.0,3272.0,0.0,4.0],
            [5800.0,10.0,2.0,-3045.0,0.0,-1.0],
            [6443.0,0.0,-7.0,-2768.0,0.0,-4.0],
            [-5774.0,-11.0,-15.0,3041.0,0.0,-5.0],
            [-5350.0,0.0,21.0,2695.0,0.0,12.0],
            [-4752.0,-11.0,-3.0,2719.0,0.0,-3.0],
            [-4940.0,-11.0,-21.0,2720.0,0.0,-9.0],
            [7350.0,0.0,-8.0,-51.0,0.0,4.0],
            [4065.0,0.0,6.0,-2206.0,0.0,1.0],
            [6579.0,0.0,-24.0,-199.0,0.0,2.0],
            [3579.0,0.0,5.0,-1900.0,0.0,1.0],
            [4725.0,0.0,-6.0,-41.0,0.0,3.0],
            [-3075.0,0.0,-2.0,1313.0,0.0,-1.0],
            [-2904.0,0.0,15.0,1233.0,0.0,7.0],
            [4348.0,0.0,-10.0,-81.0,0.0,2.0],
            [-2878.0,0.0,8.0,1232.0,0.0,4.0],
            [-4230.0,0.0,5.0,-20.0,0.0,-2.0],
            [-2819.0,0.0,7.0,1207.0,0.0,3.0],
            [-4056.0,0.0,5.0,40.0,0.0,-2.0],
            [-2647.0,0.0,11.0,1129.0,0.0,5.0],
            [-2294.0,0.0,-10.0,1266.0,0.0,-4.0],
            [2481.0,0.0,-7.0,-1062.0,0.0,-3.0],
            [2179.0,0.0,-2.0,-1129.0,0.0,-2.0],
            [3276.0,0.0,1.0,-9.0,0.0,0.0],
            [-3389.0,0.0,5.0,35.0,0.0,-2.0],
            [3339.0,0.0,-13.0,-107.0,0.0,1.0],
            [-1987.0,0.0,-6.0,1073.0,0.0,-2.0],
            [-1981.0,0.0,0.0,854.0,0.0,0.0],
            [4026.0,0.0,-353.0,-553.0,0.0,-139.0],
            [1660.0,0.0,-5.0,-710.0,0.0,-2.0],
            [-1521.0,0.0,9.0,647.0,0.0,4.0],
            [1314.0,0.0,0.0,-700.0,0.0,0.0],
            [-1283.0,0.0,0.0,672.0,0.0,0.0],
            [-1331.0,0.0,8.0,663.0,0.0,4.0],
            [1383.0,0.0,-2.0,-594.0,0.0,-2.0],
            [1405.0,0.0,4.0,-610.0,0.0,2.0],
            [1290.0,0.0,0.0,-556.0,0.0,0.0],
            [-1214.0,0.0,5.0,518.0,0.0,2.0],
            [1146.0,0.0,-3.0,-490.0,0.0,-1.0],
            [1019.0,0.0,-1.0,-527.0,0.0,-1.0],
            [-1100.0,0.0,9.0,465.0,0.0,4.0],
            [-970.0,0.0,2.0,496.0,0.0,1.0],
            [1575.0,0.0,-6.0,-50.0,0.0,0.0],
            [934.0,0.0,-3.0,-399.0,0.0,-1.0],
            [922.0,0.0,-1.0,-395.0,0.0,-1.0],
            [815.0,0.0,-1.0,-422.0,0.0,-1.0],
            [834.0,0.0,2.0,-440.0,0.0,1.0],
            [1248.0,0.0,0.0,-170.0,0.0,1.0],
            [1338.0,0.0,-5.0,-39.0,0.0,0.0],
            [716.0,0.0,-2.0,-389.0,0.0,-1.0],
            [1282.0,0.0,-3.0,-23.0,0.0,1.0],
            [742.0,0.0,1.0,-391.0,0.0,0.0],
            [1020.0,0.0,-25.0,-495.0,0.0,-10.0],
            [715.0,0.0,-4.0,-326.0,0.0,2.0],
            [-666.0,0.0,-3.0,369.0,0.0,-1.0],
            [-667.0,0.0,1.0,346.0,0.0,1.0],
            [-704.0,0.0,0.0,304.0,0.0,0.0],
            [-694.0,0.0,5.0,294.0,0.0,2.0],
            [-1014.0,0.0,-1.0,4.0,0.0,-1.0],
            [-585.0,0.0,-2.0,316.0,0.0,-1.0],
            [-949.0,0.0,1.0,8.0,0.0,-1.0],
            [-595.0,0.0,0.0,258.0,0.0,0.0],
            [528.0,0.0,0.0,-279.0,0.0,0.0],
            [-590.0,0.0,4.0,252.0,0.0,2.0],
            [570.0,0.0,-2.0,-244.0,0.0,-1.0],
            [-502.0,0.0,3.0,250.0,0.0,2.0],
            [-875.0,0.0,1.0,29.0,0.0,0.0],
            [-492.0,0.0,-3.0,275.0,0.0,-1.0],
            [535.0,0.0,-2.0,-228.0,0.0,-1.0],
            [-467.0,0.0,1.0,240.0,0.0,1.0],
            [591.0,0.0,0.0,-253.0,0.0,0.0],
            [-453.0,0.0,-1.0,244.0,0.0,-1.0],
            [766.0,0.0,1.0,9.0,0.0,0.0],
            [-446.0,0.0,2.0,225.0,0.0,1.0],
            [-488.0,0.0,2.0,207.0,0.0,1.0],
            [-468.0,0.0,0.0,201.0,0.0,0.0],
            [-421.0,0.0,1.0,216.0,0.0,1.0],
            [463.0,0.0,0.0,-200.0,0.0,0.0],
            [-673.0,0.0,2.0,14.0,0.0,0.0],
            [658.0,0.0,0.0,-2.0,0.0,0.0],
            [-438.0,0.0,0.0,188.0,0.0,0.0],
            [-390.0,0.0,0.0,205.0,0.0,0.0],
            [639.0,-11.0,-2.0,-19.0,0.0,0.0],
            [412.0,0.0,-2.0,-176.0,0.0,-1.0],
            [-361.0,0.0,0.0,189.0,0.0,0.0],
            [360.0,0.0,-1.0,-185.0,0.0,-1.0],
            [588.0,0.0,-3.0,-24.0,0.0,0.0],
            [-578.0,0.0,1.0,5.0,0.0,0.0],
            [-396.0,0.0,0.0,171.0,0.0,0.0],
            [565.0,0.0,-1.0,-6.0,0.0,0.0],
            [-335.0,0.0,-1.0,184.0,0.0,-1.0],
            [357.0,0.0,1.0,-154.0,0.0,0.0],
            [321.0,0.0,1.0,-174.0,0.0,0.0],
            [-301.0,0.0,-1.0,162.0,0.0,0.0],
            [-334.0,0.0,0.0,144.0,0.0,0.0],
            [493.0,0.0,-2.0,-15.0,0.0,0.0],
            [494.0,0.0,-2.0,-19.0,0.0,0.0],
            [337.0,0.0,-1.0,-143.0,0.0,-1.0],
            [280.0,0.0,-1.0,-144.0,0.0,0.0],
            [309.0,0.0,1.0,-134.0,0.0,0.0],
            [-263.0,0.0,2.0,131.0,0.0,1.0],
            [253.0,0.0,1.0,-138.0,0.0,0.0],
            [245.0,0.0,0.0,-128.0,0.0,0.0],
            [416.0,0.0,-2.0,-17.0,0.0,0.0],
            [-229.0,0.0,0.0,128.0,0.0,0.0],
            [231.0,0.0,0.0,-120.0,0.0,0.0],
            [-259.0,0.0,2.0,109.0,0.0,1.0],
            [375.0,0.0,-1.0,-8.0,0.0,0.0],
            [252.0,0.0,0.0,-108.0,0.0,0.0],
            [-245.0,0.0,1.0,104.0,0.0,0.0],
            [243.0,0.0,-1.0,-104.0,0.0,0.0],
            [208.0,0.0,1.0,-112.0,0.0,0.0],
            [199.0,0.0,0.0,-102.0,0.0,0.0],
            [-208.0,0.0,1.0,105.0,0.0,0.0],
            [335.0,0.0,-2.0,-14.0,0.0,0.0],
            [-325.0,0.0,1.0,7.0,0.0,0.0],
            [-187.0,0.0,0.0,96.0,0.0,0.0],
            [197.0,0.0,-1.0,-100.0,0.0,0.0],
            [-192.0,0.0,2.0,94.0,0.0,1.0],
            [-188.0,0.0,0.0,83.0,0.0,0.0],
            [276.0,0.0,0.0,-2.0,0.0,0.0],
            [-286.0,0.0,1.0,6.0,0.0,0.0],
            [186.0,0.0,-1.0,-79.0,0.0,0.0],
            [-219.0,0.0,0.0,43.0,0.0,0.0],
            [276.0,0.0,0.0,2.0,0.0,0.0],
            [-153.0,0.0,-1.0,84.0,0.0,0.0],
            [-156.0,0.0,0.0,81.0,0.0,0.0],
            [-154.0,0.0,1.0,78.0,0.0,0.0],
            [-174.0,0.0,1.0,75.0,0.0,0.0],
            [-163.0,0.0,2.0,69.0,0.0,1.0],
            [-228.0,0.0,0.0,1.0,0.0,0.0],
            [91.0,0.0,-4.0,-54.0,0.0,-2.0],
            [175.0,0.0,0.0,-75.0,0.0,0.0],
            [-159.0,0.0,0.0,69.0,0.0,0.0],
            [141.0,0.0,0.0,-72.0,0.0,0.0],
            [147.0,0.0,0.0,-75.0,0.0,0.0],
            [-132.0,0.0,0.0,69.0,0.0,0.0],
            [159.0,0.0,-28.0,-54.0,0.0,11.0],
            [213.0,0.0,0.0,-4.0,0.0,0.0],
            [123.0,0.0,0.0,-64.0,0.0,0.0],
            [-118.0,0.0,-1.0,66.0,0.0,0.0],
            [144.0,0.0,-1.0,-61.0,0.0,0.0],
            [-121.0,0.0,1.0,60.0,0.0,0.0],
            [-134.0,0.0,1.0,56.0,0.0,1.0],
            [-105.0,0.0,0.0,57.0,0.0,0.0],
            [-102.0,0.0,0.0,56.0,0.0,0.0],
            [120.0,0.0,0.0,-52.0,0.0,0.0],
            [101.0,0.0,0.0,-54.0,0.0,0.0],
            [-113.0,0.0,0.0,59.0,0.0,0.0],
            [-106.0,0.0,0.0,61.0,0.0,0.0],
            [-129.0,0.0,1.0,55.0,0.0,0.0],
            [-114.0,0.0,0.0,57.0,0.0,0.0],
            [113.0,0.0,-1.0,-49.0,0.0,0.0],
            [-102.0,0.0,0.0,44.0,0.0,0.0],
            [-94.0,0.0,0.0,51.0,0.0,0.0],
            [-100.0,0.0,-1.0,56.0,0.0,0.0],
            [87.0,0.0,0.0,-47.0,0.0,0.0],
            [161.0,0.0,0.0,-1.0,0.0,0.0],
            [96.0,0.0,0.0,-50.0,0.0,0.0],
            [151.0,0.0,-1.0,-5.0,0.0,0.0],
            [-104.0,0.0,0.0,44.0,0.0,0.0],
            [-110.0,0.0,0.0,48.0,0.0,0.0],
            [-100.0,0.0,1.0,50.0,0.0,0.0],
            [92.0,0.0,-5.0,12.0,0.0,-2.0],
            [82.0,0.0,0.0,-45.0,0.0,0.0],
            [82.0,0.0,0.0,-45.0,0.0,0.0],
            [-78.0,0.0,0.0,41.0,0.0,0.0],
            [-77.0,0.0,0.0,43.0,0.0,0.0],
            [2.0,0.0,0.0,54.0,0.0,0.0],
            [94.0,0.0,0.0,-40.0,0.0,0.0],
            [-93.0,0.0,0.0,40.0,0.0,0.0],
            [-83.0,0.0,10.0,40.0,0.0,-2.0],
            [83.0,0.0,0.0,-36.0,0.0,0.0],
            [-91.0,0.0,0.0,39.0,0.0,0.0],
            [128.0,0.0,0.0,-1.0,0.0,0.0],
            [-79.0,0.0,0.0,34.0,0.0,0.0],
            [-83.0,0.0,0.0,47.0,0.0,0.0],
            [84.0,0.0,0.0,-44.0,0.0,0.0],
            [83.0,0.0,0.0,-43.0,0.0,0.0],
            [91.0,0.0,0.0,-39.0,0.0,0.0],
            [-77.0,0.0,0.0,39.0,0.0,0.0],
            [84.0,0.0,0.0,-43.0,0.0,0.0],
            [-92.0,0.0,1.0,39.0,0.0,0.0],
            [-92.0,0.0,1.0,39.0,0.0,0.0],
            [-94.0,0.0,0.0,0.0,0.0,0.0],
            [68.0,0.0,0.0,-36.0,0.0,0.0],
            [-61.0,0.0,0.0,32.0,0.0,0.0],
            [71.0,0.0,0.0,-31.0,0.0,0.0],
            [62.0,0.0,0.0,-34.0,0.0,0.0],
            [-63.0,0.0,0.0,33.0,0.0,0.0],
            [-73.0,0.0,0.0,32.0,0.0,0.0],
            [115.0,0.0,0.0,-2.0,0.0,0.0],
            [-103.0,0.0,0.0,2.0,0.0,0.0],
            [63.0,0.0,0.0,-28.0,0.0,0.0],
            [74.0,0.0,0.0,-32.0,0.0,0.0],
            [-103.0,0.0,-3.0,3.0,0.0,-1.0],
            [-69.0,0.0,0.0,30.0,0.0,0.0],
            [57.0,0.0,0.0,-29.0,0.0,0.0],
            [94.0,0.0,0.0,-4.0,0.0,0.0],
            [64.0,0.0,0.0,-33.0,0.0,0.0],
            [-63.0,0.0,0.0,26.0,0.0,0.0],
            [-38.0,0.0,0.0,20.0,0.0,0.0],
            [-43.0,0.0,0.0,24.0,0.0,0.0],
            [-45.0,0.0,0.0,23.0,0.0,0.0],
            [47.0,0.0,0.0,-24.0,0.0,0.0],
            [-48.0,0.0,0.0,25.0,0.0,0.0],
            [45.0,0.0,0.0,-26.0,0.0,0.0],
            [56.0,0.0,0.0,-25.0,0.0,0.0],
            [88.0,0.0,0.0,2.0,0.0,0.0],
            [-75.0,0.0,0.0,0.0,0.0,0.0],
            [85.0,0.0,0.0,0.0,0.0,0.0],
            [49.0,0.0,0.0,-26.0,0.0,0.0],
            [-74.0,0.0,-3.0,-1.0,0.0,-1.0],
            [-39.0,0.0,0.0,21.0,0.0,0.0],
            [45.0,0.0,0.0,-20.0,0.0,0.0],
            [51.0,0.0,0.0,-22.0,0.0,0.0],
            [-40.0,0.0,0.0,21.0,0.0,0.0],
            [41.0,0.0,0.0,-21.0,0.0,0.0],
            [-42.0,0.0,0.0,24.0,0.0,0.0],
            [-51.0,0.0,0.0,22.0,0.0,0.0],
            [-42.0,0.0,0.0,22.0,0.0,0.0],
            [39.0,0.0,0.0,-21.0,0.0,0.0],
            [46.0,0.0,0.0,-18.0,0.0,0.0],
            [-53.0,0.0,0.0,22.0,0.0,0.0],
            [82.0,0.0,0.0,-4.0,0.0,0.0],
            [81.0,0.0,-1.0,-4.0,0.0,0.0],
            [47.0,0.0,0.0,-19.0,0.0,0.0],
            [53.0,0.0,0.0,-23.0,0.0,0.0],
            [-45.0,0.0,0.0,22.0,0.0,0.0],
            [-44.0,0.0,0.0,-2.0,0.0,0.0],
            [-33.0,0.0,0.0,16.0,0.0,0.0],
            [-61.0,0.0,0.0,1.0,0.0,0.0],
            [28.0,0.0,0.0,-15.0,0.0,0.0],
            [-38.0,0.0,0.0,19.0,0.0,0.0],
            [-33.0,0.0,0.0,21.0,0.0,0.0],
            [-60.0,0.0,0.0,0.0,0.0,0.0],
            [48.0,0.0,0.0,-10.0,0.0,0.0],
            [27.0,0.0,0.0,-14.0,0.0,0.0],
            [38.0,0.0,0.0,-20.0,0.0,0.0],
            [31.0,0.0,0.0,-13.0,0.0,0.0],
            [-29.0,0.0,0.0,15.0,0.0,0.0],
            [28.0,0.0,0.0,-15.0,0.0,0.0],
            [-32.0,0.0,0.0,15.0,0.0,0.0],
            [45.0,0.0,0.0,-8.0,0.0,0.0],
            [-44.0,0.0,0.0,19.0,0.0,0.0],
            [28.0,0.0,0.0,-15.0,0.0,0.0],
            [-51.0,0.0,0.0,0.0,0.0,0.0],
            [-36.0,0.0,0.0,20.0,0.0,0.0],
            [44.0,0.0,0.0,-19.0,0.0,0.0],
            [26.0,0.0,0.0,-14.0,0.0,0.0],
            [-60.0,0.0,0.0,2.0,0.0,0.0],
            [35.0,0.0,0.0,-18.0,0.0,0.0],
            [-27.0,0.0,0.0,11.0,0.0,0.0],
            [47.0,0.0,0.0,-1.0,0.0,0.0],
            [36.0,0.0,0.0,-15.0,0.0,0.0],
            [-36.0,0.0,0.0,20.0,0.0,0.0],
            [-35.0,0.0,0.0,19.0,0.0,0.0],
            [-37.0,0.0,0.0,19.0,0.0,0.0],
            [32.0,0.0,0.0,-16.0,0.0,0.0],
            [35.0,0.0,0.0,-14.0,0.0,0.0],
            [32.0,0.0,0.0,-13.0,0.0,0.0],
            [65.0,0.0,0.0,-2.0,0.0,0.0],
            [47.0,0.0,0.0,-1.0,0.0,0.0],
            [32.0,0.0,0.0,-16.0,0.0,0.0],
            [37.0,0.0,0.0,-16.0,0.0,0.0],
            [-30.0,0.0,0.0,15.0,0.0,0.0],
            [-32.0,0.0,0.0,16.0,0.0,0.0],
            [-31.0,0.0,0.0,13.0,0.0,0.0],
            [37.0,0.0,0.0,-16.0,0.0,0.0],
            [31.0,0.0,0.0,-13.0,0.0,0.0],
            [49.0,0.0,0.0,-2.0,0.0,0.0],
            [32.0,0.0,0.0,-13.0,0.0,0.0],
            [23.0,0.0,0.0,-12.0,0.0,0.0],
            [-43.0,0.0,0.0,18.0,0.0,0.0],
            [26.0,0.0,0.0,-11.0,0.0,0.0],
            [-32.0,0.0,0.0,14.0,0.0,0.0],
            [-29.0,0.0,0.0,14.0,0.0,0.0],
            [-27.0,0.0,0.0,12.0,0.0,0.0],
            [30.0,0.0,0.0,0.0,0.0,0.0],
            [-11.0,0.0,0.0,5.0,0.0,0.0],
            [-21.0,0.0,0.0,10.0,0.0,0.0],
            [-34.0,0.0,0.0,15.0,0.0,0.0],
            [-10.0,0.0,0.0,6.0,0.0,0.0],
            [-36.0,0.0,0.0,0.0,0.0,0.0],
            [-9.0,0.0,0.0,4.0,0.0,0.0],
            [-12.0,0.0,0.0,5.0,0.0,0.0],
            [-21.0,0.0,0.0,5.0,0.0,0.0],
            [-29.0,0.0,0.0,-1.0,0.0,0.0],
            [-15.0,0.0,0.0,3.0,0.0,0.0],
            [-20.0,0.0,0.0,0.0,0.0,0.0],
            [28.0,0.0,0.0,0.0,0.0,-2.0],
            [17.0,0.0,0.0,0.0,0.0,0.0],
            [-22.0,0.0,0.0,12.0,0.0,0.0],
            [-14.0,0.0,0.0,7.0,0.0,0.0],
            [24.0,0.0,0.0,-11.0,0.0,0.0],
            [11.0,0.0,0.0,-6.0,0.0,0.0],
            [14.0,0.0,0.0,-6.0,0.0,0.0],
            [24.0,0.0,0.0,0.0,0.0,0.0],
            [18.0,0.0,0.0,-8.0,0.0,0.0],
            [-38.0,0.0,0.0,0.0,0.0,0.0],
            [-31.0,0.0,0.0,0.0,0.0,0.0],
            [-16.0,0.0,0.0,8.0,0.0,0.0],
            [29.0,0.0,0.0,0.0,0.0,0.0],
            [-18.0,0.0,0.0,10.0,0.0,0.0],
            [-10.0,0.0,0.0,5.0,0.0,0.0],
            [-17.0,0.0,0.0,10.0,0.0,0.0],
            [9.0,0.0,0.0,-4.0,0.0,0.0],
            [16.0,0.0,0.0,-6.0,0.0,0.0],
            [22.0,0.0,0.0,-12.0,0.0,0.0],
            [20.0,0.0,0.0,0.0,0.0,0.0],
            [-13.0,0.0,0.0,6.0,0.0,0.0],
            [-17.0,0.0,0.0,9.0,0.0,0.0],
            [-14.0,0.0,0.0,8.0,0.0,0.0],
            [0.0,0.0,0.0,-7.0,0.0,0.0],
            [14.0,0.0,0.0,0.0,0.0,0.0],
            [19.0,0.0,0.0,-10.0,0.0,0.0],
            [-34.0,0.0,0.0,0.0,0.0,0.0],
            [-20.0,0.0,0.0,8.0,0.0,0.0],
            [9.0,0.0,0.0,-5.0,0.0,0.0],
            [-18.0,0.0,0.0,7.0,0.0,0.0],
            [13.0,0.0,0.0,-6.0,0.0,0.0],
            [17.0,0.0,0.0,0.0,0.0,0.0],
            [-12.0,0.0,0.0,5.0,0.0,0.0],
            [15.0,0.0,0.0,-8.0,0.0,0.0],
            [-11.0,0.0,0.0,3.0,0.0,0.0],
            [13.0,0.0,0.0,-5.0,0.0,0.0],
            [-18.0,0.0,0.0,0.0,0.0,0.0],
            [-35.0,0.0,0.0,0.0,0.0,0.0],
            [9.0,0.0,0.0,-4.0,0.0,0.0],
            [-19.0,0.0,0.0,10.0,0.0,0.0],
            [-26.0,0.0,0.0,11.0,0.0,0.0],
            [8.0,0.0,0.0,-4.0,0.0,0.0],
            [-10.0,0.0,0.0,4.0,0.0,0.0],
            [10.0,0.0,0.0,-6.0,0.0,0.0],
            [-21.0,0.0,0.0,9.0,0.0,0.0],
            [-15.0,0.0,0.0,0.0,0.0,0.0],
            [9.0,0.0,0.0,-5.0,0.0,0.0],
            [-29.0,0.0,0.0,0.0,0.0,0.0],
            [-19.0,0.0,0.0,10.0,0.0,0.0],
            [12.0,0.0,0.0,-5.0,0.0,0.0],
            [22.0,0.0,0.0,-9.0,0.0,0.0],
            [-10.0,0.0,0.0,5.0,0.0,0.0],
            [-20.0,0.0,0.0,11.0,0.0,0.0],
            [-20.0,0.0,0.0,0.0,0.0,0.0],
            [-17.0,0.0,0.0,7.0,0.0,0.0],
            [15.0,0.0,0.0,-3.0,0.0,0.0],
            [8.0,0.0,0.0,-4.0,0.0,0.0],
            [14.0,0.0,0.0,0.0,0.0,0.0],
            [-12.0,0.0,0.0,6.0,0.0,0.0],
            [25.0,0.0,0.0,0.0,0.0,0.0],
            [-13.0,0.0,0.0,6.0,0.0,0.0],
            [-14.0,0.0,0.0,8.0,0.0,0.0],
            [13.0,0.0,0.0,-5.0,0.0,0.0],
            [-17.0,0.0,0.0,9.0,0.0,0.0],
            [-12.0,0.0,0.0,6.0,0.0,0.0],
            [-10.0,0.0,0.0,5.0,0.0,0.0],
            [10.0,0.0,0.0,-6.0,0.0,0.0],
            [-15.0,0.0,0.0,0.0,0.0,0.0],
            [-22.0,0.0,0.0,0.0,0.0,0.0],
            [28.0,0.0,0.0,-1.0,0.0,0.0],
            [15.0,0.0,0.0,-7.0,0.0,0.0],
            [23.0,0.0,0.0,-10.0,0.0,0.0],
            [12.0,0.0,0.0,-5.0,0.0,0.0],
            [29.0,0.0,0.0,-1.0,0.0,0.0],
            [-25.0,0.0,0.0,1.0,0.0,0.0],
            [22.0,0.0,0.0,0.0,0.0,0.0],
            [-18.0,0.0,0.0,0.0,0.0,0.0],
            [15.0,0.0,0.0,3.0,0.0,0.0],
            [-23.0,0.0,0.0,0.0,0.0,0.0],
            [12.0,0.0,0.0,-5.0,0.0,0.0],
            [-8.0,0.0,0.0,4.0,0.0,0.0],
            [-19.0,0.0,0.0,0.0,0.0,0.0],
            [-10.0,0.0,0.0,4.0,0.0,0.0],
            [21.0,0.0,0.0,-9.0,0.0,0.0],
            [23.0,0.0,0.0,-1.0,0.0,0.0],
            [-16.0,0.0,0.0,8.0,0.0,0.0],
            [-19.0,0.0,0.0,9.0,0.0,0.0],
            [-22.0,0.0,0.0,10.0,0.0,0.0],
            [27.0,0.0,0.0,-1.0,0.0,0.0],
            [16.0,0.0,0.0,-8.0,0.0,0.0],
            [19.0,0.0,0.0,-8.0,0.0,0.0],
            [9.0,0.0,0.0,-4.0,0.0,0.0],
            [-9.0,0.0,0.0,4.0,0.0,0.0],
            [-9.0,0.0,0.0,4.0,0.0,0.0],
            [-8.0,0.0,0.0,4.0,0.0,0.0],
            [18.0,0.0,0.0,-9.0,0.0,0.0],
            [16.0,0.0,0.0,-1.0,0.0,0.0],
            [-10.0,0.0,0.0,4.0,0.0,0.0],
            [-23.0,0.0,0.0,9.0,0.0,0.0],
            [16.0,0.0,0.0,-1.0,0.0,0.0],
            [-12.0,0.0,0.0,6.0,0.0,0.0],
            [-8.0,0.0,0.0,4.0,0.0,0.0],
            [30.0,0.0,0.0,-2.0,0.0,0.0],
            [24.0,0.0,0.0,-10.0,0.0,0.0],
            [10.0,0.0,0.0,-4.0,0.0,0.0],
            [-16.0,0.0,0.0,7.0,0.0,0.0],
            [-16.0,0.0,0.0,7.0,0.0,0.0],
            [17.0,0.0,0.0,-7.0,0.0,0.0],
            [-24.0,0.0,0.0,10.0,0.0,0.0],
            [-12.0,0.0,0.0,5.0,0.0,0.0],
            [-24.0,0.0,0.0,11.0,0.0,0.0],
            [-23.0,0.0,0.0,9.0,0.0,0.0],
            [-13.0,0.0,0.0,5.0,0.0,0.0],
            [-15.0,0.0,0.0,7.0,0.0,0.0],
            [0.0,0.0,-1988.0,0.0,0.0,-1679.0],
            [0.0,0.0,-63.0,0.0,0.0,-27.0],
            [-4.0,0.0,0.0,0.0,0.0,0.0],
            [0.0,0.0,5.0,0.0,0.0,4.0],
            [5.0,0.0,0.0,-3.0,0.0,0.0],
            [0.0,0.0,364.0,0.0,0.0,176.0],
            [0.0,0.0,-1044.0,0.0,0.0,-891.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [0.0,0.0,330.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [3.0,0.0,0.0,-2.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [-5.0,0.0,0.0,2.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [0.0,0.0,5.0,0.0,0.0,0.0],
            [0.0,0.0,0.0,1.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [6.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [-7.0,0.0,0.0,0.0,0.0,0.0],
            [-12.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-3.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [-5.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [-7.0,0.0,0.0,3.0,0.0,0.0],
            [7.0,0.0,0.0,-4.0,0.0,0.0],
            [0.0,0.0,-12.0,0.0,0.0,-10.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [3.0,0.0,0.0,-2.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [-7.0,0.0,0.0,3.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [0.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [7.0,0.0,0.0,-3.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [-5.0,0.0,0.0,3.0,0.0,0.0],
            [5.0,0.0,0.0,0.0,0.0,0.0],
            [-5.0,0.0,0.0,2.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [-8.0,0.0,0.0,3.0,0.0,0.0],
            [9.0,0.0,0.0,0.0,0.0,0.0],
            [6.0,0.0,0.0,-3.0,0.0,0.0],
            [-5.0,0.0,0.0,2.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [-7.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [5.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [-5.0,0.0,0.0,2.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [9.0,0.0,0.0,-3.0,0.0,0.0],
            [4.0,0.0,0.0,0.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [9.0,0.0,0.0,-3.0,0.0,0.0],
            [-4.0,0.0,0.0,0.0,0.0,0.0],
            [-4.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,-2.0,0.0,0.0],
            [8.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [6.0,0.0,0.0,-3.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [-7.0,0.0,0.0,0.0,0.0,0.0],
            [9.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [-3.0,0.0,0.0,0.0,0.0,0.0],
            [-4.0,0.0,0.0,0.0,0.0,0.0],
            [-5.0,0.0,0.0,3.0,0.0,0.0],
            [-13.0,0.0,0.0,0.0,0.0,0.0],
            [-7.0,0.0,0.0,0.0,0.0,0.0],
            [10.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [10.0,0.0,13.0,6.0,0.0,-5.0],
            [0.0,0.0,30.0,0.0,0.0,14.0],
            [0.0,0.0,-162.0,0.0,0.0,-138.0],
            [0.0,0.0,75.0,0.0,0.0,0.0],
            [-7.0,0.0,0.0,4.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [5.0,0.0,0.0,-3.0,0.0,0.0],
            [-3.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [-5.0,0.0,0.0,2.0,0.0,0.0],
            [6.0,0.0,0.0,0.0,0.0,0.0],
            [9.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,0.0,0.0,0.0],
            [-7.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [7.0,0.0,0.0,0.0,0.0,0.0],
            [-4.0,0.0,0.0,0.0,0.0,0.0],
            [4.0,0.0,0.0,0.0,0.0,0.0],
            [-6.0,0.0,-3.0,3.0,0.0,1.0],
            [0.0,0.0,-3.0,0.0,0.0,-2.0],
            [11.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [11.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [-1.0,0.0,3.0,3.0,0.0,-1.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [0.0,0.0,-13.0,0.0,0.0,-11.0],
            [3.0,0.0,6.0,0.0,0.0,0.0],
            [-7.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-3.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-3.0,0.0,0.0],
            [-7.0,0.0,0.0,3.0,0.0,0.0],
            [8.0,0.0,0.0,-3.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [11.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [8.0,0.0,0.0,-4.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [11.0,0.0,0.0,0.0,0.0,0.0],
            [-6.0,0.0,0.0,3.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [-8.0,0.0,0.0,4.0,0.0,0.0],
            [-7.0,0.0,0.0,3.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [6.0,0.0,0.0,-3.0,0.0,0.0],
            [-6.0,0.0,0.0,3.0,0.0,0.0],
            [6.0,0.0,0.0,0.0,0.0,0.0],
            [6.0,0.0,0.0,-1.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [-5.0,0.0,0.0,2.0,0.0,0.0],
            [-4.0,0.0,0.0,0.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [4.0,0.0,0.0,0.0,0.0,0.0],
            [6.0,0.0,0.0,-3.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [0.0,0.0,-26.0,0.0,0.0,-11.0],
            [0.0,0.0,-10.0,0.0,0.0,-5.0],
            [5.0,0.0,0.0,-3.0,0.0,0.0],
            [-13.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,-2.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [7.0,0.0,0.0,-3.0,0.0,0.0],
            [4.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [-6.0,0.0,0.0,2.0,0.0,0.0],
            [-5.0,0.0,0.0,2.0,0.0,0.0],
            [-7.0,0.0,0.0,3.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [13.0,0.0,0.0,0.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [-3.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [-11.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [4.0,0.0,0.0,0.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [6.0,0.0,0.0,-3.0,0.0,0.0],
            [3.0,0.0,0.0,-2.0,0.0,0.0],
            [-12.0,0.0,0.0,0.0,0.0,0.0],
            [4.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,0.0,0.0,0.0],
            [-4.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [0.0,0.0,-5.0,0.0,0.0,-2.0],
            [-7.0,0.0,0.0,4.0,0.0,0.0],
            [6.0,0.0,0.0,-3.0,0.0,0.0],
            [-3.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-3.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [3.0,0.0,0.0,0.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [-5.0,0.0,0.0,3.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [12.0,0.0,0.0,0.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [4.0,0.0,0.0,0.0,0.0,0.0],
            [6.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-3.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [-6.0,0.0,0.0,3.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [6.0,0.0,0.0,-3.0,0.0,0.0],
            [6.0,0.0,0.0,0.0,0.0,0.0],
            [-6.0,0.0,0.0,3.0,0.0,0.0],
            [3.0,0.0,0.0,-2.0,0.0,0.0],
            [7.0,0.0,0.0,-4.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [-5.0,0.0,0.0,2.0,0.0,0.0],
            [5.0,0.0,0.0,0.0,0.0,0.0],
            [-6.0,0.0,0.0,3.0,0.0,0.0],
            [-6.0,0.0,0.0,3.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [10.0,0.0,0.0,0.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [7.0,0.0,0.0,0.0,0.0,0.0],
            [7.0,0.0,0.0,-3.0,0.0,0.0],
            [4.0,0.0,0.0,0.0,0.0,0.0],
            [11.0,0.0,0.0,0.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [-6.0,0.0,0.0,2.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [3.0,0.0,0.0,-2.0,0.0,0.0],
            [5.0,0.0,0.0,-2.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [-4.0,0.0,0.0,2.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0],
            [4.0,0.0,0.0,-2.0,0.0,0.0],
            [3.0,0.0,0.0,-1.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [-3.0,0.0,0.0,1.0,0.0,0.0],
            [-3.0,0.0,0.0,2.0,0.0,0.0]]
    
    #行星章动参数
    #*   L   L'  F   D   Om  Me  Ve  E  Ma  Ju  Sa  Ur  Ne  pre
    NAPL=[[0,0,0,0,0,0,0,8,-16,4,5,0,0,0],
            [0,0,0,0,0,0,0,-8,16,-4,-5,0,0,2],
            [0,0,0,0,0,0,0,8,-16,4,5,0,0,2],
            [0,0,0,0,0,0,0,0,0,0,0,-1,2,2],
            [0,0,0,0,0,0,0,-4,8,-1,-5,0,0,2],
            [0,0,0,0,0,0,0,4,-8,3,0,0,0,1],
            [0,0,1,-1,1,0,0,3,-8,3,0,0,0,0],
            [-1,0,0,0,0,0,10,-3,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,-2,6,-3,0,2],
            [0,0,0,0,0,0,0,4,-8,3,0,0,0,0],
            [0,0,1,-1,1,0,0,-5,8,-3,0,0,0,0],
            [0,0,0,0,0,0,0,-4,8,-3,0,0,0,1],
            [0,0,0,0,0,0,0,4,-8,1,5,0,0,2],
            [0,0,0,0,0,0,-5,6,4,0,0,0,0,2],
            [0,0,0,0,0,0,0,0,0,2,-5,0,0,2],
            [0,0,0,0,0,0,0,0,0,2,-5,0,0,1],
            [0,0,1,-1,1,0,0,-1,0,2,-5,0,0,0],
            [0,0,0,0,0,0,0,0,0,2,-5,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,-2,5,0,0,0],
            [0,0,0,0,0,0,0,0,0,-2,5,0,0,1],
            [0,0,0,0,0,0,0,0,0,-2,5,0,0,2],
            [2,0,-1,-1,0,0,0,3,-7,0,0,0,0,0],
            [1,0,0,-2,0,0,19,-21,3,0,0,0,0,0],
            [0,0,1,-1,1,0,2,-4,0,-3,0,0,0,0],
            [1,0,0,-1,1,0,0,-1,0,2,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,-4,10,0,0,0],
            [-2,0,0,2,1,0,0,2,0,0,-5,0,0,0],
            [0,0,0,0,0,0,3,-7,4,0,0,0,0,0],
            [0,0,-1,1,0,0,0,1,0,1,-1,0,0,0],
            [-2,0,0,2,1,0,0,2,0,-2,0,0,0,0],
            [-1,0,0,0,0,0,18,-16,0,0,0,0,0,0],
            [-2,0,1,1,2,0,0,1,0,-2,0,0,0,0],
            [-1,0,1,-1,1,0,18,-17,0,0,0,0,0,0],
            [-1,0,0,1,1,0,0,2,-2,0,0,0,0,0],
            [0,0,0,0,0,0,-8,13,0,0,0,0,0,2],
            [0,0,2,-2,2,0,-8,11,0,0,0,0,0,0],
            [0,0,0,0,0,0,-8,13,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-8,12,0,0,0,0,0,0],
            [0,0,0,0,0,0,8,-13,0,0,0,0,0,0],
            [0,0,1,-1,1,0,8,-14,0,0,0,0,0,0],
            [0,0,0,0,0,0,8,-13,0,0,0,0,0,1],
            [-2,0,0,2,1,0,0,2,0,-4,5,0,0,0],
            [-2,0,0,2,2,0,3,-3,0,0,0,0,0,0],
            [-2,0,0,2,0,0,0,2,0,-3,1,0,0,0],
            [0,0,0,0,1,0,3,-5,0,2,0,0,0,0],
            [-2,0,0,2,0,0,0,2,0,-4,3,0,0,0],
            [0,0,-1,1,0,0,0,0,2,0,0,0,0,0],
            [0,0,0,0,1,0,0,-1,2,0,0,0,0,0],
            [0,0,1,-1,2,0,0,-2,2,0,0,0,0,0],
            [-1,0,1,0,1,0,3,-5,0,0,0,0,0,0],
            [-1,0,0,1,0,0,3,-4,0,0,0,0,0,0],
            [-2,0,0,2,0,0,0,2,0,-2,-2,0,0,0],
            [-2,0,2,0,2,0,0,-5,9,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,0,-1,0,0],
            [0,0,0,0,0,0,0,0,0,0,0,1,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,0,0,2,0],
            [0,0,0,0,0,0,0,0,0,0,0,0,2,1],
            [0,0,0,0,0,0,0,0,0,0,0,0,2,2],
            [-1,0,0,1,0,0,0,3,-4,0,0,0,0,0],
            [0,0,-1,1,0,0,0,1,0,0,2,0,0,0],
            [0,0,1,-1,2,0,0,-1,0,0,2,0,0,0],
            [0,0,0,0,1,0,0,-9,17,0,0,0,0,0],
            [0,0,0,0,2,0,-3,5,0,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,-1,2,0,0,0],
            [0,0,0,0,0,0,0,0,0,1,-2,0,0,0],
            [1,0,0,-2,0,0,17,-16,0,-2,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,1,-3,0,0,0],
            [-2,0,0,2,1,0,0,5,-6,0,0,0,0,0],
            [0,0,-2,2,0,0,0,9,-13,0,0,0,0,0],
            [0,0,1,-1,2,0,0,-1,0,0,1,0,0,0],
            [0,0,0,0,1,0,0,0,0,0,1,0,0,0],
            [0,0,-1,1,0,0,0,1,0,0,1,0,0,0],
            [0,0,-2,2,0,0,5,-6,0,0,0,0,0,0],
            [0,0,-1,1,1,0,5,-7,0,0,0,0,0,0],
            [-2,0,0,2,0,0,6,-8,0,0,0,0,0,0],
            [2,0,1,-3,1,0,-6,7,0,0,0,0,0,0],
            [0,0,0,0,2,0,0,0,0,1,0,0,0,0],
            [0,0,-1,1,1,0,0,1,0,1,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,0,2,0,0],
            [0,0,0,0,0,0,0,0,0,0,0,2,0,1],
            [0,0,0,0,0,0,0,0,0,0,0,2,0,2],
            [0,0,0,0,0,0,0,-8,15,0,0,0,0,2],
            [0,0,0,0,0,0,0,-8,15,0,0,0,0,1],
            [0,0,1,-1,1,0,0,-9,15,0,0,0,0,0],
            [0,0,0,0,0,0,0,8,-15,0,0,0,0,0],
            [1,0,-1,-1,0,0,0,8,-15,0,0,0,0,0],
            [2,0,0,-2,0,0,2,-5,0,0,0,0,0,0],
            [-2,0,0,2,0,0,0,2,0,-5,5,0,0,0],
            [2,0,0,-2,1,0,0,-6,8,0,0,0,0,0],
            [2,0,0,-2,1,0,0,-2,0,3,0,0,0,0],
            [-2,0,1,1,0,0,0,1,0,-3,0,0,0,0],
            [-2,0,1,1,1,0,0,1,0,-3,0,0,0,0],
            [-2,0,0,2,0,0,0,2,0,-3,0,0,0,0],
            [-2,0,0,2,0,0,0,6,-8,0,0,0,0,0],
            [-2,0,0,2,0,0,0,2,0,-1,-5,0,0,0],
            [-1,0,0,1,0,0,0,1,0,-1,0,0,0,0],
            [-1,0,1,1,1,0,-20,20,0,0,0,0,0,0],
            [1,0,0,-2,0,0,20,-21,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,8,-15,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-10,15,0,0,0,0,0],
            [0,0,-1,1,0,0,0,1,0,1,0,0,0,0],
            [0,0,0,0,1,0,0,0,0,1,0,0,0,0],
            [0,0,1,-1,2,0,0,-1,0,1,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,-2,4,0,0,0],
            [2,0,0,-2,1,0,-6,8,0,0,0,0,0,0],
            [0,0,-2,2,1,0,5,-6,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,0,-1,0,0,1],
            [0,0,1,-1,1,0,0,-1,0,0,-1,0,0,0],
            [0,0,0,0,0,0,0,0,0,0,1,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,1,0,0,0],
            [0,0,0,0,0,0,0,0,0,0,1,0,0,1],
            [0,0,0,0,0,0,0,0,0,0,1,0,0,2],
            [0,0,2,-2,1,0,0,-9,13,0,0,0,0,0],
            [0,0,0,0,1,0,0,7,-13,0,0,0,0,0],
            [-2,0,0,2,0,0,0,5,-6,0,0,0,0,0],
            [0,0,0,0,0,0,0,9,-17,0,0,0,0,0],
            [0,0,0,0,0,0,0,-9,17,0,0,0,0,2],
            [1,0,0,-1,1,0,0,-3,4,0,0,0,0,0],
            [1,0,0,-1,1,0,-3,4,0,0,0,0,0,0],
            [0,0,0,0,2,0,0,-1,2,0,0,0,0,0],
            [0,0,-1,1,1,0,0,0,2,0,0,0,0,0],
            [0,0,-2,2,0,1,0,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,3,-5,0,2,0,0,0,0],
            [-2,0,0,2,1,0,0,2,0,-3,1,0,0,0],
            [-2,0,0,2,1,0,3,-3,0,0,0,0,0,0],
            [0,0,0,0,1,0,8,-13,0,0,0,0,0,0],
            [0,0,-1,1,0,0,8,-12,0,0,0,0,0,0],
            [0,0,2,-2,1,0,-8,11,0,0,0,0,0,0],
            [-1,0,0,1,0,0,0,2,-2,0,0,0,0,0],
            [-1,0,0,0,1,0,18,-16,0,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,-1,1,0,0,0],
            [0,0,0,0,1,0,3,-7,4,0,0,0,0,0],
            [-2,0,1,1,1,0,0,-3,7,0,0,0,0,0],
            [0,0,1,-1,2,0,0,-1,0,-2,5,0,0,0],
            [0,0,0,0,1,0,0,0,0,-2,5,0,0,0],
            [0,0,0,0,1,0,0,-4,8,-3,0,0,0,0],
            [1,0,0,0,1,0,-10,3,0,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-2,0,0,0,0,0,0],
            [-1,0,0,0,1,0,10,-3,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,4,-8,3,0,0,0,0],
            [0,0,0,0,1,0,0,0,0,2,-5,0,0,0],
            [0,0,-1,1,0,0,0,1,0,2,-5,0,0,0],
            [2,0,-1,-1,1,0,0,3,-7,0,0,0,0,0],
            [-2,0,0,2,0,0,0,2,0,0,-5,0,0,0],
            [0,0,0,0,1,0,-3,7,-4,0,0,0,0,0],
            [-2,0,0,2,0,0,0,2,0,-2,0,0,0,0],
            [1,0,0,0,1,0,-18,16,0,0,0,0,0,0],
            [-2,0,1,1,1,0,0,1,0,-2,0,0,0,0],
            [0,0,1,-1,2,0,-8,12,0,0,0,0,0,0],
            [0,0,0,0,1,0,-8,13,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,-2,0,0,0,0,1],
            [0,0,1,-1,1,0,0,0,-2,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,-2,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-2,2,0,0,0,0,0],
            [0,0,0,0,0,0,0,-1,2,0,0,0,0,1],
            [-1,0,0,1,1,0,3,-4,0,0,0,0,0,0],
            [-1,0,0,1,1,0,0,3,-4,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,-2,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,2,0,0,0],
            [0,0,0,0,0,0,0,0,0,0,2,0,0,1],
            [0,0,0,0,0,0,0,0,0,0,2,0,0,2],
            [0,0,1,-1,0,0,3,-6,0,0,0,0,0,0],
            [0,0,0,0,1,0,-3,5,0,0,0,0,0,0],
            [0,0,1,-1,2,0,-3,4,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,-2,4,0,0,0,0,0],
            [0,0,2,-2,1,0,-5,6,0,0,0,0,0,0],
            [0,0,-1,1,0,0,5,-7,0,0,0,0,0,0],
            [0,0,0,0,1,0,5,-8,0,0,0,0,0,0],
            [-2,0,0,2,1,0,6,-8,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,-8,15,0,0,0,0,0],
            [-2,0,0,2,1,0,0,2,0,-3,0,0,0,0],
            [-2,0,0,2,1,0,0,6,-8,0,0,0,0,0],
            [1,0,0,-1,1,0,0,-1,0,1,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,3,-5,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,-1,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,-1,0,0,0,1],
            [0,0,0,0,0,0,0,0,0,1,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,1,0,0,0,1],
            [0,0,1,-1,1,0,0,-1,0,1,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,1,0,0,0,1],
            [0,0,0,0,0,0,0,0,0,1,0,0,0,2],
            [0,0,1,-1,2,0,0,-1,0,0,-1,0,0,0],
            [0,0,0,0,1,0,0,0,0,0,-1,0,0,0],
            [0,0,-1,1,0,0,0,1,0,0,-1,0,0,0],
            [0,0,0,0,0,0,0,-7,13,0,0,0,0,2],
            [0,0,0,0,0,0,0,7,-13,0,0,0,0,0],
            [2,0,0,-2,1,0,0,-5,6,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-8,11,0,0,0,0,0],
            [0,0,2,-2,1,-1,0,2,0,0,0,0,0,0],
            [-2,0,0,2,0,0,0,4,-4,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,2,-2,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,3,0,0,0],
            [0,0,0,0,0,0,0,0,0,0,3,0,0,1],
            [0,0,0,0,0,0,0,0,0,0,3,0,0,2],
            [-2,0,0,2,0,0,3,-3,0,0,0,0,0,0],
            [0,0,0,0,2,0,0,-4,8,-3,0,0,0,0],
            [0,0,0,0,2,0,0,4,-8,3,0,0,0,0],
            [2,0,0,-2,1,0,0,-2,0,2,0,0,0,0],
            [0,0,1,-1,2,0,0,-1,0,2,0,0,0,0],
            [0,0,1,-1,2,0,0,0,-2,0,0,0,0,0],
            [0,0,0,0,1,0,0,1,-2,0,0,0,0,0],
            [0,0,-1,1,0,0,0,2,-2,0,0,0,0,0],
            [0,0,-1,1,0,0,0,1,0,0,-2,0,0,0],
            [0,0,2,-2,1,0,0,-2,0,0,2,0,0,0],
            [0,0,1,-1,1,0,3,-6,0,0,0,0,0,0],
            [0,0,0,0,0,0,3,-5,0,0,0,0,0,1],
            [0,0,0,0,0,0,3,-5,0,0,0,0,0,0],
            [0,0,1,-1,1,0,-3,4,0,0,0,0,0,0],
            [0,0,0,0,0,0,-3,5,0,0,0,0,0,1],
            [0,0,0,0,0,0,-3,5,0,0,0,0,0,2],
            [0,0,2,-2,2,0,-3,3,0,0,0,0,0,0],
            [0,0,0,0,0,0,-3,5,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,-4,0,0,0,0,1],
            [0,0,1,-1,1,0,0,1,-4,0,0,0,0,0],
            [0,0,0,0,0,0,0,2,-4,0,0,0,0,0],
            [0,0,0,0,0,0,0,-2,4,0,0,0,0,1],
            [0,0,1,-1,1,0,0,-3,4,0,0,0,0,0],
            [0,0,0,0,0,0,0,-2,4,0,0,0,0,1],
            [0,0,0,0,0,0,0,-2,4,0,0,0,0,2],
            [0,0,0,0,0,0,-5,8,0,0,0,0,0,2],
            [0,0,2,-2,2,0,-5,6,0,0,0,0,0,0],
            [0,0,0,0,0,0,-5,8,0,0,0,0,0,2],
            [0,0,0,0,0,0,-5,8,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-5,7,0,0,0,0,0,0],
            [0,0,0,0,0,0,-5,8,0,0,0,0,0,1],
            [0,0,0,0,0,0,5,-8,0,0,0,0,0,0],
            [0,0,1,-1,2,0,0,-1,0,-1,0,0,0,0],
            [0,0,0,0,1,0,0,0,0,-1,0,0,0,0],
            [0,0,-1,1,0,0,0,1,0,-1,0,0,0,0],
            [0,0,2,-2,1,0,0,-2,0,1,0,0,0,0],
            [0,0,0,0,0,0,0,-6,11,0,0,0,0,2],
            [0,0,0,0,0,0,0,6,-11,0,0,0,0,0],
            [0,0,0,0,0,-1,0,4,0,0,0,0,0,2],
            [0,0,0,0,0,1,0,-4,0,0,0,0,0,0],
            [2,0,0,-2,1,0,-3,3,0,0,0,0,0,0],
            [-2,0,0,2,0,0,0,2,0,0,-2,0,0,0],
            [0,0,2,-2,1,0,0,-7,9,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,4,-5,0,0,2],
            [0,0,0,0,0,0,0,0,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,2,0,0,0,1],
            [0,0,1,-1,1,0,0,-1,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,2,0,0,0,1],
            [0,0,0,0,0,0,0,0,0,2,0,0,0,2],
            [0,0,2,-2,2,0,0,-2,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,0,5,0,0,2],
            [0,0,0,0,1,0,3,-5,0,0,0,0,0,0],
            [0,0,-1,1,0,0,3,-4,0,0,0,0,0,0],
            [0,0,2,-2,1,0,-3,3,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,2,-4,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-4,4,0,0,0,0,0],
            [0,0,1,-1,2,0,-5,7,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,3,-6,0,0,0,0,0],
            [0,0,0,0,0,0,0,-3,6,0,0,0,0,1],
            [0,0,1,-1,1,0,0,-4,6,0,0,0,0,0],
            [0,0,0,0,0,0,0,-3,6,0,0,0,0,1],
            [0,0,0,0,0,0,0,-3,6,0,0,0,0,2],
            [0,0,-1,1,0,0,2,-2,0,0,0,0,0,0],
            [0,0,0,0,1,0,2,-3,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,-5,9,0,0,0,0,2],
            [0,0,0,0,0,0,0,-5,9,0,0,0,0,1],
            [0,0,0,0,0,0,0,5,-9,0,0,0,0,0],
            [0,0,-1,1,0,0,0,1,0,-2,0,0,0,0],
            [0,0,2,-2,1,0,0,-2,0,2,0,0,0,0],
            [-2,0,1,1,1,0,0,1,0,0,0,0,0,0],
            [0,0,-2,2,0,0,3,-3,0,0,0,0,0,0],
            [0,0,0,0,0,0,-6,10,0,0,0,0,0,1],
            [0,0,0,0,0,0,-6,10,0,0,0,0,0,2],
            [0,0,0,0,0,0,-2,3,0,0,0,0,0,2],
            [0,0,0,0,0,0,-2,3,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-2,2,0,0,0,0,0,0],
            [0,0,0,0,0,0,2,-3,0,0,0,0,0,0],
            [0,0,0,0,0,0,2,-3,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,0,0,3,0,0,0,1],
            [0,0,1,-1,1,0,0,-1,0,3,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,3,0,0,0,1],
            [0,0,0,0,0,0,0,0,0,3,0,0,0,2],
            [0,0,0,0,0,0,0,4,-8,0,0,0,0,0],
            [0,0,0,0,0,0,0,-4,8,0,0,0,0,2],
            [0,0,-2,2,0,0,0,2,0,-2,0,0,0,0],
            [0,0,0,0,0,0,0,-4,7,0,0,0,0,2],
            [0,0,0,0,0,0,0,-4,7,0,0,0,0,1],
            [0,0,0,0,0,0,0,4,-7,0,0,0,0,0],
            [0,0,0,0,1,0,-2,3,0,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-2,0,3,0,0,0,0],
            [0,0,0,0,0,0,0,-5,10,0,0,0,0,2],
            [0,0,0,0,1,0,-1,2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,4,0,0,0,2],
            [0,0,0,0,0,0,0,-3,5,0,0,0,0,2],
            [0,0,0,0,0,0,0,-3,5,0,0,0,0,1],
            [0,0,0,0,0,0,0,3,-5,0,0,0,0,0],
            [0,0,0,0,0,0,1,-2,0,0,0,0,0,1],
            [0,0,1,-1,1,0,1,-3,0,0,0,0,0,0],
            [0,0,0,0,0,0,1,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,-1,2,0,0,0,0,0,1],
            [0,0,0,0,0,0,-1,2,0,0,0,0,0,2],
            [0,0,0,0,0,0,-7,11,0,0,0,0,0,2],
            [0,0,0,0,0,0,-7,11,0,0,0,0,0,1],
            [0,0,-2,2,0,0,4,-4,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,2,-3,0,0,0,0,0],
            [0,0,2,-2,1,0,-4,4,0,0,0,0,0,0],
            [0,0,-1,1,0,0,4,-5,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,-1,0,0,0,0,0],
            [0,0,0,0,0,0,-4,7,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-4,6,0,0,0,0,0,0],
            [0,0,0,0,0,0,-4,7,0,0,0,0,0,2],
            [0,0,0,0,0,0,-4,6,0,0,0,0,0,2],
            [0,0,0,0,0,0,-4,6,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-4,5,0,0,0,0,0,0],
            [0,0,0,0,0,0,-4,6,0,0,0,0,0,1],
            [0,0,0,0,0,0,4,-6,0,0,0,0,0,0],
            [-2,0,0,2,0,0,2,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,1,0,0,0,0,0],
            [0,0,-1,1,0,0,1,0,0,0,0,0,0,0],
            [0,0,0,0,1,0,1,-1,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,-1,0,5,0,0,0,2],
            [0,0,0,0,0,0,0,1,-3,0,0,0,0,0],
            [0,0,0,0,0,0,0,-1,3,0,0,0,0,2],
            [0,0,0,0,0,0,0,-7,12,0,0,0,0,2],
            [0,0,0,0,0,0,-1,1,0,0,0,0,0,2],
            [0,0,0,0,0,0,-1,1,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-1,0,0,0,0,0,0,0],
            [0,0,0,0,0,0,1,-1,0,0,0,0,0,0],
            [0,0,0,0,0,0,1,-1,0,0,0,0,0,1],
            [0,0,1,-1,1,0,1,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,-2,5,0,0,0,0,2],
            [0,0,0,0,0,0,0,-1,0,4,0,0,0,2],
            [0,0,0,0,0,0,0,1,0,-4,0,0,0,0],
            [0,0,0,0,1,0,-1,1,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,-6,10,0,0,0,0,2],
            [0,0,0,0,0,0,0,-6,10,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-3,0,3,0,0,0,0],
            [0,0,0,0,0,0,0,-3,7,0,0,0,0,2],
            [-2,0,0,2,0,0,4,-4,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,-5,8,0,0,0,0,2],
            [0,0,0,0,0,0,0,5,-8,0,0,0,0,0],
            [0,0,0,0,0,0,0,-1,0,3,0,0,0,2],
            [0,0,0,0,0,0,0,-1,0,3,0,0,0,1],
            [0,0,0,0,0,0,0,1,0,-3,0,0,0,0],
            [0,0,0,0,0,0,2,-4,0,0,0,0,0,0],
            [0,0,0,0,0,0,-2,4,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-2,3,0,0,0,0,0,0],
            [0,0,0,0,0,0,-2,4,0,0,0,0,0,2],
            [0,0,0,0,0,0,-6,9,0,0,0,0,0,2],
            [0,0,0,0,0,0,-6,9,0,0,0,0,0,1],
            [0,0,0,0,0,0,6,-9,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,1,0,-2,0,0,0,0],
            [0,0,2,-2,1,0,-2,2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,-4,6,0,0,0,0,2],
            [0,0,0,0,0,0,0,4,-6,0,0,0,0,0],
            [0,0,0,0,1,0,3,-4,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,-1,0,2,0,0,0,2],
            [0,0,0,0,0,0,0,1,0,-2,0,0,0,0],
            [0,0,0,0,1,0,0,1,0,-1,0,0,0,0],
            [0,0,0,0,0,0,-5,9,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,-4,0,0,0,0,0],
            [0,0,0,0,0,0,-3,4,0,0,0,0,0,2],
            [0,0,0,0,0,0,-3,4,0,0,0,0,0,1],
            [0,0,0,0,0,0,3,-4,0,0,0,0,0,0],
            [0,0,0,0,0,0,3,-4,0,0,0,0,0,1],
            [0,0,0,0,1,0,0,2,-2,0,0,0,0,0],
            [0,0,0,0,1,0,0,-1,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,0,-3,0,0,0],
            [0,0,0,0,0,0,0,1,0,1,-5,0,0,0],
            [0,0,0,0,0,0,0,-1,0,1,0,0,0,1],
            [0,0,0,0,0,0,0,1,0,-1,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,-1,0,0,0,1],
            [0,0,0,0,0,0,0,1,0,-3,5,0,0,0],
            [0,0,0,0,1,0,-3,4,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,0,-2,0,0,0],
            [0,0,0,0,0,0,0,2,-2,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,0,-1,0,0,0],
            [0,0,0,0,1,0,0,-1,0,1,0,0,0,0],
            [0,0,0,0,1,0,0,-2,2,0,0,0,0,0],
            [0,0,0,0,0,0,-8,14,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,1,0,2,-5,0,0,0],
            [0,0,0,0,0,0,0,5,-8,3,0,0,0,0],
            [0,0,0,0,0,0,0,5,-8,3,0,0,0,2],
            [0,0,0,0,0,0,0,-1,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,1,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,3,-8,3,0,0,0,0],
            [0,0,0,0,0,0,0,-3,8,-3,0,0,0,2],
            [0,0,0,0,0,0,0,1,0,-2,5,0,0,2],
            [0,0,0,0,0,0,-8,12,0,0,0,0,0,2],
            [0,0,0,0,0,0,-8,12,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,1,-2,0,0,0],
            [0,0,0,0,0,0,0,1,0,0,1,0,0,2],
            [0,0,0,0,0,0,0,0,2,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,2,0,0,0,0,2],
            [0,0,0,0,0,0,0,1,0,0,2,0,0,2],
            [0,0,2,-2,1,0,-5,5,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,1,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,1,0,0,0,1],
            [0,0,0,0,0,0,0,1,0,1,0,0,0,2],
            [0,0,0,0,0,0,3,-6,0,0,0,0,0,0],
            [0,0,0,0,0,0,-3,6,0,0,0,0,0,1],
            [0,0,0,0,0,0,-3,6,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,-1,4,0,0,0,0,2],
            [0,0,0,0,0,0,-5,7,0,0,0,0,0,2],
            [0,0,0,0,0,0,-5,7,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-5,6,0,0,0,0,0,0],
            [0,0,0,0,0,0,5,-7,0,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-1,0,1,0,0,0,0],
            [0,0,0,0,0,0,0,-1,0,1,0,0,0,0],
            [0,0,0,0,0,-1,0,3,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,1,0,2,0,0,0,2],
            [0,0,0,0,0,0,0,-2,6,0,0,0,0,2],
            [0,0,0,0,1,0,2,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,-6,9,0,0,0,0,2],
            [0,0,0,0,0,0,0,6,-9,0,0,0,0,0],
            [0,0,0,0,0,0,-2,2,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-2,1,0,0,0,0,0,0],
            [0,0,0,0,0,0,2,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,2,-2,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,1,0,3,0,0,0,2],
            [0,0,0,0,0,0,0,-5,7,0,0,0,0,2],
            [0,0,0,0,0,0,0,5,-7,0,0,0,0,0],
            [0,0,0,0,1,0,-2,2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,4,-5,0,0,0,0,0],
            [0,0,0,0,0,0,1,-3,0,0,0,0,0,0],
            [0,0,0,0,0,0,-1,3,0,0,0,0,0,1],
            [0,0,1,-1,1,0,-1,2,0,0,0,0,0,0],
            [0,0,0,0,0,0,-1,3,0,0,0,0,0,2],
            [0,0,0,0,0,0,-7,10,0,0,0,0,0,2],
            [0,0,0,0,0,0,-7,10,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,3,-3,0,0,0,0,0],
            [0,0,0,0,0,0,-4,8,0,0,0,0,0,2],
            [0,0,0,0,0,0,-4,5,0,0,0,0,0,2],
            [0,0,0,0,0,0,-4,5,0,0,0,0,0,1],
            [0,0,0,0,0,0,4,-5,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,1,0,0,0,0,2],
            [0,0,0,0,0,0,0,-2,0,5,0,0,0,2],
            [0,0,0,0,0,0,0,0,3,0,0,0,0,2],
            [0,0,0,0,0,0,1,0,0,0,0,0,0,0],
            [0,0,0,0,0,0,1,0,0,0,0,0,0,2],
            [0,0,0,0,0,0,-9,13,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,-1,5,0,0,0,0,2],
            [0,0,0,0,0,0,0,-2,0,4,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,-4,0,0,0,0],
            [0,0,0,0,0,0,0,-2,7,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,-3,0,0,0,0],
            [0,0,0,0,0,0,-2,5,0,0,0,0,0,1],
            [0,0,0,0,0,0,-2,5,0,0,0,0,0,2],
            [0,0,0,0,0,0,-6,8,0,0,0,0,0,2],
            [0,0,0,0,0,0,-6,8,0,0,0,0,0,1],
            [0,0,0,0,0,0,6,-8,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,2,0,-2,0,0,0,0],
            [0,0,0,0,0,0,0,-3,9,0,0,0,0,2],
            [0,0,0,0,0,0,0,5,-6,0,0,0,0,0],
            [0,0,0,0,0,0,0,5,-6,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,-2,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,-2,0,0,0,1],
            [0,0,0,0,0,0,0,2,0,-2,0,0,0,2],
            [0,0,0,0,0,0,-5,10,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,4,-4,0,0,0,0,0],
            [0,0,0,0,0,0,0,4,-4,0,0,0,0,2],
            [0,0,0,0,0,0,-3,3,0,0,0,0,0,1],
            [0,0,0,0,0,0,3,-3,0,0,0,0,0,0],
            [0,0,0,0,0,0,3,-3,0,0,0,0,0,1],
            [0,0,0,0,0,0,3,-3,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,0,-3,0,0,0],
            [0,0,0,0,0,0,0,-5,13,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,-1,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,-1,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,0,-2,0,0,0],
            [0,0,0,0,0,0,0,2,0,0,-2,0,0,1],
            [0,0,0,0,0,0,0,3,-2,0,0,0,0,0],
            [0,0,0,0,0,0,0,3,-2,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,0,-1,0,0,2],
            [0,0,0,0,0,0,0,-6,15,0,0,0,0,2],
            [0,0,0,0,0,0,-8,15,0,0,0,0,0,2],
            [0,0,0,0,0,0,-3,9,-4,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,2,-5,0,0,2],
            [0,0,0,0,0,0,0,-2,8,-1,-5,0,0,2],
            [0,0,0,0,0,0,0,6,-8,3,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,0,0,0,0,1],
            [0,0,1,-1,1,0,0,1,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,2,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,-6,16,-4,-5,0,0,2],
            [0,0,0,0,0,0,0,-2,8,-3,0,0,0,2],
            [0,0,0,0,0,0,0,-2,8,-3,0,0,0,2],
            [0,0,0,0,0,0,0,6,-8,1,5,0,0,2],
            [0,0,0,0,0,0,0,2,0,-2,5,0,0,2],
            [0,0,0,0,0,0,3,-5,4,0,0,0,0,2],
            [0,0,0,0,0,0,-8,11,0,0,0,0,0,2],
            [0,0,0,0,0,0,-8,11,0,0,0,0,0,1],
            [0,0,0,0,0,0,-8,11,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,11,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,0,1,0,0,2],
            [0,0,0,0,0,0,3,-3,0,2,0,0,0,2],
            [0,0,2,-2,1,0,0,4,-8,3,0,0,0,0],
            [0,0,1,-1,0,0,0,1,0,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-4,8,-3,0,0,0,0],
            [0,0,0,0,0,0,0,1,2,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,1,0,0,0,2],
            [0,0,0,0,0,0,-3,7,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,0,4,0,0,0,0,2],
            [0,0,0,0,0,0,-5,6,0,0,0,0,0,2],
            [0,0,0,0,0,0,-5,6,0,0,0,0,0,1],
            [0,0,0,0,0,0,5,-6,0,0,0,0,0,0],
            [0,0,0,0,0,0,5,-6,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,2,0,0,0,2],
            [0,0,0,0,0,0,0,-1,6,0,0,0,0,2],
            [0,0,0,0,0,0,0,7,-9,0,0,0,0,2],
            [0,0,0,0,0,0,2,-1,0,0,0,0,0,0],
            [0,0,0,0,0,0,2,-1,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,6,-7,0,0,0,0,2],
            [0,0,0,0,0,0,0,5,-5,0,0,0,0,2],
            [0,0,0,0,0,0,-1,4,0,0,0,0,0,1],
            [0,0,0,0,0,0,-1,4,0,0,0,0,0,2],
            [0,0,0,0,0,0,-7,9,0,0,0,0,0,2],
            [0,0,0,0,0,0,-7,9,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,4,-3,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,-1,0,0,0,0,2],
            [0,0,0,0,0,0,-4,4,0,0,0,0,0,1],
            [0,0,0,0,0,0,4,-4,0,0,0,0,0,0],
            [0,0,0,0,0,0,4,-4,0,0,0,0,0,1],
            [0,0,0,0,0,0,4,-4,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,1,0,0,0,0,2],
            [0,0,0,0,0,0,0,-3,0,5,0,0,0,2],
            [0,0,0,0,0,0,1,1,0,0,0,0,0,0],
            [0,0,0,0,0,0,1,1,0,0,0,0,0,1],
            [0,0,0,0,0,0,1,1,0,0,0,0,0,2],
            [0,0,0,0,0,0,-9,12,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,-4,0,0,0,0],
            [0,0,2,-2,1,0,1,-1,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,7,-8,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,-3,0,0,0,0],
            [0,0,0,0,0,0,0,3,0,-3,0,0,0,2],
            [0,0,0,0,0,0,-2,6,0,0,0,0,0,2],
            [0,0,0,0,0,0,-6,7,0,0,0,0,0,1],
            [0,0,0,0,0,0,6,-7,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,6,-6,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,-2,0,0,0,0],
            [0,0,0,0,0,0,0,3,0,-2,0,0,0,2],
            [0,0,0,0,0,0,0,5,-4,0,0,0,0,2],
            [0,0,0,0,0,0,3,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,3,-2,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,-1,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,-1,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,0,-2,0,0,2],
            [0,0,0,0,0,0,0,4,-2,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,0,-1,0,0,2],
            [0,0,2,-2,1,0,0,1,0,-1,0,0,0,0],
            [0,0,0,0,0,0,-8,16,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,2,-5,0,0,2],
            [0,0,0,0,0,0,0,7,-8,3,0,0,0,2],
            [0,0,0,0,0,0,0,-5,16,-4,-5,0,0,2],
            [0,0,0,0,0,0,0,3,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,-1,8,-3,0,0,0,2],
            [0,0,0,0,0,0,-8,10,0,0,0,0,0,2],
            [0,0,0,0,0,0,-8,10,0,0,0,0,0,1],
            [0,0,0,0,0,0,-8,10,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,2,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,1,0,0,0,2],
            [0,0,0,0,0,0,-3,8,0,0,0,0,0,2],
            [0,0,0,0,0,0,-5,5,0,0,0,0,0,1],
            [0,0,0,0,0,0,5,-5,0,0,0,0,0,0],
            [0,0,0,0,0,0,5,-5,0,0,0,0,0,1],
            [0,0,0,0,0,0,5,-5,0,0,0,0,0,2],
            [0,0,0,0,0,0,2,0,0,0,0,0,0,0],
            [0,0,0,0,0,0,2,0,0,0,0,0,0,1],
            [0,0,0,0,0,0,2,0,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,7,-7,0,0,0,0,2],
            [0,0,0,0,0,0,0,7,-7,0,0,0,0,2],
            [0,0,0,0,0,0,0,6,-5,0,0,0,0,2],
            [0,0,0,0,0,0,7,-8,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,5,-3,0,0,0,0,2],
            [0,0,0,0,0,0,4,-3,0,0,0,0,0,2],
            [0,0,0,0,0,0,1,2,0,0,0,0,0,2],
            [0,0,0,0,0,0,-9,11,0,0,0,0,0,2],
            [0,0,0,0,0,0,-9,11,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,4,0,-4,0,0,0,2],
            [0,0,0,0,0,0,0,4,0,-3,0,0,0,2],
            [0,0,0,0,0,0,-6,6,0,0,0,0,0,1],
            [0,0,0,0,0,0,6,-6,0,0,0,0,0,0],
            [0,0,0,0,0,0,6,-6,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,4,0,-2,0,0,0,2],
            [0,0,0,0,0,0,0,6,-4,0,0,0,0,2],
            [0,0,0,0,0,0,3,-1,0,0,0,0,0,0],
            [0,0,0,0,0,0,3,-1,0,0,0,0,0,1],
            [0,0,0,0,0,0,3,-1,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,4,0,-1,0,0,0,2],
            [0,0,0,0,0,0,0,4,0,0,-2,0,0,2],
            [0,0,0,0,0,0,0,5,-2,0,0,0,0,2],
            [0,0,0,0,0,0,0,4,0,0,0,0,0,0],
            [0,0,0,0,0,0,8,-9,0,0,0,0,0,0],
            [0,0,0,0,0,0,5,-4,0,0,0,0,0,2],
            [0,0,0,0,0,0,2,1,0,0,0,0,0,2],
            [0,0,0,0,0,0,2,1,0,0,0,0,0,1],
            [0,0,0,0,0,0,2,1,0,0,0,0,0,1],
            [0,0,0,0,0,0,-7,7,0,0,0,0,0,1],
            [0,0,0,0,0,0,7,-7,0,0,0,0,0,0],
            [0,0,0,0,0,0,4,-2,0,0,0,0,0,1],
            [0,0,0,0,0,0,4,-2,0,0,0,0,0,2],
            [0,0,0,0,0,0,4,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,4,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,5,0,-4,0,0,0,2],
            [0,0,0,0,0,0,0,5,0,-3,0,0,0,2],
            [0,0,0,0,0,0,0,5,0,-2,0,0,0,2],
            [0,0,0,0,0,0,3,0,0,0,0,0,0,2],
            [0,0,0,0,0,0,-8,8,0,0,0,0,0,1],
            [0,0,0,0,0,0,8,-8,0,0,0,0,0,0],
            [0,0,0,0,0,0,5,-3,0,0,0,0,0,1],
            [0,0,0,0,0,0,5,-3,0,0,0,0,0,2],
            [0,0,0,0,0,0,-9,9,0,0,0,0,0,1],
            [0,0,0,0,0,0,-9,9,0,0,0,0,0,1],
            [0,0,0,0,0,0,-9,9,0,0,0,0,0,1],
            [0,0,0,0,0,0,9,-9,0,0,0,0,0,0],
            [0,0,0,0,0,0,6,-4,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,0,0,0,0,0,0,2],
            [1,0,0,-2,0,0,0,2,0,-2,0,0,0,0],
            [1,0,0,-2,0,0,2,-2,0,0,0,0,0,0],
            [1,0,0,-2,0,0,0,1,0,-1,0,0,0,0],
            [1,0,0,-2,0,0,1,-1,0,0,0,0,0,0],
            [-1,0,0,0,0,0,3,-3,0,0,0,0,0,0],
            [-1,0,0,0,0,0,0,2,0,-2,0,0,0,0],
            [-1,0,0,2,0,0,0,4,-8,3,0,0,0,0],
            [1,0,0,-2,0,0,0,4,-8,3,0,0,0,0],
            [-2,0,0,2,0,0,0,4,-8,3,0,0,0,0],
            [-1,0,0,0,0,0,0,2,0,-3,0,0,0,0],
            [-1,0,0,0,0,0,0,1,0,-1,0,0,0,0],
            [-1,0,0,0,0,0,1,-1,0,0,0,0,0,0],
            [-1,0,0,2,0,0,2,-2,0,0,0,0,0,0],
            [1,0,-1,1,0,0,0,1,0,0,0,0,0,0],
            [-1,0,0,2,0,0,0,2,0,-3,0,0,0,0],
            [-2,0,0,0,0,0,0,2,0,-3,0,0,0,0],
            [1,0,0,0,0,0,0,4,-8,3,0,0,0,0],
            [-1,0,1,-1,1,0,0,-1,0,0,0,0,0,0],
            [1,0,1,-1,1,0,0,-1,0,0,0,0,0,0],
            [-1,0,0,0,0,0,0,4,-8,3,0,0,0,0],
            [-1,0,0,2,1,0,0,2,0,-2,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,-2,0,0,0,0],
            [-1,0,0,2,0,0,0,2,0,-2,0,0,0,0],
            [-1,0,0,2,0,0,3,-3,0,0,0,0,0,0],
            [1,0,0,-2,1,0,0,-2,0,2,0,0,0,0],
            [1,0,2,-2,2,0,-3,3,0,0,0,0,0,0],
            [1,0,2,-2,2,0,0,-2,0,2,0,0,0,0],
            [1,0,0,0,0,0,1,-1,0,0,0,0,0,0],
            [1,0,0,0,0,0,0,1,0,-1,0,0,0,0],
            [0,0,0,-2,0,0,2,-2,0,0,0,0,0,0],
            [0,0,0,-2,0,0,0,1,0,-1,0,0,0,0],
            [0,0,2,0,2,0,-2,2,0,0,0,0,0,0],
            [0,0,2,0,2,0,0,-1,0,1,0,0,0,0],
            [0,0,2,0,2,0,-1,1,0,0,0,0,0,0],
            [0,0,2,0,2,0,-2,3,0,0,0,0,0,0],
            [0,0,0,2,0,0,0,2,0,-2,0,0,0,0],
            [0,0,1,1,2,0,0,1,0,0,0,0,0,0],
            [1,0,2,0,2,0,0,1,0,0,0,0,0,0],
            [-1,0,2,0,2,0,10,-3,0,0,0,0,0,0],
            [0,0,1,1,1,0,0,1,0,0,0,0,0,0],
            [1,0,2,0,2,0,0,1,0,0,0,0,0,0],
            [0,0,2,0,2,0,0,4,-8,3,0,0,0,0],
            [0,0,2,0,2,0,0,-4,8,-3,0,0,0,0],
            [-1,0,2,0,2,0,0,-4,8,-3,0,0,0,0],
            [2,0,2,-2,2,0,0,-2,0,3,0,0,0,0],
            [1,0,2,0,1,0,0,-2,0,3,0,0,0,0],
            [0,0,1,1,0,0,0,1,0,0,0,0,0,0],
            [-1,0,2,0,1,0,0,1,0,0,0,0,0,0],
            [-2,0,2,2,2,0,0,2,0,-2,0,0,0,0],
            [0,0,2,0,2,0,2,-3,0,0,0,0,0,0],
            [0,0,2,0,2,0,1,-1,0,0,0,0,0,0],
            [0,0,2,0,2,0,0,1,0,-1,0,0,0,0],
            [0,0,2,0,2,0,2,-2,0,0,0,0,0,0],
            [-1,0,2,2,2,0,0,-1,0,1,0,0,0,0],
            [1,0,2,0,2,0,-1,1,0,0,0,0,0,0],
            [-1,0,2,2,2,0,0,2,0,-3,0,0,0,0],
            [2,0,2,0,2,0,0,2,0,-3,0,0,0,0],
            [1,0,2,0,2,0,0,-4,8,-3,0,0,0,0],
            [1,0,2,0,2,0,0,4,-8,3,0,0,0,0],
            [1,0,1,1,1,0,0,1,0,0,0,0,0,0],
            [0,0,2,0,2,0,0,1,0,0,0,0,0,0],
            [2,0,2,0,1,0,0,1,0,0,0,0,0,0],
            [-1,0,2,2,2,0,0,2,0,-2,0,0,0,0],
            [-1,0,2,2,2,0,3,-3,0,0,0,0,0,0],
            [1,0,2,0,2,0,1,-1,0,0,0,0,0,0],
            [0,0,2,2,2,0,0,2,0,-2,0,0,0,0]]
    
    #行星章动系数，单位：0.1微角秒
    #*  longitude (sin, cos), obliquity (sin, cos)
    ICPL=[[1440,0,0,0],
            [56,-117,-42,-40],
            [125,-43,0,-54],
            [0,5,0,0],
            [3,-7,-3,0],
            [3,0,0,-2],
            [-114,0,0,61],
            [-219,89,0,0],
            [-3,0,0,0],
            [-462,1604,0,0],
            [99,0,0,-53],
            [-3,0,0,2],
            [0,6,2,0],
            [3,0,0,0],
            [-12,0,0,0],
            [14,-218,117,8],
            [31,-481,-257,-17],
            [-491,128,0,0],
            [-3084,5123,2735,1647],
            [-1444,2409,-1286,-771],
            [11,-24,-11,-9],
            [26,-9,0,0],
            [103,-60,0,0],
            [0,-13,-7,0],
            [-26,-29,-16,14],
            [9,-27,-14,-5],
            [12,0,0,-6],
            [-7,0,0,0],
            [0,24,0,0],
            [284,0,0,-151],
            [226,101,0,0],
            [0,-8,-2,0],
            [0,-6,-3,0],
            [5,0,0,-3],
            [-41,175,76,17],
            [0,15,6,0],
            [425,212,-133,269],
            [1200,598,319,-641],
            [235,334,0,0],
            [11,-12,-7,-6],
            [5,-6,3,3],
            [-5,0,0,3],
            [6,0,0,-3],
            [15,0,0,0],
            [13,0,0,-7],
            [-6,-9,0,0],
            [266,-78,0,0],
            [-460,-435,-232,246],
            [0,15,7,0],
            [-3,0,0,2],
            [0,131,0,0],
            [4,0,0,0],
            [0,3,0,0],
            [0,4,2,0],
            [0,3,0,0],
            [-17,-19,-10,9],
            [-9,-11,6,-5],
            [-6,0,0,3],
            [-16,8,0,0],
            [0,3,0,0],
            [11,24,11,-5],
            [-3,-4,-2,1],
            [3,0,0,-1],
            [0,-8,-4,0],
            [0,3,0,0],
            [0,5,0,0],
            [0,3,2,0],
            [-6,4,2,3],
            [-3,-5,0,0],
            [-5,0,0,2],
            [4,24,13,-2],
            [-42,20,0,0],
            [-10,233,0,0],
            [-3,0,0,1],
            [78,-18,0,0],
            [0,3,1,0],
            [0,-3,-1,0],
            [0,-4,-2,1],
            [0,-8,-4,-1],
            [0,-5,3,0],
            [-7,0,0,3],
            [-14,8,3,6],
            [0,8,-4,0],
            [0,19,10,0],
            [45,-22,0,0],
            [-3,0,0,0],
            [0,-3,0,0],
            [0,3,0,0],
            [3,5,3,-2],
            [89,-16,-9,-48],
            [0,3,0,0],
            [-3,7,4,2],
            [-349,-62,0,0],
            [-15,22,0,0],
            [-3,0,0,0],
            [-53,0,0,0],
            [5,0,0,-3],
            [0,-8,0,0],
            [15,-7,-4,-8],
            [-3,0,0,1],
            [-21,-78,0,0],
            [20,-70,-37,-11],
            [0,6,3,0],
            [5,3,2,-2],
            [-17,-4,-2,9],
            [0,6,3,0],
            [32,15,-8,17],
            [174,84,45,-93],
            [11,56,0,0],
            [-66,-12,-6,35],
            [47,8,4,-25],
            [0,8,4,0],
            [10,-22,-12,-5],
            [-3,0,0,2],
            [-24,12,0,0],
            [5,-6,0,0],
            [3,0,0,-2],
            [4,3,1,-2],
            [0,29,15,0],
            [-5,-4,-2,2],
            [8,-3,-1,-5],
            [0,-3,0,0],
            [10,0,0,0],
            [3,0,0,-2],
            [-5,0,0,3],
            [46,66,35,-25],
            [-14,7,0,0],
            [0,3,2,0],
            [-5,0,0,0],
            [-68,-34,-18,36],
            [0,14,7,0],
            [10,-6,-3,-5],
            [-5,-4,-2,3],
            [-3,5,2,1],
            [76,17,9,-41],
            [84,298,159,-45],
            [3,0,0,-1],
            [-3,0,0,2],
            [-3,0,0,1],
            [-82,292,156,44],
            [-73,17,9,39],
            [-9,-16,0,0],
            [3,0,-1,-2],
            [-3,0,0,0],
            [-9,-5,-3,5],
            [-439,0,0,0],
            [57,-28,-15,-30],
            [0,-6,-3,0],
            [-4,0,0,2],
            [-40,57,30,21],
            [23,7,3,-13],
            [273,80,43,-146],
            [-449,430,0,0],
            [-8,-47,-25,4],
            [6,47,25,-3],
            [0,23,13,0],
            [-3,0,0,2],
            [3,-4,-2,-2],
            [-48,-110,-59,26],
            [51,114,61,-27],
            [-133,0,0,57],
            [0,4,0,0],
            [-21,-6,-3,11],
            [0,-3,-1,0],
            [-11,-21,-11,6],
            [-18,-436,-233,9],
            [35,-7,0,0],
            [0,5,3,0],
            [11,-3,-1,-6],
            [-5,-3,-1,3],
            [-53,-9,-5,28],
            [0,3,2,1],
            [4,0,0,-2],
            [0,-4,0,0],
            [-50,194,103,27],
            [-13,52,28,7],
            [-91,248,0,0],
            [6,49,26,-3],
            [-6,-47,-25,3],
            [0,5,3,0],
            [52,23,10,-23],
            [-3,0,0,1],
            [0,5,3,0],
            [-4,0,0,0],
            [-4,8,3,2],
            [10,0,0,0],
            [3,0,0,-2],
            [0,8,4,0],
            [0,8,4,1],
            [-4,0,0,0],
            [-4,0,0,0],
            [-8,4,2,4],
            [8,-4,-2,-4],
            [0,15,7,0],
            [-138,0,0,0],
            [0,-7,-3,0],
            [0,-7,-3,0],
            [54,0,0,-29],
            [0,10,4,0],
            [-7,0,0,3],
            [-37,35,19,20],
            [0,4,0,0],
            [-4,9,0,0],
            [8,0,0,-4],
            [-9,-14,-8,5],
            [-3,-9,-5,3],
            [-145,47,0,0],
            [-10,40,21,5],
            [11,-49,-26,-7],
            [-2150,0,0,932],
            [-12,0,0,5],
            [85,0,0,-37],
            [4,0,0,-2],
            [3,0,0,-2],
            [-86,153,0,0],
            [-6,9,5,3],
            [9,-13,-7,-5],
            [-8,12,6,4],
            [-51,0,0,22],
            [-11,-268,-116,5],
            [0,12,5,0],
            [0,7,3,0],
            [31,6,3,-17],
            [140,27,14,-75],
            [57,11,6,-30],
            [-14,-39,0,0],
            [0,-6,-2,0],
            [4,15,8,-2],
            [0,4,0,0],
            [-3,0,0,1],
            [0,11,5,0],
            [9,6,0,0],
            [-4,10,4,2],
            [5,3,0,0],
            [16,0,0,-9],
            [-3,0,0,0],
            [0,3,2,-1],
            [7,0,0,-3],
            [-25,22,0,0],
            [42,223,119,-22],
            [-27,-143,-77,14],
            [9,49,26,-5],
            [-1166,0,0,505],
            [-5,0,0,2],
            [-6,0,0,3],
            [-8,0,1,4],
            [0,-4,0,0],
            [117,0,0,-63],
            [-4,8,4,2],
            [3,0,0,-2],
            [-5,0,0,2],
            [0,31,0,0],
            [-5,0,1,3],
            [4,0,0,-2],
            [-4,0,0,2],
            [-24,-13,-6,10],
            [3,0,0,0],
            [0,-32,-17,0],
            [8,12,5,-3],
            [3,0,0,-1],
            [7,13,0,0],
            [-3,16,0,0],
            [50,0,0,-27],
            [0,-5,-3,0],
            [13,0,0,0],
            [0,5,3,1],
            [24,5,2,-11],
            [5,-11,-5,-2],
            [30,-3,-2,-16],
            [18,0,0,-9],
            [8,614,0,0],
            [3,-3,-1,-2],
            [6,17,9,-3],
            [-3,-9,-5,2],
            [0,6,3,-1],
            [-127,21,9,55],
            [3,5,0,0],
            [-6,-10,-4,3],
            [5,0,0,0],
            [16,9,4,-7],
            [3,0,0,-2],
            [0,22,0,0],
            [0,19,10,0],
            [7,0,0,-4],
            [0,-5,-2,0],
            [0,3,1,0],
            [-9,3,1,4],
            [17,0,0,-7],
            [0,-3,-2,-1],
            [-20,34,0,0],
            [-10,0,1,5],
            [-4,0,0,2],
            [22,-87,0,0],
            [-4,0,0,2],
            [-3,-6,-2,1],
            [-16,-3,-1,7],
            [0,-3,-2,0],
            [4,0,0,0],
            [-68,39,0,0],
            [27,0,0,-14],
            [0,-4,0,0],
            [-25,0,0,0],
            [-12,-3,-2,6],
            [3,0,0,-1],
            [3,66,29,-1],
            [490,0,0,-213],
            [-22,93,49,12],
            [-7,28,15,4],
            [-3,13,7,2],
            [-46,14,0,0],
            [-5,0,0,0],
            [2,1,0,0],
            [0,-3,0,0],
            [-28,0,0,15],
            [5,0,0,-2],
            [0,3,0,0],
            [-11,0,0,5],
            [0,3,1,0],
            [-3,0,0,1],
            [25,106,57,-13],
            [5,21,11,-3],
            [1485,0,0,0],
            [-7,-32,-17,4],
            [0,5,3,0],
            [-6,-3,-2,3],
            [30,-6,-2,-13],
            [-4,4,0,0],
            [-19,0,0,10],
            [0,4,2,-1],
            [0,3,0,0],
            [4,0,0,-2],
            [0,-3,-1,0],
            [-3,0,0,0],
            [5,3,1,-2],
            [0,11,0,0],
            [118,0,0,-52],
            [0,-5,-3,0],
            [-28,36,0,0],
            [5,-5,0,0],
            [14,-59,-31,-8],
            [0,9,5,1],
            [-458,0,0,198],
            [0,-45,-20,0],
            [9,0,0,-5],
            [0,-3,0,0],
            [0,-4,-2,-1],
            [11,0,0,-6],
            [6,0,0,-2],
            [-16,23,0,0],
            [0,-4,-2,0],
            [-5,0,0,2],
            [-166,269,0,0],
            [15,0,0,-8],
            [10,0,0,-4],
            [-78,45,0,0],
            [0,-5,-2,0],
            [7,0,0,-4],
            [-5,328,0,0],
            [3,0,0,-2],
            [5,0,0,-2],
            [0,3,1,0],
            [-3,0,0,0],
            [-3,0,0,0],
            [0,-4,-2,0],
            [-1223,-26,0,0],
            [0,7,3,0],
            [3,0,0,0],
            [0,3,2,0],
            [-6,20,0,0],
            [-368,0,0,0],
            [-75,0,0,0],
            [11,0,0,-6],
            [3,0,0,-2],
            [-3,0,0,1],
            [-13,-30,0,0],
            [21,3,0,0],
            [-3,0,0,1],
            [-4,0,0,2],
            [8,-27,0,0],
            [-19,-11,0,0],
            [-4,0,0,2],
            [0,5,2,0],
            [-6,0,0,2],
            [-8,0,0,0],
            [-1,0,0,0],
            [-14,0,0,6],
            [6,0,0,0],
            [-74,0,0,32],
            [0,-3,-1,0],
            [4,0,0,-2],
            [8,11,0,0],
            [0,3,2,0],
            [-262,0,0,114],
            [0,-4,0,0],
            [-7,0,0,4],
            [0,-27,-12,0],
            [-19,-8,-4,8],
            [202,0,0,-87],
            [-8,35,19,5],
            [0,4,2,0],
            [16,-5,0,0],
            [5,0,0,-3],
            [0,-3,0,0],
            [1,0,0,0],
            [-35,-48,-21,15],
            [-3,-5,-2,1],
            [6,0,0,-3],
            [3,0,0,-1],
            [0,-5,0,0],
            [12,55,29,-6],
            [0,5,3,0],
            [-598,0,0,0],
            [-3,-13,-7,1],
            [-5,-7,-3,2],
            [3,0,0,-1],
            [5,-7,0,0],
            [4,0,0,-2],
            [16,-6,0,0],
            [8,-3,0,0],
            [8,-31,-16,-4],
            [0,3,1,0],
            [113,0,0,-49],
            [0,-24,-10,0],
            [4,0,0,-2],
            [27,0,0,0],
            [-3,0,0,1],
            [0,-4,-2,0],
            [5,0,0,-2],
            [0,-3,0,0],
            [-13,0,0,6],
            [5,0,0,-2],
            [-18,-10,-4,8],
            [-4,-28,0,0],
            [-5,6,3,2],
            [-3,0,0,1],
            [-5,-9,-4,2],
            [17,0,0,-7],
            [11,4,0,0],
            [0,-6,-2,0],
            [83,15,0,0],
            [-4,0,0,2],
            [0,-114,-49,0],
            [117,0,0,-51],
            [-5,19,10,2],
            [-3,0,0,0],
            [-3,0,0,2],
            [0,-3,-1,0],
            [3,0,0,0],
            [0,-6,-2,0],
            [393,3,0,0],
            [-4,21,11,2],
            [-6,0,-1,3],
            [-3,8,4,1],
            [8,0,0,0],
            [18,-29,-13,-8],
            [8,34,18,-4],
            [89,0,0,0],
            [3,12,6,-1],
            [54,-15,-7,-24],
            [0,3,0,0],
            [3,0,0,-1],
            [0,35,0,0],
            [-154,-30,-13,67],
            [15,0,0,0],
            [0,4,2,0],
            [0,9,0,0],
            [80,-71,-31,-35],
            [0,-20,-9,0],
            [11,5,2,-5],
            [61,-96,-42,-27],
            [14,9,4,-6],
            [-11,-6,-3,5],
            [0,-3,-1,0],
            [123,-415,-180,-53],
            [0,0,0,-35],
            [-5,0,0,0],
            [7,-32,-17,-4],
            [0,-9,-5,0],
            [0,-4,2,0],
            [-89,0,0,38],
            [0,-86,-19,-6],
            [0,0,-19,6],
            [-123,-416,-180,53],
            [0,-3,-1,0],
            [12,-6,-3,-5],
            [-13,9,4,6],
            [0,-15,-7,0],
            [3,0,0,-1],
            [-62,-97,-42,27],
            [-11,5,2,5],
            [0,-19,-8,0],
            [-3,0,0,1],
            [0,4,2,0],
            [0,3,0,0],
            [0,4,2,0],
            [-85,-70,-31,37],
            [163,-12,-5,-72],
            [-63,-16,-7,28],
            [-21,-32,-14,9],
            [0,-3,-1,0],
            [3,0,0,-2],
            [0,8,0,0],
            [3,10,4,-1],
            [3,0,0,-1],
            [0,-7,-3,0],
            [0,-4,-2,0],
            [6,19,0,0],
            [5,-173,-75,-2],
            [0,-7,-3,0],
            [7,-12,-5,-3],
            [-3,0,0,2],
            [3,-4,-2,-1],
            [74,0,0,-32],
            [-3,12,6,2],
            [26,-14,-6,-11],
            [19,0,0,-8],
            [6,24,13,-3],
            [83,0,0,0],
            [0,-10,-5,0],
            [11,-3,-1,-5],
            [3,0,1,-1],
            [3,0,0,-1],
            [-4,0,0,0],
            [5,-23,-12,-3],
            [-339,0,0,147],
            [0,-10,-5,0],
            [5,0,0,0],
            [3,0,0,-1],
            [0,-4,-2,0],
            [18,-3,0,0],
            [9,-11,-5,-4],
            [-8,0,0,4],
            [3,0,0,-1],
            [0,9,0,0],
            [6,-9,-4,-2],
            [-4,-12,0,0],
            [67,-91,-39,-29],
            [30,-18,-8,-13],
            [0,0,0,0],
            [0,-114,-50,0],
            [0,0,0,23],
            [517,16,7,-224],
            [0,-7,-3,0],
            [143,-3,-1,-62],
            [29,0,0,-13],
            [-4,0,0,2],
            [-6,0,0,3],
            [5,12,5,-2],
            [-25,0,0,11],
            [-3,0,0,1],
            [0,4,2,0],
            [-22,12,5,10],
            [50,0,0,-22],
            [0,7,4,0],
            [0,3,1,0],
            [-4,4,2,2],
            [-5,-11,-5,2],
            [0,4,2,0],
            [4,17,9,-2],
            [59,0,0,0],
            [0,-4,-2,0],
            [-8,0,0,4],
            [-3,0,0,0],
            [4,-15,-8,-2],
            [370,-8,0,-160],
            [0,0,-3,0],
            [0,3,1,0],
            [-6,3,1,3],
            [0,6,0,0],
            [-10,0,0,4],
            [0,9,4,0],
            [4,17,7,-2],
            [34,0,0,-15],
            [0,5,3,0],
            [-5,0,0,2],
            [-37,-7,-3,16],
            [3,13,7,-2],
            [40,0,0,0],
            [0,-3,-2,0],
            [-184,-3,-1,80],
            [-3,0,0,1],
            [-3,0,0,0],
            [0,-10,-6,-1],
            [31,-6,0,-13],
            [-3,-32,-14,1],
            [-7,0,0,3],
            [0,-8,-4,0],
            [3,-4,0,0],
            [0,4,0,0],
            [0,3,1,0],
            [19,-23,-10,2],
            [0,0,0,-10],
            [0,3,2,0],
            [0,9,5,-1],
            [28,0,0,0],
            [0,-7,-4,0],
            [8,-4,0,-4],
            [0,0,-2,0],
            [0,3,0,0],
            [-3,0,0,1],
            [-9,0,1,4],
            [3,12,5,-1],
            [17,-3,-1,0],
            [0,7,4,0],
            [19,0,0,0],
            [0,-5,-3,0],
            [14,-3,0,-1],
            [0,0,-1,0],
            [0,0,0,-5],
            [0,5,3,0],
            [13,0,0,0],
            [0,-3,-2,0],
            [2,9,4,3],
            [0,0,0,-4],
            [8,0,0,0],
            [0,4,2,0],
            [6,0,0,-3],
            [6,0,0,0],
            [0,3,1,0],
            [5,0,0,-2],
            [3,0,0,-1],
            [-3,0,0,0],
            [6,0,0,0],
            [7,0,0,0],
            [-4,0,0,0],
            [4,0,0,0],
            [6,0,0,0],
            [0,-4,0,0],
            [0,-4,0,0],
            [5,0,0,0],
            [-3,0,0,0],
            [4,0,0,0],
            [-5,0,0,0],
            [4,0,0,0],
            [0,3,0,0],
            [13,0,0,0],
            [21,11,0,0],
            [0,-5,0,0],
            [0,-5,-2,0],
            [0,5,3,0],
            [0,-5,0,0],
            [-3,0,0,2],
            [20,10,0,0],
            [-34,0,0,0],
            [-19,0,0,0],
            [3,0,0,-2],
            [-3,0,0,1],
            [-6,0,0,3],
            [-4,0,0,0],
            [3,0,0,0],
            [3,0,0,0],
            [4,0,0,0],
            [3,0,0,-1],
            [6,0,0,-3],
            [-8,0,0,3],
            [0,3,1,0],
            [-3,0,0,0],
            [0,-3,-2,0],
            [126,-63,-27,-55],
            [-5,0,1,2],
            [-3,28,15,2],
            [5,0,1,-2],
            [0,9,4,1],
            [0,9,4,-1],
            [-126,-63,-27,55],
            [3,0,0,-1],
            [21,-11,-6,-11],
            [0,-4,0,0],
            [-21,-11,-6,11],
            [-3,0,0,1],
            [0,3,1,0],
            [8,0,0,-4],
            [-6,0,0,3],
            [-3,0,0,1],
            [3,0,0,-1],
            [-3,0,0,1],
            [-5,0,0,2],
            [24,-12,-5,-11],
            [0,3,1,0],
            [0,3,1,0],
            [0,3,2,0],
            [-24,-12,-5,10],
            [4,0,-1,-2],
            [13,0,0,-6],
            [7,0,0,-3],
            [3,0,0,-1],
            [3,0,0,-1]]
    
    #给定日期与参考日期之间的时间间隔，单位：儒略世纪数
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #日月章动
    #基本参数（Delaunay）

    #月亮的平均近点角
    EL=pymFal03(T)
    
    #太阳的平均近点角
    A=1287104.79305+T*(129596581.0481+T*(-0.5532+\
        T*(0.000136+T*(-0.00001149))))
    ELP=(A%TURNAS)*DAS2R
    
    #月亮的平黄经减去升交点的平黄经
    F=pymFaf03(T)
    
    #月亮相对于太阳的平均距角
    B=1072260.70369+T*(1602961601.2090+T*(-6.3706+\
        T*(0.006593+T*(-0.00003169))))
    D=(B%TURNAS)*DAS2R
    
    #月亮升交点的平黄经
    OM=pymFaom03(T)
    
    #初始化章动量
    DP=0.0
    DE=0.0
    
    #从后到前对日月章动的各项求和
    for I in range(NLS-1,-1,-1):
        C=float(NALS[I][0])*EL+float(NALS[I][1])*ELP+\
            float(NALS[I][2])*F+float(NALS[I][3])*D+float(NALS[I][4])*OM
        ARG=C%D2PI
        SARG=ma.sin(ARG)
        CARG=ma.cos(ARG)
    
        #Term.
        DP=DP+(CLS[I][0]+CLS[I][1]*T)*SARG+CLS[I][2]*CARG
        DE=DE+(CLS[I][3]+CLS[I][4]*T)*CARG+CLS[I][5]*SARG
    
    #从0.1微弧秒单位转换为弧度。
    DPSILS=DP*U2R
    DEPSLS=DE*U2R
    
    #行星章动

    #月亮的平近点角(MHB2000).
    AL=(2.35555598+8328.6914269554*T)%D2PI
    
    #太阳的平近点角(MHB2000).
    ALSU=(6.24006013+628.301955*T)%D2PI
    
    #月亮的平黄经减去升交点的平黄经(MHB2000).
    AF=(1.627905234+8433.466158131*T)%D2PI

    #月亮相对于太阳的平均距角(MHB2000).
    AD=(5.198466741+7771.3771468121*T)%D2PI

    #月亮升交点的平黄经(MHB2000).
    AOM=(2.18243920-33.757045*T)%D2PI

    #黄经的一般累计进动(IERS 2003).
    APA=pymFapa03(T)
    
    #行星的平黄经，水星到天王星(IERS 2003).
    ALME=pymFame03(T)
    ALVE=pymFave03(T)
    ALEA=pymFae03(T)
    ALMA=pymFama03(T)
    ALJU=pymFaju03(T)
    ALSA=pymFasa03(T)
    ALUR=pymFaur03(T)
    
    #海王星的平黄经(MHB2000).
    ALNE=(5.321159000+3.8127774000*T)%D2PI
    
    #初始化章动量
    DP=0.0
    DE=0.0
    
    #从后到前对行星章动的各项求和
    for I in range(NPL-1,-1,-1):
        D=float(NAPL[I][0])*AL+float(NAPL[I][1])*ALSU+float(NAPL[I][2])*AF+\
            float(NAPL[I][3])*AD+float(NAPL[I][4])*AOM+float(NAPL[I][5])*ALME+\
            float(NAPL[I][6])*ALVE+float(NAPL[I][7])*ALEA+\
            float(NAPL[I][8])*ALMA+float(NAPL[I][9])*ALJU+\
            float(NAPL[I][10])*ALSA+float(NAPL[I][11])*ALUR+\
            float(NAPL[I][12])*ALNE+float(NAPL[I][13])*APA
        ARG=D%D2PI
        SARG=ma.sin(ARG)
        CARG=ma.cos(ARG)
    
        #Term.
        DP=DP+float(ICPL[I][0])*SARG+float(ICPL[I][1])*CARG
        DE=DE+float(ICPL[I][2])*SARG+float(ICPL[I][3])*CARG
    
    #从0.1微弧秒单位转换为弧度。
    DPSIPL=DP*U2R
    DEPSPL=DE*U2R

    #日月章动和行星章动相加
    DPSI=DPSILS+DPSIPL
    DEPS=DEPSLS+DEPSPL
    
    return(DPSI,DEPS)

def pymNut06a(DATE1,DATE2):
    '''
    IAU 2000A nutation with adjustments to match the IAU 2006
    precession.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    dpsi : float
        nutation, luni-solar + planetary    
    deps : float
        nutation, luni-solar + planetary

    '''
    #参考历元(J2000.0), JD
    DJ00=2451545.0
    
    #1儒略世纪的天数
    DJC=36525.0
    
    #相对于参考历元的时间间隔，单位：儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #根据IAU 2006，对于J2的长期变化进行改正
    FJ2=-2.7774e-6*T
    
    #调用pymNut00a 获得IAU 2000A 的章动.
    DP,DE=pymNut00a(DATE1, DATE2)
    
    #应用改正项(Wallace & Capitaine, 2006, Eqs.5).
    DPSI=DP+DP*(0.4697e-6+FJ2)
    DEPS=DE+DE*FJ2
    
    return(DPSI,DEPS)

def pymObl06(DATE1,DATE2):
    '''
    Mean obliquity of the ecliptic, IAU 2006 precession model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        obliquity of the ecliptic (radians)

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0
    
    #1儒略世纪的天数
    DJC=36525.0
    
    #相对于参考历元的时间间隔，单位：儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #黄道平均倾角.
    OBL=(84381.406+(-46.836769+(-0.0001831+(0.00200340+(-0.000000576+\
            (-0.0000000434)*T)*T)*T)*T)*T)*DAS2R

    return(OBL)

def pymPfw06(DATE1,DATE2):
    '''
    Precession angles, IAU 2006 (Fukushima-Williams 4-angle formulation).

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    gamb : flaot
        F-W angle gamma_bar (radians)    
    phib : float
        F-W angle phi_bar (radians)    
    psib : float
        F-W angle psi_bar (radians)    
    epsa : float
        F-W angle epsilon_A (radians)

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0
    
    #1儒略世纪的天数
    DJC=36525.0
    
    #相对于参考历元的时间间隔，单位：儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #P03极移+进动角.
    GAMB=(-0.052928+(10.556378+(0.4932044+(-0.00031238+\
           (-0.000002788+(0.0000000260)*T)*T)*T)*T)*T)*DAS2R
    PHIB=(84381.412819+(-46.811016+(0.0511268+(0.00053289+\
           (-0.000000440+(-0.0000000176)*T)*T)*T)*T)*T)*DAS2R
    PSIB=(-0.041775+(5038.481484+(1.5584175+(-0.00018522+\
           (-0.000026452+(-0.0000000148)*T)*T)*T)*T)*T)*DAS2R      
    EPSA=pymObl06(DATE1,DATE2)
    
    return(GAMB,PHIB,PSIB,EPSA)

def pymPnm06a(DATE1,DATE2):
    '''
    Form the matrix of precession-nutation for a given date (including
    frame bias), equinox based, IAU 2006 precession and IAU 2000A
    nutation models.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    rbpn : list(3,3)
        bias-precession-nutation matrix

    '''
    #调用pymPfw06获得四个进动角
    GAMB,PHIB,PSIB,EPSA=pymPfw06(DATE1, DATE2)
    
    #调用pymNut06a获得章动参数
    DP,DE=pymNut06a(DATE1,DATE2)
    
    #结合进动角，章动参数，获得用于计算的极移进动章动矩阵
    RBPN=pymFw2m(GAMB,PHIB,PSIB+DP,EPSA+DE)
    
    return(RBPN)

def pymAnp(A):
    '''
    Normalize angle into the range 0 <= a < 2pi.

    Parameters
    ----------
    a : float
        angle (radians)

    Returns
    -------
    function value : float
        angle in range 0-2pi

    '''
    W=np.abs(A)%(2*ma.pi)
    if A<0:
        W=-W
    if (W<0.0):
        W=W+2*ma.pi
    
    return(W)

def pymEra00(DJ1,DJ2):
    '''
    Earth rotation angle (IAU 2000 model).

    Parameters
    ----------
    dj1 : float
        UT1 as a 2-part Julian Date    
    dj2 : float
        UT1 as a 2-part Julian Date

    Returns
    -------
    function value : float
        Earth rotation angle (radians), range 0-2pi

    '''
    #参考历元(J2000.0), JD
    DJ00=2451545.0
    
    #自参考历元经过的天数
    if (DJ1<=DJ2):
        D1=DJ1
        D2=DJ2
    else:
        D1=DJ2
        D2=DJ1
    T=D1+(D2-DJ00)

    #时间的小数部分(days).
    F=D1%1.0+D2%1.0
    
    #在这个UT1时刻的地球自转角.
    A=2*ma.pi*(F+0.7790572732640+0.00273781191135448*T)
    ERA=pymAnp(A)
    
    return(ERA)

def pymSp00(DATE1,DATE2):
    '''
    The TIO locator s', positioning the Terrestrial Intermediate Origin
    on the equator of the Celestial Intermediate Pole.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        the TIO locator s' in radians

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0
    
    #1儒略世纪的天数
    DJC=36525.0
    
    #相对于J2000.0的儒略世纪数
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #近似TIO定位角
    SP=-47e-6*T*DAS2R
    
    return(SP)

def pymPom00(XP,YP,SP):
    '''
    Form the matrix of polar motion for a given date, IAU 2000.

    Parameters
    ----------
    xp : float
        coordinates of the pole (radians)    
    yp : float
        coordinates of the pole (radians)    
    sp : float
        the TIO locator s' (radians)

    Returns
    -------
    rpom : list(3,3)
        polar-motion matrix

    '''
    RPOM=pymIr()
    RPOM=pymRz(SP,RPOM)
    RPOM=pymRy(-XP,RPOM)
    RPOM=pymRx(-YP,RPOM)
    
    return(RPOM)

def pymS06(DATE1,DATE2,X,Y):
    '''
    The CIO locator s, positioning the Celestial Intermediate Origin on
    the equator of the Celestial Intermediate Pole, given the CIP's X,Y
    coordinates.  Compatible with IAU 2006/2000A precession-nutation.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date    
    x : float
        CIP coordinates    
    y : float
        CIP coordinates

    Returns
    -------
    function value : float
        the CIO locator s in radians

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0
    
    #1儒略世纪对应的天数
    DJC=36525.0
    
    #自J2000.0起经过的儒略世纪数，此处为定义
    T=0
    
    #基本的参数
    FA=[0 for i in range(8)]
    
    # *  ---------------------
    # *  The series for s+XY/2
    # *  ---------------------
    
    NSP=6
    NS0=33
    NS1=3
    NS2=25
    NS3=4
    NS4=1
    
    #多项式系数
    SP=[94e-6,3808.65e-6,-122.68e-6,-72574.11e-6,27.98e-6,15.62e-6]
    
    #Coefficients of l,l',F,D,Om,LVe,LE,pA
    #Argument coefficients for t^0
    KS0=[[0,0,0,0,1,0,0,0],
            [0,0,0,0,2,0,0,0],
            [0,0,2,-2,3,0,0,0],
            [0,0,2,-2,1,0,0,0],
            [0,0,2,-2,2,0,0,0],
            [0,0,2,0,3,0,0,0],
            [0,0,2,0,1,0,0,0],
            [0,0,0,0,3,0,0,0],
            [0,1,0,0,1,0,0,0],
            [0,1,0,0,-1,0,0,0],
            [1,0,0,0,-1,0,0,0],
            [1,0,0,0,1,0,0,0],
            [0,1,2,-2,3,0,0,0],
            [0,1,2,-2,1,0,0,0],
            [0,0,4,-4,4,0,0,0],
            [0,0,1,-1,1,-8,12,0],
            [0,0,2,0,0,0,0,0],
            [0,0,2,0,2,0,0,0],
            [1,0,2,0,3,0,0,0],
            [1,0,2,0,1,0,0,0],
            [0,0,2,-2,0,0,0,0],
            [0,1,-2,2,-3,0,0,0],
            [0,1,-2,2,-1,0,0,0],
            [0,0,0,0,0,8,-13,-1],
            [0,0,0,2,0,0,0,0],
            [2,0,-2,0,-1,0,0,0],
            [0,1,2,-2,2,0,0,0],
            [1,0,0,-2,1,0,0,0],
            [1,0,0,-2,-1,0,0,0],
            [0,0,4,-2,4,0,0,0],
            [0,0,2,-2,4,0,0,0],
            [1,0,-2,0,-3,0,0,0],
            [1,0,-2,0,-1,0,0,0]]
    
    #Argument coefficients for t^1
    KS1=[[0,0,0,0,2,0,0,0],
            [0,0,0,0,1,0,0,0],
            [0,0,2,-2,3,0,0,0]]
    
    #Argument coefficients for t^2
    KS2=[[0,0,0,0,1,0,0,0],
            [0,0,2,-2,2,0,0,0],
            [0,0,2,0,2,0,0,0],
            [0,0,0,0,2,0,0,0],
            [0,1,0,0,0,0,0,0],
            [1,0,0,0,0,0,0,0],
            [0,1,2,-2,2,0,0,0],
            [0,0,2,0,1,0,0,0],
            [1,0,2,0,2,0,0,0],
            [0,1,-2,2,-2,0,0,0],
            [1,0,0,-2,0,0,0,0],
            [0,0,2,-2,1,0,0,0],
            [1,0,-2,0,-2,0,0,0],
            [0,0,0,2,0,0,0,0],
            [1,0,0,0,1,0,0,0],
            [1,0,-2,-2,-2,0,0,0],
            [1,0,0,0,-1,0,0,0],
            [1,0,2,0,1,0,0,0],
            [2,0,0,-2,0,0,0,0],
            [2,0,-2,0,-1,0,0,0],
            [0,0,2,2,2,0,0,0],
            [2,0,2,0,2,0,0,0],
            [2,0,0,0,0,0,0,0],
            [1,0,2,-2,2,0,0,0],
            [0,0,2,0,0,0,0,0]]
    
    #Argument coefficients for t^3
    KS3=[[0,0,0,0,1,0,0,0],
            [0,0,2,-2,2,0,0,0],
            [0,0,2,0,2,0,0,0],
            [0,0,0,0,2,0,0,0]]
    
    #Argument coefficients for t^4
    KS4=[[0,0,0,0,1,0,0,0]]
 
    #Sine and cosine coefficients
    #Sine and cosine coefficients for t^0
    SS0=[[-2640.73e-6,+0.39e-6],
            [-63.53e-6,+0.02e-6],
            [-11.75e-6,-0.01e-6],
            [-11.21e-6,-0.01e-6],
            [+4.57e-6,0.00e-6],
            [-2.02e-6,0.00e-6],
            [-1.98e-6,0.00e-6],
            [+1.72e-6,0.00e-6],
            [+1.41e-6,+0.01e-6],
            [+1.26e-6,+0.01e-6],
            [+0.63e-6,0.00e-6],
            [+0.63e-6,0.00e-6],
            [-0.46e-6,0.00e-6],
            [-0.45e-6,0.00e-6],
            [-0.36e-6,0.00e-6],
            [+0.24e-6,+0.12e-6],
            [-0.32e-6,0.00e-6],
            [-0.28e-6,0.00e-6],
            [-0.27e-6,0.00e-6],
            [-0.26e-6,0.00e-6],
            [+0.21e-6,0.00e-6],
            [-0.19e-6,0.00e-6],
            [-0.18e-6,0.00e-6],
            [+0.10e-6,-0.05e-6],
            [-0.15e-6,0.00e-6],
            [+0.14e-6,0.00e-6],
            [+0.14e-6,0.00e-6],
            [-0.14e-6,0.00e-6],
            [-0.14e-6,0.00e-6],
            [-0.13e-6,0.00e-6],
            [+0.11e-6,0.00e-6],
            [-0.11e-6,0.00e-6],
            [-0.11e-6,0.00e-6]]
    
    #Sine and cosine coefficients for t^1
    SS1=[[-0.07e-6,+3.57e-6],
            [+1.73e-6,-0.03e-6],
            [0.00e-6,+0.48e-6]]
    
    #Sine and cosine coefficients for t^2
    SS2=[[+743.52e-6,-0.17e-6],
            [+56.91e-6,+0.06e-6],
            [+9.84e-6,-0.01e-6],
            [-8.85e-6,+0.01e-6],
            [-6.38e-6,-0.05e-6],
            [-3.07e-6,0.00e-6],
            [+2.23e-6,0.00e-6],
            [+1.67e-6,0.00e-6],
            [+1.30e-6,0.00e-6],
            [+0.93e-6,0.00e-6],
            [+0.68e-6,0.00e-6],
            [-0.55e-6,0.00e-6],
            [+0.53e-6,0.00e-6],
            [-0.27e-6,0.00e-6],
            [-0.27e-6,0.00e-6],
            [-0.26e-6,0.00e-6],
            [-0.25e-6,0.00e-6],
            [+0.22e-6,0.00e-6],
            [-0.21e-6,0.00e-6],
            [+0.20e-6,0.00e-6],
            [+0.17e-6,0.00e-6],
            [+0.13e-6,0.00e-6],
            [-0.13e-6,0.00e-6],
            [-0.12e-6,0.00e-6],
            [-0.11e-6,0.00e-6]]
    
    #Sine and cosine coefficients for t^3
    SS3=[[+0.30e-6,-23.42e-6],
            [-0.03e-6,-1.46e-6],
            [-0.01e-6,-0.25e-6],
            [0.00e-6,+0.23e-6]]
    
    #Sine and cosine coefficients for t^4
    SS4=[[-0.26e-6,-0.01e-6]]
    
    #参考历元J2000.0和当前日期之间的间隔，儒略世纪数
    T=((DATE1-DJ00)+DATE2)/DJC

    #Fundamental Arguments (from IERS Conventions 2003)

    #月亮的平近点角
    FA[0]=pymFal03(T)
    
    #太阳的平近点角
    FA[1]=pymFalp03(T)

    #月亮的平黄经减去升交点黄经
    FA[2]=pymFaf03(T)

    #月亮到太阳的平均距角
    FA[3]=pymFad03(T)

    #月亮的升交点平黄经
    FA[4]=pymFaom03(T)

    #金星的平黄经
    FA[5]=pymFave03(T)

    #地球的平黄经
    FA[6]=pymFae03(T)

    #黄经上的累计进动
    FA[7]=pymFapa03(T)
    
    #估值 s.
    S0 = SP[0]
    S1 = SP[1]
    S2 = SP[2]
    S3 = SP[3]
    S4 = SP[4]
    S5 = SP[5]
    
    for I in range(NS0-1,-1,-1):
        A=0.0
        for J in range(8):
            A=A+float(KS0[I][J])*FA[J]
        S0=S0+(SS0[I][0]*ma.sin(A)+SS0[I][1]*ma.cos(A))
    
    for I in range(NS1-1,-1,-1):
        A=0.0    
        for J in range(8):
            A=A+float(KS1[I][J])*FA[J]
        S1=S1+(SS1[I][0]*ma.sin(A)+SS1[I][1]*ma.cos(A))
    
    for I in range(NS2-1,-1,-1):
        A=0.0
        for J in range(8):
            A=A+float(KS2[I][J])*FA[J]
        S2=S2+(SS2[I][0]*ma.sin(A)+SS2[I][1]*ma.cos(A))
    
    for I in range(NS3-1,-1,-1):
        A=0.0    
        for J in range(8):
            A=A+float(KS3[I][J])*FA[J]
        S3=S3+(SS3[I][0]*ma.sin(A)+SS3[I][1]*ma.cos(A))
    
    for I in range(NS4-1,-1,-1):
        A=0.0    
        for J in range(8):
            A=A+float(KS4[I][J])*FA[J]
        S4=S4+(SS4[I][0]*ma.sin(A)+SS4[I][1]*ma.cos(A))
        
    S06=(S0+(S1+(S2+(S3+(S4+S5*T)*T)*T)*T)*T)*DAS2R-X*Y/2.0
        
    return(S06)

def pymEors(RNPB,S):
    '''
    Equation of the origins, given the classical NPB matrix and the
    quantity s.

    Parameters
    ----------
    rnpb : list(3,3) 
        classical nutation x precession x bias matrix    
    s : float
        the quantity s (the CIO locator) in radians

    Returns
    -------
    function value : float
        the equation of the origins in radians

    '''
    #由 Wallace & Capitaine (2006) expression (16)估计.
    X=RNPB[2][0]
    AX=X/(1.0+RNPB[2][2])
    XS=1.0-AX*X
    YS=-AX*RNPB[2][1]
    ZS=-X
    P=RNPB[0][0]*XS+RNPB[0][1]*YS+RNPB[0][2]*ZS
    Q=RNPB[1][0]*XS+RNPB[1][1]*YS+RNPB[1][2]*ZS
    
    if (P!=0.0)|(Q!=0.0):
        EORS=S-ma.atan2(Q,P)
    else:
        EORS=S
    
    return(EORS)

def pymBpn2xy(RBPN):
    '''
    Extract from the bias-precession-nutation matrix the X,Y coordinates
    of the Celestial Intermediate Pole.

    Parameters
    ----------
    rbpn : list(3,3)
        celestial-to-true matrix

    Returns
    -------
    x : float
        Celestial Intermediate Pole    
    y : float
        Celestial Intermediate Pole

    '''
    #分离出X,Y坐标
    X=RBPN[2][0]
    Y=RBPN[2][1]

    return(X,Y)

def pymBi00():
    '''
    Frame bias components of IAU 2000 precession-nutation models;  part
    of the Mathews-Herring-Buffett (MHB2000) nutation series, with
    additions.

    Returns
    -------
    dpsibi : float
        longitude and obliquity corrections    
    depsbi : float
        longitude and obliquity corrections    
    dra : flaot
        the ICRS RA of the J2000.0 mean equinox

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #在经度和倾角上的框架极移改正
    DPBIAS=-0.041775*DAS2R
    DEBIAS=-0.0068192*DAS2R
    
    #J2000.0平春分点的ICRS赤经(Chapront et al., 2002)
    DRA0=-0.0146*DAS2R
    
    #输出结果.
    DPSIBI=DPBIAS
    DEPSBI=DEBIAS
    DRA=DRA0
    
    return(DPSIBI,DEPSBI,DRA)

def pymPr00(DATE1,DATE2):
    '''
    Precession-rate part of the IAU 2000 precession-nutation models
    (part of MHB2000).

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    dpsipr : float 
        precession corrections    
    depspr : float
        precession corrections

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #参考历元(J2000.0), JD
    DJ00=2451545

    #1儒略世纪的天数
    DJC=36525.0
    
    # 进动和倾角的改正，单位：弧度/世纪
    PRECOR=-0.29965*DAS2R
    OBLCOR=-0.02524*DAS2R
    
    #与参考历元之间的时间间隔，单位：儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #进动速率
    DPSIPR=PRECOR*T
    DEPSPR=OBLCOR*T
    
    return(DPSIPR,DEPSPR)

def pymCp(P):
    '''
    Copy a p-vector.

    Parameters
    ----------
    p : list(3)
        p-vector to be copied

    Returns
    -------
    c : list(3)
        copy

    '''
    C=[P[i] for i in range(3)]
  
    return(C)

def pymCr(R):
    '''
    Copy an r-matrix.

    Parameters
    ----------
    r : list(3,3)
        r-matrix to be copied

    Returns
    -------
    c : list(3,3)
        r-matrix to be copied

    '''
    C=[0,0,0]
    
    for k in range(3):
        C[k]=pymCp(R[k])
    
    return(C)

def pymBp00(DATE1,DATE2):
    '''
    Frame bias and precession, IAU 2000.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    rb : list(3,3)
        frame bias matrix     
    rp : list(3,3)   
        precession matrix    
    rbp : list(3,3)
        bias-precession matrix
    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0

    #1儒略世纪对应的天数
    DJC=36525.0

    #J2000.0 倾角(Lieske et al. 1977)
    EPS0=84381.448e0*DAS2R
    
    #与参考历元之间的时间间隔，单位：儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #参考架偏差
    DPSIBI,DEPSBI,DRA0=pymBi00()

    #岁差角(Lieske et al. 1977)
    PSIA77=(5038.7784+(-1.07259+(-0.001147)*T)*T)*T*DAS2R
    OMA77=EPS0+((0.05127+(-0.007726)*T)*T)*T*DAS2R
    CHIA=(10.5526+(-2.38064+(-0.001125)*T)*T)*T*DAS2R
    
    #应用IAU 2000岁差改正.
    DPSIPR, DEPSPR=pymPr00(DATE1, DATE2)
    PSIA=PSIA77+DPSIPR
    OMA=OMA77+DEPSPR
    
    #参考架偏差矩阵: GCRS to J2000.0.
    RBW=pymIr()
    RBW=pymRz(DRA0,RBW)
    RBW=pymRy(DPSIBI*ma.sin(EPS0),RBW)
    RBW=pymRx(-DEPSBI,RBW)
    RB=pymCr(RBW)
    
    #岁差矩阵: J2000.0 to mean of date.
    RP=pymIr()
    RP=pymRx(EPS0,RP)
    RP=pymRz(-PSIA,RP)
    RP=pymRx(-OMA,RP)
    RP=pymRz(CHIA,RP)
    
    #偏差-岁差矩阵: GCRS to mean of date.
    RBP=pymRxr(RP,RBW)
    
    return(RB,RP,RBP)

def pymRxr(A,B):
    '''
    Multiply two r-matrices.

    Parameters
    ----------
    a : list(3,3)
        first r-matrix    
    b : list(3,3)
        second r-matrix

    Returns
    -------
    atb : list(3,3)
        a * b

    '''
    WM=[[0 for i in range(3)] for j in range(3)]
    
    for i in range(3):
        for j in range(3):
            W=0.0
            for k in range(3):
                W=W+A[i][k]*B[k][j]
            WM[i][j]=W
    ATB=pymCr(WM)
    
    return(ATB)

def pymPmat06 (DATE1,DATE2):
    '''
    Precession matrix (including frame bias) from GCRS to a specified
    date, IAU 2006 model.

    Parameters
    ----------
    date1 : flaot
        TT as a 2-part Julian Date     
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    rbp : list(3,3)
        bias-precession matrix

    '''

    #Fukushima-Williams 偏差-岁差角
    GAMB,PHIB,PSIB,EPSA=pymPfw06(DATE1,DATE2)
    
    #构建矩阵
    RBP=pymFw2m(GAMB, PHIB, PSIB, EPSA)
    
    return(RBP)

def pymBp06(DATE1,DATE2):
    '''
    Frame bias and precession, IAU 2006.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    rb : list(3,3)
        frame bias matrix    
    rp : list(3,3)
        precession matrix    
    rbp : list(3,3)
        bias-precession matrix

    '''
    #约简儒略日零点
    DJM0=2400000.5
    
    #参考历元(J2000.0), 约简儒略日
    DJM00 = 51544.5
    
    #B matrix.
    GAMB,PHIB,PSIB,EPSA=pymPfw06(DJM0,DJM00)
    RB=pymFw2m(GAMB,PHIB,PSIB,EPSA)
    
    #PxB matrix (temporary).
    RBPW=pymPmat06(DATE1, DATE2)
    
    #P matrix.
    RBT=pymTr(RB)
    RP=pymRxr(RBPW,RBT)
    
    #PxB matrix.
    RBP=pymCr(RBPW)
    
    return(RB,RP,RBP)

def pymTr(R):
    '''
    Transpose an r-matrix.

    Parameters
    ----------
    r : list(3,3)
        r-matrix

    Returns
    -------
    rt : list(3,3)
        transpose

    '''
    WM=[[0 for i in range(3)] for j in range(3)]
    
    for i in range(3):
        for j in range(3):
            WM[i][j]=R[j][i]
    
    RT=pymCr(WM)    
    
    return(RT)

def pymObl80(DATE1,DATE2):
    '''
    Mean obliquity of the ecliptic, IAU 1980 model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        obliquity of the ecliptic (radians)

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0

    #1儒略世纪对应的天数
    DJC=36525.0
    
    #与参考历元之间的时间间隔(JC).
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #平均倾角
    OBL=DAS2R*(84381.448+(-46.8150+(-0.00059+0.001813*T)*T)*T)
    
    return(OBL)

def pymNumat(EPSA,DPSI,DEPS):
    '''
    Form the matrix of nutation.

    Parameters
    ----------
    epsa : float 
        mean obliquity of date    
    dpsi : float
        nutation    
    deps : float
        nutation

    Returns
    -------
    rmatn : list(3,3)
        nutation matrix

    '''
    #构建旋转矩阵
    RMATN=pymIr()
    RMATN=pymRx(EPSA,RMATN)
    RMATN=pymRz(-DPSI,RMATN)
    RMATN=pymRx(-(EPSA+DEPS),RMATN)    
    
    return(RMATN)

def pymPn00(DATE1,DATE2,DPSI,DEPS):
    '''
    Precession-nutation, IAU 2000 model:  a multi-purpose function,
    supporting classical (equinox-based) use directly and CIO-based
    use indirectly.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date    
    dpsi : float
        nutation    
    deps : float
        nutation

    Returns
    -------
    epsa : flaot
        mean obliquity    
    rb : list(3,3)
        frame bias matrix    
    rp : list(3,3)
        precession matrix    
    rbp : list(3,3)
        bias-precession matrix    
    rn : list(3,3)
        nutation matrix    
    rbpn : list(3,3)
        GCRS-to-true matrix
        
    '''  
    #IAU 2000 岁差速率修正.
    DPSIPR,DEPSPR=pymPr00(DATE1,DATE2)
    
    #平均倾角，与IAU2000的岁差章动模型一致   
    EPSA=pymObl80(DATE1,DATE2)+DEPSPR
    
    #参考架偏差，岁差，以及两者的乘积
    RB,RP,RBP=pymBp00(DATE1,DATE2)
    
    #章动矩阵
    RN=pymNumat(EPSA,DPSI,DEPS)
    
    #参考架偏差-岁差-章动矩阵(经典).
    RBPN=pymRxr(RN,RBP)
    
    return(EPSA,RB,RP,RBP,RN,RBPN)

def pymPn00a(DATE1,DATE2):
    '''
    Precession-nutation, IAU 2000A model:  a multi-purpose function,
    supporting classical (equinox-based) use directly and CIO-based
    use indirectly.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date    

    Returns
    -------
    dpsi : float
        nutation    
    deps : float
        nutation    
    epsa : flaot
        mean obliquity    
    rb : list(3,3)
        frame bias matrix    
    rp : list(3,3)
        precession matrix    
    rbp : list(3,3)
        bias-precession matrix    
    rn : list(3,3)
        nutation matrix    
    rbpn : list(3,3)
        GCRS-to-true matrix

    '''
    #章动.
    DPSI,DEPS=pymNut00a(DATE1,DATE2)
    
    #调用pymPn00
    EPSA,RB,RP,RBP,RN,RBPN=pymPn00(DATE1,DATE2,DPSI,DEPS)
    
    return(DPSI,DEPS,EPSA,RB,RP,RBP,RN,RBPN)

def pymPnm00a(DATE1,DATE2):
    '''
    Form the matrix of precession-nutation for a given date (including
    frame bias), equinox based, IAU 2000A model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    rbpn : list(3,3)
        bias-precession-nutation matrix

    '''
    #调用pymPn00a获得矩阵，舍去其他参数
    DPSI,DEPS,EPSA,RB,RP,RBP,RN,RBPN=pymPn00a(DATE1, DATE2)
    
    return(RBPN)

def pymS00(DATE1,DATE2,X,Y):
    '''
    The CIO locator s, positioning the Celestial Intermediate Origin on
    the equator of the Celestial Intermediate Pole, given the CIP's X,Y
    coordinates.  Compatible with IAU 2000A precession-nutation.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date    
    x : float
        CIP coordinates    
    y : float
        CIP coordinates

    Returns
    -------
    Tfunction value : float
        the CIO locator s in radians

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6

    #参考历元(J2000.0), JD
    DJ00=2451545.0

    #1儒略世纪对应的天数
    DJC=36525.0
    
    #与参考历元之间的时间间隔，占位，单位：儒略世纪数
    T=0
    FA=[0 for i in range(8)]
    
    # s+XY/2的系数
    #各系数的项数
    NSP=6
    NS0=33
    NS1=3
    NS2=25
    NS3=4
    NS4=1
    
    #多项式系数
    SP=[94e-6,3808.35e-6,-119.94e-6,-72574.09e-6,27.70e-6,15.61e-6]
    
    #参数系数  t^0
    KS0=[[0,0,0,0,1,0,0,0],
            [0,0,0,0,2,0,0,0],
            [0,0,2,-2,3,0,0,0],
            [0,0,2,-2,1,0,0,0],
            [0,0,2,-2,2,0,0,0],
            [0,0,2,0,3,0,0,0],
            [0,0,2,0,1,0,0,0],
            [0,0,0,0,3,0,0,0],
            [0,1,0,0,1,0,0,0],
            [0,1,0,0,-1,0,0,0],
            [1,0,0,0,-1,0,0,0],
            [1,0,0,0,1,0,0,0],
            [0,1,2,-2,3,0,0,0],
            [0,1,2,-2,1,0,0,0],
            [0,0,4,-4,4,0,0,0],
            [0,0,1,-1,1,-8,12,0],
            [0,0,2,0,0,0,0,0],
            [0,0,2,0,2,0,0,0],
            [1,0,2,0,3,0,0,0],
            [1,0,2,0,1,0,0,0],
            [0,0,2,-2,0,0,0,0],
            [0,1,-2,2,-3,0,0,0],
            [0,1,-2,2,-1,0,0,0],
            [0,0,0,0,0,8,-13,-1],
            [0,0,0,2,0,0,0,0],
            [2,0,-2,0,-1,0,0,0],
            [0,1,2,-2,2,0,0,0],
            [1,0,0,-2,1,0,0,0],
            [1,0,0,-2,-1,0,0,0],
            [0,0,4,-2,4,0,0,0],
            [0,0,2,-2,4,0,0,0],
            [1,0,-2,0,-3,0,0,0],
            [1,0,-2,0,-1,0,0,0]]
    
    #参数系数 t^1
    KS1=[[0,0,0,0,2,0,0,0],
             [0,0,0,0,1,0,0,0],
             [0,0,2,-2,3,0,0,0]]
    
    #参数系数 t^2
    KS2=[[0,0,0,0,1,0,0,0],
            [0,0,2,-2,2,0,0,0],
            [0,0,2,0,2,0,0,0],
            [0,0,0,0,2,0,0,0],
            [0,1,0,0,0,0,0,0],
            [1,0,0,0,0,0,0,0],
            [0,1,2,-2,2,0,0,0],
            [0,0,2,0,1,0,0,0],
            [1,0,2,0,2,0,0,0],
            [0,1,-2,2,-2,0,0,0],
            [1,0,0,-2,0,0,0,0],
            [0,0,2,-2,1,0,0,0],
            [1,0,-2,0,-2,0,0,0],
            [0,0,0,2,0,0,0,0],
            [1,0,0,0,1,0,0,0],
            [1,0,-2,-2,-2,0,0,0],
            [1,0,0,0,-1,0,0,0],
            [1,0,2,0,1,0,0,0],
            [2,0,0,-2,0,0,0,0],
            [2,0,-2,0,-1,0,0,0],
            [0,0,2,2,2,0,0,0],
            [2,0,2,0,2,0,0,0],
            [2,0,0,0,0,0,0,0],
            [1,0,2,-2,2,0,0,0],
            [0,0,2,0,0,0,0,0]]
    
    #参数系数 t^3
    KS3=[[0,0,0,0,1,0,0,0],
            [0,0,2,-2,2,0,0,0],
            [0,0,2,0,2,0,0,0],
            [0,0,0,0,2,0,0,0]]
    
    #参数系数 t^4
    KS4=[[0,0,0,0,1,0,0,0]]

    #正余弦系数 t^0
    SS0=[[-2640.73e-6,+0.39e-6],
            [-63.53e-6,+0.02e-6],
            [-11.75e-6,-0.01e-6],
            [-11.21e-6,-0.01e-6],
            [+4.57e-6,0.00e-6],
            [-2.02e-6,0.00e-6],
            [-1.98e-6,0.00e-6],
            [+1.72e-6,0.00e-6],
            [+1.41e-6,+0.01e-6],
            [+1.26e-6,+0.01e-6],
            [+0.63e-6,0.00e-6],
            [+0.63e-6,0.00e-6],
            [-0.46e-6,0.00e-6],
            [-0.45e-6,0.00e-6],
            [-0.36e-6,0.00e-6],
            [+0.24e-6,+0.12e-6],
            [-0.32e-6,0.00e-6],
            [-0.28e-6,0.00e-6],
            [-0.27e-6,0.00e-6],
            [-0.26e-6,0.00e-6],
            [+0.21e-6,0.00e-6],
            [-0.19e-6,0.00e-6],
            [-0.18e-6,0.00e-6],
            [+0.10e-6,-0.05e-6],
            [-0.15e-6,0.00e-6],
            [+0.14e-6,0.00e-6],
            [+0.14e-6,0.00e-6],
            [-0.14e-6,0.00e-6],
            [-0.14e-6,0.00e-6],
            [-0.13e-6,0.00e-6],
            [+0.11e-6,0.00e-6],
            [-0.11e-6,0.00e-6],
            [-0.11e-6,0.00e-6]]
    
    #正余弦系数 t^1
    SS1=[[-0.07e-6,+3.57e-6],
            [+1.71e-6,-0.03e-6],
            [0.00e-6,+0.48e-6]]
    
    #正余弦系数 t^2
    SS2=[[+743.53e-6,-0.17e-6],
            [+56.91e-6,+0.06e-6],
            [+9.84e-6,-0.01e-6],
            [-8.85e-6,+0.01e-6],
            [-6.38e-6,-0.05e-6],
            [-3.07e-6,0.00e-6],
            [+2.23e-6,0.00e-6],
            [+1.67e-6,0.00e-6],
            [+1.30e-6,0.00e-6],
            [+0.93e-6,0.00e-6],
            [+0.68e-6,0.00e-6],
            [-0.55e-6,0.00e-6],
            [+0.53e-6,0.00e-6],
            [-0.27e-6,0.00e-6],
            [-0.27e-6,0.00e-6],
            [-0.26e-6,0.00e-6],
            [-0.25e-6,0.00e-6],
            [+0.22e-6,0.00e-6],
            [-0.21e-6,0.00e-6],
            [+0.20e-6,0.00e-6],
            [+0.17e-6,0.00e-6],
            [+0.13e-6,0.00e-6],
            [-0.13e-6,0.00e-6],
            [-0.12e-6,0.00e-6],
            [-0.11e-6,0.00e-6]]
    
    #正余弦系数 t^3
    SS3=[[+0.30e-6,-23.51e-6],
            [-0.03e-6,-1.39e-6],
            [-0.01e-6,-0.24e-6],
            [0.00e-6,+0.22e-6]]
    
    #正余弦系数 t^4
    SS4=[[-0.26e-6,-0.01e-6]]
    
    #当前日期和参考历元之间的时间间隔，儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #基本参数(from IERS Conventions 2003)
    #月亮的平近点角
    FA[0]=pymFal03(T)
    
    #太阳的的平近点角
    FA[1]=pymFalp03(T)

    #月球的平黄经减去其上升点的平黄经
    FA[2]=pymFaf03(T)

    #月亮到太阳的平均距角
    FA[3]=pymFad03(T)

    #月亮升交点的平黄经
    FA[4]=pymFaom03(T)

    #金星的平黄经
    FA[5]=pymFave03(T)

    #地球的平黄经
    FA[6]=pymFae03(T)

    #黄经的一般累计岁差
    FA[7]=pymFapa03(T)

    #估计 s.
    S0=SP[0]
    S1=SP[1]
    S2=SP[2]
    S3=SP[3]
    S4=SP[4]
    S5=SP[5]

    for i in range(NS0-1,-1,-1):
        A=0.0
        for j in range(8):
            A=A+float(KS0[i][j])*FA[j]
        S0=S0+(SS0[i][0]*ma.sin(A)+SS0[i][1]*ma.cos(A))
    
    for i in range(NS1-1,-1,-1):
        A=0.0
        for j in range(8):
            A=A+float(KS1[i][j])*FA[j]
        S1=S1+(SS1[i][0]*ma.sin(A)+SS1[i][1]*ma.cos(A))

    for i in range(NS2-1,-1,-1):
        A=0.0
        for j in range(8):
            A=A+float(KS2[i][j])*FA[j]
        S2=S2+(SS2[i][0]*ma.sin(A)+SS2[i][1]*ma.cos(A))
        
    for i in range(NS3-1,-1,-1):
        A=0.0
        for j in range(8):
            A=A+float(KS3[i][j])*FA[j]
        S3=S3+(SS3[i][0]*ma.sin(A)+SS3[i][1]*ma.cos(A))
        
    for i in range(NS4-1,-1,-1):
        A=0.0
        for j in range(8):
            A=A+float(KS4[i][j])*FA[j]
        S4=S4+(SS4[i][0]*ma.sin(A)+SS4[i][1]*ma.cos(A))

    S00=(S0+(S1+(S2+(S3+(S4+S5*T)*T)*T)*T)*T)*DAS2R-X*Y/2.0
    
    return(S00)

def pymC2ixy(DATE1,DATE2,X,Y):
    '''
    Form the celestial to intermediate-frame-of-date matrix for a given
    date when the CIP X,Y coordinates are known.  IAU 2000.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date    
    x : float
        Celestial Intermediate Pole    
    y : float
        Celestial Intermediate Pole

    Returns
    -------
    rc2i : list(3,3)
        celestial-to-intermediate matrix

    '''
    #计算s，并获得矩阵
    A=pymS00(DATE1,DATE2,X,Y)
    RC2I=pymC2ixys(X,Y,A)
    
    return(RC2I)

def pymC2ibpn(DATE1,DATE2,RBPN):
    '''
    Form the celestial-to-intermediate matrix for a given date given
    the bias-precession-nutation matrix.  IAU 2000.
    
    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date     
    date2 : float
        TT as a 2-part Julian Date    
    rbpn : list(3,3)
        celestial-to-true matrix
    
    Returns
    -------
    rc2i : list(3,3)
        celestial-to-intermediate matrix
    
    '''
    #分离出CIP的X,Y分量
    X,Y=pymBpn2xy(RBPN)
    
    #调用pymC2ixy，获得天球到中间参考系的矩阵  
    RC2I=pymC2ixy(DATE1,DATE2,X,Y)
    
    return(RC2I)

def pymC2i00a(DATE1,DATE2):
    '''
    Form the celestial-to-intermediate matrix for a given date using the
    IAU 2000A precession-nutation model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date     
    date2 : float
        TT as a 2-part Julian Date   

    Returns
    -------
    rc2i : list(3,3)
        celestial-to-intermediate matrix

    '''
    #调用pymPnm00a获得参考架偏差-岁差-章动矩阵(IAU 2000A).
    RBPN=pymPnm00a(DATE1,DATE2)
    
    #构建天球到中间参考系的矩阵
    RC2I=pymC2ibpn(DATE1,DATE2,RBPN)
    
    return(RC2I)

def pymNut00b(DATE1,DATE2):
    '''
    Nutation, IAU 2000B model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    dpsi : float
        nutation, luni-solar + planetary    
    deps : float
        nutation, luni-solar + planetary

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6

    #1毫角秒对应的弧度
    DMAS2R=DAS2R/1e3
    
    #360度对应的角秒
    TURNAS=1296000.0
    
    #2Pi
    D2PI=6.283185307179586476925287
    
    #0.1微角秒对应的弧度
    U2R=DAS2R/1e7
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0

    #1儒略世纪对应的天数
    DJC=36525.0
    
    #日月章动模型
    #模型中的项数
    NLS=77
    
    #基本参数的系数
    NALS=[]
    
    #经度和倾角系数
    CLS=[]
    
    #修正的偏移代替行星项，弧度制
    DPPLAN=-0.135*DMAS2R
    DEPLAN=+0.388*DMAS2R
    
    #各项参数的系数表
    
    # *  Luni-Solar argument multipliers:
    # *
    # *               L     L'    F     D     Om
    NALS=[[0,0,0,0,1],
            [0,0,2,-2,2],
            [0,0,2,0,2],
            [0,0,0,0,2],
            [0,1,0,0,0],
            [0,1,2,-2,2],
            [1,0,0,0,0],
            [0,0,2,0,1],
            [1,0,2,0,2],
            [0,-1,2,-2,2],
            [0,0,2,-2,1],
            [-1,0,2,0,2],
            [-1,0,0,2,0],
            [1,0,0,0,1],
            [-1,0,0,0,1],
            [-1,0,2,2,2],
            [1,0,2,0,1],
            [-2,0,2,0,1],
            [0,0,0,2,0],
            [0,0,2,2,2],
            [0,-2,2,-2,2],
            [-2,0,0,2,0],
            [2,0,2,0,2],
            [1,0,2,-2,2],
            [-1,0,2,0,1],
            [2,0,0,0,0],
            [0,0,2,0,0],
            [0,1,0,0,1],
            [-1,0,0,2,1],
            [0,2,2,-2,2],
            [0,0,-2,2,0],
            [1,0,0,-2,1],
            [0,-1,0,0,1],
            [-1,0,2,2,1],
            [0,2,0,0,0],
            [1,0,2,2,2],
            [-2,0,2,0,0],
            [0,1,2,0,2],
            [0,0,2,2,1],
            [0,-1,2,0,2],
            [0,0,0,2,1],
            [1,0,2,-2,1],
            [2,0,2,-2,2],
            [-2,0,0,2,1],
            [2,0,2,0,1],
            [0,-1,2,-2,1],
            [0,0,0,-2,1],
            [-1,-1,0,2,0],
            [2,0,0,-2,1],
            [1,0,0,2,0],
            [0,1,2,-2,1],
            [1,-1,0,0,0],
            [-2,0,2,0,2],
            [3,0,2,0,2],
            [0,-1,0,2,0],
            [1,-1,2,0,2],
            [0,0,0,1,0],
            [-1,-1,2,2,2],
            [-1,0,2,0,0],
            [0,-1,2,2,2],
            [-2,0,0,0,1],
            [1,1,2,0,2],
            [2,0,0,0,1],
            [-1,1,0,1,0],
            [1,1,0,0,0],
            [1,0,2,0,0],
            [-1,0,2,-2,1],
            [1,0,0,0,2],
            [-1,0,0,1,0],
            [0,0,2,1,2],
            [-1,0,2,4,2],
            [-1,1,0,1,1],
            [0,-2,2,-2,1],
            [1,0,2,2,1],
            [-2,0,2,2,2],
            [-1,0,0,0,2],
            [1,1,2,-2,2]]
    
    # *  Luni-Solar nutation coefficients, unit 1e-7 arcsec:
    # *  longitude (sin, t*sin, cos), obliquity (cos, t*cos, sin)
    CLS=[[-172064161e0,-174666e0,33386e0,92052331e0,9086e0,15377e0],
            [-13170906e0,-1675e0,-13696e0,5730336e0,-3015e0,-4587e0],
            [-2276413e0,-234e0,2796e0,978459e0,-485e0,1374e0],
            [2074554e0,207e0,-698e0,-897492e0,470e0,-291e0],
            [1475877e0,-3633e0,11817e0,73871e0,-184e0,-1924e0],
            [-516821e0,1226e0,-524e0,224386e0,-677e0,-174e0],
            [711159e0,73e0,-872e0,-6750e0,0e0,358e0],
            [-387298e0,-367e0,380e0,200728e0,18e0,318e0],
            [-301461e0,-36e0,816e0,129025e0,-63e0,367e0],
            [215829e0,-494e0,111e0,-95929e0,299e0,132e0],
            [128227e0,137e0,181e0,-68982e0,-9e0,39e0],
            [123457e0,11e0,19e0,-53311e0,32e0,-4e0],
            [156994e0,10e0,-168e0,-1235e0,0e0,82e0],
            [63110e0,63e0,27e0,-33228e0,0e0,-9e0],
            [-57976e0,-63e0,-189e0,31429e0,0e0,-75e0],
            [-59641e0,-11e0,149e0,25543e0,-11e0,66e0],
            [-51613e0,-42e0,129e0,26366e0,0e0,78e0],
            [45893e0,50e0,31e0,-24236e0,-10e0,20e0],
            [63384e0,11e0,-150e0,-1220e0,0e0,29e0],
            [-38571e0,-1e0,158e0,16452e0,-11e0,68e0],
            [32481e0,0e0,0e0,-13870e0,0e0,0e0],
            [-47722e0,0e0,-18e0,477e0,0e0,-25e0],
            [-31046e0,-1e0,131e0,13238e0,-11e0,59e0],
            [28593e0,0e0,-1e0,-12338e0,10e0,-3e0],
            [20441e0,21e0,10e0,-10758e0,0e0,-3e0],
            [29243e0,0e0,-74e0,-609e0,0e0,13e0],
            [25887e0,0e0,-66e0,-550e0,0e0,11e0],
            [-14053e0,-25e0,79e0,8551e0,-2e0,-45e0],
            [15164e0,10e0,11e0,-8001e0,0e0,-1e0],
            [-15794e0,72e0,-16e0,6850e0,-42e0,-5e0],
            [21783e0,0e0,13e0,-167e0,0e0,13e0],
            [-12873e0,-10e0,-37e0,6953e0,0e0,-14e0],
            [-12654e0,11e0,63e0,6415e0,0e0,26e0],
            [-10204e0,0e0,25e0,5222e0,0e0,15e0],
            [16707e0,-85e0,-10e0,168e0,-1e0,10e0],
            [-7691e0,0e0,44e0,3268e0,0e0,19e0],
            [-11024e0,0e0,-14e0,104e0,0e0,2e0],
            [7566e0,-21e0,-11e0,-3250e0,0e0,-5e0],
            [-6637e0,-11e0,25e0,3353e0,0e0,14e0],
            [-7141e0,21e0,8e0,3070e0,0e0,4e0],
            [-6302e0,-11e0,2e0,3272e0,0e0,4e0],
            [5800e0,10e0,2e0,-3045e0,0e0,-1e0],
            [6443e0,0e0,-7e0,-2768e0,0e0,-4e0],
            [-5774e0,-11e0,-15e0,3041e0,0e0,-5e0],
            [-5350e0,0e0,21e0,2695e0,0e0,12e0],
            [-4752e0,-11e0,-3e0,2719e0,0e0,-3e0],
            [-4940e0,-11e0,-21e0,2720e0,0e0,-9e0],
            [7350e0,0e0,-8e0,-51e0,0e0,4e0],
            [4065e0,0e0,6e0,-2206e0,0e0,1e0],
            [6579e0,0e0,-24e0,-199e0,0e0,2e0],
            [3579e0,0e0,5e0,-1900e0,0e0,1e0],
            [4725e0,0e0,-6e0,-41e0,0e0,3e0],
            [-3075e0,0e0,-2e0,1313e0,0e0,-1e0],
            [-2904e0,0e0,15e0,1233e0,0e0,7e0],
            [4348e0,0e0,-10e0,-81e0,0e0,2e0],
            [-2878e0,0e0,8e0,1232e0,0e0,4e0],
            [-4230e0,0e0,5e0,-20e0,0e0,-2e0],
            [-2819e0,0e0,7e0,1207e0,0e0,3e0],
            [-4056e0,0e0,5e0,40e0,0e0,-2e0],
            [-2647e0,0e0,11e0,1129e0,0e0,5e0],
            [-2294e0,0e0,-10e0,1266e0,0e0,-4e0],
            [2481e0,0e0,-7e0,-1062e0,0e0,-3e0],
            [2179e0,0e0,-2e0,-1129e0,0e0,-2e0],
            [3276e0,0e0,1e0,-9e0,0e0,0e0],
            [-3389e0,0e0,5e0,35e0,0e0,-2e0],
            [3339e0,0e0,-13e0,-107e0,0e0,1e0],
            [-1987e0,0e0,-6e0,1073e0,0e0,-2e0],
            [-1981e0,0e0,0e0,854e0,0e0,0e0],
            [4026e0,0e0,-353e0,-553e0,0e0,-139e0],
            [1660e0,0e0,-5e0,-710e0,0e0,-2e0],
            [-1521e0,0e0,9e0,647e0,0e0,4e0],
            [1314e0,0e0,0e0,-700e0,0e0,0e0],
            [-1283e0,0e0,0e0,672e0,0e0,0e0],
            [-1331e0,0e0,8e0,663e0,0e0,4e0],
            [1383e0,0e0,-2e0,-594e0,0e0,-2e0],
            [1405e0,0e0,4e0,-610e0,0e0,2e0],
            [1290e0,0e0,0e0,-556e0,0e0,0e0]]
    
    #给定日期与参考历元之间的时间间隔，儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #日月章动
    
    #Fundamental (Delaunay) arguments from Simon et al. (1994)
    
    #月亮的平近点角
    A=(485868.249036+(+1717915923.2178)*T)%TURNAS
    EL=A*DAS2R

    #太阳的平近点角
    A=(1287104.79305+(+129596581.0481)*T)%TURNAS
    ELP=A*DAS2R

    #月亮的平均升交角距
    A=(335779.526232+(+1739527262.8478)*T)%TURNAS
    F=A*DAS2R

    #月亮到太阳的平均角距
    A=(1072260.70369+(+1602961601.2090)*T)%TURNAS
    D=A*DAS2R

    #月亮的升交点平黄经
    A=(450160.398036+(-6962890.5431)*T)%TURNAS
    OM=A*DAS2R
    
    #初始化章动量
    DP=0.0
    DE=0.0
    
    #日月章动级数求和(倒序)
    for i in range(NLS-1,-1,-1):
        B=float(NALS[i][0])*EL+float(NALS[i][1])*ELP+\
           float(NALS[i][2])*F+float(NALS[i][3])*D+\
           float(NALS[i][4])*OM
        ARG=B%D2PI
        SARG=ma.sin(ARG)
        CARG=ma.cos(ARG)
    
        #Term.
        DP=DP+(CLS[i][0]+CLS[i][1]*T)*SARG+CLS[i][2]*CARG
        DE=DE+(CLS[i][3]+CLS[i][4]*T)*CARG+CLS[i][5]*SARG
    
    #从0.1微弧秒单位转换为弧度。
    DPSILS=DP*U2R
    DEPSLS=DE*U2R
    
    #代替行星章动项
    
    #修正偏移量，以纠正截断系数中缺失的项
    DPSIPL=DPPLAN
    DEPSPL=DEPLAN
    
    #日月和行星章动相加
    DPSI=DPSILS+DPSIPL
    DEPS=DEPSLS+DEPSPL
    
    return(DPSI,DEPS)

def pymPn00b(DATE1,DATE2):
    '''
    Precession-nutation, IAU 2000B model:  a multi-purpose function,
    supporting classical (equinox-based) use directly and CIO-based
    use indirectly.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date    

    Returns
    -------
    dpsi : float
        nutation    
    deps : float
        nutation    
    epsa : flaot
        mean obliquity    
    rb : list(3,3)
        frame bias matrix    
    rp : list(3,3)
        precession matrix    
    rbp : list(3,3)
        bias-precession matrix    
    rn : list(3,3)
        nutation matrix    
    rbpn : list(3,3)
        GCRS-to-true matrix

    '''  
    
    #章动，调用pymNut00b
    DPSI,DEPS=pymNut00b(DATE1,DATE2)
    
    #Remaining results.
    EPSA,RB,RP,RBP,RN,RBPN=pymPn00(DATE1,DATE2,DPSI,DEPS)
    
    return(DPSI,DEPS,EPSA,RB,RP,RBP,RN,RBPN)

def pymPnm00b(DATE1,DATE2):
    '''
    Form the matrix of precession-nutation for a given date (including
    frame bias), equinox-based, IAU 2000B model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    rbpn : list(3,3)
        bias-precession-nutation matrix

    '''
    #获得需要的矩阵，舍去其他无关参数
    DPSI,DEPS,EPSA,RB,RP,RBP,RN,RBPN=pymPn00b(DATE1,DATE2)
    
    return(RBPN)

def pymC2i00b(DATE1,DATE2):
    '''
    Form the celestial-to-intermediate matrix for a given date using the
    IAU 2000B precession-nutation model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date     
    date2 : float
        TT as a 2-part Julian Date   

    Returns
    -------
    rc2i : list(3,3)
        celestial-to-intermediate matrix

    '''
    #获得天球到真赤道和春分点的矩阵 (IAU 2000B).
    RBPN=pymPnm00b(DATE1,DATE2)
    
    #构建天球到中间参考系的矩阵
    RC2I=pymC2ibpn(DATE1,DATE2,RBPN)
    
    return(RC2I)

def pymC2i06a(DATE1,DATE2):
    '''
    Form the celestial-to-intermediate matrix for a given date using the
    IAU 2006 precession and IAU 2000A nutation models.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    rc2i : list(3,3)
        celestial-to-intermediate matrix

    '''
    #获得天球到真赤道和春分点的矩阵(IAU 2006/2000A).
    RBPN=pymPnm06a(DATE1,DATE2)
    
    #分离出CIP的X,Y分量
    X,Y=pymBpn2xy(RBPN)
    
    #获得CIO定位角s
    S=pymS06(DATE1,DATE2,X,Y)

    #构建天球到中间参考系的矩阵
    RC2I=pymC2ixys(X,Y,S)
    
    return(RC2I)

def pymC2tcio(RC2I,ERA,RPOM):
    '''
    Assemble the celestial to terrestrial matrix from CIO-based
    components (the celestial-to-intermediate matrix, the Earth Rotation
    Angle and the polar motion matrix).

    Parameters
    ----------
    rc2i : list(3,3)
        celestial-to-intermediate matrix     
    era : float
        Earth rotation angle (radians)     
    rpom : list(3,3)
        polar-motion matrix

    Returns
    -------
    rc2t : list(3,3)
        celestial-to-terrestrial matrix

    '''    
    #构建矩阵
    R=pymCr(RC2I)
    R=pymRz(ERA,R)
    RC2T=pymRxr(RPOM,R)
    
    return(RC2T)

def pymC2t00a(TTA,TTB,UTA,UTB,XP,YP):
    '''
    Form the celestial to terrestrial matrix given the date, the UT1 and
    the polar motion, using the IAU 2000A precession-nutation model.

    Parameters
    ----------
    tta : float
        TT as a 2-part Julian Date    
    ttb : float
        TT as a 2-part Julian Date    
    uta : float
        UT1 as a 2-part Julian Date    
    utb : float
        UT1 as a 2-part Julian Date    
    xp : float
        coordinates of the pole    
    yp : float
        coordinates of the pole  

    Returns
    -------
    rc2t : list(3,3)
        celestial-to-terrestrial matrix

    '''
    #构建在TT时刻的天球到中间参考系的矩阵(IAU 2000A).
    RC2I=pymC2i00a(TTA,TTB)
    
    #调用pymEra00预测在UT1时刻的地球自转角
    ERA=pymEra00(UTA,UTB)
    
    #预测TIO定位角 s'.
    SP=pymSp00(TTA,TTB)
    
    #构建极移矩阵
    RPOM=pymPom00(XP,YP,SP)
    
    #组合构建天球到地球的矩阵
    RC2T=pymC2tcio(RC2I,ERA,RPOM)
    
    return(RC2T)

def pymC2t00b(TTA,TTB,UTA,UTB,XP,YP):
    '''
    Form the celestial to terrestrial matrix given the date, the UT1 and
    the polar motion, using the IAU 2000B precession-nutation model.

    Parameters
    ----------
    tta : float
        TT as a 2-part Julian Date    
    ttb : float
        TT as a 2-part Julian Date    
    uta : float
        UT1 as a 2-part Julian Date    
    utb : float
        UT1 as a 2-part Julian Date    
    xp : float
        coordinates of the pole    
    yp : float
        coordinates of the pole  

    Returns
    -------
    rc2t : list(3,3)
        celestial-to-terrestrial matrix
        
    '''
    #构建在TT时刻的天球到中间参考系的矩阵(IAU 2000B).
    RC2I=pymC2i00b(TTA,TTB)
    
    #调用pymEra00预测在UT1时刻的地球自转角
    ERA=pymEra00(UTA,UTB)
    
    #构建极移矩阵，忽略TIO定位角s'
    RPOM=pymPom00(XP,YP,0.0)
    
    #组合构建天球到地球的矩阵
    RC2T=pymC2tcio(RC2I,ERA,RPOM)
    
    return(RC2T)

def pymC2t06a(TTA,TTB,UTA,UTB,XP,YP):
    '''
    Form the celestial to terrestrial matrix given the date, the UT1 and
    the polar motion, using the IAU 2006/2000A precession-nutation
    model.

    Parameters
    ----------
    tta : float
        TT as a 2-part Julian Date     
    ttb : float
        TT as a 2-part Julian Date    
    uta : float
        UT1 as a 2-part Julian Date    
    utb : float
        UT1 as a 2-part Julian Date     
    xp : float
        coordinates of the pole (radians)    
    yp : float
        coordinates of the pole (radians)

    Returns
    -------
    rc2t : list(3,3)
        celestial-to-terrestrial matrix

    '''
    #构建在TT时刻的天球到中间参考系的矩阵(IAU 2006/2000A).
    RC2I=pymC2i06a(TTA,TTB)
    
    #调用pymEra00预测在UT1时刻的地球自转角
    ERA=pymEra00(UTA,UTB)
    
    #预测TIO定位角 s'.
    SP=pymSp00(TTA,TTB)
    
    #构建极移矩阵
    RPOM=pymPom00(XP,YP,SP)
    
    #组合构建天球到地球的矩阵
    RC2T=pymC2tcio(RC2I,ERA,RPOM)
    
    return(RC2T)

def pymC2teqx(RBPN,GST,RPOM):
    '''
    Assemble the celestial to terrestrial matrix from equinox-based
    components (the celestial-to-true matrix, the Greenwich Apparent
    Sidereal Time and the polar motion matrix).

    Parameters
    ----------
    rbpn : list(3,3)
        celestial-to-true matrix
    gst : float
        Greenwich (apparent) Sidereal Time (radians)
    rpom : list(3,3)
        polar-motion matrix

    Returns
    -------
    rc2t : list(3,3)
        celestial-to-terrestrial matrix

    '''
    #构建矩阵
    R=pymCr(RBPN)
    R=pymRz(GST,R)
    RC2T=pymRxr(RPOM,R)
    
    return(RC2T)

def pymEect00(DATE1,DATE2):
    '''
    Equation of the equinoxes complementary terms, consistent with
    IAU 2000 resolutions.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        complementary terms

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6

    #参考历元(J2000.0), JD
    DJ00=2451545.0

    #1儒略世纪的天数
    DJC=36525.0
    
    #基本参数
    FA=[0 for i in range(14)]
    
    # EE补充项部分
    
    #项数
    NE0=33
    NE1=1
    
    #Argument coefficients for t^0
    KE0=[[0,0,0,0,1,0,0,0],
            [0,0,0,0,2,0,0,0],
            [0,0,2,-2,3,0,0,0],
            [0,0,2,-2,1,0,0,0],
            [0,0,2,-2,2,0,0,0],
            [0,0,2,0,3,0,0,0],
            [0,0,2,0,1,0,0,0],
            [0,0,0,0,3,0,0,0],
            [0,1,0,0,1,0,0,0],
            [0,1,0,0,-1,0,0,0],
            [1,0,0,0,-1,0,0,0],
            [1,0,0,0,1,0,0,0],
            [0,1,2,-2,3,0,0,0],
            [0,1,2,-2,1,0,0,0],
            [0,0,4,-4,4,0,0,0],
            [0,0,1,-1,1,-8,12,0],
            [0,0,2,0,0,0,0,0],
            [0,0,2,0,2,0,0,0],
            [1,0,2,0,3,0,0,0],
            [1,0,2,0,1,0,0,0],
            [0,0,2,-2,0,0,0,0],
            [0,1,-2,2,-3,0,0,0],
            [0,1,-2,2,-1,0,0,0],
            [0,0,0,0,0,8,-13,-1],
            [0,0,0,2,0,0,0,0],
            [2,0,-2,0,-1,0,0,0],
            [1,0,0,-2,1,0,0,0],
            [0,1,2,-2,2,0,0,0],
            [1,0,0,-2,-1,0,0,0],
            [0,0,4,-2,4,0,0,0],
            [0,0,2,-2,4,0,0,0],
            [1,0,-2,0,-3,0,0,0],
            [1,0,-2,0,-1,0,0,0]]
    
    #Argument coefficients for t^1
    KE1=[[0,0,0,0,1,0,0,0]]
    
    #Sine and cosine coefficients for t^0
    SE0=[[+2640.96e-6,-0.39e-6],
            [+63.52e-6,-0.02e-6],
            [+11.75e-6,+0.01e-6],
            [+11.21e-6,+0.01e-6],
            [-4.55e-6,+0.00e-6],
            [+2.02e-6,+0.00e-6],
            [+1.98e-6,+0.00e-6],
            [-1.72e-6,+0.00e-6],
            [-1.41e-6,-0.01e-6],
            [-1.26e-6,-0.01e-6],
            [-0.63e-6,+0.00e-6],
            [-0.63e-6,+0.00e-6],
            [+0.46e-6,+0.00e-6],
            [+0.45e-6,+0.00e-6],
            [+0.36e-6,+0.00e-6],
            [-0.24e-6,-0.12e-6],
            [+0.32e-6,+0.00e-6],
            [+0.28e-6,+0.00e-6],
            [+0.27e-6,+0.00e-6],
            [+0.26e-6,+0.00e-6],
            [-0.21e-6,+0.00e-6],
            [+0.19e-6,+0.00e-6],
            [+0.18e-6,+0.00e-6],
            [-0.10e-6,+0.05e-6],
            [+0.15e-6,+0.00e-6],
            [-0.14e-6,+0.00e-6],
            [+0.14e-6,+0.00e-6],
            [-0.14e-6,+0.00e-6],
            [+0.14e-6,+0.00e-6],
            [+0.13e-6,+0.00e-6],
            [-0.11e-6,+0.00e-6],
            [+0.11e-6,+0.00e-6],
            [+0.11e-6,+0.00e-6]]
    
    #Sine and cosine coefficients for t^1
    SE1=[[-0.87e-6,+0.00e-6]]
    
    #给定日期和参考历元之间的时间间隔儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #基本参数(依据IERS Conventions 2003)
    #月亮的平近点角
    FA[0]=pymFal03(T)
    
    #太阳的平近点角
    FA[1]=pymFalp03(T)

    #月亮的平黄经减去升交点黄经
    FA[2]=pymFaf03(T)

    #月亮到太阳的平均距角
    FA[3]=pymFad03(T)

    #月亮的升交点平黄经
    FA[4]=pymFaom03(T)

    #金星的平黄经
    FA[5]=pymFave03(T)

    #地球的平黄经
    FA[6]=pymFae03(T)

    #黄经上的累计岁差
    FA[7]=pymFapa03(T)
    
    #估计EE补充项
    S0=0.0
    S1=0.0
    
    for i in range(NE0-1,-1,-1):
        A=0.0
        for j in range(8):
            A=A+float(KE0[i][j])*FA[j]
        S0=S0+(SE0[i][0]*ma.sin(A)+SE0[i][1]*ma.cos(A))
    
    for i in range(NE1-1,-1,-1):
        A=0.0
        for j in range(8):
            A=A+float(KE1[i][j])*FA[j]
        S1=S1+(SE1[i][0]*ma.sin(A)+SE1[i][1]*ma.cos(A))
    
    EECT=(S0+S1*T)*DAS2R    
    
    return(EECT)

def pymEe00(DATE1,DATE2,EPSA,DPSI):
    '''
    The equation of the equinoxes, compatible with IAU 2000 resolutions,
    given the nutation in longitude and the mean obliquity.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date    
    epsa : float
        mean obliquity    
    dpsi : float
        nutation in longitude

    Returns
    -------
    function value : float
        equation of the equinoxes

    '''
    #赤经章动
    EE=DPSI*ma.cos(EPSA)+pymEect00(DATE1,DATE2)
    
    return(EE)

def pymGmst00(UTA,UTB,TTA,TTB):
    '''
    Greenwich mean sidereal time (model consistent with IAU 2000
    resolutions).

    Parameters
    ----------
    uta : float
        UT1 as a 2-part Julian Date    
    utb : float
        UT1 as a 2-part Julian Date    
    tta : float
        TT as a 2-part Julian Date    
    ttb : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        Greenwich mean sidereal time (radians)

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6

    #参考历元(J2000.0), JD
    DJ00=2451545.0

    #1儒略世纪对应的天数
    DJC=36525.0
    
    #自参考历元起的世界时间隔
    T=((TTA-DJ00)+TTB)/DJC
    
    #格林尼治平恒星时, IAU 2000.
    A=pymEra00(UTA,UTB)
    B=A+(0.014506+(4612.15739966+(+1.39667721+(-0.00009344+(+0.00001882)
                                               *T)*T)*T)*T)*DAS2R
    GMST=pymAnp(B)  
    
    return(GMST)

def pymC2tpe(TTA,TTB,UTA,UTB,DPSI,DEPS,XP,YP):
    '''
    Assemble the celestial to terrestrial matrix from equinox-based
    components (the celestial-to-true matrix, the Greenwich Apparent
    Sidereal Time and the polar motion matrix).

    Parameters
    ----------
    rbpn : list(3,3)
        celestial-to-true matrix
    gst : float
        Greenwich (apparent) Sidereal Time (radians)
    rpom : list(3,3)
        polar-motion matrix

    Returns
    -------
    rc2t : list(3,3)
        celestial-to-terrestrial matrix

    '''
    #构建给定时间的从天球到真赤道的矩阵
    EPSA,RB,RP,RBP,RN,RBPN=pymPn00(TTA,TTB,DPSI,DEPS)
    
    #预测给定时间的格林尼治平恒星时.
    GMST=pymGmst00(UTA,UTB,TTA,TTB)
    
    #基于给定的时间和章动量，预测赤经章动的补充项.
    EE=pymEe00(TTA,TTB,EPSA,DPSI)
    
    #估计TIO定位角s'.
    SP=pymSp00(TTA,TTB)
    
    #构建极移矩阵
    RPOM=pymPom00(XP,YP,SP)
    
    #组合构建天球到地球的矩阵
    RC2T=pymC2teqx(RBPN,GMST+EE,RPOM)
    
    return(RC2T)

def pymC2txy(TTA,TTB,UTA,UTB,X,Y,XP,YP):
    '''
    Form the celestial to terrestrial matrix given the date, the UT1,
    the CIP coordinates and the polar motion.  IAU 2000.

    Parameters
    ----------
    tta : float
        TT as a 2-part Julian Date    
    ttb : float
        TT as a 2-part Julian Date    
    uta : float
        UT1 as a 2-part Julian Date    
    utb : float
        UT1 as a 2-part Julian Date    
    x : float
        Celestial Intermediate Pole    
    y : float
        Celestial Intermediate Pole    
    xp : float
        coordinates of the pole    
    yp : float
        coordinates of the pole    

    Returns
    -------
    rc2t : list(3,3)
        celestial-to-terrestrial matrix

    '''
    #构建给定时间的从天球到中间参考架的矩阵.
    RC2I=pymC2ixy(TTA,TTB,X,Y)
    
    #预测在UT1时刻的地球自转角
    ERA=pymEra00(UTA,UTB)
    
    #估计 s'.
    SP=pymSp00(TTA,TTB)
 
    #构建极移矩阵
    RPOM=pymPom00(XP,YP,SP)
    
    #组合构建天球到地球的矩阵
    RC2T=pymC2tcio(RC2I,ERA,RPOM)
    
    return(RC2T)

def pymEe00a(DATE1,DATE2):
    '''
    Equation of the equinoxes, compatible with IAU 2000 resolutions.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        equation of the equinoxes

    '''
    #IAU 2000岁差速率改正.
    DPSIPR,DEPSPR=pymPr00(DATE1,DATE2)
    
    #平均倾角，与IAU 2000岁差章动模型一致.
    EPSA=pymObl80(DATE1,DATE2)+DEPSPR
    
    #经度上的章动
    DPSI,DEPS=pymNut00a(DATE1,DATE2)
    
    #赤经章动
    EE=pymEe00(DATE1,DATE2,EPSA,DPSI)
    
    return(EE)

def pymEe00b(DATE1,DATE2):
    '''
    Equation of the equinoxes, compatible with IAU 2000 resolutions but
    using the truncated nutation model IAU 2000B.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        equation of the equinoxes

    '''
    #IAU 2000岁差速率改正.
    DPSIPR,DEPSPR=pymPr00(DATE1,DATE2)
    
    #平均倾角，与IAU 2000岁差章动模型一致.
    EPSA=pymObl80(DATE1,DATE2)+DEPSPR
    
    #经度上的章动，采用删减后的章动模型IAU 2000B  
    DPSI,DEPS=pymNut00b(DATE1,DATE2)
    
    #赤经章动
    EE=pymEe00(DATE1,DATE2,EPSA,DPSI)
    
    return(EE)

def pymGst06(UTA,UTB,TTA,TTB,RNPB):
    '''
    Greenwich apparent sidereal time, IAU 2006, given the NPB matrix.

    Parameters
    ----------
    uta : float
        UT1 as a 2-part Julian Date    
    utb : float
        UT1 as a 2-part Julian Date    
    tta : float
        TT as a 2-part Julian Date     
    ttb : float
        TT as a 2-part Julian Date    
    rnpb : float
        nutation x precession x bias matrix

    Returns
    -------
    function value : float
        Greenwich apparent sidereal time (radians)

    '''
    #从CIP坐标中分离出X,Y
    X,Y=pymBpn2xy(RNPB)
    
    #CIO定位角, s.
    S=pymS06(TTA,TTB,X,Y)
    
    #格林尼治视恒星时
    GST=pymAnp(pymEra00(UTA,UTB)-pymEors(RNPB,S))
    
    return(GST)

def pymGst06a(UTA,UTB,TTA,TTB):
    '''
    Greenwich apparent sidereal time (consistent with IAU 2000 and 2006
    resolutions).

    Parameters
    ----------
    uta : float
        UT1 as a 2-part Julian Date    
    utb : float
        UT1 as a 2-part Julian Date     
    tta : float
        TT as a 2-part Julian Date    
    ttb : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        Greenwich apparent sidereal time (radians)

    '''
    #调用pymPnm06a获得NPB（参考架偏差-岁差-章动）矩阵, IAU 2000A/2006.
    RNPB=pymPnm06a(TTA,TTB)
    
    #格林尼治视恒星时
    GST=pymGst06(UTA,UTB,TTA,TTB,RNPB)
    
    return(GST)

def pymGmst06(UTA,UTB,TTA,TTB):
    '''
    Greenwich mean sidereal time (consistent with IAU 2006 precession).

    Parameters
    ----------
    uta : float
        UT1 as a 2-part Julian Date    
    utb : float
        UT1 as a 2-part Julian Date    
    tta : float
        TT as a 2-part Julian Date    
    ttb : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        Greenwich mean sidereal time (radians)

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6

    #参考历元(J2000.0), JD
    DJ00=2451545.0

    #1儒略世纪对应的天数
    DJC=36525.0
    
    #自参考历元起的世界时间隔
    T=((TTA-DJ00)+TTB)/DJC
    
    #格林尼治平恒星时, IAU 2006.
    A=pymEra00(UTA,UTB)
    GMST=pymAnp(A+(0.014506+(4612.156534+(1.3915817+(-0.00000044+
                    (-0.000029956+(-0.0000000368)*T)*T)*T)*T)*T)*DAS2R)
    
    return(GMST)

def pymEe06a(DATE1,DATE2):
    '''
    Equation of the equinoxes, compatible with IAU 2000 resolutions and
    IAU 2006/2000A precession-nutation.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        equation of the equinoxes

    '''
    #赤经章动
    A=pymGst06a(0.0,0.0,DATE1,DATE2)-pymGmst06(0.0,0.0,DATE1,DATE2)
    EE=pymAnpm(A)
    
    return(EE)

def pymEo06a (DATE1,DATE2):
    '''
    Equation of the origins, IAU 2006 precession and IAU 2000A nutation.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        the equation of the origins in radians

    '''
    #NPB（参考架偏差-岁差-章动）矩阵
    R=pymPnm06a(DATE1,DATE2)
    
    #从CIP中分离出X,Y
    X,Y=pymBpn2xy(R)
    
    #CIO定位角, s.
    S=pymS06(DATE1,DATE2,X,Y)
    
    #求解EO
    EO=pymEors(R,S)
    
    return(EO)

def pymNut80(DATE1,DATE2):
    '''
    Nutation, IAU 1980 model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    dpsi : float
        nutation in longitude (radians)    
    deps : float
        nutation in obliquity (radians)

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6

    #2Pi
    D2PI=6.283185307179586476925287
    
    #0.1微角秒对应的弧度
    U2R=DAS2R/1e4
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0

    #1儒略世纪对应的天数
    DJC=36525.0
    
    # 参数和多项式系数表
    #系数单位为0.1微角秒，变化速率为角秒/千儒略年
    # *                Multiple of            Longitude        Obliquity
    # *           L    L'   F    D  Omega   coeff. of sin    coeff. of cos
    # *                                         1       t        1     t
    
    X=[[0.,0.,0.,0.,1.,-171996.,-1742.,92025.,89.],
            [0.,0.,0.,0.,2.,2062.,2.,-895.,5.],
            [-2.,0.,2.,0.,1.,46.,0.,-24.,0.],
            [2.,0.,-2.,0.,0.,11.,0.,0.,0.],
            [-2.,0.,2.,0.,2.,-3.,0.,1.,0.],
            [1.,-1.,0.,-1.,0.,-3.,0.,0.,0.],
            [0.,-2.,2.,-2.,1.,-2.,0.,1.,0.],
            [2.,0.,-2.,0.,1.,1.,0.,0.,0.],
            [0.,0.,2.,-2.,2.,-13187.,-16.,5736.,-31.],
            [0.,1.,0.,0.,0.,1426.,-34.,54.,-1.],
            [0.,1.,2.,-2.,2.,-517.,12.,224.,-6.],
            [0.,-1.,2.,-2.,2.,217.,-5.,-95.,3.],
            [0.,0.,2.,-2.,1.,129.,1.,-70.,0.],
            [2.,0.,0.,-2.,0.,48.,0.,1.,0.],
            [0.,0.,2.,-2.,0.,-22.,0.,0.,0.],
            [0.,2.,0.,0.,0.,17.,-1.,0.,0.],
            [0.,1.,0.,0.,1.,-15.,0.,9.,0.],
            [0.,2.,2.,-2.,2.,-16.,1.,7.,0.],
            [0.,-1.,0.,0.,1.,-12.,0.,6.,0.],
            [-2.,0.,0.,2.,1.,-6.,0.,3.,0.],
            [0.,-1.,2.,-2.,1.,-5.,0.,3.,0.],
            [2.,0.,0.,-2.,1.,4.,0.,-2.,0.],
            [0.,1.,2.,-2.,1.,4.,0.,-2.,0.],
            [1.,0.,0.,-1.,0.,-4.,0.,0.,0.],
            [2.,1.,0.,-2.,0.,1.,0.,0.,0.],
            [0.,0.,-2.,2.,1.,1.,0.,0.,0.],
            [0.,1.,-2.,2.,0.,-1.,0.,0.,0.],
            [0.,1.,0.,0.,2.,1.,0.,0.,0.],
            [-1.,0.,0.,1.,1.,1.,0.,0.,0.],
            [0.,1.,2.,-2.,0.,-1.,0.,0.,0.],
            [0.,0.,2.,0.,2.,-2274.,-2.,977.,-5.],
            [1.,0.,0.,0.,0.,712.,1.,-7.,0.],
            [0.,0.,2.,0.,1.,-386.,-4.,200.,0.],
            [1.,0.,2.,0.,2.,-301.,0.,129.,-1.],
            [1.,0.,0.,-2.,0.,-158.,0.,-1.,0.],
            [-1.,0.,2.,0.,2.,123.,0.,-53.,0.],
            [0.,0.,0.,2.,0.,63.,0.,-2.,0.],
            [1.,0.,0.,0.,1.,63.,1.,-33.,0.],
            [-1.,0.,0.,0.,1.,-58.,-1.,32.,0.],
            [-1.,0.,2.,2.,2.,-59.,0.,26.,0.],
            [1.,0.,2.,0.,1.,-51.,0.,27.,0.],
            [0.,0.,2.,2.,2.,-38.,0.,16.,0.],
            [2.,0.,0.,0.,0.,29.,0.,-1.,0.],
            [1.,0.,2.,-2.,2.,29.,0.,-12.,0.],
            [2.,0.,2.,0.,2.,-31.,0.,13.,0.],
            [0.,0.,2.,0.,0.,26.,0.,-1.,0.],
            [-1.,0.,2.,0.,1.,21.,0.,-10.,0.],
            [-1.,0.,0.,2.,1.,16.,0.,-8.,0.],
            [1.,0.,0.,-2.,1.,-13.,0.,7.,0.],
            [-1.,0.,2.,2.,1.,-10.,0.,5.,0.],
            [1.,1.,0.,-2.,0.,-7.,0.,0.,0.],
            [0.,1.,2.,0.,2.,7.,0.,-3.,0.],
            [0.,-1.,2.,0.,2.,-7.,0.,3.,0.],
            [1.,0.,2.,2.,2.,-8.,0.,3.,0.],
            [1.,0.,0.,2.,0.,6.,0.,0.,0.],
            [2.,0.,2.,-2.,2.,6.,0.,-3.,0.],
            [0.,0.,0.,2.,1.,-6.,0.,3.,0.],
            [0.,0.,2.,2.,1.,-7.,0.,3.,0.],
            [1.,0.,2.,-2.,1.,6.,0.,-3.,0.],
            [0.,0.,0.,-2.,1.,-5.,0.,3.,0.],
            [1.,-1.,0.,0.,0.,5.,0.,0.,0.],
            [2.,0.,2.,0.,1.,-5.,0.,3.,0.],
            [0.,1.,0.,-2.,0.,-4.,0.,0.,0.],
            [1.,0.,-2.,0.,0.,4.,0.,0.,0.],
            [0.,0.,0.,1.,0.,-4.,0.,0.,0.],
            [1.,1.,0.,0.,0.,-3.,0.,0.,0.],
            [1.,0.,2.,0.,0.,3.,0.,0.,0.],
            [1.,-1.,2.,0.,2.,-3.,0.,1.,0.],
            [-1.,-1.,2.,2.,2.,-3.,0.,1.,0.],
            [-2.,0.,0.,0.,1.,-2.,0.,1.,0.],
            [3.,0.,2.,0.,2.,-3.,0.,1.,0.],
            [0.,-1.,2.,2.,2.,-3.,0.,1.,0.],
            [1.,1.,2.,0.,2.,2.,0.,-1.,0.],
            [-1.,0.,2.,-2.,1.,-2.,0.,1.,0.],
            [2.,0.,0.,0.,1.,2.,0.,-1.,0.],
            [1.,0.,0.,0.,2.,-2.,0.,1.,0.],
            [3.,0.,0.,0.,0.,2.,0.,0.,0.],
            [0.,0.,2.,1.,2.,2.,0.,-1.,0.],
            [-1.,0.,0.,0.,2.,1.,0.,-1.,0.],
            [1.,0.,0.,-4.,0.,-1.,0.,0.,0.],
            [-2.,0.,2.,2.,2.,1.,0.,-1.,0.],
            [-1.,0.,2.,4.,2.,-2.,0.,1.,0.],
            [2.,0.,0.,-4.,0.,-1.,0.,0.,0.],
            [1.,1.,2.,-2.,2.,1.,0.,-1.,0.],
            [1.,0.,2.,2.,1.,-1.,0.,1.,0.],
            [-2.,0.,2.,4.,2.,-1.,0.,1.,0.],
            [-1.,0.,4.,0.,2.,1.,0.,0.,0.],
            [1.,-1.,0.,-2.,0.,1.,0.,0.,0.],
            [2.,0.,2.,-2.,1.,1.,0.,-1.,0.],
            [2.,0.,2.,2.,2.,-1.,0.,0.,0.],
            [1.,0.,0.,2.,1.,-1.,0.,0.,0.],
            [0.,0.,4.,-2.,2.,1.,0.,0.,0.],
            [3.,0.,2.,-2.,2.,1.,0.,0.,0.],
            [1.,0.,2.,-2.,0.,-1.,0.,0.,0.],
            [0.,1.,2.,0.,1.,1.,0.,0.,0.],
            [-1.,-1.,0.,2.,1.,1.,0.,0.,0.],
            [0.,0.,-2.,0.,1.,-1.,0.,0.,0.],
            [0.,0.,2.,-1.,2.,-1.,0.,0.,0.],
            [0.,1.,0.,2.,0.,-1.,0.,0.,0.],
            [1.,0.,-2.,-2.,0.,-1.,0.,0.,0.],
            [0.,-1.,2.,0.,1.,-1.,0.,0.,0.],
            [1.,1.,0.,-2.,1.,-1.,0.,0.,0.],
            [1.,0.,-2.,2.,0.,-1.,0.,0.,0.],
            [2.,0.,0.,2.,0.,1.,0.,0.,0.],
            [0.,0.,2.,4.,2.,-1.,0.,0.,0.],
            [0.,1.,0.,1.,0.,1.,0.,0.,0.]]

    #给定日期与参考历元之间的时间间隔，儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #FK5参考系中的基本参数
    #月球的平黄经减去月球近地点的平黄经。
    C=np.abs(1325.0*T)%1.0
    if (1325.0*T)<0:
        B=-C
    else:
        B=C
    A=(485866.733+(715922.633+(31.310+0.064*T)*T)*T)*DAS2R+B*D2PI
    EL=pymAnpm(A)
    
    #太阳的平黄经减去近地点的平黄经。
    C=np.abs(99.0*T)%1.0
    if (99.0*T)<0:
        B=-C
    else:
        B=C
    A=(1287099.804+(1292581.224+(-0.577-0.012*T)*T)*T)*DAS2R+B*D2PI
    ELP=pymAnpm(A)
    
    #月球的平黄经减去月球交点的平黄经。
    C=np.abs(1342.0*T)%1.0
    if (1342.0*T)<0:
        B=-C
    else:
        B=C
    A=(335778.877+(295263.137+(-13.257+0.011*T)*T)*T)*DAS2R+B*D2PI
    F=pymAnpm(A)
    
    #月亮与太阳之间的平角距
    C=np.abs(1236.0*T)%1.0
    if (1236.0*T)<0:
        B=-C
    else:
        B=C
    A=(1072261.307+(1105601.328+(-6.891+0.019*T)*T)*T)*DAS2R+B*D2PI
    D=pymAnpm(A)
    
    #月球轨道在黄道上的升交点平黄经，起量点为春分点
    C=np.abs(-5.0*T)%1.0
    if (-5.0*T)<0:
        B=-C
    else:
        B=C
    A=(450160.280+(-482890.539+(7.455+0.008*T)*T)*T)*DAS2R+B*D2PI
    OM=pymAnpm(A)
    
    # 章动部分
    
    #将时间单位由世纪转换为千年
    T=T/10.0
    
    #初始化章动参数
    DP=0.0
    DE=0.0
    
    #对章动项求和，从小到大
    for j in range(105,-1,-1):
        
        #构建当前项的参数
        ARG=float(X[j][0])*EL+float(X[j][1])*ELP+float(X[j][2])*F+\
            float(X[j][3])*D+float(X[j][4])*OM
            
        #累加当前章动项。
        S=float(X[j][5])+float(X[j][6])*T
        C=float(X[j][7])+float(X[j][8])*T
        if (S!=0.0):
            DP=DP+S*ma.sin(ARG)
        if (C!=0.0):
            DE=DE+C*ma.cos(ARG)
        
    #转换单位到弧度制
    DPSI=DP*U2R
    DEPS=DE*U2R
    
    return(DPSI,DEPS)

def pymAnpm(A):
    '''
    Normalize angle into the range -pi <= a < +pi.

    Parameters
    ----------
    a : float
        angle (radians)

    Returns
    -------
    function value : float
        angle in range +/-pi

    '''
    W=np.abs(A)%(2*ma.pi)
    if A<0:
        W=-W
    if (np.abs(W)>=ma.pi):
        if A>=0:
            W=W-2*ma.pi
        else:
            W=W+2*ma.pi
        
    return(W)

def pymEqeq94(DATE1,DATE2):
    '''
    Equation of the equinoxes, IAU 1994 model.

    Parameters
    ----------
    date1 : float
        TDB date    
    date2 : float
        TDB date

    Returns
    -------
    function value : float
        equation of the equinoxes

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6

    #2Pi
    D2PI=6.283185307179586476925287
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0

    #1儒略世纪的天数
    DJC=36525.0
    
    #给定日期和参考历元之间的时间间隔，儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #月球轨道在黄道上的升交点平黄经，起量点为春分点
    C=np.abs(-5.0*T)%1.0
    if (-5.0*T)<0:
        B=-C
    else:
        B=C
    A=(450160.280+(-482890.539+(7.455+0.008*T)*T)*T)*DAS2R+B*D2PI
    OM=pymAnpm(A)
    
    #章动参数和平均倾角
    DPSI,DEPS=pymNut80(DATE1,DATE2)
    EPS0=pymObl80(DATE1,DATE2)
    
    #赤经章动
    EQEQ=DPSI*ma.cos(EPS0)+DAS2R*(0.00264*ma.sin(OM)+0.000063*ma.sin(OM+OM))
    
    return(EQEQ)

def pymFw2xy(GAMB,PHIB,PSI,EPS):
    '''
    CIP X,Y given Fukushima-Williams bias-precession-nutation angles.

    Parameters
    ----------
    gamb : float
        F-W angle gamma_bar (radians)    
    phib : float
        F-W angle phi_bar (radians)    
    psi : float
        F-W angle psi (radians)    
    eps : float
        F-W angle epsilon (radians)

    Returns
    -------
    x : float
        CIP unit vector X,Y     
    y : float
        CIP unit vector X,Y

    '''
    #构建NxPxB矩阵.
    R=pymFw2m(GAMB,PHIB,PSI,EPS)
    
    #从CIP中分解出X,Y
    X,Y=pymBpn2xy(R)
    
    return(X,Y)

def pymLtpequ(EPJ):
    '''
    Long-term precession of the equator.

    Parameters
    ----------
    epj : float
        Julian epoch (TT)   

    Returns
    -------
    veq : list(3)
        equator pole unit vector

    '''
    VEQ=[0,0,0]
    
    #1角秒对应的弧度
    AS2R=4.848136811095359935899141e-6
    
    #2Pi
    D2PI=6.283185307179586476925287
    
    #多项式数目
    NPOL=4
    
    #周期项数目
    NPER=14
    
    #多项式
    XYPOL=[[+5453.282155,+0.4252841,-0.00037173,-0.000000152],
           [-73750.930350,-0.7675452,-0.00018725,+0.000000231]]
    
    #周期
    XYPER=[[256.75,-819.940624,75004.344875,81491.287984,1558.515853],
           [708.15,-8444.676815,624.033993,787.163481,7774.939698],
           [274.20,2600.009459,1251.136893,1251.296102,-2219.534038],
           [241.45,2755.175630,-1102.212834,-1257.950837,-2523.969396],
           [2309.00,-167.659835,-2660.664980,-2966.799730,247.850422],
           [492.20,871.855056,699.291817,639.744522,-846.485643],
           [396.10,44.769698,153.167220,131.600209,-1393.124055],
           [288.90,-512.313065,-950.865637,-445.040117,368.526116],
           [231.10,-819.415595,499.754645,584.522874,749.045012],
           [1610.00,-538.071099,-145.188210,-89.756563,444.704518],
           [620.00,-189.793622,558.116553,524.429630,235.934465],
           [157.87,-402.922932,-23.923029,-13.549067,374.049623],
           [220.30,179.516345,-165.405086,-210.157124,-171.330180],
           [1200.00,-9.814756,9.344131,-44.919798,-22.899655]]
    
    #自J2000的世纪数.
    T=(EPJ-2000.0)/100.0
    
    #初始化X,Y
    X=0.0
    Y=0.0
    
    #周期项
    W=D2PI*T
    for i in range(NPER):
        A=W/XYPER[i][0]
        S=ma.sin(A)
        C=ma.cos(A)
        X=X+C*XYPER[i][1]+S*XYPER[i][3]
        Y=Y+C*XYPER[i][2]+S*XYPER[i][4]
    
    #多项式
    W=1.0
    for i in range(NPOL):
        X=X+XYPOL[0][i]*W
        Y=Y+XYPOL[1][i]*W
        W=W*T
    
    #X,Y（方向余弦）
    X=X*AS2R
    Y=Y*AS2R

    #构建赤道极向量
    VEQ[0]=X
    VEQ[1]=Y
    VEQ[2]=ma.sqrt(max(1.0-X*X-Y*Y,0.0))
    
    return(VEQ)

def pymLtpecl(EPJ):
    '''
    Long-term precession of the ecliptic.

    Parameters
    ----------
    epj : float
        Julian epoch (TT)    
        
    Returns
    -------
    vec : list(3)
        ecliptic pole unit vector    

    '''
    VEC=[0,0,0]
    
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #2Pi
    D2PI=6.283185307179586476925287
    
    #J2000.0 时的倾角，弧度制.
    EPS0=84381.406*DAS2R
    
    #多项式数目
    NPOL=4
    
    #周期项数目
    NPER=8
    
    #多项式
    PQPOL=[[+5851.607687,-0.1189000,-0.00028913,+0.000000101],
           [-1600.886300,+1.1689818,-0.00000020,-0.000000437]]
    
    #周期
    PQPER=[[708.15,-5486.751211,-684.661560,667.666730,-5523.863691],
           [2309.00,-17.127623,2446.283880,-2354.886252,-549.747450],
           [1620.00,-617.517403,399.671049,-428.152441,-310.998056],
           [492.20,413.442940,-356.652376,376.202861,421.535876],
           [1183.00,78.614193,-186.387003,184.778874,-36.776172],
           [622.00,-180.732815,-316.800070,335.321713,-145.278396],
           [882.00,-87.676083,198.296701,-185.138669,-34.744450],
           [547.00,46.140315,101.135679,-120.972830,22.885731]]
    
    #自J2000的世纪数.
    T=(EPJ-2000.0)/100.0
    
    #初始化P,Q
    P=0.0
    Q=0.0
    
    #周期项
    W=D2PI*T
    for i in range(NPER):
        A=W/PQPER[i][0]
        S=ma.sin(A)
        C=ma.cos(A)
        P=P+C*PQPER[i][1]+S*PQPER[i][3]
        Q=Q+C*PQPER[i][2]+S*PQPER[i][4]
    
    #多项式
    W = 1.0
    for i in range(NPOL):
        P=P+PQPOL[0][i]*W
        Q=Q+PQPOL[1][i]*W
        W=W*T
    
    #P,Q，弧度制.
    P=P*DAS2R
    Q=Q*DAS2R
    
    #构建黄道极向量
    W=ma.sqrt(max(1.0-P*P-Q*Q,0.0))
    S=ma.sin(EPS0)
    C=ma.cos(EPS0)
    VEC[0]=P
    VEC[1]=-Q*C-W*S
    VEC[2]=-Q*S+W*C
    
    return(VEC)

def pymPxp(A,B):
    ''''
    p-vector outer (=vector=cross) product.

    Parameters
    ----------
    a : list(3) 
        first p-vector    
    b : list(3) 
        second p-vector

    Returns
    -------
    axb : list(3) 
        a x b

    '''

    AXB=[0,0,0]
    
    XA=A[0]
    YA=A[1]
    ZA=A[2]
    XB=B[0]
    YB=B[1]
    ZB=B[2]
    AXB[0]=YA*ZB-ZA*YB
    AXB[1]=ZA*XB-XA*ZB
    AXB[2]=XA*YB-YA*XB
       
    return(AXB)

def pymPm(P):
    '''
    Modulus of p-vector.

    Parameters
    ----------
    p : list(3)
        p-vector

    Returns
    -------
    function value : float
        modulus

    '''
    W=0.0
    for i in range(3):
        C=P[i]
        W+=C**2
    R=ma.sqrt(W)
        
    return(R)

def pymZp():
    '''
    Zero a p-vector

    Parameters
    ----------

    Returns
    -------
    p : list(3)
        zero p-vector
    '''

    P=[0,0,0]
    
    return(P)

def pymSxp(S,P):
    '''
    Multiply a p-vector by a scalar.

    Parameters
    ----------
    s : float
        scalar    
    p : list(3)
        p-vector

    Returns
    -------
    sp : list(3)
        s * p

    '''
    SP=[0,0,0]
    
    for i in range(3):
        SP[i]=S*P[i]
    
    return(SP)

def pymPn(P):
    '''
    Convert a p-vector into modulus and unit vector.

    Parameters
    ----------
    p : list(3)
        p-vector

    Returns
    -------
    r : float
        modulus    
    u : list(3)
        unit vector

    '''
    U=[0,0,0]
    
    #调用pymPm函数获得向量的模，并判断是否为零向量
    W=pymPm(P)
    if (W==0.0):
        
        #零向量
        U=pymZp()
    else:
        
        #单位向量
        U=pymSxp(1.0/W, P)
    
    R=W
    return(R,U)

def pymLtp(EPJ):
    '''
    Long-term precession matrix.

    Parameters
    ----------
    epj : float
        Julian epoch (TT)     

    Returns
    -------
    rp : list(3,3)
        precession matrix, J2000.0 to date

    '''
    RP=[[0 for i in range(3)] for j in range(3)]
    
    #赤道极
    PEQR=pymLtpequ(EPJ)
    
    #黄道极
    PECL=pymLtpecl(EPJ)
    
    #春分点
    V=pymPxp(PEQR,PECL)
    W,EQX=pymPn(V)
    
    #矩阵中间行
    V=pymPxp(PEQR,EQX)
    
    #组建矩阵
    for i in range(3):
        RP[0][i]=EQX[i]
        RP[1][i]=V[i]
        RP[2][i]=PEQR[i]
    
    return(RP)

def pymLtpb(EPJ):
    '''
    Long-term precession matrix, including ICRS frame bias.

    Parameters
    ----------
    epj : float
        Julian epoch (TT)    

    Returns
    -------
    rpb : list(3,3)
        precession-bias matrix, J2000.0 to date

    '''
    RPB=[[0 for i in range(3)] for j in range(3)]
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #参考架偏差(IERS Conventions 2010, Eqs. 5.21 and 5.33)
    DX=-0.016617*DAS2R
    DE=-0.0068192*DAS2R
    DR=-0.0146*DAS2R
    
    #岁差矩阵
    RP=pymLtp(EPJ)

    #应用参考架偏差
    for i in range(3):
        RPB[i][0] =  RP[i][0]    - RP[i][1]*DR + RP[i][2]*DX
        RPB[i][1] =  RP[i][0]*DR + RP[i][1]    + RP[i][2]*DE
        RPB[i][2] = -RP[i][0]*DX - RP[i][1]*DE + RP[i][2]
    
    return(RPB)

def pymMoon98(DATE1,DATE2):
    '''
    Approximate geocentric position and velocity of the Moon.

    Parameters
    ----------
    date1 : float
        TT date part A    
    date2 : float
        TT date part A

    Returns
    -------
    pv : list(2,3)
        Moon p,v, GCRS (AU, AU/d)

    '''
    #天文单位(m)
    DAU=149597870.7e3
    
    #参考历元
    DJ00=2451545.0
    
    #1度对应的弧度
    DD2R=1.745329251994329576923691e-2
    
    #1儒略世纪对应的天数
    DJC=36525.0
    
    #基本参数的系数
    #时间在儒略世纪数上的次方，单位：度
    
    #月亮的平黄经(基于给定日期的平春分点和倾角)
    ELP0=218.31665436
    ELP1=481267.88123421
    ELP2=-0.0015786
    ELP3=1.0/538841.0
    ELP4=-1.0/65194000.0
    
    #月亮的平近点角
    EMP0=134.9633964
    EMP1=477198.8675055
    EMP2=0.0087414
    EMP3=1.0/69699.0
    EMP4=-1.0/14712000.0
    
    #月亮的平距角
    D0=297.8501921
    D1=445267.1114034
    D2=-0.0018819
    D3=1.0/545868
    D4=1.0/113065000.0
    
    #太阳的平近点角
    EM0=357.5291092
    EM1=35999.0502909
    EM2=-0.0001536
    EM3=1/2449000
    EM4=0.0
    
    #月亮到其升交点的平均距离
    F0=93.2720950
    F1=483202.0175233
    F2=-0.0036539
    F3=1.0/3526000.0
    F4=1.0/863310000 
    
    #其他参数
    
    #Meeus A_1, 金星相关 (deg)
    A10=119.75
    A11=131.849
    
    #Meeus A_2, 木星相关 (deg)
    A20=53.09
    A21=479264.290
    
    #Meeus A_3, due to sidereal motion of the Moon in longitude (deg)
    A30=313.45
    A31=481266.484
    
    #Coefficients for Meeus "additive terms" (deg)
    AL1=0.003958
    AL2=0.001962
    AL3=0.000318
    
    AB1=-0.002235
    AB2=0.000382
    AB3=0.000175
    AB4=0.000175
    AB5=0.000127
    AB6=-0.000115
    
    #Fixed term in distance (m)
    R0=385000560.0
    
    #Coefficients for (dimensionless) E factor
    E1=-0.002516
    E2=-0.0000074
    
    # *  Coefficients for Moon position series (L,B,R)
    # *
    # *   TLR(1,N)      =  coefficient of L sine term (deg)
    # *   TLR(2,N)      =  coefficient of R cosine term (m)
    # *   TB(N)         =  coefficient B sine term (deg)
    # *   ITx(1-4,N)    =  coefficients of D, M, M', F in argument
    NLR=60
    NB=60
    
    #Longitude and distance series
    TLR=[[6.288774e0,-20905355e0],
            [1.274027e0,-3699111e0],
            [0.658314e0,-2955968e0],
            [0.213618e0,-569925e0],
            [-0.185116e0,48888e0],
            [-0.114332e0,-3149e0],
            [0.058793e0,246158e0],
            [0.057066e0,-152138e0],
            [0.053322e0,-170733e0],
            [0.045758e0,-204586e0],
            [-0.040923e0,-129620e0],
            [-0.034720e0,108743e0],
            [-0.030383e0,104755e0],
            [0.015327e0,10321e0],
            [-0.012528e0,0e0],
            [0.010980e0,79661e0],
            [0.010675e0,-34782e0],
            [0.010034e0,-23210e0],
            [0.008548e0,-21636e0],
            [-0.007888e0,24208e0],
            [-0.006766e0,30824e0],
            [-0.005163e0,-8379e0],
            [0.004987e0,-16675e0],
            [0.004036e0,-12831e0],
            [0.003994e0,-10445e0],
            [0.003861e0,-11650e0],
            [0.003665e0,14403e0],
            [-0.002689e0,-7003e0],
            [-0.002602e0,0e0],
            [0.002390e0,10056e0],
            [-0.002348e0,6322e0],
            [0.002236e0,-9884e0],
            [-0.002120e0,5751e0],
            [-0.002069e0,0e0],
            [0.002048e0,-4950e0],
            [-0.001773e0,4130e0],
            [-0.001595e0,0e0],
            [0.001215e0,-3958e0],
            [-0.001110e0,0e0],
            [-0.000892e0,3258e0],
            [-0.000810e0,2616e0],
            [0.000759e0,-1897e0],
            [-0.000713e0,-2117e0],
            [-0.000700e0,2354e0],
            [0.000691e0,0e0],
            [0.000596e0,0e0],
            [0.000549e0,-1423e0],
            [0.000537e0,-1117e0],
            [0.000520e0,-1571e0],
            [-0.000487e0,-1739e0],
            [-0.000399e0,0e0],
            [-0.000381e0,-4421e0],
            [0.000351e0,0e0],
            [-0.000340e0,0e0],
            [0.000330e0,0e0],
            [0.000327e0,0e0],
            [-0.000323e0,1165e0],
            [0.000299e0,0e0],
            [0.000294e0,0e0],
            [0.000000e0,8752e0]]
    
    #D   M   M'  F
    ITLR=[[0,0,1,0],
            [2,0,-1,0],
            [2,0,0,0],
            [0,0,2,0],
            [0,1,0,0],
            [0,0,0,2],
            [2,0,-2,0],
            [2,-1,-1,0],
            [2,0,1,0],
            [2,-1,0,0],
            [0,1,-1,0],
            [1,0,0,0],
            [0,1,1,0],
            [2,0,0,-2],
            [0,0,1,2],
            [0,0,1,-2],
            [4,0,-1,0],
            [0,0,3,0],
            [4,0,-2,0],
            [2,1,-1,0],
            [2,1,0,0],
            [1,0,-1,0],
            [1,1,0,0],
            [2,-1,1,0],
            [2,0,2,0],
            [4,0,0,0],
            [2,0,-3,0],
            [0,1,-2,0],
            [2,0,-1,2],
            [2,-1,-2,0],
            [1,0,1,0],
            [2,-2,0,0],
            [0,1,2,0],
            [0,2,0,0],
            [2,-2,-1,0],
            [2,0,1,-2],
            [2,0,0,2],
            [4,-1,-1,0],
            [0,0,2,2],
            [3,0,-1,0],
            [2,1,1,0],
            [4,-1,-2,0],
            [0,2,-1,0],
            [2,2,-1,0],
            [2,1,-2,0],
            [2,-1,0,-2],
            [4,0,1,0],
            [0,0,4,0],
            [4,-1,0,0],
            [1,0,-2,0],
            [2,1,0,-2],
            [0,0,2,-2],
            [1,1,1,0],
            [3,0,-2,0],
            [4,0,-3,0],
            [2,-1,2,0],
            [0,2,1,0],
            [1,1,-1,0],
            [2,0,3,0],
            [2,0,-1,-2]]
    
    #Latitude series
    TB=[5.128122e0,
            0.280602e0,
            0.277693e0,
            0.173237e0,
            0.055413e0,
            0.046271e0,
            0.032573e0,
            0.017198e0,
            0.009266e0,
            0.008822e0,
            0.008216e0,
            0.004324e0,
            0.004200e0,
            -0.003359e0,
            0.002463e0,
            0.002211e0,
            0.002065e0,
            -0.001870e0,
            0.001828e0,
            -0.001794e0,
            -0.001749e0,
            -0.001565e0,
            -0.001491e0,
            -0.001475e0,
            -0.001410e0,
            -0.001344e0,
            -0.001335e0,
            0.001107e0,
            0.001021e0,
            0.000833e0,
            0.000777e0,
            0.000671e0,
            0.000607e0,
            0.000596e0,
            0.000491e0,
            -0.000451e0,
            0.000439e0,
            0.000422e0,
            0.000421e0,
            -0.000366e0,
            -0.000351e0,
            0.000331e0,
            0.000315e0,
            0.000302e0,
            -0.000283e0,
            -0.000229e0,
            0.000223e0,
            0.000223e0,
            -0.000220e0,
            -0.000220e0,
            -0.000185e0,
            0.000181e0,
            -0.000177e0,
            0.000176e0,
            0.000166e0,
            -0.000164e0,
            0.000132e0,
            -0.000119e0,
            0.000115e0,
            0.000107e0]
    
    #D   M   M'  F
    ITB=[[0,0,0,1],
            [0,0,1,1],
            [0,0,1,-1],
            [2,0,0,-1],
            [2,0,-1,1],
            [2,0,-1,-1],
            [2,0,0,1],
            [0,0,2,1],
            [2,0,1,-1],
            [0,0,2,-1],
            [2,-1,0,-1],
            [2,0,-2,-1],
            [2,0,1,1],
            [2,1,0,-1],
            [2,-1,-1,1],
            [2,-1,0,1],
            [2,-1,-1,-1],
            [0,1,-1,-1],
            [4,0,-1,-1],
            [0,1,0,1],
            [0,0,0,3],
            [0,1,-1,1],
            [1,0,0,1],
            [0,1,1,1],
            [0,1,1,-1],
            [0,1,0,-1],
            [1,0,0,-1],
            [0,0,3,1],
            [4,0,0,-1],
            [4,0,-1,1],
            [0,0,1,-3],
            [4,0,-2,1],
            [2,0,0,-3],
            [2,0,2,-1],
            [2,-1,1,-1],
            [2,0,-2,1],
            [0,0,3,-1],
            [2,0,2,1],
            [2,0,-3,-1],
            [2,1,-1,1],
            [2,1,0,1],
            [4,0,0,1],
            [2,-1,1,1],
            [2,-2,0,-1],
            [0,0,1,3],
            [2,1,1,-1],
            [1,1,0,-1],
            [1,1,0,1],
            [0,1,-2,-1],
            [2,1,-1,-1],
            [1,0,1,1],
            [2,-1,-2,-1],
            [0,1,2,1],
            [4,0,-2,-1],
            [4,-1,-1,-1],
            [1,0,1,-1],
            [4,0,1,-1],
            [1,0,-1,-1],
            [4,-1,0,-1],
            [2,-2,0,1]]
    
    #自J2000其的儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #基本参数
    
    #给定日期的参数(弧度)和导数(每儒略世纪的弧度)

    #月亮的平黄经
    A=ELP0+(ELP1+(ELP2+(ELP3+ELP4*T)*T)*T)*T
    ELP=DD2R*(A%360.0)
    DELP=DD2R*(ELP1+(ELP2*2.0+(ELP3*3.0+ELP4*4.0*T)*T)*T)
    
    #月亮的平距角
    A=D0+(D1+(D2+(D3+D4*T)*T)*T)*T
    D=DD2R*(A%360.0)
    DD=DD2R*(D1+(D2*2.0+(D3*3.0+D4*4.0*T)*T)*T)

    #太阳的平近点角
    A=EM0+(EM1+(EM2+(EM3+EM4*T)*T)*T)*T
    EM=DD2R*(A%360.0)
    DEM=DD2R*(EM1+(EM2*2.0+(EM3*3.0+EM4*4.0*T)*T)*T)

    #月亮的平近点角.
    A=EMP0+(EMP1+(EMP2+(EMP3+EMP4*T)*T)*T)*T
    EMP=DD2R*(A%360.0)
    DEMP=DD2R*(EMP1+(EMP2*2.0+(EMP3*3.0+EMP4*4.0*T)*T)*T)
    
    #月亮到其升交点的平均距离
    A=F0+(F1+(F2+(F3+F4*T)*T)*T)*T
    F=DD2R*(A%360.0)
    DF=DD2R*(F1+(F2*2.0+(F3*3.0+F4*4.0*T)*T)*T)

    #Meeus更进一步的参数.
    A1=DD2R*(A10+A11*T)
    DA1=DD2R*AL1
    A2=DD2R*(A20+A21*T)
    DA2=DD2R*A21
    A3=DD2R*(A30+A31*T)
    DA3=DD2R*A31

    #E-因子和平方项
    E=1.0+(E1+E2*T)*T
    DE=E1+2.0*E2*T
    ESQ=E*E
    DESQ=2.0*E*DE

    #Use the Meeus additive terms (deg) to start off the summations.
    ELPMF=ELP-F
    DELPMF=DELP-DF;
    VEL=AL1*ma.sin(A1)+AL2*ma.sin(ELPMF)+AL3*ma.sin(A2)
    VDEL=AL1*ma.cos(A1)*DA1+AL2*ma.cos(ELPMF)*DELPMF+AL3*ma.cos(A2)*DA2

    VR=0.0
    VDR=0.0
    
    A1MF=A1-F
    DA1MF=DA1-DF
    A1PF=A1+F
    DA1PF=DA1+DF
    DLPMP=ELP-EMP
    SLPMP=ELP+EMP
    VB=AB1*ma.sin(ELP)+AB2*ma.sin(A3)+AB3*ma.sin(A1MF)+AB4*ma.sin(A1PF)\
        +AB5*ma.sin(DLPMP)+AB6*ma.sin(SLPMP)
    VDB=AB1*ma.cos(ELP)*DELP+AB2*ma.cos(A3)*DA3+AB3*ma.cos(A1MF)*DA1MF\
        +AB4*ma.cos(A1PF)*DA1PF+AB5*ma.cos(DLPMP)*(DELP-DEMP)\
            +AB6*ma.cos(SLPMP)*(DELP+DEMP)

    #级数展开
    
    #Longitude and distance plus derivatives.
    for N in range(NLR-1,-1,-1):
        DN=float(ITLR[N][0])
        I=ITLR[N][1]
        EMN=float(I)
        EMPN=float(ITLR[N][2])
        FN=float(ITLR[N][3])
        I=np.abs(I)
        if (I==1):
            EN=E
            DEN=DE
        elif (I==2):
            EN=ESQ
            DEN=DESQ
        else:
            EN=1.0
            DEN=0.0
        ARG=DN*D+EMN*EM+EMPN*EMP+FN*F
        DARG=DN*DD+EMN*DEM+EMPN*DEMP+FN*DF
        FARG=ma.sin(ARG)
        V=FARG*EN
        DV=ma.cos(ARG)*DARG*EN+FARG*DEN
        COEFF=TLR[N][0]
        VEL=VEL+COEFF*V
        VDEL=VDEL+COEFF*DV
        FARG=ma.cos(ARG)
        V=FARG*EN
        DV=-ma.sin(ARG)*DARG*EN+FARG*DEN
        COEFF=TLR[N][1]
        VR=VR+COEFF*V
        VDR=VDR+COEFF*DV
    EL=ELP+DD2R*VEL
    DEL=(DELP+DD2R*VDEL)/DJC
    R=(VR+R0)/DAU
    DR=VDR/DAU/DJC

    #Latitude plus derivative.
    for N in range(NB-1,-1,-1):
        DN=float(ITB[N][0])
        I=ITB[N][1]
        EMN=float(I)
        EMPN=float(ITB[N][2])
        FN=float(ITB[N][3])
        I=np.abs(I)
        if (I==1):
            EN=E
            DEN=DE
        elif (I==2):
            EN=ESQ
            DEN=DESQ
        else:
            EN=1.0
            DEN=0.0
        ARG=DN*D+EMN*EM+EMPN*EMP+FN*F
        DARG=DN*DD+EMN*DEM+EMPN*DEMP+FN*DF
        FARG=ma.sin(ARG)
        V=FARG*EN
        DV=ma.cos(ARG)*DARG*EN+FARG*DEN
        COEFF=TB[N]
        VB=VB+COEFF*V
        VDB=VDB+COEFF*DV
    B=VB*DD2R
    DB=VDB*DD2R/DJC
    
    #转换到最终形式
    
    #Longitude, latitude to x, y, z (AU).
    PV=pymS2pv(EL,B,R,DEL,DB,DR)
    
    #IAU 2006 Fukushima-Williams bias+precession angles.
    GAMB,PHIB,PSIB,EPSA=pymPfw06(DATE1,DATE2)
    
    #Mean ecliptic coordinates to GCRS rotation matrix.
    RM=pymIr()
    RM=pymRz(PSIB,RM)
    RM=pymRx(-PHIB,RM)
    RM=pymRz(-GAMB,RM)

    #Rotate the Moon position and velocity into GCRS (Note 7).
    PV=pymRxpv(RM,PV)
    
    return(PV)

def pymRxp(R,P):
    '''
    Multiply a p-vector by an r-matrix.

    Parameters
    ----------
    r : list(3,3)
        r-matrix    
    p : list(3)
        p-vector    

    Returns
    -------
    rp : list(3,3)
        r * p

    '''
    WRP=[0,0,0]

    for j in range(3):
        W=0.0
        for i in range(3):
            W=W+R[j][i]*P[i]
        WRP[j]=W
    RP=pymCp(WRP)   
    
    return(RP)

def pymRxpv(R,PV):
    '''
    Multiply a pv-vector by an r-matrix.

    Parameters
    ----------
    r : list(3,3)
        r-matrix    
    pv : list(2,3)
        pv-vector

    Returns
    -------
    rpv : list(2,3)
        r * pv

    '''
    RPV=[0,0]
    
    RPV[0]=pymRxp(R,PV[0])
    RPV[1]=pymRxp(R,PV[1])    
    
    return(RPV)

def pymS2pv(THETA,PHI,R,TD,PD,RD):
    '''
    Convert position/velocity from spherical to Cartesian coordinates.

    Parameters
    ----------
    theta : float
        longitude angle (radians)    
    phi : float
        latitude angle (radians)    
    r : float
        radial distance    
    td : float
        rate of change of theta    
    pd : float
        rate of change of phi    
    rd : float
        rate of change of r

    Returns
    -------
    pv : list(2,3)
        pv-vector

    '''
    PV=[[0 for i in range(3)] for j in range(2)]
    
    ST=ma.sin(THETA)
    CT=ma.cos(THETA)
    SP=ma.sin(PHI)
    CP=ma.cos(PHI)
    RCP=R*CP
    X=RCP*CT
    Y=RCP*ST
    RPD=R*PD
    W=RPD*SP-CP*RD

    PV[0][0]=X
    PV[0][1]=Y
    PV[0][2]=R*SP
    PV[1][0]=-Y*TD-W*CT
    PV[1][1]=X*TD-W*ST
    PV[1][2]=RPD*CP+SP*RD

    return(PV)

def pymNum00a(DATE1,DATE2):
    '''
    Form the matrix of nutation for a given date, IAU 2000A model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    rmatn : list(3,3)
        nutation matrix

    '''
    #调用au_PN00A获得所需的矩阵，忽略其他项
    DPSI,DEPS,EPSA,RB,RP,RBP,RMATN,RBPN=pymPn00a(DATE1, DATE2)
    
    return(RMATN)

def pymNum00b (DATE1,DATE2):
    '''
    Form the matrix of nutation for a given date, IAU 2000B model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    rmatn : list(3,3)
        nutation matrix

    '''
    #调用au_PN00B获得所需的矩阵，忽略其他项
    DPSI,DEPS,EPSA,RB,RP,RBP,RMATN,RBPN=pymPn00b(DATE1,DATE2)
    
    return(RMATN)

def pymNum06a(DATE1,DATE2):
    '''
    Form the matrix of nutation for a given date, IAU 2006/2000A model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    rmatn : list(3,3)
        nutation matrix

    '''
    #平均倾角
    EPS=pymObl06(DATE1,DATE2)

    #章动量
    DP,DE=pymNut06a(DATE1,DATE2)

    #章动矩阵
    RMATN=pymNumat(EPS,DP,DE)
    
    return(RMATN)

def pymNutm80(DATE1,DATE2):
    '''
    Form the matrix of nutation for a given date, IAU 1980 model.

    Parameters
    ----------
    date1 : float
        TDB date      
    date2 : float
        TDB date

    Returns
    -------
    rmatn : list(3,3)
        nutation matrix

    '''
    #章动量以及平均倾角
    DPSI,DEPS=pymNut80(DATE1,DATE2)
    EPSA=pymObl80(DATE1, DATE2)
    
    #构建旋转矩阵
    RMATN=pymNumat(EPSA,DPSI,DEPS)
    
    return(RMATN)

def pymP06e(DATE1,DATE2):
    '''
    Precession angles, IAU 2006, equinox based.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    eps0 : float
        epsilon_0    
    psia : float
        psi_A    
    oma : float
        omega_A    
    bpa : float
        P_A    
    bqa : float
        Q_A    
    pia : float
        pi_A    
    bpia : float
        Pi_A    
    epsa : float
        obliquity epsilon_A    
    chia : float
        chi_A     
    za : float
        z_A     
    zetaa : float
        zeta_A    
    thetaa : float 
        theta_A    
    pa : float
        p_A    
    gma : float
        F-W angle gamma_J2000    
    phi : float
        F-W angle phi_J2000    
    psi : float
        F-W angle psi_J2000    

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6
    
    #参考历元(J2000.0), JD
    DJ00=2451545.0
    
    #1儒略世纪对应的天数
    DJC=36525.0
    
    #给定日期到参考历元的时间间隔，儒略世纪数
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #倾角J2000.0.
    EPS0=84381.406*DAS2R
    
    #日月岁差
    PSIA=(5038.481507+(-1.0790069+(-0.00114045+(0.000132851+(-0.0000000951)\
                                                 *T)*T)*T)*T)*T*DAS2R
    
    #平赤道相对于J2000.0黄道的倾角。
    OMA=EPS0+(-0.025754+(0.0512623+(-0.00772503+(-0.000000467+(0.0000003337)\
                                                 *T)*T)*T)*T)*T*DAS2R

    #黄极 x, J2000.0 
    BPA=(4.199094+(0.1939873+(-0.00022466+(-0.000000912+(0.0000000120)\
                                           *T)*T)*T)*T)*T*DAS2R

    #黄极 -y, J2000.0 ecliptic triad.
    BQA=(-46.811015+(0.0510283+(0.00052413+(-0.000000646+(-0.0000000172)\
                                            *T)*T)*T)*T)*T*DAS2R

    #变化的黄道与J2000.0黄道之间的夹角
    PIA=(46.998973+(-0.0334926+(-0.00012559+(0.000000113+(-0.0000000022)\
                                             *T)*T)*T)*T)*T*DAS2R

    #变化的黄道的升交点黄经.
    BPIA=(629546.7936+(-867.95758+(0.157992+(-0.0005371+(-0.00004797\
                                        +(0.000000072)*T)*T)*T)*T)*T)*DAS2R

    #黄道的平均倾角
    EPSA=pymObl06(DATE1,DATE2)

    #行星岁差.
    CHIA=(10.556403+(-2.3814292+(-0.00121197+(0.000170663+(-0.0000000560)\
                                              *T)*T)*T)*T)*T*DAS2R

    #赤道岁差: 负的第三个 323欧拉角.
    ZA=(-2.650545+(2306.077181+(1.0927348+(0.01826837+(-0.000028596\
                                       +(-0.0000002904)*T)*T)*T)*T)*T)*DAS2R

    #赤道岁差: 负的第一个 323欧拉角.
    ZETAA=(2.650545+(2306.083227+(0.2988499+(0.01801828+(-0.000005971\
                                     +(-0.0000003173)*T)*T)*T)*T)*T)*DAS2R

    #赤道岁差: 第二个 323欧拉角.
    THETAA=(2004.191903+(-0.4294934+(-0.04182264+(-0.000007089\
                                      +(-0.0000001274)*T)*T)*T)*T)*T*DAS2R

    #累计岁差
    PA=(5028.796195+(1.1054348+(0.00007964+(-0.000023857+(-0.0000000383)\
                                            *T)*T)*T)*T)*T*DAS2R
          

    #Fukushima-Williams岁差角
    GAM=(10.556403+(0.4932044+(-0.00031238+(-0.000002788+(0.0000000260)\
                                           *T)*T)*T)*T)*T*DAS2R

    PHI=EPS0+(-46.811015+(0.0511269+(0.00053289+(-0.000000440\
                                     +(-0.0000000176)*T)*T)*T)*T)*T*DAS2R

    PSI=(5038.481507+(1.5584176+(-0.00018522+(-0.000026452+(-0.0000000148)\
                                              *T)*T)*T)*T)*T*DAS2R

    return(EPS0,PSIA,OMA,BPA,BQA,PIA,BPIA,EPSA,CHIA,
           ZA,ZETAA,THETAA,PA,GAM,PHI,PSI)

def pymPas(AL,AP,BL,BP):
    '''
    Position-angle from spherical coordinates. 

    Parameters
    ----------
    al : float
        longitude of point A (e.g. RA) in radians     
    ap : float
        latitude of point A (e.g. Dec) in radians    
    bl : float
        longitude of point B     
    bp : float
        latitude of point B     

    Returns
    -------
    THETA : float
        position angle of B with respect to A

    '''
    DL=BL-AL
    Y=ma.sin(DL)*ma.cos(BP)
    X=ma.sin(BP)*ma.cos(AP)-ma.cos(BP)*ma.sin(AP)*ma.cos(DL)
    if (X!=0.0)|(Y!=0.0):
        THETA=ma.atan2(Y,X)
    else:
        THETA=0.0
    
    return(THETA)

def pymPb06(DATE1,DATE2):
    '''
    This function forms three Euler angles which implement general
    precession from epoch J2000.0, using the IAU 2006 model.  Frame
    bias (the offset between ICRS and mean J2000.0) is included.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    bzeta : float
        1st rotation: radians cw around z    
    ca : float
        3rd rotation: radians cw around z     
    btheta : float
        2nd rotation: radians ccw around y

    '''
    #岁差矩阵基于 Fukushima-Williams angles.
    R=pymPmat06(DATE1,DATE2)
    
    #求解z，选取+/-
    Y=R[1][2]
    X=-R[0][2]
    if (X<0.0):
        Y=-Y
        X=-X
    if (X!=0.0)|(Y!=0.0):
        BZ=-ma.atan2(Y,X)
    else:
        BZ=0.0
        
    #把它从矩阵中划去
    R=pymRz(BZ,R)
    
    #解出剩余两个角
    Y=R[0][2]
    X=R[2][2]
    if (X!=0.0)|(Y!=0.0):
        BTHETA=-ma.atan2(Y,X)
    else:
        BTHETA=0.0
    
    Y=-R[1][0]
    X=R[1][1]
    if (X!=0.0)|(Y!=0.0):
        BZETA=-ma.atan2(Y,X)
    else:
        BZETA=0.0
    
    return(BZETA,BZ,BTHETA)

def pymPlan94(DATE1,DATE2,NP):
    '''
    Approximate heliocentric position and velocity of a nominated major
    planet:  Mercury, Venus, EMB, Mars, Jupiter, Saturn, Uranus or
    Neptune (but not the Earth itself).

    Parameters
    ----------
    date1 : float
        TDB date part A    
    date2 : float
        TDB date part B    
    npp : int
        planet (1=Mercury, 2=Venus, 3=EMB, 4=Mars,
        5=Jupiter, 6=Saturn, 7=Uranus, 8=Neptune)

    Returns
    -------
    pv : list(2,3)
        planet p,v (heliocentric, J2000.0, au,au/d)
    J : ValueError
       -1 = illegal NP (outside 1-8)
        0 = OK
       +1 = warning: date outside 1000-3000 CE
       +2 = warning: solution failed to converge
    '''  
    PV=[[0 for i in range(3)] for j in range(2)]
    
    #解开普勒方程的最大迭代数
    KMAX=10
    
    #2Pi
    D2PI=6.283185307179586476925287
    
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6

    #参考历元(J2000.0), JD
    DJ00=2451545.0
    
    #1千年的儒略日数
    DJM=365250.0
    
    #J2000.0平倾角的正余弦 (IAU 1976)
    SINEPS=0.3977771559319137
    COSEPS=0.9174820620691818
    
    #高斯常数
    GK=0.017202098950
    
    #行星质量倒数
    AMAS=[6023600,408523.5,328900.5,3098710,1047.355,3498.5,22869.0,19314.0]
    
    # Tables giving the mean Keplerian elements, limited to T**2 terms:
    
    #          A       semi-major axis (au)
    #          DLM     mean longitude (degree and arcsecond)
    #          E       eccentricity
    #          PI      longitude of the perihelion (degree and arcsecond)
    #          DINC    inclination (degree and arcsecond)
    #          OMEGA   longitude of the ascending node (degree and arcsecond)
    
    A=[[0.3870983098,0.0,0.0],
        [0.7233298200,0.0,0.0],
        [1.0000010178,0.0,0.0],
        [1.5236793419,3e-10,0.0],
        [5.2026032092,19132e-10,-39e-10],
        [9.5549091915,-0.0000213896,444e-10],
        [19.2184460618,-3716e-10,979e-10],
        [30.1103868694,-16635e-10,686e-10]]

    DLM=[[252.25090552e0,5381016286.88982e0,-1.92789e0],
        [181.97980085e0,2106641364.33548e0,0.59381e0],
        [100.46645683e0,1295977422.83429e0,-2.04411e0],
        [355.43299958e0,689050774.93988e0,0.94264e0],
        [34.35151874e0,109256603.77991e0,-30.60378e0],
        [50.07744430e0,43996098.55732e0,75.61614e0],
        [314.05500511e0,15424811.93933e0,-1.75083e0],
        [304.34866548e0,7865503.20744e0,0.21103e0]]
            
    E=[[0.2056317526e0,0.0002040653e0,-28349e-10],
        [0.0067719164e0,-0.0004776521e0,98127e-10],
        [0.0167086342e0,-0.0004203654e0,-0.0000126734e0],
        [0.0934006477e0,0.0009048438e0,-80641e-10],
        [0.0484979255e0,0.0016322542e0,-0.0000471366e0],
        [0.0555481426e0,-0.0034664062e0,-0.0000643639e0],
        [0.0463812221e0,-0.0002729293e0,0.0000078913e0],
        [0.0094557470e0,0.0000603263e0,0e0]]
    
    PI=[[77.45611904e0,5719.11590e0,-4.83016e0],
        [131.56370300e0,175.48640e0,-498.48184e0],
        [102.93734808e0,11612.35290e0,53.27577e0],
        [336.06023395e0,15980.45908e0,-62.32800e0],
        [14.33120687e0,7758.75163e0,259.95938e0],
        [93.05723748e0,20395.49439e0,190.25952e0],
        [173.00529106e0,3215.56238e0,-34.09288e0],
        [48.12027554e0,1050.71912e0,27.39717e0]]
    
    DINC=[[7.00498625e0,-214.25629e0,0.28977e0],
        [3.39466189e0,-30.84437e0,-11.67836e0],
        [0e0,469.97289e0,-3.35053e0],
        [1.84972648e0,-293.31722e0,-8.11830e0],
        [1.30326698e0,-71.55890e0,11.95297e0],
        [2.48887878e0,91.85195e0,-17.66225e0],
        [0.77319689e0,-60.72723e0,1.25759e0],
        [1.76995259e0,8.12333e0,0.08135e0]]
    
    OMEGA=[[48.33089304e0,-4515.21727e0,-31.79892e0],
        [76.67992019e0,-10008.48154e0,-51.32614e0],
        [174.87317577e0,-8679.27034e0,15.34191e0],
        [49.55809321e0,-10620.90088e0,-230.57416e0],
        [100.46440702e0,6362.03561e0,326.52178e0],
        [113.66550252e0,-9240.19942e0,-66.23743e0],
        [74.00595701e0,2669.15033e0,145.93964e0],
        [131.78405702e0,-221.94322e0,-0.78728e0]]
    
    # Tables for trigonometric terms to be added to the mean elements
    # of the semi-major axes.
    
    KP=[[69613,75645,88306,59899,15746,71087,142173,3086,0],
        [21863,32794,26934,10931,26250,43725,53867,28939,0],
        [16002,21863,32004,10931,14529,16368,15318,32794,0],
        [6345,7818,15636,7077,8184,14163,1107,4872,0],
        [1760,1454,1167,880,287,2640,19,2047,1454],
        [574,0,880,287,19,1760,1167,306,574],
        [204,0,177,1265,4,385,200,208,204],
        [0,102,106,4,98,1367,487,204,0]]
    
    CA=[[4,-13,11,-9,-9,-3,-1,4,0],
        [-156,59,-42,6,19,-20,-10,-12,0],
        [64,-152,62,-8,32,-41,19,-11,0],
        [124,621,-145,208,54,-57,30,15,0],
        [-23437,-2634,6601,6259,-1507,-1821,2620,-2115,-1489],
        [62911,-119919,79336,17814,-24241,12068,8306,-4893,8902],
        [389061,-262125,-44088,8387,-22976,-2093,-615,-9720,6633],
        [-412235,-157046,-31430,37817,-9740,-13,-7449,9644,0]]
    
    SA=[[-29,-1,9,6,-6,5,4,0,0],
        [-48,-125,-26,-37,18,-13,-20,-2,0],
        [-150,-46,68,54,14,24,-28,22,0],
        [-621,532,-694,-20,192,-94,71,-73,0],
        [-14614,-19828,-5869,1881,-4372,-2255,782,930,913],
        [139737,0,24667,51123,-5102,7429,-4095,-1976,-9566],
        [-138081,0,37205,-49039,-41901,-33872,-27037,-12474,18797],
        [0,28492,133236,69654,52322,-49577,-26430,-3593,0]]
    
    # Tables giving the trigonometric terms to be added to the mean
    # elements of the mean longitudes.
    
    KQ=[[3086,15746,69613,59899,75645,88306,12661,2658,0,0],
        [21863,32794,10931,73,4387,26934,1473,2157,0,0],
        [10,16002,21863,10931,1473,32004,4387,73,0,0],
        [10,6345,7818,1107,15636,7077,8184,532,10,0],
        [19,1760,1454,287,1167,880,574,2640,19,1454],
        [19,574,287,306,1760,12,31,38,19,574],
        [4,204,177,8,31,200,1265,102,4,204],
        [4,102,106,8,98,1367,487,204,4,102]]
    
    CL=[[21,-95,-157,41,-5,42,23,30,0,0],
        [-160,-313,-235,60,-74,-76,-27,34,0,0],
        [-325,-322,-79,232,-52,97,55,-41,0,0],
        [2268,-979,802,602,-668,-33,345,201,-55,0],
        [7610,-4997,-7689,-5841,-2617,1115,-748,-607,6074,354],
        [-18549,30125,20012,-730,824,23,1289,-352,-14767,-2062],
        [-135245,-14594,4197,-4030,-5630,-2898,2540,-306,2939,1986],
        [89948,2103,8963,2695,3682,1648,866,-154,-1963,-283]]
    
    SL=[[-342,136,-23,62,66,-52,-33,17,0,0],
        [524,-149,-35,117,151,122,-71,-62,0,0],
        [-105,-137,258,35,-116,-88,-112,-80,0,0],
        [854,-205,-936,-240,140,-341,-97,-232,536,0],
        [-56980,8016,1012,1448,-3024,-3710,318,503,3767,577],
        [138606,-13478,-4964,1441,-1319,-1482,427,1236,-9167,-1918],
        [71234,-41116,5334,-4935,-1848,66,434,-1748,3780,-701],
        [-47645,11647,2166,3194,679,0,-244,-419,-2531,48]]
    
    #验证行星编号
    if (NP<1)|(NP>8):
        JSTAT=-1
        
        #验证行星编号
        for k in range(2):
            for i in range(3):
                PV[k][i]=0.0
    
    else:
        
        #时间：自 J2000.0起，千儒略年.
        T=((DATE1-DJ00)+DATE2)/DJM
        
        #判断日期是否超出限制.
        if (np.abs(T)<=1.0):
            JSTAT=0
        else:
            JSTAT=1
        
        #计算平均参数
        DA=A[NP-1][0]+(A[NP-1][1]+A[NP-1][2]*T)*T
        DL=(3600.0*DLM[NP-1][0]+(DLM[NP-1][1]+DLM[NP-1][2]*T)*T)*DAS2R
        DE=E[NP-1][0]+(E[NP-1][1]+E[NP-1][2]*T)*T
        DP=pymAnpm((3600.0*PI[NP-1][0]+(PI[NP-1][1]+PI[NP-1][2]*T)*T)*DAS2R)
        DI=(3600.0*DINC[NP-1][0]+(DINC[NP-1][1]+DINC[NP-1][2]*T)*T)*DAS2R
        DOM=pymAnpm((3600.0*OMEGA[NP-1][0]+(OMEGA[NP-1][1]+OMEGA[NP-1][2]\
                                             *T)*T)*DAS2R)
        
        #应用三角函数项
        DMU=0.35953620*T
        for K in range(8):
            ARGA=KP[NP-1][K]*DMU
            ARGL=KQ[NP-1][K]*DMU
            DA=DA+(CA[NP-1][K]*ma.cos(ARGA)+SA[NP-1][K]*ma.sin(ARGA))*1e-7
            DL=DL+(CL[NP-1][K]*ma.cos(ARGL)+SL[NP-1][K]*ma.sin(ARGL))*1e-7
            
        ARGA=KP[NP-1][8]*DMU
        DA=DA+T*(CA[NP-1][8]*ma.cos(ARGA)+SA[NP-1][8]*ma.sin(ARGA))*1e-7
        
        for K in range(8,10,1):
            ARGL=KQ[NP-1][K]*DMU
            DL=DL+T*(CL[NP-1][K]*ma.cos(ARGL)+SL[NP-1][K]*ma.sin(ARGL))*1e-7
        C=np.abs(DL)%D2PI
        if DL<0:
            B=-C
        else:
            B=C
        DL=B
        
        #迭代求解开普勒方程获得偏近点角
        AM=DL-DP
        AE=AM+DE*ma.sin(AM)
        K=0
        DAE=1
        while (K<KMAX)&(np.abs(DAE)>1e-12):
            DAE=(AM-AE+DE*ma.sin(AE))/(1.0-DE*ma.cos(AE))
            AE=AE+DAE
            K=K+1
            if (K>=KMAX):
                JSTAT=2
            
        #真实近点角.
        AE2=AE/2.0
        AT=2.0*ma.atan2(ma.sqrt((1.0+DE)/(1.0-DE))*ma.sin(AE2),ma.cos(AE2))
        
        #距离(au) 和速度(弧度/day).
        R=DA*(1.0-DE*ma.cos(AE))
        V=GK*ma.sqrt((1.0+1.0/AMAS[NP])/(DA*DA*DA))
        
        SI2=ma.sin(DI/2.0)
        XQ=SI2*ma.cos(DOM)
        XP=SI2*ma.sin(DOM)
        TL=AT+DP
        XSW=ma.sin(TL)
        XCW=ma.cos(TL)
        XM2=2.0*(XP*XCW-XQ*XSW)
        XF=DA/ma.sqrt(1.0-DE*DE)
        CI2=ma.cos(DI/2.0)
        XMS=(DE*ma.sin(DP)+XSW)*XF
        XMC=(DE*ma.cos(DP)+XCW)*XF
        XPXQ2=2.0*XP*XQ

        #位置(J2000.0 ecliptic x,y,z in au).
        X=R*(XCW-XM2*XP)
        Y=R*(XSW+XM2*XQ)
        Z=R*(-XM2*CI2)
        
        #旋转到赤道
        PV[0][0]=X
        PV[0][1]=Y*COSEPS-Z*SINEPS
        PV[0][2]=Y*SINEPS+Z*COSEPS
        
        #速度 (J2000.0 ecliptic xdot,ydot,zdot in au/d).
        X=V*((-1.0+2.0*XP*XP)*XMS+XPXQ2*XMC)
        Y=V*((1.0-2.0*XQ*XQ)*XMC-XPXQ2*XMS)
        Z=V*(2.0*CI2*(XP*XMS+XQ*XMC))

        #旋转到赤道
        PV[1][0]=X
        PV[1][1]=Y*COSEPS-Z*SINEPS
        PV[1][2]=Y*SINEPS+Z*COSEPS
        
    J=JSTAT

    return(PV,J)

def pymXy06(DATE1,DATE2):
    '''
    X,Y coordinates of celestial intermediate pole from series based
    on IAU 2006 precession and IAU 2000A nutation.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date     
    date2 : float
        TT as a 2-part Julian Date     

    Returns
    -------
    x : float
        CIP X,Y coordinates    
    y : float
        CIP X,Y coordinates

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6

    #参考历元(J2000.0), JD
    DJ00=2451545.0

    #1儒略世纪对应的天数
    DJC=36525.0
    
    #T在X和Y的多项式中的最大次幂
    MAXPT=5
    
    #频率数:日月、行星、总计
    NFLS=653
    NFPL=656
    NF=NFLS+NFPL
    
    #振幅系数数
    NA=4755
    
    PT=[0 for i in range(MAXPT+1)]
    FA=[0 for i in range(14)]
    XYPR=[0,0]
    XYPL=[0,0]
    XYLS=[0,0]
    SC=[0,0]
    
    #多项式系数(角秒).
    XYP=[[-0.016617,+2004.191898,-0.4297829,-0.19861834,
          +0.000007578,+0.0000059285],
         [-0.006951,-0.025896,-22.4072747,+0.00190059,
          +0.001112526,+0.0000001358]]
    
    #基本参数:日月项.
    MFALS=[[0,0,0,0,1],
            [0,0,2,-2,2],
            [0,0,2,0,2],
            [0,0,0,0,2],
            [0,1,0,0,0],
            [0,1,2,-2,2],
            [1,0,0,0,0],
            [0,0,2,0,1],
            [1,0,2,0,2],
            [0,1,-2,2,-2],
            [0,0,2,-2,1],
            [1,0,-2,0,-2],
            [1,0,0,-2,0],
            [1,0,0,0,1],
            [1,0,0,0,-1],
            [1,0,-2,-2,-2],
            [1,0,2,0,1],
            [2,0,-2,0,-1],
            [0,0,0,2,0],
            [0,0,2,2,2],
            [2,0,0,-2,0],
            [0,2,-2,2,-2],
            [2,0,2,0,2],
            [1,0,2,-2,2],
            [1,0,-2,0,-1],
            [2,0,0,0,0],
            [0,0,2,0,0],
            [0,1,0,0,1],
            [1,0,0,-2,-1],
            [0,2,2,-2,2],
            [0,0,2,-2,0],
            [1,0,0,-2,1],
            [0,1,0,0,-1],
            [0,2,0,0,0],
            [1,0,-2,-2,-1],
            [1,0,2,2,2],
            [0,1,2,0,2],
            [2,0,-2,0,0],
            [0,0,2,2,1],
            [0,1,-2,0,-2],
            [0,0,0,2,1],
            [1,0,2,-2,1],
            [2,0,0,-2,-1],
            [2,0,2,-2,2],
            [2,0,2,0,1],
            [0,0,0,2,-1],
            [0,1,-2,2,-1],
            [1,1,0,-2,0],
            [2,0,0,-2,1],
            [1,0,0,2,0],
            [0,1,2,-2,1],
            [1,-1,0,0,0],
            [0,1,-1,1,-1],
            [2,0,-2,0,-2],
            [0,1,0,-2,0],
            [1,0,0,-1,0],
            [3,0,2,0,2],
            [0,0,0,1,0],
            [1,-1,2,0,2],
            [1,1,-2,-2,-2],
            [1,0,-2,0,0],
            [2,0,0,0,-1],
            [0,1,-2,-2,-2],
            [1,1,2,0,2],
            [2,0,0,0,1],
            [1,1,0,0,0],
            [1,0,-2,2,-1],
            [1,0,2,0,0],
            [1,-1,0,-1,0],
            [1,0,0,0,2],
            [1,0,-1,0,-1],
            [0,0,2,1,2],
            [1,0,-2,-4,-2],
            [1,-1,0,-1,-1],
            [1,0,2,2,1],
            [0,2,-2,2,-1],
            [1,0,0,0,-2],
            [2,0,-2,-2,-2],
            [1,1,2,-2,2],
            [2,0,-2,-4,-2],
            [1,0,-4,0,-2],
            [2,0,2,-2,1],
            [1,0,0,-1,-1],
            [2,0,2,2,2],
            [3,0,0,0,0],
            [1,0,0,2,1],
            [0,0,2,-2,-1],
            [3,0,2,-2,2],
            [0,0,4,-2,2],
            [1,0,0,-4,0],
            [0,1,2,0,1],
            [2,0,0,-4,0],
            [1,1,0,-2,-1],
            [2,0,-2,0,1],
            [0,0,2,0,-1],
            [0,1,-2,0,-1],
            [0,1,0,0,2],
            [0,0,2,-1,2],
            [0,0,2,4,2],
            [2,1,0,-2,0],
            [1,1,0,-2,1],
            [1,-1,0,-2,0],
            [1,-1,0,-1,-2],
            [1,-1,0,0,1],
            [0,1,-2,2,0],
            [0,1,0,0,-2],
            [1,-1,2,2,2],
            [1,0,0,2,-1],
            [1,-1,-2,-2,-2],
            [3,0,2,0,1],
            [0,1,2,2,2],
            [1,0,2,-2,0],
            [1,1,-2,-2,-1],
            [1,0,2,-4,1],
            [0,1,-2,-2,-1],
            [2,-1,2,0,2],
            [0,0,0,2,2],
            [1,-1,2,0,1],
            [1,-1,-2,0,-2],
            [0,1,0,2,0],
            [0,1,2,-2,0],
            [0,0,0,1,1],
            [1,0,-2,-2,0],
            [0,3,2,-2,2],
            [2,1,2,0,2],
            [1,1,0,0,1],
            [2,0,0,2,0],
            [1,1,2,0,1],
            [1,0,0,-2,-2],
            [1,0,-2,2,0],
            [1,0,-1,0,-2],
            [0,1,0,-2,1],
            [0,1,0,1,0],
            [0,0,0,1,-1],
            [1,0,-2,2,-2],
            [1,-1,0,0,-1],
            [0,0,0,4,0],
            [1,-1,0,2,0],
            [1,0,2,1,2],
            [1,0,2,-1,2],
            [0,0,2,1,1],
            [1,0,0,-2,2],
            [1,0,-2,0,1],
            [1,0,-2,-4,-1],
            [0,0,2,2,0],
            [1,1,2,-2,1],
            [1,0,-2,1,-1],
            [0,0,1,0,1],
            [2,0,-2,-2,-1],
            [4,0,2,0,2],
            [2,-1,0,0,0],
            [2,1,2,-2,2],
            [0,1,2,1,2],
            [1,0,4,-2,2],
            [1,1,0,0,-1],
            [2,0,2,0,0],
            [2,0,-2,-4,-1],
            [1,0,-1,0,0],
            [1,0,0,1,0],
            [0,1,0,2,1],
            [1,0,-4,0,-1],
            [1,0,0,-4,-1],
            [2,0,2,2,1],
            [2,1,0,0,0],
            [0,0,2,-3,2],
            [1,2,0,-2,0],
            [0,3,0,0,0],
            [0,0,4,0,2],
            [0,0,2,-4,1],
            [2,0,0,-2,-2],
            [1,1,-2,-4,-2],
            [0,1,0,-2,-1],
            [0,0,0,4,1],
            [3,0,2,-2,1],
            [1,0,2,4,2],
            [1,1,-2,0,-2],
            [0,0,4,-2,1],
            [2,-2,0,-2,0],
            [2,1,0,-2,-1],
            [0,2,0,-2,0],
            [1,0,0,-1,1],
            [1,1,2,2,2],
            [3,0,0,0,-1],
            [2,0,0,-4,-1],
            [3,0,2,2,2],
            [0,0,2,4,1],
            [0,2,-2,-2,-2],
            [1,-1,0,-2,-1],
            [0,0,2,-1,1],
            [2,0,0,2,1],
            [1,-1,-2,2,-1],
            [0,0,0,2,-2],
            [2,0,0,-4,1],
            [1,0,0,-4,1],
            [2,0,2,-4,1],
            [4,0,2,-2,2],
            [2,1,-2,0,-1],
            [2,1,-2,-4,-2],
            [3,0,0,-4,0],
            [1,-1,2,2,1],
            [1,-1,-2,0,-1],
            [0,2,0,0,1],
            [1,2,-2,-2,-2],
            [1,1,0,-4,0],
            [2,0,0,-2,2],
            [0,2,2,-2,1],
            [1,0,2,0,-1],
            [2,1,0,-2,1],
            [2,-1,-2,0,-1],
            [1,-1,-2,-2,-1],
            [0,1,-2,1,-2],
            [1,0,-4,2,-2],
            [0,1,2,2,1],
            [3,0,0,0,1],
            [2,-1,2,2,2],
            [0,1,-2,-4,-2],
            [1,0,-2,-3,-2],
            [2,0,0,0,2],
            [1,-1,0,-2,-2],
            [2,0,-2,2,-1],
            [0,2,-2,0,-2],
            [3,0,-2,0,-1],
            [2,-1,2,0,1],
            [1,0,-2,-1,-2],
            [0,0,2,0,3],
            [2,0,-4,0,-2],
            [2,1,0,-4,0],
            [1,1,-2,1,-1],
            [0,2,2,0,2],
            [1,-1,2,-2,2],
            [1,-1,0,-2,1],
            [2,1,2,0,1],
            [1,0,2,-4,2],
            [1,1,-2,0,-1],
            [1,1,0,2,0],
            [1,0,0,-3,0],
            [2,0,2,-1,2],
            [0,2,0,0,-1],
            [2,-1,0,-2,0],
            [4,0,0,0,0],
            [2,1,-2,-2,-2],
            [0,2,-2,2,0],
            [1,0,2,1,1],
            [1,0,-1,0,-3],
            [3,-1,2,0,2],
            [2,0,2,-2,0],
            [1,-2,0,0,0],
            [2,0,0,0,-2],
            [1,0,0,4,0],
            [0,1,0,1,1],
            [1,0,2,2,0],
            [0,1,0,2,-1],
            [0,1,0,1,-1],
            [0,0,2,-2,3],
            [3,1,2,0,2],
            [1,1,2,1,2],
            [1,1,-2,2,-1],
            [2,-1,2,-2,2],
            [1,-2,2,0,2],
            [1,0,2,-4,0],
            [0,0,1,0,0],
            [1,0,2,-3,1],
            [1,-2,0,-2,0],
            [2,0,0,2,-1],
            [1,1,2,-4,1],
            [4,0,2,0,1],
            [0,1,2,1,1],
            [1,2,2,-2,2],
            [2,0,2,1,2],
            [2,1,2,-2,1],
            [1,0,2,-1,1],
            [1,0,4,-2,1],
            [1,-1,2,-2,1],
            [0,1,0,-4,0],
            [3,0,-2,-2,-2],
            [0,0,4,-4,2],
            [2,0,-4,-2,-2],
            [2,-2,0,-2,-1],
            [1,0,2,-2,-1],
            [2,0,-2,-6,-2],
            [1,0,-2,1,-2],
            [1,0,-2,2,1],
            [1,-1,0,2,-1],
            [1,0,-2,1,0],
            [2,-1,0,-2,1],
            [1,-1,0,2,1],
            [2,0,-2,-2,0],
            [1,0,2,-3,2],
            [0,0,0,4,-1],
            [2,-1,0,0,1],
            [2,0,4,-2,2],
            [0,0,2,3,2],
            [0,1,4,-2,2],
            [0,1,-2,2,1],
            [1,1,0,2,1],
            [1,0,0,4,1],
            [0,0,4,0,1],
            [2,0,0,-3,0],
            [1,0,0,-1,-2],
            [1,-2,-2,-2,-2],
            [3,0,0,2,0],
            [2,0,2,-4,2],
            [1,1,-2,-4,-1],
            [1,0,-2,-6,-2],
            [2,-1,0,0,-1],
            [2,-1,0,2,0],
            [0,1,2,-2,-1],
            [1,1,0,1,0],
            [1,2,0,-2,-1],
            [1,0,0,1,-1],
            [0,0,1,0,2],
            [3,1,2,-2,2],
            [1,0,-4,-2,-2],
            [1,0,2,4,1],
            [1,-2,2,2,2],
            [1,-1,-2,-4,-2],
            [0,0,2,-4,2],
            [0,0,2,-3,1],
            [2,1,-2,0,0],
            [3,0,-2,-2,-1],
            [2,0,2,4,2],
            [0,0,0,0,3],
            [2,-1,-2,-2,-2],
            [2,0,0,-1,0],
            [3,0,2,-4,2],
            [2,1,2,2,2],
            [0,0,3,0,3],
            [1,1,2,2,1],
            [2,1,0,0,-1],
            [1,2,0,-2,1],
            [3,0,2,2,1],
            [1,-1,-2,2,-2],
            [1,1,0,-1,0],
            [1,2,0,0,0],
            [1,0,4,0,2],
            [1,-1,2,4,2],
            [2,1,0,0,1],
            [1,0,0,2,2],
            [1,-1,-2,2,0],
            [0,2,-2,-2,-1],
            [2,0,-2,0,2],
            [5,0,2,0,2],
            [3,0,-2,-6,-2],
            [1,-1,2,-1,2],
            [3,0,0,-4,-1],
            [1,0,0,1,1],
            [1,0,-4,2,-1],
            [0,1,2,-4,1],
            [1,2,2,0,2],
            [0,1,0,-2,-2],
            [0,0,2,-1,0],
            [1,0,1,0,1],
            [0,2,0,-2,1],
            [3,0,2,0,0],
            [1,1,-2,1,0],
            [2,1,-2,-4,-1],
            [3,-1,0,0,0],
            [2,-1,-2,0,0],
            [4,0,2,-2,1],
            [2,0,-2,2,0],
            [1,1,2,-2,0],
            [1,0,-2,4,-1],
            [1,0,-2,-2,1],
            [2,0,2,-4,0],
            [1,1,0,-2,-2],
            [1,1,-2,-2,0],
            [1,0,1,-2,1],
            [2,-1,-2,-4,-2],
            [3,0,-2,0,-2],
            [0,1,-2,-2,0],
            [3,0,0,-2,-1],
            [1,0,-2,-3,-1],
            [0,1,0,-4,-1],
            [1,-2,2,-2,1],
            [0,1,-2,1,-1],
            [1,-1,0,0,2],
            [2,0,0,1,0],
            [1,-2,0,2,0],
            [1,2,-2,-2,-1],
            [0,0,4,-4,1],
            [0,1,2,4,2],
            [0,1,-4,2,-2],
            [3,0,-2,0,0],
            [2,-1,2,2,1],
            [0,1,-2,-4,-1],
            [4,0,2,2,2],
            [2,0,-2,-3,-2],
            [2,0,0,-6,0],
            [1,0,2,0,3],
            [3,1,0,0,0],
            [3,0,0,-4,1],
            [1,-1,2,0,0],
            [1,-1,0,-4,0],
            [2,0,-2,2,-2],
            [1,1,0,-2,2],
            [4,0,0,-2,0],
            [2,2,0,-2,0],
            [0,1,2,0,0],
            [1,1,0,-4,1],
            [1,0,0,-4,-2],
            [0,0,0,1,2],
            [3,0,0,2,1],
            [1,1,0,-4,-1],
            [0,0,2,2,-1],
            [1,1,2,0,0],
            [1,-1,2,-4,1],
            [1,1,0,0,2],
            [0,0,2,6,2],
            [4,0,-2,-2,-1],
            [2,1,0,-4,-1],
            [0,0,0,3,1],
            [1,-1,-2,0,0],
            [0,0,2,1,0],
            [1,0,0,2,-2],
            [3,-1,2,2,2],
            [3,-1,2,-2,2],
            [1,0,0,-1,2],
            [1,-2,2,-2,2],
            [0,1,0,2,2],
            [0,1,-2,-1,-2],
            [1,1,-2,0,0],
            [0,2,2,-2,0],
            [3,-1,-2,-1,-2],
            [1,0,0,-6,0],
            [1,0,-2,-4,0],
            [2,1,0,-4,1],
            [2,0,2,0,-1],
            [2,0,-4,0,-1],
            [0,0,3,0,2],
            [2,1,-2,-2,-1],
            [1,-2,0,0,1],
            [2,-1,0,-4,0],
            [0,0,0,3,0],
            [5,0,2,-2,2],
            [1,2,-2,-4,-2],
            [1,0,4,-4,2],
            [0,0,4,-1,2],
            [3,1,0,-4,0],
            [3,0,0,-6,0],
            [2,0,0,2,2],
            [2,-2,2,0,2],
            [1,0,0,-3,1],
            [1,-2,-2,0,-2],
            [1,-1,-2,-3,-2],
            [0,0,2,-2,-2],
            [2,0,-2,-4,0],
            [1,0,-4,0,0],
            [0,1,0,-1,0],
            [4,0,0,0,-1],
            [3,0,2,-1,2],
            [3,-1,2,0,1],
            [2,0,2,-1,1],
            [1,2,2,-2,1],
            [1,1,0,2,-1],
            [0,2,2,0,1],
            [3,1,2,0,1],
            [1,1,2,1,1],
            [1,1,0,-1,1],
            [1,-2,0,-2,-1],
            [4,0,0,-4,0],
            [2,1,0,2,0],
            [1,-1,0,4,0],
            [0,1,0,-2,2],
            [0,0,2,0,-2],
            [1,0,-1,0,1],
            [3,0,2,-2,0],
            [2,0,2,2,0],
            [1,2,0,-4,0],
            [1,-1,0,-3,0],
            [0,1,0,4,0],
            [0,1,-2,0,0],
            [2,2,2,-2,2],
            [0,0,0,1,-2],
            [0,2,-2,0,-1],
            [4,0,2,-4,2],
            [2,0,-4,2,-2],
            [2,-1,-2,0,-2],
            [1,1,4,-2,2],
            [1,1,2,-4,2],
            [1,0,2,3,2],
            [1,0,0,4,-1],
            [0,0,0,4,2],
            [2,0,0,4,0],
            [1,1,-2,2,0],
            [2,1,2,1,2],
            [2,1,2,-4,1],
            [2,0,2,1,1],
            [2,0,-4,-2,-1],
            [2,0,-2,-6,-1],
            [2,-1,2,-1,2],
            [1,-2,2,0,1],
            [1,-2,0,-2,1],
            [1,-1,0,-4,-1],
            [0,2,2,2,2],
            [0,2,-2,-4,-2],
            [0,1,2,3,2],
            [0,1,0,-4,1],
            [3,0,0,-2,1],
            [2,1,-2,0,1],
            [2,0,4,-2,1],
            [2,0,0,-3,-1],
            [2,-2,0,-2,1],
            [2,-1,2,-2,1],
            [1,0,0,-6,-1],
            [1,-2,0,0,-1],
            [1,-2,-2,-2,-1],
            [0,1,4,-2,1],
            [0,0,2,3,1],
            [2,-1,0,-1,0],
            [1,3,0,-2,0],
            [0,3,0,-2,0],
            [2,-2,2,-2,2],
            [0,0,4,-2,0],
            [4,-1,2,0,2],
            [2,2,-2,-4,-2],
            [4,1,2,0,2],
            [4,-1,-2,-2,-2],
            [2,1,0,-2,-2],
            [2,1,-2,-6,-2],
            [2,0,0,-1,1],
            [2,-1,-2,2,-1],
            [1,1,-2,2,-2],
            [1,1,-2,-3,-2],
            [1,0,3,0,3],
            [1,0,-2,1,1],
            [1,0,-2,0,2],
            [1,-1,2,1,2],
            [1,-1,0,0,-2],
            [1,-1,-4,2,-2],
            [0,3,-2,-2,-2],
            [0,1,0,4,1],
            [0,0,4,2,2],
            [3,0,-2,-2,0],
            [2,-2,0,0,0],
            [1,1,2,-4,0],
            [1,1,0,-3,0],
            [1,0,2,-3,0],
            [1,-1,2,-2,0],
            [0,2,0,2,0],
            [0,0,2,4,0],
            [1,0,1,0,0],
            [3,1,2,-2,1],
            [3,0,4,-2,2],
            [3,0,2,1,2],
            [3,0,0,2,-1],
            [3,0,0,0,2],
            [3,0,-2,2,-1],
            [2,0,4,-4,2],
            [2,0,2,-3,2],
            [2,0,0,4,1],
            [2,0,0,-3,1],
            [2,0,-4,2,-1],
            [2,0,-2,-2,1],
            [2,-2,2,2,2],
            [2,-2,0,-2,-2],
            [2,-1,0,2,1],
            [2,-1,0,2,-1],
            [1,1,2,4,2],
            [1,1,0,1,1],
            [1,1,0,1,-1],
            [1,1,-2,-6,-2],
            [1,0,0,-3,-1],
            [1,0,-4,-2,-1],
            [1,0,-2,-6,-1],
            [1,-2,2,2,1],
            [1,-2,-2,2,-1],
            [1,-1,-2,-4,-1],
            [0,2,0,0,2],
            [0,1,2,-4,2],
            [0,1,-2,4,-1],
            [5,0,0,0,0],
            [3,0,0,-3,0],
            [2,2,0,-4,0],
            [1,-1,2,2,0],
            [0,1,0,3,0],
            [4,0,-2,0,-1],
            [3,0,-2,-6,-1],
            [3,0,-2,-1,-1],
            [2,1,2,2,1],
            [2,1,0,2,1],
            [2,0,2,4,1],
            [2,0,2,-6,1],
            [2,0,2,-2,-1],
            [2,0,0,-6,-1],
            [2,-1,-2,-2,-1],
            [1,2,2,0,1],
            [1,2,0,0,1],
            [1,0,4,0,1],
            [1,0,2,-6,1],
            [1,0,2,-4,-1],
            [1,0,-1,-2,-1],
            [1,-1,2,4,1],
            [1,-1,2,-3,1],
            [1,-1,0,4,1],
            [1,-1,-2,1,-1],
            [0,1,2,-2,3],
            [3,0,0,-2,0],
            [1,0,1,-2,0],
            [0,2,0,-4,0],
            [0,0,2,-4,0],
            [0,0,1,-1,0],
            [0,0,0,6,0],
            [0,2,0,0,-2],
            [0,1,-2,2,-3],
            [4,0,0,2,0],
            [3,0,0,-1,0],
            [3,-1,0,2,0],
            [2,1,0,1,0],
            [2,1,0,-6,0],
            [2,-1,2,0,0],
            [1,0,2,-1,0],
            [1,-1,0,1,0],
            [1,-1,-2,-2,0],
            [0,1,2,2,0],
            [0,0,2,-3,0],
            [2,2,0,-2,-1],
            [2,-1,-2,0,1],
            [1,2,2,-4,1],
            [0,1,4,-4,2],
            [0,0,0,3,2],
            [5,0,2,0,1],
            [4,1,2,-2,2],
            [4,0,-2,-2,0],
            [3,1,2,2,2],
            [3,1,0,-2,0],
            [3,1,-2,-6,-2],
            [3,0,0,0,-2],
            [3,0,-2,-4,-2],
            [3,-1,0,-3,0],
            [3,-1,0,-2,0],
            [2,1,2,0,0],
            [2,1,2,-4,2],
            [2,1,2,-2,0],
            [2,1,0,-3,0],
            [2,1,-2,0,-2],
            [2,0,0,-4,2],
            [2,0,0,-4,-2],
            [2,0,-2,-5,-2],
            [2,-1,2,4,2],
            [2,-1,0,-2,2],
            [1,3,-2,-2,-2],
            [1,1,0,0,-2],
            [1,1,0,-6,0],
            [1,1,-2,1,-2],
            [1,1,-2,-1,-2],
            [1,0,2,1,0],
            [1,0,0,3,0],
            [1,0,0,-4,2],
            [1,0,-2,4,-2],
            [1,-2,0,-1,0],
            [0,1,-4,2,-1],
            [1,0,-2,0,-3],
            [0,0,4,-4,4]]
    
    #基本参数:行星项.
    MFAPL=[[0,0,1,-1,1,0,0,-1,0,-2,5,0,0,0],
            [0,0,0,0,0,0,0,0,0,2,-5,0,0,-1],
            [0,0,0,0,0,0,3,-5,0,0,0,0,0,-2],
            [0,0,1,-1,1,0,-8,12,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,2,0,0,0,2],
            [0,0,0,0,0,0,0,4,-8,3,0,0,0,0],
            [0,0,0,0,0,0,1,-1,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,8,-16,4,5,0,0,0],
            [0,0,0,0,0,0,0,1,0,-1,0,0,0,0],
            [0,0,0,0,1,0,0,-1,2,0,0,0,0,0],
            [0,0,0,0,0,0,8,-13,0,0,0,0,0,-1],
            [0,0,1,-1,1,0,0,-1,0,2,-5,0,0,0],
            [0,0,2,-2,1,0,-5,6,0,0,0,0,0,0],
            [0,0,0,0,0,0,4,-6,0,0,0,0,0,-2],
            [0,0,0,0,0,0,0,3,0,-1,0,0,0,2],
            [0,0,0,0,0,0,0,2,-8,3,0,0,0,-2],
            [0,0,0,0,0,0,2,-4,0,0,0,0,0,-2],
            [0,0,0,0,0,0,0,6,-8,3,0,0,0,2],
            [0,0,0,0,0,0,0,1,-2,0,0,0,0,0],
            [0,0,0,0,0,0,2,-3,0,0,0,0,0,0],
            [0,0,0,0,0,0,2,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,2,0,0,0,0,0,0,2],
            [0,0,0,0,1,0,0,-4,8,-3,0,0,0,0],
            [0,0,0,0,1,0,0,4,-8,3,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,2,-5,0,0,0],
            [0,0,0,0,0,0,1,1,0,0,0,0,0,2],
            [0,0,1,-1,1,0,0,0,-2,0,0,0,0,0],
            [2,0,0,-2,-1,0,0,-2,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,2,0,0,0,1],
            [2,0,0,-2,0,0,0,-2,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,-2,0,0,0,0],
            [0,0,0,0,0,0,8,-13,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,1,0,0,0,2],
            [0,0,0,0,0,0,5,-8,0,0,0,0,0,-2],
            [0,0,0,0,0,0,0,2,-2,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,2,-5,0,0,1],
            [2,0,0,-2,0,0,0,-2,0,3,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,-1,0,0,0,0],
            [0,0,0,0,0,0,3,-4,0,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,-1,0,0,0],
            [0,0,0,0,0,0,0,1,0,-2,0,0,0,0],
            [0,0,0,0,0,0,5,-7,0,0,0,0,0,-2],
            [0,0,1,-1,0,0,0,0,-2,0,0,0,0,0],
            [0,0,0,0,0,0,0,4,0,-2,0,0,0,2],
            [0,0,0,0,0,0,8,-13,0,0,0,0,0,-2],
            [0,0,0,0,0,0,0,0,0,1,0,0,0,0],
            [0,0,0,0,0,0,2,-1,0,0,0,0,0,2],
            [1,0,0,0,0,0,-18,16,0,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,1,0,0,0,2],
            [0,0,1,-1,1,0,-5,7,0,0,0,0,0,0],
            [1,0,0,0,0,0,-10,3,0,0,0,0,0,0],
            [0,0,2,-2,0,0,-5,6,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,-1,0,0,0,2],
            [1,0,2,0,2,0,0,1,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,4,-2,0,0,0,0,2],
            [0,0,0,0,0,0,0,0,0,0,2,0,0,1],
            [1,0,-2,0,-2,0,0,4,-8,3,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,2,0,0,0],
            [0,0,2,-2,1,0,-3,3,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,0,2,0,0,2],
            [0,0,0,0,0,0,0,8,-16,4,5,0,0,-2],
            [0,0,1,-1,1,0,0,3,-8,3,0,0,0,0],
            [0,0,0,0,0,0,8,-11,0,0,0,0,0,-2],
            [0,0,0,0,0,0,0,0,0,3,0,0,0,2],
            [0,0,0,0,0,0,0,8,-16,4,5,0,0,2],
            [0,0,0,0,0,0,1,-1,0,0,0,0,0,-1],
            [0,0,0,0,0,0,4,-6,0,0,0,0,0,-1],
            [0,0,0,0,0,0,0,1,0,-3,0,0,0,-2],
            [0,0,0,0,0,0,0,2,-4,0,0,0,0,0],
            [0,0,0,0,0,0,6,-8,0,0,0,0,0,-2],
            [0,0,0,0,0,0,3,-2,0,0,0,0,0,2],
            [0,0,0,0,0,0,8,-15,0,0,0,0,0,-2],
            [0,0,0,0,0,0,2,-5,0,0,0,0,0,-2],
            [0,0,0,0,0,0,1,-3,0,0,0,0,0,-2],
            [0,0,0,0,0,0,0,3,0,-2,0,0,0,2],
            [0,0,1,-1,1,0,0,-5,8,-3,0,0,0,0],
            [0,0,0,0,0,0,0,1,2,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,-2,0,0,0,0,2],
            [0,0,0,0,0,0,3,-5,0,0,0,0,0,0],
            [2,0,0,-2,1,0,0,-2,0,3,0,0,0,0],
            [0,0,0,0,0,0,5,-8,0,0,0,0,0,-1],
            [2,0,0,-2,0,0,-3,3,0,0,0,0,0,0],
            [0,0,0,0,1,0,8,-13,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,0,0,-2,5,0,0,0],
            [1,0,0,-1,0,0,-3,4,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,0,0,0,0,2],
            [1,0,0,0,-1,0,-18,16,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,0,0,2,-5,0,0,0],
            [0,0,0,0,1,0,0,0,0,1,0,0,0,0],
            [1,0,0,-2,0,0,19,-21,3,0,0,0,0,0],
            [0,0,0,0,1,0,-8,13,0,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,1,0,0,0],
            [0,0,0,0,0,0,7,-9,0,0,0,0,0,-2],
            [0,0,0,0,0,0,0,0,2,0,0,0,0,2],
            [1,0,0,0,1,0,-18,16,0,0,0,0,0,0],
            [0,0,0,0,0,0,2,-4,0,0,0,0,0,-1],
            [0,0,0,0,0,0,0,6,-16,4,5,0,0,-2],
            [0,0,0,0,0,0,4,-7,0,0,0,0,0,-2],
            [0,0,0,0,0,0,3,-7,0,0,0,0,0,-2],
            [0,0,0,0,0,0,2,-2,0,0,0,0,0,-1],
            [0,0,0,0,0,0,0,0,0,1,0,0,0,1],
            [2,0,0,-2,1,0,0,-2,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,1,0,0,0,-1],
            [0,0,0,0,0,0,0,3,-4,0,0,0,0,0],
            [0,0,0,0,0,0,1,-2,0,0,0,0,0,0],
            [2,0,0,-2,-1,0,0,-2,0,3,0,0,0,0],
            [0,0,0,0,0,0,3,-3,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,2,0,0,0,2],
            [0,0,0,0,1,0,0,1,-2,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,1,0,0,0,2],
            [0,0,2,-2,1,0,0,-2,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,-3,0,0,0,0],
            [0,0,0,0,0,0,3,-5,0,0,0,0,0,-1],
            [0,0,0,0,0,0,3,-3,0,0,0,0,0,2],
            [0,0,0,0,0,0,4,-4,0,0,0,0,0,0],
            [0,0,1,-1,0,0,0,-1,0,-1,0,0,0,0],
            [2,0,0,-2,0,0,-6,8,0,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-2,2,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,0,1,0,0,1],
            [0,0,1,-1,1,0,0,-1,0,1,0,0,0,0],
            [0,0,0,0,0,0,0,1,-2,0,0,0,0,-1],
            [0,0,0,0,0,0,0,2,-3,0,0,0,0,0],
            [0,0,0,0,0,0,0,2,-4,0,0,0,0,-2],
            [0,0,0,0,0,0,0,1,0,0,-1,0,0,0],
            [0,0,0,0,0,0,8,-10,0,0,0,0,0,-2],
            [0,0,1,-1,1,0,-3,4,0,0,0,0,0,0],
            [0,0,0,0,0,0,6,-9,0,0,0,0,0,-2],
            [1,0,0,-1,1,0,0,-1,0,2,0,0,0,0],
            [0,0,0,0,0,0,5,-7,0,0,0,0,0,-1],
            [0,0,0,0,0,0,5,-5,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,0,1,0,0,-1],
            [0,0,0,0,0,0,3,-3,0,0,0,0,0,-1],
            [0,0,0,0,0,0,0,0,0,0,1,0,0,0],
            [0,0,0,0,0,0,0,0,4,0,0,0,0,2],
            [0,0,0,0,0,0,0,4,0,-3,0,0,0,2],
            [0,0,0,0,0,0,1,-1,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,2,0,0,0,0,0,1],
            [0,0,0,0,1,0,2,-3,0,0,0,0,0,0],
            [1,0,0,-1,0,0,0,-1,0,1,0,0,0,0],
            [0,0,0,0,0,0,1,-3,0,0,0,0,0,-1],
            [0,0,0,0,0,0,0,5,-4,0,0,0,0,2],
            [0,0,0,0,0,0,0,4,-4,0,0,0,0,2],
            [0,0,0,0,0,0,9,-11,0,0,0,0,0,-2],
            [0,0,0,0,0,0,2,-3,0,0,0,0,0,-1],
            [0,0,0,0,0,0,0,8,-15,0,0,0,0,0],
            [0,0,1,-1,1,0,-4,5,0,0,0,0,0,0],
            [0,0,0,0,0,0,4,-6,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,4,0,-1,0,0,0,2],
            [1,0,0,-1,1,0,-3,4,0,0,0,0,0,0],
            [0,0,1,1,1,0,0,1,0,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,-4,10,0,0,0],
            [0,0,0,0,1,0,1,-1,0,0,0,0,0,0],
            [0,0,1,-1,0,0,0,-1,0,0,-1,0,0,0],
            [0,0,0,0,0,0,0,1,0,-3,0,0,0,0],
            [0,0,0,0,0,0,3,-1,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,1,0,-4,0,0,0,-2],
            [0,0,0,0,0,0,0,0,0,2,-5,0,0,-2],
            [0,0,2,-2,1,0,-4,4,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,3,0,0,-1,0,0,2],
            [0,0,0,0,0,0,0,4,-3,0,0,0,0,2],
            [0,0,1,-1,1,0,0,-1,0,0,0,0,2,0],
            [0,0,0,0,0,0,4,-4,0,0,0,0,0,-1],
            [0,0,0,0,0,0,0,2,-4,0,0,0,0,-1],
            [0,0,0,0,0,0,5,-8,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,-2,0,0,0,0,1],
            [0,0,0,0,1,0,0,0,0,0,1,0,0,0],
            [0,0,2,-2,1,0,0,-9,13,0,0,0,0,0],
            [2,0,2,0,2,0,0,2,0,-3,0,0,0,0],
            [0,0,0,0,0,0,3,-6,0,0,0,0,0,-2],
            [0,0,1,-1,2,0,0,-1,0,0,2,0,0,0],
            [1,0,0,-1,-1,0,-3,4,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,3,-6,0,0,0,0,-2],
            [0,0,0,0,0,0,6,-6,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,3,0,0,0,1],
            [1,0,2,0,1,0,0,-2,0,3,0,0,0,0],
            [1,0,-2,0,-1,0,0,-1,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,-2,4,0,0,0,0,0],
            [0,0,0,0,0,0,0,3,-5,0,0,0,0,0],
            [0,0,0,0,0,0,2,1,0,0,0,0,0,2],
            [0,0,0,0,0,0,1,1,0,0,0,0,0,1],
            [0,0,2,0,2,0,0,1,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,-8,3,0,0,0,-2],
            [0,0,0,0,0,0,6,-10,0,0,0,0,0,-2],
            [0,0,0,0,0,0,0,7,-8,3,0,0,0,2],
            [0,0,0,0,1,0,-3,5,0,0,0,0,0,0],
            [0,0,1,-1,1,0,-1,0,0,0,0,0,0,0],
            [0,0,1,-1,0,0,-5,7,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,-2,0,0,0,1],
            [0,0,0,0,0,0,0,2,0,-1,0,0,0,0],
            [0,0,0,0,0,0,7,-10,0,0,0,0,0,-2],
            [1,0,0,-2,0,0,0,-2,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,2,-5,0,0,0],
            [0,0,0,0,0,0,6,-8,0,0,0,0,0,-1],
            [0,0,1,-1,1,0,0,-9,15,0,0,0,0,0],
            [0,0,0,0,1,0,-2,3,0,0,0,0,0,0],
            [0,0,0,0,1,0,-1,1,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,3,-6,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,-4,0,0,0,0,-2],
            [0,0,0,0,0,0,0,0,3,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,0,-1,0,0,2],
            [2,0,0,-2,1,0,-6,8,0,0,0,0,0,0],
            [0,0,0,0,0,0,5,-5,0,0,0,0,0,-1],
            [0,0,1,-1,1,0,3,-6,0,0,0,0,0,0],
            [0,0,1,-1,1,0,-2,2,0,0,0,0,0,0],
            [0,0,1,-1,1,0,8,-14,0,0,0,0,0,0],
            [0,0,0,0,0,0,1,0,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,8,-15,0,0,0,0,0],
            [0,0,0,0,0,0,0,4,-6,0,0,0,0,0],
            [0,0,0,0,0,0,7,-7,0,0,0,0,0,0],
            [2,0,0,-2,1,0,-3,3,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,3,-1,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,0,1,0,0,2],
            [2,0,-1,-1,0,0,0,3,-7,0,0,0,0,0],
            [0,0,0,0,0,0,0,4,-7,0,0,0,0,-2],
            [0,0,0,0,0,0,0,3,-3,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-3,4,0,0,0,0,0],
            [2,0,0,-2,0,0,0,-6,8,0,0,0,0,0],
            [2,0,0,-2,0,0,0,-5,6,0,0,0,0,0],
            [0,0,0,0,1,0,0,0,0,-1,0,0,0,0],
            [0,0,0,0,0,0,2,0,0,0,0,0,0,1],
            [0,0,0,0,0,0,2,1,0,0,0,0,0,1],
            [0,0,0,0,0,0,1,2,0,0,0,0,0,2],
            [0,0,0,0,1,0,0,1,0,-1,0,0,0,0],
            [0,0,0,0,0,0,0,1,-1,0,0,0,0,0],
            [0,0,0,0,0,0,3,-9,4,0,0,0,0,-2],
            [0,0,0,0,0,0,0,3,-5,0,0,0,0,-2],
            [0,0,0,0,0,0,0,2,0,-4,0,0,0,-2],
            [0,0,0,0,0,0,0,0,0,0,0,0,2,1],
            [0,0,0,0,0,0,7,-11,0,0,0,0,0,-2],
            [0,0,0,0,0,0,3,-5,4,0,0,0,0,2],
            [0,0,1,-1,0,0,0,-1,0,-1,1,0,0,0],
            [2,0,0,0,0,0,0,-2,0,3,0,0,0,0],
            [0,0,0,0,0,0,0,8,-15,0,0,0,0,-2],
            [0,0,1,-1,2,0,0,-2,2,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,0,3,0,0,2],
            [0,0,0,0,0,0,6,-6,0,0,0,0,0,-1],
            [0,0,1,-1,1,0,0,-1,0,-1,1,0,0,0],
            [0,0,0,0,0,0,2,-2,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,4,-7,0,0,0,0,0],
            [0,0,0,0,0,0,0,3,-8,3,0,0,0,0],
            [0,0,1,-1,1,0,2,-4,0,-3,0,0,0,0],
            [0,0,0,0,1,0,3,-5,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,3,0,-3,0,0,0,2],
            [0,0,2,-2,2,0,-8,11,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,5,-8,3,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,0,-2,0,0,0],
            [0,0,0,0,0,0,0,1,0,0,1,0,0,2],
            [0,0,0,0,0,0,0,5,-9,0,0,0,0,-2],
            [0,0,0,0,0,0,0,5,-5,0,0,0,0,2],
            [0,0,0,0,0,0,7,-9,0,0,0,0,0,-1],
            [0,0,0,0,0,0,4,-7,0,0,0,0,0,-1],
            [0,0,0,0,0,0,2,-1,0,0,0,0,0,0],
            [1,0,-2,-2,-2,0,0,-2,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,1,1,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,-2,5,0,0,2],
            [0,0,0,0,0,0,3,-3,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,2,-5,0,0,2],
            [2,0,0,-2,-1,0,0,-2,0,0,5,0,0,0],
            [2,0,0,-2,-1,0,-6,8,0,0,0,0,0,0],
            [1,0,0,-2,0,0,-3,3,0,0,0,0,0,0],
            [0,0,0,0,0,0,8,-8,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,3,0,2,-5,0,0,2],
            [0,0,0,0,1,0,3,-7,4,0,0,0,0,0],
            [0,0,2,-2,1,0,-2,2,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,-1,0,1,0,0,0,0],
            [0,0,1,-1,0,0,0,-1,0,-2,5,0,0,0],
            [0,0,0,0,0,0,0,3,0,-3,0,0,0,0],
            [0,0,0,0,0,0,3,-1,0,0,0,0,0,1],
            [0,0,0,0,0,0,2,-3,0,0,0,0,0,-2],
            [0,0,0,0,0,0,0,11,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,6,-15,0,0,0,0,-2],
            [0,0,0,0,0,0,0,3,0,1,0,0,0,2],
            [1,0,0,-1,0,0,0,-3,4,0,0,0,0,0],
            [0,0,0,0,1,0,-3,7,-4,0,0,0,0,0],
            [0,0,0,0,0,0,0,5,0,-2,0,0,0,2],
            [0,0,0,0,0,0,3,-5,0,0,0,0,0,1],
            [0,0,2,-2,2,0,-5,6,0,0,0,0,0,0],
            [0,0,2,-2,2,0,-3,3,0,0,0,0,0,0],
            [0,0,0,0,0,0,3,0,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,0],
            [0,0,0,0,0,0,4,-4,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,4,-8,0,0,0,0,-2],
            [0,0,0,0,0,0,0,4,-5,0,0,0,0,0],
            [0,0,0,0,0,0,5,-7,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,6,-11,0,0,0,0,-2],
            [0,0,0,0,0,0,0,1,-3,0,0,0,0,-2],
            [0,0,1,-1,1,0,0,-1,0,3,0,0,0,0],
            [0,0,1,-1,0,0,0,-1,0,2,0,0,0,0],
            [0,0,0,0,0,0,1,-2,0,0,0,0,0,1],
            [0,0,0,0,0,0,9,-12,0,0,0,0,0,-2],
            [0,0,0,0,0,0,4,-4,0,0,0,0,0,1],
            [0,0,1,-1,0,0,-8,12,0,0,0,0,0,0],
            [0,0,1,-1,1,0,-2,3,0,0,0,0,0,0],
            [0,0,0,0,0,0,7,-7,0,0,0,0,0,-1],
            [0,0,0,0,0,0,0,3,-6,0,0,0,0,-1],
            [0,0,0,0,0,0,0,6,-6,0,0,0,0,2],
            [0,0,0,0,0,1,0,-4,0,0,0,0,0,-2],
            [0,0,1,-1,1,0,0,1,0,0,0,0,0,0],
            [0,0,0,0,0,0,6,-9,0,0,0,0,0,-1],
            [0,0,1,-1,-1,0,0,0,-2,0,0,0,0,0],
            [0,0,0,0,0,0,0,1,-5,0,0,0,0,-2],
            [2,0,0,-2,0,0,0,-2,0,3,-1,0,0,0],
            [0,0,0,0,0,0,0,2,0,0,-2,0,0,0],
            [0,0,0,0,0,0,0,5,-9,0,0,0,0,0],
            [0,0,0,0,0,0,5,-6,0,0,0,0,0,2],
            [0,0,0,0,0,0,9,-9,0,0,0,0,0,-1],
            [0,0,1,-1,1,0,0,-1,0,0,3,0,0,0],
            [0,0,0,0,1,0,0,2,-4,0,0,0,0,0],
            [0,0,0,0,0,0,5,-3,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,0,0,0,3,0,0,1],
            [0,0,1,-1,2,0,0,-1,0,2,0,0,0,0],
            [0,0,0,0,0,0,5,-9,0,0,0,0,0,-2],
            [0,0,0,0,0,0,0,5,-3,0,0,0,0,2],
            [0,0,0,0,0,0,0,0,0,4,0,0,0,2],
            [0,0,2,0,2,0,0,4,-8,3,0,0,0,0],
            [0,0,2,0,2,0,0,-4,8,-3,0,0,0,0],
            [0,0,0,0,0,0,0,5,0,-3,0,0,0,2],
            [0,0,0,0,0,0,0,1,0,1,0,0,0,0],
            [2,0,-1,-1,-1,0,0,-1,0,3,0,0,0,0],
            [0,0,0,0,0,0,4,-3,0,0,0,0,0,2],
            [0,0,0,0,0,0,4,-2,0,0,0,0,0,2],
            [0,0,0,0,0,0,5,-10,0,0,0,0,0,-2],
            [0,0,0,0,0,0,8,-13,0,0,0,0,0,1],
            [0,0,2,-2,1,-1,0,2,0,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,0,2,0,0],
            [0,0,0,0,1,0,3,-5,0,0,0,0,0,0],
            [1,0,0,-2,0,0,0,-2,0,3,0,0,0,0],
            [0,0,2,-2,0,0,-3,3,0,0,0,0,0,0],
            [0,0,0,0,0,0,9,-9,0,0,0,0,0,0],
            [0,0,2,0,2,0,1,-1,0,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-8,11,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-2,0,0,2,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,-1,2,0,0,0],
            [0,0,0,0,0,0,5,-5,0,0,0,0,0,2],
            [0,0,0,0,0,0,2,-6,0,0,0,0,0,-2],
            [0,0,0,0,0,0,0,8,-15,0,0,0,0,-1],
            [0,0,0,0,0,0,0,5,-2,0,0,0,0,2],
            [0,0,0,0,0,0,0,0,0,0,1,0,0,2],
            [0,0,0,0,0,0,0,7,-13,0,0,0,0,-2],
            [0,0,0,0,0,0,0,3,0,-2,0,0,0,0],
            [0,0,0,0,0,0,0,1,0,3,0,0,0,2],
            [0,0,2,-2,1,0,0,-2,0,3,0,0,0,0],
            [0,0,0,0,0,0,8,-8,0,0,0,0,0,-1],
            [0,0,0,0,0,0,8,-10,0,0,0,0,0,-1],
            [0,0,0,0,0,0,4,-2,0,0,0,0,0,1],
            [0,0,0,0,0,0,3,-6,0,0,0,0,0,-1],
            [0,0,0,0,0,0,3,-4,0,0,0,0,0,-1],
            [0,0,0,0,0,0,0,0,0,2,-5,0,0,2],
            [0,0,0,0,0,0,1,0,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,-4,0,0,0,0],
            [2,0,0,-2,-1,0,0,-5,6,0,0,0,0,0],
            [0,0,0,0,0,0,0,2,-5,0,0,0,0,-2],
            [2,0,-1,-1,-1,0,0,3,-7,0,0,0,0,0],
            [0,0,0,0,0,0,0,5,-8,0,0,0,0,0],
            [0,0,2,0,2,0,-1,1,0,0,0,0,0,0],
            [2,0,0,-2,0,0,0,-2,0,4,-3,0,0,0],
            [0,0,0,0,0,0,0,6,-11,0,0,0,0,0],
            [2,0,0,-2,1,0,0,-6,8,0,0,0,0,0],
            [0,0,0,0,0,0,0,4,-8,1,5,0,0,-2],
            [0,0,0,0,0,0,0,6,-5,0,0,0,0,2],
            [1,0,-2,-2,-2,0,-3,3,0,0,0,0,0,0],
            [0,0,1,-1,2,0,0,0,-2,0,0,0,0,0],
            [0,0,0,0,2,0,0,4,-8,3,0,0,0,0],
            [0,0,0,0,2,0,0,-4,8,-3,0,0,0,0],
            [0,0,0,0,0,0,0,6,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,6,-7,0,0,0,0,2],
            [0,0,0,0,0,0,0,4,0,0,-2,0,0,2],
            [0,0,0,0,0,0,0,3,0,0,-2,0,0,2],
            [0,0,0,0,0,0,0,1,0,-1,0,0,0,1],
            [0,0,0,0,0,0,0,1,-6,0,0,0,0,-2],
            [0,0,0,0,0,0,0,0,0,4,-5,0,0,2],
            [0,0,0,0,0,0,0,0,0,0,0,2,0,2],
            [0,0,0,0,0,0,3,-5,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,7,-13,0,0,0,0,0],
            [0,0,0,0,0,0,0,2,0,-2,0,0,0,2],
            [0,0,1,-1,0,0,0,-1,0,0,2,0,0,0],
            [0,0,0,0,1,0,0,-8,15,0,0,0,0,0],
            [2,0,0,-2,-2,0,-3,3,0,0,0,0,0,0],
            [2,0,-1,-1,-1,0,0,-1,0,2,0,0,0,0],
            [1,0,2,-2,2,0,0,-2,0,2,0,0,0,0],
            [1,0,-1,1,-1,0,-18,17,0,0,0,0,0,0],
            [0,0,2,0,2,0,0,1,0,-1,0,0,0,0],
            [0,0,2,0,2,0,0,-1,0,1,0,0,0,0],
            [0,0,2,-2,-1,0,-5,6,0,0,0,0,0,0],
            [0,0,1,-1,2,0,0,-1,0,1,0,0,0,0],
            [0,0,0,0,1,0,2,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,8,-16,0,0,0,0,0,-2],
            [0,0,0,0,0,0,0,0,0,0,5,0,0,2],
            [0,0,0,0,0,0,0,0,0,0,0,0,2,2],
            [0,0,0,0,2,0,0,-1,2,0,0,0,0,0],
            [2,0,-1,-1,-2,0,0,-1,0,2,0,0,0,0],
            [0,0,0,0,0,0,6,-10,0,0,0,0,0,-1],
            [0,0,1,-1,1,0,0,-1,0,-2,4,0,0,0],
            [0,0,0,0,0,0,0,2,2,0,0,0,0,2],
            [2,0,0,-2,-1,0,0,-2,0,4,-5,0,0,0],
            [2,0,0,-2,-1,0,-3,3,0,0,0,0,0,0],
            [2,0,-1,-1,-1,0,0,-1,0,0,0,0,0,0],
            [1,0,1,-1,1,0,0,-1,0,0,0,0,0,0],
            [1,0,0,-1,-1,0,0,-2,2,0,0,0,0,0],
            [1,0,-1,-1,-1,0,20,-20,0,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-1,0,1,0,0,0,0],
            [0,0,1,-1,1,0,1,-2,0,0,0,0,0,0],
            [0,0,1,-1,1,0,-2,1,0,0,0,0,0,0],
            [0,0,0,0,1,0,5,-8,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,0,0,0,-1,0,0,0],
            [0,0,0,0,0,0,9,-11,0,0,0,0,0,-1],
            [0,0,0,0,0,0,5,-3,0,0,0,0,0,1],
            [0,0,0,0,0,0,0,1,0,-3,0,0,0,-1],
            [0,0,0,0,0,0,0,0,0,0,0,2,0,1],
            [0,0,0,0,0,0,6,-7,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,3,-2,0,0,0,0,0],
            [0,0,0,0,0,0,1,-2,0,0,0,0,0,-2],
            [0,0,1,-1,1,0,0,-1,0,0,-2,0,0,0],
            [0,0,1,-1,2,0,0,-1,0,-2,5,0,0,0],
            [0,0,0,0,0,0,0,5,-7,0,0,0,0,0],
            [0,0,0,0,0,0,1,-3,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,5,-8,0,0,0,0,-2],
            [0,0,0,0,0,0,0,2,-6,0,0,0,0,-2],
            [1,0,0,-2,0,0,20,-21,0,0,0,0,0,0],
            [0,0,0,0,0,0,8,-12,0,0,0,0,0,0],
            [0,0,0,0,0,0,5,-6,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,4,-4,0,0,0,0,0],
            [0,0,1,-1,2,0,0,-1,0,-1,0,0,0,0],
            [0,0,0,0,0,0,8,-12,0,0,0,0,0,-2],
            [0,0,0,0,0,0,0,9,-17,0,0,0,0,0],
            [0,0,0,0,0,0,0,5,-6,0,0,0,0,2],
            [0,0,0,0,0,0,0,4,-8,1,5,0,0,2],
            [0,0,0,0,0,0,0,4,-6,0,0,0,0,-2],
            [0,0,0,0,0,0,0,2,-7,0,0,0,0,-2],
            [1,0,0,-1,1,0,0,-3,4,0,0,0,0,0],
            [1,0,-2,0,-2,0,-10,3,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,-9,17,0,0,0,0,0],
            [0,0,0,0,0,0,1,-4,0,0,0,0,0,-2],
            [1,0,-2,-2,-2,0,0,-2,0,3,0,0,0,0],
            [1,0,-1,1,-1,0,0,1,0,0,0,0,0,0],
            [0,0,2,-2,2,0,0,-2,0,2,0,0,0,0],
            [0,0,1,-1,2,0,0,-1,0,0,1,0,0,0],
            [0,0,1,-1,2,0,-5,7,0,0,0,0,0,0],
            [0,0,0,0,1,0,0,2,-2,0,0,0,0,0],
            [0,0,0,0,0,0,4,-5,0,0,0,0,0,-1],
            [0,0,0,0,0,0,3,-4,0,0,0,0,0,-2],
            [0,0,0,0,0,0,2,-4,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,5,-10,0,0,0,0,-2],
            [0,0,0,0,0,0,0,4,0,-4,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,-5,0,0,0,-2],
            [0,0,0,0,0,0,0,1,0,-5,0,0,0,-2],
            [0,0,0,0,0,0,0,1,0,-2,5,0,0,2],
            [0,0,0,0,0,0,0,1,0,-2,0,0,0,-2],
            [0,0,0,0,0,0,2,-3,0,0,0,0,0,1],
            [1,0,0,-2,0,0,0,1,0,-1,0,0,0,0],
            [0,0,0,0,0,0,3,-7,4,0,0,0,0,0],
            [2,0,2,0,1,0,0,1,0,0,0,0,0,0],
            [0,0,1,-1,-1,0,0,-1,0,-1,0,0,0,0],
            [0,0,0,0,1,0,0,1,0,-2,0,0,0,0],
            [0,0,0,0,0,0,0,6,-10,0,0,0,0,-2],
            [1,0,0,-1,1,0,0,-1,0,1,0,0,0,0],
            [0,0,2,-2,1,0,0,4,-8,3,0,0,0,0],
            [0,0,2,-2,1,0,0,1,0,-1,0,0,0,0],
            [0,0,2,-2,1,0,0,-4,8,-3,0,0,0,0],
            [0,0,2,-2,1,0,0,-3,0,3,0,0,0,0],
            [0,0,2,-2,1,0,-5,5,0,0,0,0,0,0],
            [0,0,1,-1,1,0,1,-3,0,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-4,6,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,0,0,-1,0,0],
            [0,0,1,-1,1,0,-5,6,0,0,0,0,0,0],
            [0,0,0,0,1,0,3,-4,0,0,0,0,0,0],
            [0,0,0,0,1,0,-2,2,0,0,0,0,0,0],
            [0,0,0,0,0,0,7,-10,0,0,0,0,0,-1],
            [0,0,0,0,0,0,5,-5,0,0,0,0,0,1],
            [0,0,0,0,0,0,4,-5,0,0,0,0,0,-2],
            [0,0,0,0,0,0,3,-8,0,0,0,0,0,-2],
            [0,0,0,0,0,0,2,-5,0,0,0,0,0,-1],
            [0,0,0,0,0,0,1,-2,0,0,0,0,0,-1],
            [0,0,0,0,0,0,0,7,-9,0,0,0,0,2],
            [0,0,0,0,0,0,0,7,-8,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,3,-8,3,0,0,0,-2],
            [0,0,0,0,0,0,0,2,0,0,-2,0,0,1],
            [0,0,0,0,0,0,0,2,-4,0,0,0,0,1],
            [0,0,0,0,0,0,0,1,0,0,0,0,0,-1],
            [0,0,0,0,0,0,0,1,0,-1,0,0,0,-1],
            [2,0,0,-2,-1,0,0,-6,8,0,0,0,0,0],
            [2,0,-1,-1,1,0,0,3,-7,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-7,9,0,0,0,0,0],
            [0,0,0,0,0,0,0,3,-5,0,0,0,0,-1],
            [0,0,1,-1,2,0,-8,12,0,0,0,0,0,0],
            [1,0,0,0,0,0,0,-2,0,2,0,0,0,0],
            [1,0,0,-2,0,0,2,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,7,-8,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,2,0,0,0,0,0],
            [2,0,0,-2,1,0,0,-5,6,0,0,0,0,0],
            [2,0,0,-2,-1,0,0,-2,0,3,-1,0,0,0],
            [1,0,1,1,1,0,0,1,0,0,0,0,0,0],
            [1,0,0,-2,1,0,0,-2,0,2,0,0,0,0],
            [1,0,0,-2,-1,0,0,-2,0,2,0,0,0,0],
            [1,0,0,-1,-1,0,0,-3,4,0,0,0,0,0],
            [1,0,-1,0,-1,0,-3,5,0,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-4,4,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-2,0,0,0,0,0,0],
            [0,0,2,-2,1,0,-8,11,0,0,0,0,0,0],
            [0,0,2,-2,0,0,0,-9,13,0,0,0,0,0],
            [0,0,1,1,2,0,0,1,0,0,0,0,0,0],
            [0,0,1,-1,1,0,0,1,-4,0,0,0,0,0],
            [0,0,1,-1,1,0,0,-1,0,1,-3,0,0,0],
            [0,0,0,0,1,0,0,7,-13,0,0,0,0,0],
            [0,0,0,0,1,0,0,2,0,-2,0,0,0,0],
            [0,0,0,0,1,0,0,-2,2,0,0,0,0,0],
            [0,0,0,0,1,0,-3,4,0,0,0,0,0,0],
            [0,0,0,0,0,1,0,-4,0,0,0,0,0,0],
            [0,0,0,0,0,0,7,-11,0,0,0,0,0,-1],
            [0,0,0,0,0,0,6,-6,0,0,0,0,0,1],
            [0,0,0,0,0,0,6,-4,0,0,0,0,0,1],
            [0,0,0,0,0,0,5,-6,0,0,0,0,0,-1],
            [0,0,0,0,0,0,4,-2,0,0,0,0,0,0],
            [0,0,0,0,0,0,3,-4,0,0,0,0,0,1],
            [0,0,0,0,0,0,1,-4,0,0,0,0,0,-1],
            [0,0,0,0,0,0,0,9,-17,0,0,0,0,-2],
            [0,0,0,0,0,0,0,7,-7,0,0,0,0,2],
            [0,0,0,0,0,0,0,4,-8,3,0,0,0,1],
            [0,0,0,0,0,0,0,4,-8,3,0,0,0,-1],
            [0,0,0,0,0,0,0,4,-8,0,0,0,0,0],
            [0,0,0,0,0,0,0,4,-7,0,0,0,0,-1],
            [0,0,0,0,0,0,0,1,0,1,0,0,0,1],
            [0,0,0,0,0,0,0,1,0,-4,0,0,0,0],
            [2,0,0,-2,0,0,0,-4,8,-3,0,0,0,0],
            [2,0,0,-2,0,0,-2,2,0,0,0,0,0,0],
            [1,0,0,0,0,0,0,4,-8,3,0,0,0,0],
            [1,0,0,0,0,0,0,-4,8,-3,0,0,0,0],
            [1,0,0,0,0,0,-1,1,0,0,0,0,0,0],
            [1,0,0,-2,0,0,17,-16,0,-2,0,0,0,0],
            [1,0,0,-1,0,0,0,-2,2,0,0,0,0,0],
            [0,0,2,-2,0,0,0,-2,0,2,0,0,0,0],
            [0,0,0,0,0,0,0,6,-9,0,0,0,0,0],
            [0,0,0,0,0,0,0,4,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,3,0,-4,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,0,0,1,-2,-2],
            [0,0,0,0,0,0,0,2,1,0,0,0,0,2],
            [2,0,0,-2,0,0,0,-4,4,0,0,0,0,0],
            [2,0,0,-2,0,0,0,-2,0,2,2,0,0,0],
            [1,0,0,0,0,0,1,-1,0,0,0,0,0,0],
            [1,0,0,0,0,0,0,-1,0,1,0,0,0,0],
            [1,0,0,0,0,0,-3,3,0,0,0,0,0,0],
            [1,0,0,-2,0,0,1,-1,0,0,0,0,0,0],
            [1,0,0,-2,0,0,0,4,-8,3,0,0,0,0],
            [1,0,0,-2,0,0,0,-4,8,-3,0,0,0,0],
            [1,0,0,-2,0,0,-2,2,0,0,0,0,0,0],
            [0,0,2,-2,0,0,-4,4,0,0,0,0,0,0],
            [0,0,1,1,0,0,0,1,0,0,0,0,0,0],
            [0,0,1,-1,0,0,3,-6,0,0,0,0,0,0],
            [0,0,1,-1,0,0,0,-2,2,0,0,0,0,0],
            [0,0,1,-1,0,0,0,-1,0,1,0,0,0,0],
            [0,0,1,-1,0,0,0,-1,0,0,1,0,0,0],
            [0,0,1,-1,0,0,-4,5,0,0,0,0,0,0],
            [0,0,1,-1,0,0,-3,4,0,0,0,0,0,0],
            [0,0,0,2,0,0,0,-1,0,1,0,0,0,0],
            [0,0,0,0,0,0,8,-9,0,0,0,0,0,0],
            [0,0,0,0,0,0,3,-6,0,0,0,0,0,0],
            [0,0,0,0,0,0,1,1,0,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,3,-5,0,0,0],
            [0,0,0,0,0,0,0,0,0,2,-2,0,0,0],
            [2,0,-2,-2,-2,0,0,-2,0,2,0,0,0,0],
            [1,0,0,0,1,0,-10,3,0,0,0,0,0,0],
            [1,0,0,0,-1,0,-10,3,0,0,0,0,0,0],
            [0,0,2,0,2,0,2,-3,0,0,0,0,0,0],
            [0,0,2,0,2,0,2,-2,0,0,0,0,0,0],
            [0,0,2,0,2,0,-2,3,0,0,0,0,0,0],
            [0,0,2,0,2,0,-2,2,0,0,0,0,0,0],
            [0,0,0,0,2,0,0,0,0,1,0,0,0,0],
            [0,0,0,0,1,0,0,-1,0,2,0,0,0,0],
            [2,0,2,-2,2,0,0,-2,0,3,0,0,0,0],
            [2,0,1,-3,1,0,-6,7,0,0,0,0,0,0],
            [2,0,0,-2,0,0,2,-5,0,0,0,0,0,0],
            [2,0,0,-2,0,0,0,-2,0,5,-5,0,0,0],
            [2,0,0,-2,0,0,0,-2,0,1,5,0,0,0],
            [2,0,0,-2,0,0,0,-2,0,0,5,0,0,0],
            [2,0,0,-2,0,0,0,-2,0,0,2,0,0,0],
            [2,0,0,-2,0,0,-4,4,0,0,0,0,0,0],
            [2,0,-2,0,-2,0,0,5,-9,0,0,0,0,0],
            [2,0,-1,-1,0,0,0,-1,0,3,0,0,0,0],
            [1,0,2,0,2,0,1,-1,0,0,0,0,0,0],
            [1,0,2,0,2,0,0,4,-8,3,0,0,0,0],
            [1,0,2,0,2,0,0,-4,8,-3,0,0,0,0],
            [1,0,2,0,2,0,-1,1,0,0,0,0,0,0],
            [1,0,2,-2,2,0,-3,3,0,0,0,0,0,0],
            [1,0,0,0,0,0,0,1,0,-1,0,0,0,0],
            [1,0,0,0,0,0,0,-2,0,3,0,0,0,0],
            [1,0,0,-2,0,0,0,2,0,-2,0,0,0,0],
            [1,0,-2,-2,-2,0,0,1,0,-1,0,0,0,0],
            [1,0,-1,1,0,0,0,1,0,0,0,0,0,0],
            [1,0,-1,-1,0,0,0,8,-15,0,0,0,0,0],
            [0,0,2,2,2,0,0,2,0,-2,0,0,0,0],
            [0,0,2,-2,1,0,1,-1,0,0,0,0,0,0],
            [0,0,2,-2,1,0,0,-2,0,1,0,0,0,0],
            [0,0,2,-2,1,0,0,-10,15,0,0,0,0,0],
            [0,0,2,-2,0,-1,0,2,0,0,0,0,0,0],
            [0,0,1,-1,2,0,0,-1,0,0,-1,0,0,0],
            [0,0,1,-1,2,0,-3,4,0,0,0,0,0,0],
            [0,0,1,-1,1,0,-4,6,0,0,0,0,0,0],
            [0,0,1,-1,1,0,-1,2,0,0,0,0,0,0],
            [0,0,1,-1,0,0,0,1,0,0,0,0,0,0],
            [0,0,1,-1,0,0,0,-1,0,0,-2,0,0,0],
            [0,0,1,-1,0,0,-2,2,0,0,0,0,0,0],
            [0,0,1,-1,0,0,-1,0,0,0,0,0,0,0],
            [0,0,1,-1,-1,0,-5,7,0,0,0,0,0,0],
            [0,0,0,2,0,0,0,2,0,-2,0,0,0,0],
            [0,0,0,2,0,0,-2,2,0,0,0,0,0,0],
            [0,0,0,0,2,0,-3,5,0,0,0,0,0,0],
            [0,0,0,0,1,0,-1,2,0,0,0,0,0,0],
            [0,0,0,0,0,0,9,-13,0,0,0,0,0,-2],
            [0,0,0,0,0,0,8,-14,0,0,0,0,0,-2],
            [0,0,0,0,0,0,8,-11,0,0,0,0,0,-1],
            [0,0,0,0,0,0,6,-9,0,0,0,0,0,0],
            [0,0,0,0,0,0,6,-8,0,0,0,0,0,0],
            [0,0,0,0,0,0,6,-7,0,0,0,0,0,-1],
            [0,0,0,0,0,0,5,-6,0,0,0,0,0,-2],
            [0,0,0,0,0,0,5,-6,-4,0,0,0,0,-2],
            [0,0,0,0,0,0,5,-4,0,0,0,0,0,2],
            [0,0,0,0,0,0,4,-8,0,0,0,0,0,-2],
            [0,0,0,0,0,0,4,-5,0,0,0,0,0,0],
            [0,0,0,0,0,0,3,-3,0,2,0,0,0,2],
            [0,0,0,0,0,0,3,-1,0,0,0,0,0,0],
            [0,0,0,0,0,0,2,0,0,0,0,0,0,0],
            [0,0,0,0,0,0,1,-1,0,0,0,0,0,-2],
            [0,0,0,0,0,0,0,7,-12,0,0,0,0,-2],
            [0,0,0,0,0,0,0,6,-9,0,0,0,0,-2],
            [0,0,0,0,0,0,0,6,-8,1,5,0,0,2],
            [0,0,0,0,0,0,0,6,-4,0,0,0,0,2],
            [0,0,0,0,0,0,0,6,-10,0,0,0,0,0],
            [0,0,0,0,0,0,0,5,0,-4,0,0,0,2],
            [0,0,0,0,0,0,0,5,-9,0,0,0,0,-1],
            [0,0,0,0,0,0,0,5,-8,3,0,0,0,2],
            [0,0,0,0,0,0,0,5,-7,0,0,0,0,-2],
            [0,0,0,0,0,0,0,5,-6,0,0,0,0,0],
            [0,0,0,0,0,0,0,5,-16,4,5,0,0,-2],
            [0,0,0,0,0,0,0,5,-13,0,0,0,0,-2],
            [0,0,0,0,0,0,0,3,0,-5,0,0,0,-2],
            [0,0,0,0,0,0,0,3,-9,0,0,0,0,-2],
            [0,0,0,0,0,0,0,3,-7,0,0,0,0,-2],
            [0,0,0,0,0,0,0,2,0,2,0,0,0,2],
            [0,0,0,0,0,0,0,2,0,0,-3,0,0,0],
            [0,0,0,0,0,0,0,2,-8,1,5,0,0,-2],
            [0,0,0,0,0,0,0,1,0,1,-5,0,0,0],
            [0,0,0,0,0,0,0,1,0,0,2,0,0,2],
            [0,0,0,0,0,0,0,1,0,0,-3,0,0,0],
            [0,0,0,0,0,0,0,1,0,-3,5,0,0,0],
            [0,0,0,0,0,0,0,1,-3,0,0,0,0,0],
            [0,0,0,0,0,0,0,0,0,2,-6,3,0,-2],
            [0,0,0,0,0,0,0,0,0,1,-2,0,0,0],
            [0,0,0,0,0,0,0,0,0,0,0,1,0,0],
            [0,0,0,0,0,0,0,0,0,0,0,0,0,2],
            [0,0,0,0,0,0,0,0,1,0,0,0,0,0]]
    
    #指向振幅数组的指针，每个频率一个指针
    NC=[1,21,37,51,65,79,91,103,115,127,
        139,151,163,172,184,196,207,219,231,240,
        252,261,273,285,297,309,318,327,339,351,
        363,372,384,396,405,415,423,435,444,452,
        460,467,474,482,490,498,506,513,521,528,
        536,543,551,559,566,574,582,590,597,605,
        613,620,628,636,644,651,658,666,674,680,
        687,695,702,710,717,725,732,739,746,753,
        760,767,774,782,790,798,805,812,819,826,
        833,840,846,853,860,867,874,881,888,895,
        901,908,914,921,928,934,941,948,955,962,
        969,976,982,989,996,1003,1010,1017,1024,1031,
        1037,1043,1050,1057,1064,1071,1078,1084,1091,1098,
        1104,1112,1118,1124,1131,1138,1145,1151,1157,1164,
        1171,1178,1185,1192,1199,1205,1212,1218,1226,1232,
        1239,1245,1252,1259,1266,1272,1278,1284,1292,1298,
        1304,1310,1316,1323,1329,1335,1341,1347,1353,1359,
        1365,1371,1377,1383,1389,1396,1402,1408,1414,1420,
        1426,1434,1440,1446,1452,1459,1465,1471,1477,1482,
        1488,1493,1499,1504,1509,1514,1520,1527,1532,1538,
        1543,1548,1553,1558,1564,1569,1574,1579,1584,1589,
        1594,1596,1598,1600,1602,1605,1608,1610,1612,1617,
        1619,1623,1625,1627,1629,1632,1634,1640,1642,1644,
        1646,1648,1650,1652,1654,1658,1660,1662,1664,1668,
        1670,1672,1673,1675,1679,1681,1683,1684,1686,1688,
        1690,1693,1695,1697,1701,1703,1705,1707,1709,1711,
        1712,1715,1717,1721,1723,1725,1727,1729,1731,1733,
        1735,1737,1739,1741,1743,1745,1747,1749,1751,1753,
        1755,1757,1759,1761,1762,1764,1766,1768,1769,1771,
        1773,1775,1777,1779,1781,1783,1785,1787,1788,1790,
        1792,1794,1796,1798,1800,1802,1804,1806,1807,1809,
        1811,1815,1817,1819,1821,1823,1825,1827,1829,1831,
        1833,1835,1837,1839,1840,1842,1844,1848,1850,1852,
        1854,1856,1858,1859,1860,1862,1864,1866,1868,1869,
        1871,1873,1875,1877,1879,1881,1883,1885,1887,1889,
        1891,1892,1896,1898,1900,1901,1903,1905,1907,1909,
        1910,1911,1913,1915,1919,1921,1923,1927,1929,1931,
        1933,1935,1937,1939,1943,1945,1947,1948,1949,1951,
        1953,1955,1957,1958,1960,1962,1964,1966,1968,1970,
        1971,1973,1974,1975,1977,1979,1980,1981,1982,1984,
        1986,1988,1990,1992,1994,1995,1997,1999,2001,2003,
        2005,2007,2008,2009,2011,2013,2015,2017,2019,2021,
        2023,2024,2025,2027,2029,2031,2033,2035,2037,2041,
        2043,2045,2046,2047,2049,2051,2053,2055,2056,2057,
        2059,2061,2063,2065,2067,2069,2070,2071,2072,2074,
        2076,2078,2080,2082,2084,2086,2088,2090,2092,2094,
        2095,2096,2097,2099,2101,2105,2106,2107,2108,2109,
        2110,2111,2113,2115,2119,2121,2123,2125,2127,2129,
        2131,2133,2135,2136,2137,2139,2141,2143,2145,2147,
        2149,2151,2153,2155,2157,2159,2161,2163,2165,2167,
        2169,2171,2173,2175,2177,2179,2181,2183,2185,2186,
        2187,2188,2192,2193,2195,2197,2199,2201,2203,2205,
        2207,2209,2211,2213,2217,2219,2221,2223,2225,2227,
        2229,2231,2233,2234,2235,2236,2237,2238,2239,2240,
        2241,2244,2246,2248,2250,2252,2254,2256,2258,2260,
        2262,2264,2266,2268,2270,2272,2274,2276,2278,2280,
        2282,2284,2286,2288,2290,2292,2294,2296,2298,2300,
        2302,2303,2304,2305,2306,2307,2309,2311,2313,2315,
        2317,2319,2321,2323,2325,2327,2329,2331,2333,2335,
        2337,2341,2343,2345,2347,2349,2351,2352,2355,2356,
        2357,2358,2359,2361,2363,2364,2365,2366,2367,2368,
        2369,2370,2371,2372,2373,2374,2376,2378,2380,2382,
        2384,2385,2386,2387,2388,2389,2390,2391,2392,2393,
        2394,2395,2396,2397,2398,2399,2400,2401,2402,2403,
        2404,2405,2406,2407,2408,2409,2410,2411,2412,2413,
        2414,2415,2417,2418,2430,2438,2445,2453,2460,2468,
        2474,2480,2488,2496,2504,2512,2520,2527,2535,2543,
        2550,2558,2566,2574,2580,2588,2596,2604,2612,2619,
        2627,2634,2642,2648,2656,2664,2671,2679,2685,2693,
        2701,2709,2717,2725,2733,2739,2747,2753,2761,2769,
        2777,2785,2793,2801,2809,2817,2825,2833,2841,2848,
        2856,2864,2872,2878,2884,2892,2898,2906,2914,2922,
        2930,2938,2944,2952,2958,2966,2974,2982,2988,2996,
        3001,3009,3017,3025,3032,3039,3045,3052,3059,3067,
        3069,3076,3083,3090,3098,3105,3109,3111,3113,3120,
        3124,3128,3132,3136,3140,3144,3146,3150,3158,3161,
        3165,3166,3168,3172,3176,3180,3182,3185,3189,3193,
        3194,3197,3200,3204,3208,3212,3216,3219,3221,3222,
        3226,3230,3234,3238,3242,3243,3247,3251,3254,3258,
        3262,3266,3270,3274,3275,3279,3283,3287,3289,3293,
        3296,3300,3303,3307,3311,3315,3319,3321,3324,3327,
        3330,3334,3338,3340,3342,3346,3350,3354,3358,3361,
        3365,3369,3373,3377,3381,3385,3389,3393,3394,3398,
        3402,3406,3410,3413,3417,3421,3425,3429,3433,3435,
        3439,3443,3446,3450,3453,3457,3458,3461,3464,3468,
        3472,3476,3478,3481,3485,3489,3493,3497,3501,3505,
        3507,3511,3514,3517,3521,3524,3525,3527,3529,3533,
        3536,3540,3541,3545,3548,3551,3555,3559,3563,3567,
        3569,3570,3574,3576,3578,3582,3586,3590,3593,3596,
        3600,3604,3608,3612,3616,3620,3623,3626,3630,3632,
        3636,3640,3643,3646,3648,3652,3656,3660,3664,3667,
        3669,3671,3675,3679,3683,3687,3689,3693,3694,3695,
        3699,3703,3705,3707,3710,3713,3717,3721,3725,3729,
        3733,3736,3740,3744,3748,3752,3754,3757,3759,3763,
        3767,3770,3773,3777,3779,3783,3786,3790,3794,3798,
        3801,3805,3809,3813,3817,3821,3825,3827,3831,3835,
        3836,3837,3840,3844,3848,3852,3856,3859,3863,3867,
        3869,3871,3875,3879,3883,3887,3890,3894,3898,3901,
        3905,3909,3913,3917,3921,3922,3923,3924,3926,3930,
        3932,3936,3938,3940,3944,3948,3952,3956,3959,3963,
        3965,3969,3973,3977,3979,3981,3982,3986,3989,3993,
        3997,4001,4004,4006,4009,4012,4016,4020,4024,4026,
        4028,4032,4036,4040,4044,4046,4050,4054,4058,4060,
        4062,4063,4064,4068,4071,4075,4077,4081,4083,4087,
        4089,4091,4095,4099,4101,4103,4105,4107,4111,4115,
        4119,4123,4127,4129,4131,4135,4139,4141,4143,4145,
        4149,4153,4157,4161,4165,4169,4173,4177,4180,4183,
        4187,4191,4195,4198,4201,4205,4209,4212,4213,4216,
        4217,4221,4223,4226,4230,4234,4236,4240,4244,4248,
        4252,4256,4258,4262,4264,4266,4268,4270,4272,4276,
        4279,4283,4285,4287,4289,4293,4295,4299,4300,4301,
        4305,4309,4313,4317,4319,4323,4325,4329,4331,4333,
        4335,4337,4341,4345,4349,4351,4353,4357,4361,4365,
        4367,4369,4373,4377,4381,4383,4387,4389,4391,4395,
        4399,4403,4407,4411,4413,4414,4415,4418,4419,4421,
        4423,4427,4429,4431,4433,4435,4437,4439,4443,4446,
        4450,4452,4456,4458,4460,4462,4466,4469,4473,4477,
        4481,4483,4487,4489,4491,4493,4497,4499,4501,4504,
        4506,4510,4513,4514,4515,4518,4521,4522,4525,4526,
        4527,4530,4533,4534,4537,4541,4542,4543,4544,4545,
        4546,4547,4550,4553,4554,4555,4558,4561,4564,4567,
        4568,4571,4574,4575,4578,4581,4582,4585,4586,4588,
        4590,4592,4596,4598,4602,4604,4608,4612,4613,4616,
        4619,4622,4623,4624,4625,4626,4629,4632,4633,4636,
        4639,4640,4641,4642,4643,4644,4645,4648,4649,4650,
        4651,4652,4653,4656,4657,4660,4661,4664,4667,4670,
        4671,4674,4675,4676,4677,4678,4681,4682,4683,4684,
        4687,4688,4689,4692,4693,4696,4697,4700,4701,4702,
        4703,4704,4707,4708,4711,4712,4715,4716,4717,4718,
        4719,4720,4721,4722,4723,4726,4729,4730,4733,4736,
        4737,4740,4741,4742,4745,4746,4749,4752,4753]
    
    #振幅系数(微角秒);使用NC数组进行索引。
    A= [-6844318.44e0,9205236.26e0,1328.67e0,1538.18e0,205833.11e0,
        153041.79e0,-3309.73e0,853.32e0,2037.98e0,-2301.27e0,81.46e0,
        120.56e0,-20.39e0,-15.22e0,1.73e0,-1.61e0,-0.1e0,0.11e0,-0.02e0,
        -0.02e0,-523908.04e0,573033.42e0,-544.75e0,-458.66e0,12814.01e0,
        11714.49e0,198.97e0,-290.91e0,155.74e0,-143.27e0,-2.75e0,-1.03e0,
        -1.27e0,-1.16e0,0e0,-0.01e0,-90552.22e0,97846.69e0,111.23e0,
        137.41e0,2187.91e0,2024.68e0,41.44e0,-51.26e0,26.92e0,-24.46e0,
        -0.46e0,-0.28e0,-0.22e0,-0.2e0,82168.76e0,-89618.24e0,-27.64e0,
        -29.05e0,-2004.36e0,-1837.32e0,-36.07e0,48e0,-24.43e0,22.41e0,
        0.47e0,0.24e0,0.2e0,0.18e0,58707.02e0,7387.02e0,470.05e0,
        -192.4e0,164.33e0,-1312.21e0,-179.73e0,-28.93e0,-17.36e0,-1.83e0,
        -0.5e0,3.57e0,0e0,0.13e0,-20557.78e0,22438.42e0,-20.84e0,
        -17.4e0,501.82e0,459.68e0,59.2e0,-67.3e0,6.08e0,-5.61e0,-1.36e0,
        -1.19e0,28288.28e0,-674.99e0,-34.69e0,35.8e0,-15.07e0,-632.54e0,
        -11.19e0,0.78e0,-8.41e0,0.17e0,0.01e0,0.07e0,-15406.85e0,
        20069.5e0,15.12e0,31.8e0,448.76e0,344.5e0,-5.77e0,1.41e0,4.59e0,
        -5.02e0,0.17e0,0.24e0,-11991.74e0,12902.66e0,32.46e0,36.7e0,
        288.49e0,268.14e0,5.7e0,-7.06e0,3.57e0,-3.23e0,-0.06e0,-0.04e0,
        -8584.95e0,-9592.72e0,4.42e0,-13.2e0,-214.5e0,192.06e0,23.87e0,
        29.83e0,2.54e0,2.4e0,0.6e0,-0.48e0,5095.5e0,-6918.22e0,
        7.19e0,3.92e0,-154.91e0,-113.94e0,2.86e0,-1.04e0,-1.52e0,
        1.73e0,-0.07e0,-0.1e0,-4910.93e0,-5331.13e0,0.76e0,0.4e0,
        -119.21e0,109.81e0,2.16e0,3.2e0,1.46e0,1.33e0,0.04e0,-0.02e0,
        -6245.02e0,-123.48e0,-6.68e0,-8.2e0,-2.76e0,139.64e0,2.71e0,
        0.15e0,1.86e0,2511.85e0,-3323.89e0,1.07e0,-0.9e0,-74.33e0,
        -56.17e0,1.16e0,-0.01e0,-0.75e0,0.83e0,-0.02e0,-0.04e0,
        2307.58e0,3143.98e0,-7.52e0,7.5e0,70.31e0,-51.6e0,1.46e0,0.16e0,
        -0.69e0,-0.79e0,0.02e0,-0.05e0,2372.58e0,2554.51e0,5.93e0,
        -6.6e0,57.12e0,-53.05e0,-0.96e0,-1.24e0,-0.71e0,-0.64e0,-0.01e0,
        -2053.16e0,2636.13e0,5.13e0,7.8e0,58.94e0,45.91e0,-0.42e0,
        -0.12e0,0.61e0,-0.66e0,0.02e0,0.03e0,-1825.49e0,-2423.59e0,
        1.23e0,-2e0,-54.19e0,40.82e0,-1.07e0,-1.02e0,0.54e0,0.61e0,
        -0.04e0,0.04e0,2521.07e0,-122.28e0,-5.97e0,2.9e0,-2.73e0,
        -56.37e0,-0.82e0,0.13e0,-0.75e0,-1534.09e0,1645.01e0,6.29e0,
        6.8e0,36.78e0,34.3e0,0.92e0,-1.25e0,0.46e0,-0.41e0,-0.02e0,
        -0.01e0,1898.27e0,47.7e0,-0.72e0,2.5e0,1.07e0,-42.45e0,-0.94e0,
        0.02e0,-0.56e0,-1292.02e0,-1387e0,0e0,0e0,-31.01e0,28.89e0,
        0.68e0,0e0,0.38e0,0.35e0,-0.01e0,-0.01e0,-1234.96e0,
        1323.81e0,5.21e0,5.9e0,29.6e0,27.61e0,0.74e0,-1.22e0,0.37e0,
        -0.33e0,-0.02e0,-0.01e0,1137.48e0,-1233.89e0,-0.04e0,-0.3e0,
        -27.59e0,-25.43e0,-0.61e0,1e0,-0.34e0,0.31e0,0.01e0,0.01e0,
        -813.13e0,-1075.6e0,0.4e0,0.3e0,-24.05e0,18.18e0,-0.4e0,-0.01e0,
        0.24e0,0.27e0,-0.01e0,0.01e0,1163.22e0,-60.9e0,-2.94e0,1.3e0,
        -1.36e0,-26.01e0,-0.58e0,0.07e0,-0.35e0,1029.7e0,-55.55e0,
        -2.63e0,1.1e0,-1.25e0,-23.02e0,-0.52e0,0.06e0,-0.31e0,
        -556.26e0,852.85e0,3.16e0,-4.48e0,19.06e0,12.44e0,-0.81e0,
        -0.27e0,0.17e0,-0.21e0,0e0,0.02e0,-603.52e0,-800.34e0,
        0.44e0,0.1e0,-17.9e0,13.49e0,-0.08e0,-0.01e0,0.18e0,0.2e0,
        -0.01e0,0.01e0,-628.24e0,684.99e0,-0.64e0,-0.5e0,15.32e0,
        14.05e0,3.18e0,-4.19e0,0.19e0,-0.17e0,-0.09e0,-0.07e0,
        -866.48e0,-16.26e0,0.52e0,-1.3e0,-0.36e0,19.37e0,0.43e0,-0.01e0,
        0.26e0,-512.37e0,695.54e0,-1.47e0,-1.4e0,15.55e0,11.46e0,
        -0.16e0,0.03e0,0.15e0,-0.17e0,0.01e0,0.01e0,506.65e0,
        643.75e0,2.54e0,-2.62e0,14.4e0,-11.33e0,-0.77e0,-0.06e0,-0.15e0,
        -0.16e0,0e0,0.01e0,664.57e0,16.81e0,-0.4e0,1e0,0.38e0,
        -14.86e0,-3.71e0,-0.09e0,-0.2e0,405.91e0,522.11e0,0.99e0,-1.5e0,
        11.67e0,-9.08e0,-0.25e0,-0.02e0,-0.12e0,-0.13e0,-305.78e0,
        326.6e0,1.75e0,1.9e0,7.3e0,6.84e0,0.2e0,-0.04e0,300.99e0,
        -325.03e0,-0.44e0,-0.5e0,-7.27e0,-6.73e0,-1.01e0,0.01e0,0e0,
        0.08e0,0e0,0.02e0,438.51e0,10.47e0,-0.56e0,-0.2e0,0.24e0,
        -9.81e0,-0.24e0,0.01e0,-0.13e0,-264.02e0,335.24e0,0.99e0,1.4e0,
        7.49e0,5.9e0,-0.27e0,-0.02e0,284.09e0,307.03e0,0.32e0,-0.4e0,
        6.87e0,-6.35e0,-0.99e0,-0.01e0,-250.54e0,327.11e0,0.08e0,0.4e0,
        7.31e0,5.6e0,-0.3e0,230.72e0,-304.46e0,0.08e0,-0.1e0,-6.81e0,
        -5.16e0,0.27e0,229.78e0,304.17e0,-0.6e0,0.5e0,6.8e0,-5.14e0,
        0.33e0,0.01e0,256.3e0,-276.81e0,-0.28e0,-0.4e0,-6.19e0,-5.73e0,
        -0.14e0,0.01e0,-212.82e0,269.45e0,0.84e0,1.2e0,6.02e0,4.76e0,
        0.14e0,-0.02e0,196.64e0,272.05e0,-0.84e0,0.9e0,6.08e0,-4.4e0,
        0.35e0,0.02e0,188.95e0,272.22e0,-0.12e0,0.3e0,6.09e0,-4.22e0,
        0.34e0,-292.37e0,-5.1e0,-0.32e0,-0.4e0,-0.11e0,6.54e0,0.14e0,
        0.01e0,161.79e0,-220.67e0,0.24e0,0.1e0,-4.93e0,-3.62e0,-0.08e0,
        261.54e0,-19.94e0,-0.95e0,0.2e0,-0.45e0,-5.85e0,-0.13e0,0.02e0,
        142.16e0,-190.79e0,0.2e0,0.1e0,-4.27e0,-3.18e0,-0.07e0,187.95e0,
        -4.11e0,-0.24e0,0.3e0,-0.09e0,-4.2e0,-0.09e0,0.01e0,0e0,
        0e0,-79.08e0,167.9e0,0.04e0,0e0,3.75e0,1.77e0,121.98e0,
        131.04e0,-0.08e0,0.1e0,2.93e0,-2.73e0,-0.06e0,-172.95e0,
        -8.11e0,-0.4e0,-0.2e0,-0.18e0,3.87e0,0.09e0,0.01e0,
        -160.15e0,-55.3e0,-14.04e0,13.9e0,-1.23e0,3.58e0,0.4e0,0.31e0,
        -115.4e0,123.2e0,0.6e0,0.7e0,2.75e0,2.58e0,0.08e0,-0.01e0,
        -168.26e0,-2e0,0.2e0,-0.2e0,-0.04e0,3.76e0,0.08e0,
        -114.49e0,123.2e0,0.32e0,0.4e0,2.75e0,2.56e0,0.07e0,-0.01e0,
        112.14e0,120.7e0,0.28e0,-0.3e0,2.7e0,-2.51e0,-0.07e0,-0.01e0,
        161.34e0,4.03e0,0.2e0,0.2e0,0.09e0,-3.61e0,-0.08e0,91.31e0,
        126.64e0,-0.4e0,0.4e0,2.83e0,-2.04e0,-0.04e0,0.01e0,105.29e0,
        112.9e0,0.44e0,-0.5e0,2.52e0,-2.35e0,-0.07e0,-0.01e0,98.69e0,
        -106.2e0,-0.28e0,-0.3e0,-2.37e0,-2.21e0,-0.06e0,0.01e0,86.74e0,
        -112.94e0,-0.08e0,-0.2e0,-2.53e0,-1.94e0,-0.05e0,-134.81e0,
        3.51e0,0.2e0,-0.2e0,0.08e0,3.01e0,0.07e0,79.03e0,107.31e0,
        -0.24e0,0.2e0,2.4e0,-1.77e0,-0.04e0,0.01e0,132.81e0,
        -10.77e0,-0.52e0,0.1e0,-0.24e0,-2.97e0,-0.07e0,0.01e0,
        -130.31e0,-0.9e0,0.04e0,0e0,0e0,2.91e0,-78.56e0,85.32e0,
        0e0,0e0,1.91e0,1.76e0,0.04e0,0e0,0e0,-41.53e0,
        89.1e0,0.02e0,0e0,1.99e0,0.93e0,66.03e0,-71e0,-0.2e0,
        -0.2e0,-1.59e0,-1.48e0,-0.04e0,60.5e0,64.7e0,0.36e0,-0.4e0,
        1.45e0,-1.35e0,-0.04e0,-0.01e0,-52.27e0,-70.01e0,0e0,0e0,
        -1.57e0,1.17e0,0.03e0,-52.95e0,66.29e0,0.32e0,0.4e0,1.48e0,
        1.18e0,0.04e0,-0.01e0,51.02e0,67.25e0,0e0,0e0,1.5e0,
        -1.14e0,-0.03e0,-55.66e0,-60.92e0,0.16e0,-0.2e0,-1.36e0,1.24e0,
        0.03e0,-54.81e0,-59.2e0,-0.08e0,0.2e0,-1.32e0,1.23e0,0.03e0,
        51.32e0,-55.6e0,0e0,0e0,-1.24e0,-1.15e0,-0.03e0,48.29e0,
        51.8e0,0.2e0,-0.2e0,1.16e0,-1.08e0,-0.03e0,-45.59e0,-49e0,
        -0.12e0,0.1e0,-1.1e0,1.02e0,0.03e0,40.54e0,-52.69e0,-0.04e0,
        -0.1e0,-1.18e0,-0.91e0,-0.02e0,-40.58e0,-49.51e0,-1e0,1e0,
        -1.11e0,0.91e0,0.04e0,0.02e0,-43.76e0,46.5e0,0.36e0,0.4e0,
        1.04e0,0.98e0,0.03e0,-0.01e0,62.65e0,-5e0,-0.24e0,0e0,
        -0.11e0,-1.4e0,-0.03e0,0.01e0,-38.57e0,49.59e0,0.08e0,0.1e0,
        1.11e0,0.86e0,0.02e0,-33.22e0,-44.04e0,0.08e0,-0.1e0,-0.98e0,
        0.74e0,0.02e0,37.15e0,-39.9e0,-0.12e0,-0.1e0,-0.89e0,-0.83e0,
        -0.02e0,36.68e0,-39.5e0,-0.04e0,-0.1e0,-0.88e0,-0.82e0,-0.02e0,
        -53.22e0,-3.91e0,-0.2e0,0e0,-0.09e0,1.19e0,0.03e0,32.43e0,
        -42.19e0,-0.04e0,-0.1e0,-0.94e0,-0.73e0,-0.02e0,-51e0,-2.3e0,
        -0.12e0,-0.1e0,0e0,1.14e0,-29.53e0,-39.11e0,0.04e0,0e0,
        -0.87e0,0.66e0,0.02e0,28.5e0,-38.92e0,-0.08e0,-0.1e0,-0.87e0,
        -0.64e0,-0.02e0,26.54e0,36.95e0,-0.12e0,0.1e0,0.83e0,-0.59e0,
        -0.01e0,26.54e0,34.59e0,0.04e0,-0.1e0,0.77e0,-0.59e0,-0.02e0,
        28.35e0,-32.55e0,-0.16e0,0.2e0,-0.73e0,-0.63e0,-0.01e0,-28e0,
        30.4e0,0e0,0e0,0.68e0,0.63e0,0.01e0,-27.61e0,29.4e0,
        0.2e0,0.2e0,0.66e0,0.62e0,0.02e0,40.33e0,0.4e0,-0.04e0,
        0.1e0,0e0,-0.9e0,-23.28e0,31.61e0,-0.08e0,-0.1e0,0.71e0,
        0.52e0,0.01e0,37.75e0,0.8e0,0.04e0,0.1e0,0e0,-0.84e0,
        23.66e0,25.8e0,0e0,0e0,0.58e0,-0.53e0,-0.01e0,21.01e0,
        -27.91e0,0e0,0e0,-0.62e0,-0.47e0,-0.01e0,-34.81e0,2.89e0,
        0.04e0,0e0,0e0,0.78e0,-23.49e0,-25.31e0,0e0,0e0,
        -0.57e0,0.53e0,0.01e0,-23.47e0,25.2e0,0.16e0,0.2e0,0.56e0,
        0.52e0,0.02e0,19.58e0,27.5e0,-0.12e0,0.1e0,0.62e0,-0.44e0,
        -0.01e0,-22.67e0,-24.4e0,-0.08e0,0.1e0,-0.55e0,0.51e0,0.01e0,
        -19.97e0,25e0,0.12e0,0.2e0,0.56e0,0.45e0,0.01e0,21.28e0,
        -22.8e0,-0.08e0,-0.1e0,-0.51e0,-0.48e0,-0.01e0,-30.47e0,0.91e0,
        0.04e0,0e0,0e0,0.68e0,18.58e0,24e0,0.04e0,-0.1e0,
        0.54e0,-0.42e0,-0.01e0,-18.02e0,24.4e0,-0.04e0,-0.1e0,0.55e0,
        0.4e0,0.01e0,17.74e0,22.5e0,0.08e0,-0.1e0,0.5e0,-0.4e0,
        -0.01e0,-19.41e0,20.7e0,0.08e0,0.1e0,0.46e0,0.43e0,0.01e0,
        -18.64e0,20.11e0,0e0,0e0,0.45e0,0.42e0,0.01e0,-16.75e0,
        21.6e0,0.04e0,0.1e0,0.48e0,0.37e0,0.01e0,-18.42e0,-20e0,
        0e0,0e0,-0.45e0,0.41e0,0.01e0,-26.77e0,1.41e0,0.08e0,
        0e0,0e0,0.6e0,-26.17e0,-0.19e0,0e0,0e0,0e0,
        0.59e0,-15.52e0,20.51e0,0e0,0e0,0.46e0,0.35e0,0.01e0,
        -25.42e0,-1.91e0,-0.08e0,0e0,-0.04e0,0.57e0,0.45e0,-17.42e0,
        18.1e0,0e0,0e0,0.4e0,0.39e0,0.01e0,16.39e0,-17.6e0,
        -0.08e0,-0.1e0,-0.39e0,-0.37e0,-0.01e0,-14.37e0,18.91e0,0e0,
        0e0,0.42e0,0.32e0,0.01e0,23.39e0,-2.4e0,-0.12e0,0e0,
        0e0,-0.52e0,14.32e0,-18.5e0,-0.04e0,-0.1e0,-0.41e0,-0.32e0,
        -0.01e0,15.69e0,17.08e0,0e0,0e0,0.38e0,-0.35e0,-0.01e0,
        -22.99e0,0.5e0,0.04e0,0e0,0e0,0.51e0,0e0,0e0,
        14.47e0,-17.6e0,-0.01e0,0e0,-0.39e0,-0.32e0,-13.33e0,18.4e0,
        -0.04e0,-0.1e0,0.41e0,0.3e0,22.47e0,-0.6e0,-0.04e0,0e0,
        0e0,-0.5e0,-12.78e0,-17.41e0,0.04e0,0e0,-0.39e0,0.29e0,
        0.01e0,-14.1e0,-15.31e0,0.04e0,0e0,-0.34e0,0.32e0,0.01e0,
        11.98e0,16.21e0,-0.04e0,0e0,0.36e0,-0.27e0,-0.01e0,19.65e0,
        -1.9e0,-0.08e0,0e0,0e0,-0.44e0,19.61e0,-1.5e0,-0.08e0,
        0e0,0e0,-0.44e0,13.41e0,-14.3e0,-0.04e0,-0.1e0,-0.32e0,
        -0.3e0,-0.01e0,-13.29e0,14.4e0,0e0,0e0,0.32e0,0.3e0,
        0.01e0,11.14e0,-14.4e0,-0.04e0,0e0,-0.32e0,-0.25e0,-0.01e0,
        12.24e0,-13.38e0,0.04e0,0e0,-0.3e0,-0.27e0,-0.01e0,10.07e0,
        -13.81e0,0.04e0,0e0,-0.31e0,-0.23e0,-0.01e0,10.46e0,13.1e0,
        0.08e0,-0.1e0,0.29e0,-0.23e0,-0.01e0,16.55e0,-1.71e0,-0.08e0,
        0e0,0e0,-0.37e0,9.75e0,-12.8e0,0e0,0e0,-0.29e0,
        -0.22e0,-0.01e0,9.11e0,12.8e0,0e0,0e0,0.29e0,-0.2e0,
        0e0,0e0,-6.44e0,-13.8e0,0e0,0e0,-0.31e0,0.14e0,
        -9.19e0,-12e0,0e0,0e0,-0.27e0,0.21e0,-10.3e0,10.9e0,
        0.08e0,0.1e0,0.24e0,0.23e0,0.01e0,14.92e0,-0.8e0,-0.04e0,
        0e0,0e0,-0.33e0,10.02e0,-10.8e0,0e0,0e0,-0.24e0,
        -0.22e0,-0.01e0,-9.75e0,10.4e0,0.04e0,0e0,0.23e0,0.22e0,
        0.01e0,9.67e0,-10.4e0,-0.04e0,0e0,-0.23e0,-0.22e0,-0.01e0,
        -8.28e0,-11.2e0,0.04e0,0e0,-0.25e0,0.19e0,13.32e0,-1.41e0,
        -0.08e0,0e0,0e0,-0.3e0,8.27e0,10.5e0,0.04e0,0e0,
        0.23e0,-0.19e0,0e0,0e0,13.13e0,0e0,0e0,0e0,
        0e0,-0.29e0,-12.93e0,0.7e0,0.04e0,0e0,0e0,0.29e0,
        7.91e0,-10.2e0,0e0,0e0,-0.23e0,-0.18e0,-7.84e0,-10e0,
        -0.04e0,0e0,-0.22e0,0.18e0,7.44e0,9.6e0,0e0,0e0,
        0.21e0,-0.17e0,-7.64e0,9.4e0,0.08e0,0.1e0,0.21e0,0.17e0,
        0.01e0,-11.38e0,0.6e0,0.04e0,0e0,0e0,0.25e0,-7.48e0,
        8.3e0,0e0,0e0,0.19e0,0.17e0,-10.98e0,-0.2e0,0e0,
        0e0,0e0,0.25e0,10.98e0,0.2e0,0e0,0e0,0e0,
        -0.25e0,7.4e0,-7.9e0,-0.04e0,0e0,-0.18e0,-0.17e0,-6.09e0,
        8.4e0,-0.04e0,0e0,0.19e0,0.14e0,-6.94e0,-7.49e0,0e0,
        0e0,-0.17e0,0.16e0,6.92e0,7.5e0,0.04e0,0e0,0.17e0,
        -0.15e0,6.2e0,8.09e0,0e0,0e0,0.18e0,-0.14e0,-6.12e0,
        7.8e0,0.04e0,0e0,0.17e0,0.14e0,5.85e0,-7.5e0,0e0,
        0e0,-0.17e0,-0.13e0,-6.48e0,6.9e0,0.08e0,0.1e0,0.15e0,
        0.14e0,0.01e0,6.32e0,6.9e0,0e0,0e0,0.15e0,-0.14e0,
        5.61e0,-7.2e0,0e0,0e0,-0.16e0,-0.13e0,9.07e0,0e0,
        0e0,0e0,0e0,-0.2e0,5.25e0,6.9e0,0e0,0e0,
        0.15e0,-0.12e0,-8.47e0,-0.4e0,0e0,0e0,0e0,0.19e0,
        6.32e0,-5.39e0,-1.11e0,1.1e0,-0.12e0,-0.14e0,0.02e0,0.02e0,
        5.73e0,-6.1e0,-0.04e0,0e0,-0.14e0,-0.13e0,4.7e0,6.6e0,
        -0.04e0,0e0,0.15e0,-0.11e0,-4.9e0,-6.4e0,0e0,0e0,
        -0.14e0,0.11e0,-5.33e0,5.6e0,0.04e0,0.1e0,0.13e0,0.12e0,
        0.01e0,-4.81e0,6e0,0.04e0,0e0,0.13e0,0.11e0,5.13e0,
        5.5e0,0.04e0,0e0,0.12e0,-0.11e0,4.5e0,5.9e0,0e0,
        0e0,0.13e0,-0.1e0,-4.22e0,6.1e0,0e0,0e0,0.14e0,
        -4.53e0,5.7e0,0e0,0e0,0.13e0,0.1e0,4.18e0,5.7e0,
        0e0,0e0,0.13e0,-4.75e0,-5.19e0,0e0,0e0,-0.12e0,
        0.11e0,-4.06e0,5.6e0,0e0,0e0,0.13e0,-3.98e0,5.6e0,
        -0.04e0,0e0,0.13e0,4.02e0,-5.4e0,0e0,0e0,-0.12e0,
        4.49e0,-4.9e0,-0.04e0,0e0,-0.11e0,-0.1e0,-3.62e0,-5.4e0,
        -0.16e0,0.2e0,-0.12e0,0e0,0.01e0,4.38e0,4.8e0,0e0,
        0e0,0.11e0,-6.4e0,-0.1e0,0e0,0e0,0e0,0.14e0,
        -3.98e0,5e0,0.04e0,0e0,0.11e0,-3.82e0,-5e0,0e0,
        0e0,-0.11e0,-3.71e0,5.07e0,0e0,0e0,0.11e0,4.14e0,
        4.4e0,0e0,0e0,0.1e0,-6.01e0,-0.5e0,-0.04e0,0e0,
        0e0,0.13e0,-4.04e0,4.39e0,0e0,0e0,0.1e0,3.45e0,
        -4.72e0,0e0,0e0,-0.11e0,3.31e0,4.71e0,0e0,0e0,
        0.11e0,3.26e0,-4.5e0,0e0,0e0,-0.1e0,-3.26e0,-4.5e0,
        0e0,0e0,-0.1e0,-3.34e0,-4.4e0,0e0,0e0,-0.1e0,
        -3.74e0,-4e0,3.7e0,4e0,3.34e0,-4.3e0,3.3e0,-4.3e0,
        -3.66e0,3.9e0,0.04e0,3.66e0,3.9e0,0.04e0,-3.62e0,-3.9e0,
        -3.61e0,3.9e0,-0.2e0,5.3e0,0e0,0e0,0.12e0,3.06e0,
        4.3e0,3.3e0,4e0,0.4e0,0.2e0,3.1e0,4.1e0,-3.06e0,
        3.9e0,-3.3e0,-3.6e0,-3.3e0,3.36e0,0.01e0,3.14e0,3.4e0,
        -4.57e0,-0.2e0,0e0,0e0,0e0,0.1e0,-2.7e0,-3.6e0,
        2.94e0,-3.2e0,-2.9e0,3.2e0,2.47e0,-3.4e0,2.55e0,-3.3e0,
        2.8e0,-3.08e0,2.51e0,3.3e0,-4.1e0,0.3e0,-0.12e0,-0.1e0,
        4.1e0,0.2e0,-2.74e0,3e0,2.46e0,3.23e0,-3.66e0,1.2e0,
        -0.2e0,0.2e0,3.74e0,-0.4e0,-2.51e0,-2.8e0,-3.74e0,2.27e0,
        -2.9e0,0e0,0e0,-2.5e0,2.7e0,-2.51e0,2.6e0,-3.5e0,
        0.2e0,3.38e0,-2.22e0,-2.5e0,3.26e0,-0.4e0,1.95e0,-2.6e0,
        3.22e0,-0.4e0,-0.04e0,-1.79e0,-2.6e0,1.91e0,2.5e0,0.74e0,
        3.05e0,-0.04e0,0.08e0,2.11e0,-2.3e0,-2.11e0,2.2e0,-1.87e0,
        -2.4e0,2.03e0,-2.2e0,-2.03e0,2.2e0,2.98e0,0e0,0e0,
        2.98e0,-1.71e0,2.4e0,2.94e0,-0.1e0,-0.12e0,0.1e0,1.67e0,
        2.4e0,-1.79e0,2.3e0,-1.79e0,2.2e0,-1.67e0,2.2e0,1.79e0,
        -2e0,1.87e0,-1.9e0,1.63e0,-2.1e0,-1.59e0,2.1e0,1.55e0,
        -2.1e0,-1.55e0,2.1e0,-2.59e0,-0.2e0,-1.75e0,-1.9e0,-1.75e0,
        1.9e0,-1.83e0,-1.8e0,1.51e0,2e0,-1.51e0,-2e0,1.71e0,
        1.8e0,1.31e0,2.1e0,-1.43e0,2e0,1.43e0,2e0,-2.43e0,
        -1.51e0,1.9e0,-1.47e0,1.9e0,2.39e0,0.2e0,-2.39e0,1.39e0,
        1.9e0,1.39e0,-1.8e0,1.47e0,-1.6e0,1.47e0,-1.6e0,1.43e0,
        -1.5e0,-1.31e0,1.6e0,1.27e0,-1.6e0,-1.27e0,1.6e0,1.27e0,
        -1.6e0,2.03e0,1.35e0,1.5e0,-1.39e0,-1.4e0,1.95e0,-0.2e0,
        -1.27e0,1.49e0,1.19e0,1.5e0,1.27e0,1.4e0,1.15e0,1.5e0,
        1.87e0,-0.1e0,-1.12e0,-1.5e0,1.87e0,-1.11e0,-1.5e0,-1.11e0,
        -1.5e0,0e0,0e0,1.19e0,1.4e0,1.27e0,-1.3e0,-1.27e0,
        -1.3e0,-1.15e0,1.4e0,-1.23e0,1.3e0,-1.23e0,-1.3e0,1.22e0,
        -1.29e0,1.07e0,-1.4e0,1.75e0,-0.2e0,-1.03e0,-1.4e0,-1.07e0,
        1.2e0,-1.03e0,1.15e0,1.07e0,1.1e0,1.51e0,-1.03e0,1.1e0,
        1.03e0,-1.1e0,0e0,0e0,-1.03e0,-1.1e0,0.91e0,-1.2e0,
        -0.88e0,-1.2e0,-0.88e0,1.2e0,-0.95e0,1.1e0,-0.95e0,-1.1e0,
        1.43e0,-1.39e0,0.95e0,-1e0,-0.95e0,1e0,-0.8e0,1.1e0,
        0.91e0,-1e0,-1.35e0,0.88e0,1e0,-0.83e0,1e0,-0.91e0,
        0.9e0,0.91e0,0.9e0,0.88e0,-0.9e0,-0.76e0,-1e0,-0.76e0,
        1e0,0.76e0,1e0,-0.72e0,1e0,0.84e0,-0.9e0,0.84e0,
        0.9e0,1.23e0,0e0,0e0,-0.52e0,-1.1e0,-0.68e0,1e0,
        1.19e0,-0.2e0,1.19e0,0.76e0,0.9e0,1.15e0,-0.1e0,1.15e0,
        -0.1e0,0.72e0,-0.9e0,-1.15e0,-1.15e0,0.68e0,0.9e0,-0.68e0,
        0.9e0,-1.11e0,0e0,0e0,0.2e0,0.79e0,0.8e0,-1.11e0,
        -0.1e0,0e0,0e0,-0.48e0,-1e0,-0.76e0,-0.8e0,-0.72e0,
        -0.8e0,-1.07e0,-0.1e0,0.64e0,0.8e0,-0.64e0,-0.8e0,0.64e0,
        0.8e0,0.4e0,0.6e0,0.52e0,-0.5e0,-0.6e0,-0.8e0,-0.71e0,
        0.7e0,-0.99e0,0.99e0,0.56e0,0.8e0,-0.56e0,0.8e0,0.68e0,
        -0.7e0,0.68e0,0.7e0,-0.95e0,-0.64e0,0.7e0,0.64e0,0.7e0,
        -0.6e0,0.7e0,-0.6e0,-0.7e0,-0.91e0,-0.1e0,-0.51e0,0.76e0,
        -0.91e0,-0.56e0,0.7e0,0.88e0,0.88e0,-0.63e0,-0.6e0,0.55e0,
        -0.6e0,-0.8e0,0.8e0,-0.8e0,-0.52e0,0.6e0,0.52e0,0.6e0,
        0.52e0,-0.6e0,-0.48e0,0.6e0,0.48e0,0.6e0,0.48e0,0.6e0,
        -0.76e0,0.44e0,-0.6e0,0.52e0,-0.5e0,-0.52e0,0.5e0,0.4e0,
        0.6e0,-0.4e0,-0.6e0,0.4e0,-0.6e0,0.72e0,-0.72e0,-0.51e0,
        -0.5e0,-0.48e0,0.5e0,0.48e0,-0.5e0,-0.48e0,0.5e0,-0.48e0,
        0.5e0,0.48e0,-0.5e0,-0.48e0,-0.5e0,-0.68e0,-0.68e0,0.44e0,
        0.5e0,-0.64e0,-0.1e0,-0.64e0,-0.1e0,-0.4e0,0.5e0,0.4e0,
        0.5e0,0.4e0,0.5e0,0e0,0e0,-0.4e0,-0.5e0,-0.36e0,
        -0.5e0,0.36e0,-0.5e0,0.6e0,-0.6e0,0.4e0,-0.4e0,0.4e0,
        0.4e0,-0.4e0,0.4e0,-0.4e0,0.4e0,-0.56e0,-0.56e0,0.36e0,
        -0.4e0,-0.36e0,0.4e0,0.36e0,-0.4e0,-0.36e0,-0.4e0,0.36e0,
        0.4e0,0.36e0,0.4e0,-0.52e0,0.52e0,0.52e0,0.32e0,0.4e0,
        -0.32e0,0.4e0,-0.32e0,0.4e0,-0.32e0,0.4e0,0.32e0,-0.4e0,
        -0.32e0,-0.4e0,0.32e0,-0.4e0,0.28e0,-0.4e0,-0.28e0,0.4e0,
        0.28e0,-0.4e0,0.28e0,0.4e0,0.48e0,-0.48e0,0.48e0,0.36e0,
        -0.3e0,-0.36e0,-0.3e0,0e0,0e0,0.2e0,0.4e0,-0.44e0,
        0.44e0,-0.44e0,-0.44e0,-0.44e0,-0.44e0,0.32e0,-0.3e0,0.32e0,
        0.3e0,0.24e0,0.3e0,-0.12e0,-0.1e0,-0.28e0,0.3e0,0.28e0,
        0.3e0,0.28e0,0.3e0,0.28e0,-0.3e0,0.28e0,-0.3e0,0.28e0,
        -0.3e0,0.28e0,0.3e0,-0.28e0,0.3e0,0.4e0,0.4e0,-0.24e0,
        0.3e0,0.24e0,-0.3e0,0.24e0,-0.3e0,-0.24e0,-0.3e0,0.24e0,
        0.3e0,0.24e0,-0.3e0,-0.24e0,0.3e0,0.24e0,-0.3e0,-0.24e0,
        -0.3e0,0.24e0,-0.3e0,0.24e0,0.3e0,-0.24e0,0.3e0,-0.24e0,
        0.3e0,0.2e0,-0.3e0,0.2e0,-0.3e0,0.2e0,-0.3e0,0.2e0,
        0.3e0,0.2e0,-0.3e0,0.2e0,-0.3e0,0.2e0,0.3e0,0.2e0,
        0.3e0,-0.2e0,-0.3e0,0.2e0,-0.3e0,0.2e0,-0.3e0,-0.36e0,
        -0.36e0,-0.36e0,-0.04e0,0.3e0,0.12e0,-0.1e0,-0.32e0,-0.24e0,
        0.2e0,0.24e0,0.2e0,0.2e0,-0.2e0,-0.2e0,-0.2e0,-0.2e0,
        -0.2e0,0.2e0,0.2e0,0.2e0,-0.2e0,0.2e0,0.2e0,0.2e0,
        0.2e0,-0.2e0,-0.2e0,0e0,0e0,-0.2e0,-0.2e0,-0.2e0,
        0.2e0,-0.2e0,0.2e0,0.2e0,-0.2e0,-0.2e0,-0.2e0,0.2e0,
        0.2e0,0.2e0,0.2e0,0.2e0,-0.2e0,0.2e0,-0.2e0,0.28e0,
        0.28e0,0.28e0,0.28e0,0.28e0,0.28e0,-0.28e0,0.28e0,0.12e0,
        0e0,0.24e0,0.16e0,-0.2e0,0.16e0,-0.2e0,0.16e0,-0.2e0,
        0.16e0,0.2e0,-0.16e0,0.2e0,0.16e0,0.2e0,-0.16e0,0.2e0,
        -0.16e0,0.2e0,-0.16e0,0.2e0,0.16e0,-0.2e0,0.16e0,0.2e0,
        0.16e0,-0.2e0,-0.16e0,0.2e0,-0.16e0,-0.2e0,-0.16e0,0.2e0,
        0.16e0,0.2e0,0.16e0,-0.2e0,0.16e0,-0.2e0,0.16e0,0.2e0,
        0.16e0,0.2e0,0.16e0,0.2e0,-0.16e0,-0.2e0,0.16e0,0.2e0,
        -0.16e0,0.2e0,0.16e0,0.2e0,-0.16e0,-0.2e0,0.16e0,-0.2e0,
        0.16e0,-0.2e0,-0.16e0,-0.2e0,0.24e0,-0.24e0,-0.24e0,0.24e0,
        0.24e0,0.12e0,0.2e0,0.12e0,0.2e0,-0.12e0,-0.2e0,0.12e0,
        -0.2e0,0.12e0,-0.2e0,-0.12e0,0.2e0,-0.12e0,0.2e0,-0.12e0,
        -0.2e0,0.12e0,0.2e0,0.12e0,0.2e0,0.12e0,-0.2e0,-0.12e0,
        0.2e0,0.12e0,-0.2e0,-0.12e0,0.2e0,0.12e0,0.2e0,0e0,
        0e0,-0.12e0,0.2e0,-0.12e0,0.2e0,0.12e0,-0.2e0,-0.12e0,
        0.2e0,0.12e0,0.2e0,0e0,-0.21e0,-0.2e0,0e0,0e0,
        0.2e0,-0.2e0,-0.2e0,-0.2e0,0.2e0,-0.16e0,-0.1e0,0e0,
        0.17e0,0.16e0,0.16e0,0.16e0,0.16e0,-0.16e0,0.16e0,0.16e0,
        -0.16e0,0.16e0,-0.16e0,0.16e0,0.12e0,0.1e0,0.12e0,-0.1e0,
        -0.12e0,0.1e0,-0.12e0,0.1e0,0.12e0,-0.1e0,-0.12e0,0.12e0,
        -0.12e0,0.12e0,-0.12e0,0.12e0,-0.12e0,-0.12e0,-0.12e0,-0.12e0,
        -0.12e0,-0.12e0,-0.12e0,0.12e0,0.12e0,0.12e0,0.12e0,-0.12e0,
        -0.12e0,0.12e0,0.12e0,0.12e0,-0.12e0,0.12e0,-0.12e0,-0.12e0,
        -0.12e0,0.12e0,-0.12e0,-0.12e0,0.12e0,0e0,0.11e0,0.11e0,
        -122.67e0,164.7e0,203.78e0,273.5e0,3.58e0,2.74e0,6.18e0,-4.56e0,
        0e0,-0.04e0,0e0,-0.07e0,57.44e0,-77.1e0,95.82e0,128.6e0,
        -1.77e0,-1.28e0,2.85e0,-2.14e0,82.14e0,89.5e0,0e0,0e0,
        2e0,-1.84e0,-0.04e0,47.73e0,-64.1e0,23.79e0,31.9e0,-1.45e0,
        -1.07e0,0.69e0,-0.53e0,-46.38e0,50.5e0,0e0,0e0,1.13e0,
        1.04e0,0.02e0,-18.38e0,0e0,63.8e0,0e0,0e0,0.41e0,
        0e0,-1.43e0,59.07e0,0e0,0e0,0e0,0e0,-1.32e0,
        57.28e0,0e0,0e0,0e0,0e0,-1.28e0,-48.65e0,0e0,
        -1.15e0,0e0,0e0,1.09e0,0e0,0.03e0,-18.3e0,24.6e0,
        -17.3e0,-23.2e0,0.56e0,0.41e0,-0.51e0,0.39e0,-16.91e0,26.9e0,
        8.43e0,13.3e0,0.6e0,0.38e0,0.31e0,-0.19e0,1.23e0,-1.7e0,
        -19.13e0,-25.7e0,-0.03e0,-0.03e0,-0.58e0,0.43e0,-0.72e0,0.9e0,
        -17.34e0,-23.3e0,0.03e0,0.02e0,-0.52e0,0.39e0,-19.49e0,-21.3e0,
        0e0,0e0,-0.48e0,0.44e0,0.01e0,20.57e0,-20.1e0,0.64e0,
        0.7e0,-0.45e0,-0.46e0,0e0,-0.01e0,4.89e0,5.9e0,-16.55e0,
        19.9e0,0.14e0,-0.11e0,0.44e0,0.37e0,18.22e0,19.8e0,0e0,
        0e0,0.44e0,-0.41e0,-0.01e0,4.89e0,-5.3e0,-16.51e0,-18e0,
        -0.11e0,-0.11e0,-0.41e0,0.37e0,-17.86e0,0e0,17.1e0,0e0,
        0e0,0.4e0,0e0,-0.38e0,0.32e0,0e0,24.42e0,0e0,
        0e0,-0.01e0,0e0,-0.55e0,-23.79e0,0e0,0e0,0e0,
        0e0,0.53e0,14.72e0,-16e0,-0.32e0,0e0,-0.36e0,-0.33e0,
        -0.01e0,0.01e0,3.34e0,-4.5e0,11.86e0,15.9e0,-0.11e0,-0.07e0,
        0.35e0,-0.27e0,-3.26e0,4.4e0,11.62e0,15.6e0,0.09e0,0.07e0,
        0.35e0,-0.26e0,-19.53e0,0e0,5.09e0,0e0,0e0,0.44e0,
        0e0,-0.11e0,-13.48e0,14.7e0,0e0,0e0,0.33e0,0.3e0,
        0.01e0,10.86e0,-14.6e0,3.18e0,4.3e0,-0.33e0,-0.24e0,0.09e0,
        -0.07e0,-11.3e0,-15.1e0,0e0,0e0,-0.34e0,0.25e0,0.01e0,
        2.03e0,-2.7e0,10.82e0,14.5e0,-0.07e0,-0.05e0,0.32e0,-0.24e0,
        17.46e0,0e0,0e0,0e0,0e0,-0.39e0,16.43e0,0e0,
        0.52e0,0e0,0e0,-0.37e0,0e0,-0.01e0,9.35e0,0e0,
        13.29e0,0e0,0e0,-0.21e0,0e0,-0.3e0,-10.42e0,11.4e0,
        0e0,0e0,0.25e0,0.23e0,0.01e0,0.44e0,0.5e0,-10.38e0,
        11.3e0,0.02e0,-0.01e0,0.25e0,0.23e0,-14.64e0,0e0,0e0,
        0e0,0e0,0.33e0,0.56e0,0.8e0,-8.67e0,11.7e0,0.02e0,
        -0.01e0,0.26e0,0.19e0,13.88e0,0e0,-2.47e0,0e0,0e0,
        -0.31e0,0e0,0.06e0,-1.99e0,2.7e0,7.72e0,10.3e0,0.06e0,
        0.04e0,0.23e0,-0.17e0,-0.2e0,0e0,13.05e0,0e0,0e0,
        0e0,0e0,-0.29e0,6.92e0,-9.3e0,3.34e0,4.5e0,-0.21e0,
        -0.15e0,0.1e0,-0.07e0,-6.6e0,0e0,10.7e0,0e0,0e0,
        0.15e0,0e0,-0.24e0,-8.04e0,-8.7e0,0e0,0e0,-0.19e0,
        0.18e0,-10.58e0,0e0,-3.1e0,0e0,0e0,0.24e0,0e0,
        0.07e0,-7.32e0,8e0,-0.12e0,-0.1e0,0.18e0,0.16e0,1.63e0,
        1.7e0,6.96e0,-7.6e0,0.03e0,-0.04e0,-0.17e0,-0.16e0,-3.62e0,
        0e0,9.86e0,0e0,0e0,0.08e0,0e0,-0.22e0,0.2e0,
        -0.2e0,-6.88e0,-7.5e0,0e0,0e0,-0.17e0,0.15e0,-8.99e0,
        0e0,4.02e0,0e0,0e0,0.2e0,0e0,-0.09e0,-1.07e0,
        1.4e0,-5.69e0,-7.7e0,0.03e0,0.02e0,-0.17e0,0.13e0,6.48e0,
        -7.2e0,-0.48e0,-0.5e0,-0.16e0,-0.14e0,-0.01e0,0.01e0,5.57e0,
        -7.5e0,1.07e0,1.4e0,-0.17e0,-0.12e0,0.03e0,-0.02e0,8.71e0,
        0e0,3.54e0,0e0,0e0,-0.19e0,0e0,-0.08e0,0.4e0,
        0e0,9.27e0,0e0,0e0,-0.01e0,0e0,-0.21e0,-6.13e0,
        6.7e0,-1.19e0,-1.3e0,0.15e0,0.14e0,-0.03e0,0.03e0,5.21e0,
        -5.7e0,-2.51e0,-2.6e0,-0.13e0,-0.12e0,-0.06e0,0.06e0,5.69e0,
        -6.2e0,-0.12e0,-0.1e0,-0.14e0,-0.13e0,-0.01e0,2.03e0,-2.7e0,
        4.53e0,6.1e0,-0.06e0,-0.05e0,0.14e0,-0.1e0,5.01e0,5.5e0,
        -2.51e0,2.7e0,0.12e0,-0.11e0,0.06e0,0.06e0,-1.91e0,2.6e0,
        -4.38e0,-5.9e0,0.06e0,0.04e0,-0.13e0,0.1e0,4.65e0,-6.3e0,
        0e0,0e0,-0.14e0,-0.1e0,-5.29e0,5.7e0,0e0,0e0,
        0.13e0,0.12e0,-2.23e0,-4e0,-4.65e0,4.2e0,-0.09e0,0.05e0,
        0.1e0,0.1e0,-4.53e0,6.1e0,0e0,0e0,0.14e0,0.1e0,
        2.47e0,2.7e0,-4.46e0,4.9e0,0.06e0,-0.06e0,0.11e0,0.1e0,
        -5.05e0,5.5e0,0.84e0,0.9e0,0.12e0,0.11e0,0.02e0,-0.02e0,
        4.97e0,-5.4e0,-1.71e0,0e0,-0.12e0,-0.11e0,0e0,0.04e0,
        -0.99e0,-1.3e0,4.22e0,-5.7e0,-0.03e0,0.02e0,-0.13e0,-0.09e0,
        0.99e0,1.4e0,4.22e0,-5.6e0,0.03e0,-0.02e0,-0.13e0,-0.09e0,
        -4.69e0,-5.2e0,0e0,0e0,-0.12e0,0.1e0,-3.42e0,0e0,
        6.09e0,0e0,0e0,0.08e0,0e0,-0.14e0,-4.65e0,-5.1e0,
        0e0,0e0,-0.11e0,0.1e0,0e0,0e0,-4.53e0,-5e0,
        0e0,0e0,-0.11e0,0.1e0,-2.43e0,-2.7e0,-3.82e0,4.2e0,
        -0.06e0,0.05e0,0.1e0,0.09e0,0e0,0e0,-4.53e0,4.9e0,
        0e0,0e0,0.11e0,0.1e0,-4.49e0,-4.9e0,0e0,0e0,
        -0.11e0,0.1e0,2.67e0,-2.9e0,-3.62e0,-3.9e0,-0.06e0,-0.06e0,
        -0.09e0,0.08e0,3.94e0,-5.3e0,0e0,0e0,-0.12e0,-3.38e0,
        3.7e0,-2.78e0,-3.1e0,0.08e0,0.08e0,-0.07e0,0.06e0,3.18e0,
        -3.5e0,-2.82e0,-3.1e0,-0.08e0,-0.07e0,-0.07e0,0.06e0,-5.77e0,
        0e0,1.87e0,0e0,0e0,0.13e0,0e0,-0.04e0,3.54e0,
        -4.8e0,-0.64e0,-0.9e0,-0.11e0,0e0,-0.02e0,-3.5e0,-4.7e0,
        0.68e0,-0.9e0,-0.11e0,0e0,-0.02e0,5.49e0,0e0,0e0,
        0e0,0e0,-0.12e0,1.83e0,-2.5e0,2.63e0,3.5e0,-0.06e0,
        0e0,0.08e0,3.02e0,-4.1e0,0.68e0,0.9e0,-0.09e0,0e0,
        0.02e0,0e0,0e0,5.21e0,0e0,0e0,0e0,0e0,
        -0.12e0,-3.54e0,3.8e0,2.7e0,3.6e0,-1.35e0,1.8e0,0.08e0,
        0e0,0.04e0,-2.9e0,3.9e0,0.68e0,0.9e0,0.09e0,0e0,
        0.02e0,0.8e0,-1.1e0,-2.78e0,-3.7e0,-0.02e0,0e0,-0.08e0,
        4.1e0,0e0,-2.39e0,0e0,0e0,-0.09e0,0e0,0.05e0,
        -1.59e0,2.1e0,2.27e0,3e0,0.05e0,0e0,0.07e0,-2.63e0,
        3.5e0,-0.48e0,-0.6e0,-2.94e0,-3.2e0,-2.94e0,3.2e0,2.27e0,
        -3e0,-1.11e0,-1.5e0,-0.07e0,0e0,-0.03e0,-0.56e0,-0.8e0,
        -2.35e0,3.1e0,0e0,-0.6e0,-3.42e0,1.9e0,-0.12e0,-0.1e0,
        2.63e0,-2.9e0,2.51e0,2.8e0,-0.64e0,0.7e0,-0.48e0,-0.6e0,
        2.19e0,-2.9e0,0.24e0,-0.3e0,2.15e0,2.9e0,2.15e0,-2.9e0,
        0.52e0,0.7e0,2.07e0,-2.8e0,-3.1e0,0e0,1.79e0,0e0,
        0e0,0.07e0,0e0,-0.04e0,0.88e0,0e0,-3.46e0,2.11e0,
        2.8e0,-0.36e0,0.5e0,3.54e0,-0.2e0,-3.5e0,-1.39e0,1.5e0,
        -1.91e0,-2.1e0,-1.47e0,2e0,1.39e0,1.9e0,2.07e0,-2.3e0,
        0.91e0,1e0,1.99e0,-2.7e0,3.3e0,0e0,0.6e0,-0.44e0,
        -0.7e0,-1.95e0,2.6e0,2.15e0,-2.4e0,-0.6e0,-0.7e0,3.3e0,
        0.84e0,0e0,-3.1e0,-3.1e0,0e0,-0.72e0,-0.32e0,0.4e0,
        -1.87e0,-2.5e0,1.87e0,-2.5e0,0.32e0,0.4e0,-0.24e0,0.3e0,
        -1.87e0,-2.5e0,-0.24e0,-0.3e0,1.87e0,-2.5e0,-2.7e0,0e0,
        1.55e0,2.03e0,2.2e0,-2.98e0,-1.99e0,-2.2e0,0.12e0,-0.1e0,
        -0.4e0,0.5e0,1.59e0,2.1e0,0e0,0e0,-1.79e0,2e0,
        -1.03e0,1.4e0,-1.15e0,-1.6e0,0.32e0,0.5e0,1.39e0,-1.9e0,
        2.35e0,-1.27e0,1.7e0,0.6e0,0.8e0,-0.32e0,-0.4e0,1.35e0,
        -1.8e0,0.44e0,0e0,2.23e0,-0.84e0,0.9e0,-1.27e0,-1.4e0,
        -1.47e0,1.6e0,-0.28e0,-0.3e0,-0.28e0,0.4e0,-1.27e0,-1.7e0,
        0.28e0,-0.4e0,-1.43e0,-1.5e0,0e0,0e0,-1.27e0,-1.7e0,
        2.11e0,-0.32e0,-0.4e0,-1.23e0,1.6e0,1.19e0,-1.3e0,-0.72e0,
        -0.8e0,0.72e0,-0.8e0,-1.15e0,-1.3e0,-1.35e0,-1.5e0,-1.19e0,
        -1.6e0,-0.12e0,0.2e0,1.79e0,0e0,-0.88e0,-0.28e0,0.4e0,
        1.11e0,1.5e0,-1.83e0,0e0,0.56e0,-0.12e0,0.1e0,-1.27e0,
        -1.4e0,0e0,0e0,1.15e0,1.5e0,-0.12e0,0.2e0,1.11e0,
        1.5e0,0.36e0,-0.5e0,-1.07e0,-1.4e0,-1.11e0,1.5e0,1.67e0,
        0e0,0.8e0,-1.11e0,0e0,1.43e0,1.23e0,-1.3e0,-0.24e0,
        -1.19e0,-1.3e0,-0.24e0,0.2e0,-0.44e0,-0.9e0,-0.95e0,1.1e0,
        1.07e0,-1.4e0,1.15e0,-1.3e0,1.03e0,-1.1e0,-0.56e0,-0.6e0,
        -0.68e0,0.9e0,-0.76e0,-1e0,-0.24e0,-0.3e0,0.95e0,-1.3e0,
        0.56e0,0.7e0,0.84e0,-1.1e0,-0.56e0,0e0,-1.55e0,0.91e0,
        -1.3e0,0.28e0,0.3e0,0.16e0,-0.2e0,0.95e0,1.3e0,0.4e0,
        -0.5e0,-0.88e0,-1.2e0,0.95e0,-1.1e0,-0.48e0,-0.5e0,0e0,
        0e0,-1.07e0,1.2e0,0.44e0,-0.5e0,0.95e0,1.1e0,0e0,
        0e0,0.92e0,-1.3e0,0.95e0,1e0,-0.52e0,0.6e0,1.59e0,
        0.24e0,-0.4e0,0.91e0,1.2e0,0.84e0,-1.1e0,-0.44e0,-0.6e0,
        0.84e0,1.1e0,-0.44e0,0.6e0,-0.44e0,0.6e0,-0.84e0,-1.1e0,
        -0.8e0,0e0,1.35e0,0.76e0,0.2e0,-0.91e0,-1e0,0.2e0,
        -0.3e0,-0.91e0,-1.2e0,-0.95e0,1e0,-0.48e0,-0.5e0,0.88e0,
        1e0,0.48e0,-0.5e0,-0.95e0,-1.1e0,0.2e0,-0.2e0,-0.99e0,
        1.1e0,-0.84e0,1.1e0,-0.24e0,-0.3e0,0.2e0,-0.3e0,0.84e0,
        1.1e0,-1.39e0,0e0,-0.28e0,-0.16e0,0.2e0,0.84e0,1.1e0,
        0e0,0e0,1.39e0,0e0,0e0,-0.95e0,1e0,1.35e0,
        -0.99e0,0e0,0.88e0,-0.52e0,0e0,-1.19e0,0.2e0,0.2e0,
        0.76e0,-1e0,0e0,0e0,0.76e0,1e0,0e0,0e0,
        0.76e0,1e0,-0.76e0,1e0,0e0,0e0,1.23e0,0.76e0,
        0.8e0,-0.32e0,0.4e0,-0.72e0,0.8e0,-0.4e0,-0.4e0,0e0,
        0e0,-0.8e0,-0.9e0,-0.68e0,0.9e0,-0.16e0,-0.2e0,-0.16e0,
        -0.2e0,0.68e0,-0.9e0,-0.36e0,0.5e0,-0.56e0,-0.8e0,0.72e0,
        -0.9e0,0.44e0,-0.6e0,-0.48e0,-0.7e0,-0.16e0,0e0,-1.11e0,
        0.32e0,0e0,-1.07e0,0.6e0,-0.8e0,-0.28e0,-0.4e0,-0.64e0,
        0e0,0.91e0,1.11e0,0.64e0,-0.9e0,0.76e0,-0.8e0,0e0,
        0e0,-0.76e0,-0.8e0,1.03e0,0e0,-0.36e0,-0.64e0,-0.7e0,
        0.36e0,-0.4e0,1.07e0,0.36e0,-0.5e0,-0.52e0,-0.7e0,0.6e0,
        0e0,0.88e0,0.95e0,0e0,0.48e0,0.16e0,-0.2e0,0.6e0,
        0.8e0,0.16e0,-0.2e0,-0.6e0,-0.8e0,0e0,-1e0,0.12e0,
        0.2e0,0.16e0,-0.2e0,0.68e0,0.7e0,0.59e0,-0.8e0,-0.99e0,
        -0.56e0,-0.6e0,0.36e0,-0.4e0,-0.68e0,-0.7e0,-0.68e0,-0.7e0,
        -0.36e0,-0.5e0,-0.44e0,0.6e0,0.64e0,0.7e0,-0.12e0,0.1e0,
        -0.52e0,0.6e0,0.36e0,0.4e0,0e0,0e0,0.95e0,-0.84e0,
        0e0,0.44e0,0.56e0,0.6e0,0.32e0,-0.3e0,0e0,0e0,
        0.6e0,0.7e0,0e0,0e0,0.6e0,0.7e0,-0.12e0,-0.2e0,
        0.52e0,-0.7e0,0e0,0e0,0.56e0,0.7e0,-0.12e0,0.1e0,
        -0.52e0,-0.7e0,0e0,0e0,0.88e0,-0.76e0,0e0,-0.44e0,
        0e0,0e0,-0.52e0,-0.7e0,0.52e0,-0.7e0,0.36e0,-0.4e0,
        -0.44e0,-0.5e0,0e0,0e0,0.6e0,0.6e0,0.84e0,0e0,
        0.12e0,-0.24e0,0e0,0.8e0,-0.56e0,0.6e0,-0.32e0,-0.3e0,
        0.48e0,-0.5e0,0.28e0,-0.3e0,-0.48e0,-0.5e0,0.12e0,0.2e0,
        0.48e0,-0.6e0,0.48e0,0.6e0,-0.12e0,0.2e0,0.24e0,0e0,
        0.76e0,-0.52e0,-0.6e0,-0.52e0,0.6e0,0.48e0,-0.5e0,-0.24e0,
        -0.3e0,0.12e0,-0.1e0,0.48e0,0.6e0,0.52e0,-0.2e0,0.36e0,
        0.4e0,-0.44e0,0.5e0,-0.24e0,-0.3e0,-0.48e0,-0.6e0,-0.44e0,
        -0.6e0,-0.12e0,0.1e0,0.76e0,0.76e0,0.2e0,-0.2e0,0.48e0,
        0.5e0,0.4e0,-0.5e0,-0.24e0,-0.3e0,0.44e0,-0.6e0,0.44e0,
        -0.6e0,0.36e0,0e0,-0.64e0,0.72e0,0e0,-0.12e0,0e0,
        -0.1e0,-0.4e0,-0.6e0,-0.2e0,-0.2e0,-0.44e0,0.5e0,-0.44e0,
        0.5e0,0.2e0,0.2e0,-0.44e0,-0.5e0,0.2e0,-0.2e0,-0.2e0,
        0.2e0,-0.44e0,-0.5e0,0.64e0,0e0,0.32e0,-0.36e0,0.5e0,
        -0.2e0,-0.3e0,0.12e0,-0.1e0,0.48e0,0.5e0,-0.12e0,0.3e0,
        -0.36e0,-0.5e0,0e0,0e0,0.48e0,0.5e0,-0.48e0,0.5e0,
        0.68e0,0e0,-0.12e0,0.56e0,-0.4e0,0.44e0,-0.5e0,-0.12e0,
        -0.1e0,0.24e0,0.3e0,-0.4e0,0.4e0,0.64e0,0e0,-0.24e0,
        0.64e0,0e0,-0.2e0,0e0,0e0,0.44e0,-0.5e0,0.44e0,
        0.5e0,-0.12e0,0.2e0,-0.36e0,-0.5e0,0.12e0,0e0,0.64e0,
        -0.4e0,0.5e0,0e0,0.1e0,0e0,0e0,-0.4e0,0.5e0,
        0e0,0e0,-0.4e0,-0.5e0,0.56e0,0e0,0.28e0,0e0,
        0.1e0,0.36e0,0.5e0,0e0,-0.1e0,0.36e0,-0.5e0,0.36e0,
        0.5e0,0e0,-0.1e0,0.24e0,-0.2e0,-0.36e0,-0.4e0,0.16e0,
        0.2e0,0.4e0,-0.4e0,0e0,0e0,-0.36e0,-0.5e0,-0.36e0,
        -0.5e0,-0.32e0,-0.5e0,-0.12e0,0.1e0,0.2e0,0.2e0,-0.36e0,
        0.4e0,-0.6e0,0.6e0,0.28e0,0e0,0.52e0,0.12e0,-0.1e0,
        0.4e0,0.4e0,0e0,-0.5e0,0.2e0,-0.2e0,-0.32e0,0.4e0,
        0.16e0,0.2e0,-0.16e0,0.2e0,0.32e0,0.4e0,0.56e0,0e0,
        -0.12e0,0.32e0,-0.4e0,-0.16e0,-0.2e0,0e0,0e0,0.4e0,
        0.4e0,-0.4e0,-0.4e0,-0.4e0,0.4e0,-0.36e0,0.4e0,0.12e0,
        0.1e0,0e0,0.1e0,0.36e0,0.4e0,0e0,-0.1e0,0.36e0,
        0.4e0,-0.36e0,0.4e0,0e0,0.1e0,0.32e0,0e0,0.44e0,
        0.12e0,0.2e0,0.28e0,-0.4e0,0e0,0e0,0.36e0,0.4e0,
        0.32e0,-0.4e0,-0.16e0,0.12e0,0.1e0,0.32e0,-0.4e0,0.2e0,
        0.3e0,-0.24e0,0.3e0,0e0,0.1e0,0.32e0,0.4e0,0e0,
        -0.1e0,-0.32e0,-0.4e0,-0.32e0,0.4e0,0e0,0.1e0,-0.52e0,
        -0.52e0,0.52e0,0.32e0,-0.4e0,0e0,0e0,0.32e0,0.4e0,
        0.32e0,-0.4e0,0e0,0e0,-0.32e0,-0.4e0,-0.32e0,0.4e0,
        0.32e0,0.4e0,0e0,0e0,0.32e0,0.4e0,0e0,0e0,
        -0.32e0,-0.4e0,0e0,0e0,0.32e0,0.4e0,0.16e0,0.2e0,
        0.32e0,-0.3e0,-0.16e0,0e0,-0.48e0,-0.2e0,0.2e0,-0.28e0,
        -0.3e0,0.28e0,-0.4e0,0e0,0e0,0.28e0,-0.4e0,0e0,
        0e0,0.28e0,-0.4e0,0e0,0e0,-0.28e0,-0.4e0,0.28e0,
        0.4e0,-0.28e0,-0.4e0,-0.48e0,-0.2e0,0.2e0,0.24e0,0.3e0,
        0.44e0,0e0,0.16e0,0.24e0,0.3e0,0.16e0,-0.2e0,0.24e0,
        0.3e0,-0.12e0,0.2e0,0.2e0,0.3e0,-0.16e0,0.2e0,0e0,
        0e0,0.44e0,-0.32e0,0.3e0,0.24e0,0e0,-0.36e0,0.36e0,
        0e0,0.24e0,0.12e0,-0.2e0,0.2e0,0.3e0,-0.12e0,0e0,
        -0.28e0,0.3e0,-0.24e0,0.3e0,0.12e0,0.1e0,-0.28e0,-0.3e0,
        -0.28e0,0.3e0,0e0,0e0,-0.28e0,-0.3e0,0e0,0e0,
        -0.28e0,-0.3e0,0e0,0e0,0.28e0,0.3e0,0e0,0e0,
        -0.28e0,-0.3e0,-0.28e0,0.3e0,0e0,0e0,-0.28e0,-0.3e0,
        0e0,0e0,0.28e0,0.3e0,0e0,0e0,-0.28e0,0.3e0,
        0.28e0,-0.3e0,-0.28e0,0.3e0,0.4e0,0.4e0,-0.24e0,0.3e0,
        0e0,-0.1e0,0.16e0,0e0,0.36e0,-0.2e0,0.3e0,-0.12e0,
        -0.1e0,-0.24e0,-0.3e0,0e0,0e0,-0.24e0,0.3e0,-0.24e0,
        0.3e0,0e0,0e0,-0.24e0,0.3e0,-0.24e0,0.3e0,0.24e0,
        -0.3e0,0e0,0e0,0.24e0,-0.3e0,0e0,0e0,0.24e0,
        0.3e0,0.24e0,-0.3e0,0.24e0,0.3e0,-0.24e0,0.3e0,-0.24e0,
        0.3e0,-0.2e0,0.2e0,-0.16e0,-0.2e0,0e0,0e0,-0.32e0,
        0.2e0,0e0,0.1e0,0.2e0,-0.3e0,0.2e0,-0.2e0,0.12e0,
        0.2e0,-0.16e0,0.2e0,0.16e0,0.2e0,0.2e0,0.3e0,0.2e0,
        0.3e0,0e0,0e0,-0.2e0,0.3e0,0e0,0e0,0.2e0,
        0.3e0,-0.2e0,-0.3e0,-0.2e0,-0.3e0,0.2e0,-0.3e0,0e0,
        0e0,0.2e0,0.3e0,0e0,0e0,0.2e0,0.3e0,0e0,
        0e0,0.2e0,0.3e0,0e0,0e0,0.2e0,0.3e0,0e0,
        0e0,0.2e0,-0.3e0,0e0,0e0,-0.2e0,-0.3e0,0e0,
        0e0,-0.2e0,0.3e0,0e0,0e0,-0.2e0,0.3e0,0e0,
        0e0,0.36e0,0e0,0e0,0.36e0,0.12e0,0.1e0,-0.24e0,
        0.2e0,0.12e0,-0.2e0,-0.16e0,-0.2e0,-0.13e0,0.1e0,0.22e0,
        0.21e0,0.2e0,0e0,-0.28e0,0.32e0,0e0,-0.12e0,-0.2e0,
        -0.2e0,0.12e0,-0.1e0,0.12e0,0.1e0,-0.2e0,0.2e0,0e0,
        0e0,-0.32e0,0.32e0,0e0,0e0,0.32e0,0.32e0,0e0,
        0e0,-0.24e0,-0.2e0,0.24e0,0.2e0,0.2e0,0e0,-0.24e0,
        0e0,0e0,-0.24e0,-0.2e0,0e0,0e0,0.24e0,0.2e0,
        -0.24e0,-0.2e0,0e0,0e0,-0.24e0,0.2e0,0.16e0,-0.2e0,
        0.12e0,0.1e0,0.2e0,0.2e0,0e0,-0.1e0,-0.12e0,0.1e0,
        -0.16e0,-0.2e0,-0.12e0,-0.1e0,-0.16e0,0.2e0,0.2e0,0.2e0,
        0e0,0e0,-0.2e0,0.2e0,-0.2e0,0.2e0,-0.2e0,0.2e0,
        -0.2e0,0.2e0,0.2e0,-0.2e0,-0.2e0,-0.2e0,0e0,0e0,
        -0.2e0,0.2e0,0.2e0,0e0,-0.2e0,0e0,0e0,-0.2e0,
        0.2e0,-0.2e0,0.2e0,-0.2e0,-0.2e0,-0.2e0,-0.2e0,0e0,
        0e0,0.2e0,0.2e0,0.2e0,0.2e0,0.12e0,-0.2e0,-0.12e0,
        -0.1e0,0.28e0,-0.28e0,0.16e0,-0.2e0,0e0,-0.1e0,0e0,
        0.1e0,-0.16e0,0.2e0,0e0,-0.1e0,-0.16e0,-0.2e0,0e0,
        -0.1e0,0.16e0,-0.2e0,0.16e0,-0.2e0,0e0,0e0,0.16e0,
        0.2e0,-0.16e0,0.2e0,0e0,0e0,0.16e0,0.2e0,0.16e0,
        -0.2e0,0.16e0,-0.2e0,-0.16e0,0.2e0,0.16e0,-0.2e0,0e0,
        0e0,0.16e0,0.2e0,0e0,0e0,0.16e0,0.2e0,0e0,
        0e0,-0.16e0,-0.2e0,0.16e0,-0.2e0,-0.16e0,-0.2e0,0e0,
        0e0,-0.16e0,-0.2e0,0e0,0e0,-0.16e0,0.2e0,0e0,
        0e0,0.16e0,-0.2e0,0.16e0,0.2e0,0.16e0,0.2e0,0e0,
        0e0,-0.16e0,-0.2e0,0e0,0e0,-0.16e0,-0.2e0,0e0,
        0e0,0.16e0,0.2e0,0.16e0,0.2e0,0e0,0e0,0.16e0,
        0.2e0,0.16e0,-0.2e0,0.16e0,0.2e0,0e0,0e0,-0.16e0,
        0.2e0,0e0,0.1e0,0.12e0,-0.2e0,0.12e0,-0.2e0,0e0,
        -0.1e0,0e0,-0.1e0,0.12e0,0.2e0,0e0,-0.1e0,-0.12e0,
        0.2e0,-0.15e0,0.2e0,-0.24e0,0.24e0,0e0,0e0,0.24e0,
        0.24e0,0.12e0,-0.2e0,-0.12e0,-0.2e0,0e0,0e0,0.12e0,
        0.2e0,0.12e0,-0.2e0,0.12e0,0.2e0,0.12e0,0.2e0,0.12e0,
        0.2e0,0.12e0,-0.2e0,-0.12e0,0.2e0,0e0,0e0,0.12e0,
        0.2e0,0.12e0,0e0,-0.2e0,0e0,0e0,-0.12e0,-0.2e0,
        0.12e0,-0.2e0,0e0,0e0,0.12e0,0.2e0,-0.12e0,0.2e0,
        -0.12e0,0.2e0,0.12e0,-0.2e0,0e0,0e0,0.12e0,0.2e0,
        0.2e0,0e0,0.12e0,0e0,0e0,-0.12e0,0.2e0,0e0,
        0e0,-0.12e0,-0.2e0,0e0,0e0,-0.12e0,-0.2e0,-0.12e0,
        -0.2e0,0e0,0e0,0.12e0,-0.2e0,0.12e0,-0.2e0,0.12e0,
        0.2e0,-0.12e0,-0.2e0,0e0,0e0,0.12e0,-0.2e0,0.12e0,
        -0.2e0,0.12e0,0.2e0,0.12e0,0e0,0.2e0,-0.12e0,-0.2e0,
        0e0,0e0,0.12e0,0.2e0,-0.16e0,0e0,0.16e0,-0.2e0,
        0.2e0,0e0,0e0,-0.2e0,0e0,0e0,-0.2e0,0.2e0,
        0e0,0e0,0.2e0,0.2e0,-0.2e0,0e0,0e0,-0.2e0,
        0.12e0,0e0,-0.16e0,0.2e0,0e0,0e0,0.2e0,0.12e0,
        -0.1e0,0e0,0.1e0,0.16e0,-0.16e0,-0.16e0,-0.16e0,-0.16e0,
        -0.16e0,0e0,0e0,-0.16e0,0e0,0e0,-0.16e0,-0.16e0,
        -0.16e0,0e0,0e0,-0.16e0,0e0,0e0,0.16e0,0e0,
        0e0,0.16e0,0e0,0e0,0.16e0,0.16e0,0e0,0e0,
        -0.16e0,0e0,0e0,-0.16e0,-0.16e0,0e0,0e0,0.16e0,
        0e0,0e0,-0.16e0,-0.16e0,0e0,0e0,-0.16e0,-0.16e0,
        0.12e0,0.1e0,0.12e0,-0.1e0,0.12e0,0.1e0,0e0,0e0,
        0.12e0,0.1e0,-0.12e0,0.1e0,0e0,0e0,0.12e0,0.1e0,
        0.12e0,-0.1e0,0e0,0e0,-0.12e0,-0.1e0,0e0,0e0,
        0.12e0,0.1e0,0.12e0,0e0,0e0,0.12e0,0e0,0e0,
        -0.12e0,0e0,0e0,0.12e0,0.12e0,0.12e0,0.12e0,0.12e0,
        0e0,0e0,0.12e0,0e0,0e0,0.12e0,0.12e0,0e0,
        0e0,0.12e0,0e0,0e0,0.12e0,-0.12e0,-0.12e0,0.12e0,
        0.12e0,-0.12e0,-0.12e0,0e0,0e0,0.12e0,-0.12e0,0.12e0,
        0.12e0,-0.12e0,-0.12e0,0e0,0e0,-0.12e0,-0.12e0,0e0,
        0e0,-0.12e0,0.12e0,0e0,0e0,0.12e0,0e0,0e0,
        0.12e0,0e0,0e0,0.12e0,-0.12e0,0e0,0e0,-0.12e0,
        0.12e0,-0.12e0,-0.12e0,0.12e0,0e0,0e0,0.12e0,0.12e0,
        0.12e0,-0.12e0,0e0,0e0,-0.12e0,-0.12e0,-0.12e0,0e0,
        0e0,-0.12e0,-0.12e0,0e0,0e0,0.12e0,0.12e0,0e0,
        0e0,-0.12e0,-0.12e0,-0.12e0,-0.12e0,0.12e0,0e0,0e0,
        0.12e0,-0.12e0,0e0,0e0,-0.12e0,-0.12e0,0e0,0e0,
        0.12e0,-0.12e0,-0.12e0,-0.12e0,-0.12e0,0.12e0,0.12e0,-0.12e0,
        -0.12e0,0e0,0e0,-0.12e0,0e0,0e0,-0.12e0,0.12e0,
        0e0,0e0,0.12e0,0e0,0e0,-0.12e0,-0.12e0,0e0,
        0e0,-0.12e0,-0.12e0,0.12e0,0e0,0e0,0.12e0,0.12e0,
        0e0,0e0,0.12e0,0e0,0e0,0.12e0,0.12e0,0.08e0,
        0e0,0.04e0]
    
    #振幅使用:X或Y, sin或cos, T的幂。
    JAXY=[0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1]
    JASC=[0,1,1,0,1,0,0,1,0,1,1,0,1,0,0,1,0,1,1,0]
    JAPT=[0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4]
    
    #给定日期与参考历元之间的时间间隔，儒略世纪数.
    T=((DATE1-DJ00)+DATE2)/DJC
    
    #T的幂.
    W=1.0
    for JPT in range(MAXPT+1):
        PT[JPT]=W
        W=W*T
    
    #日月基本(Delaunay)参数(IERS 2003)
    
    #月亮的平近点角.
    FA[0]=pymFal03(T)

    #太阳的平近点角.
    FA[1]=pymFalp03(T)

    #月球纬度的平均辐角.
    FA[2]=pymFaf03(T)

    #月亮到太阳的平均距角
    FA[3]=pymFad03(T)

    #月亮升交点的平黄经
    FA[4]=pymFaom03(T)

    #行星的平黄经，水星到海王星
    FA[5]=pymFame03(T)
    FA[6]=pymFave03(T)
    FA[7]=pymFae03(T)
    FA[8]=pymFama03(T)
    FA[9]=pymFaju03(T)
    FA[10]=pymFasa03(T)
    FA[11]=pymFaur03(T)
    FA[12]=pymFane03(T)
    
    #黄经上的累计岁差
    FA[13]=pymFapa03(T)
    
    #岁差章动的多项式部分
    for JXY in range(2):
        XYPR[JXY]=0.0
        for J in range(MAXPT,-1,-1):
            XYPR[JXY]=XYPR[JXY]+XYP[JXY][J]*PT[J]
            
    #章动的周期项，行星
    
    #初始化X,Y.
    for JXY in range(2):
        XYPL[JXY]=0.0
        
    #每个频率的系数从后往前处理
    IALAST=NA
    for IFREQ in range(NFPL-1,-1,-1):
        
        #获得系数方程
        ARG=0.0
        for I in range(14):
            M=MFAPL[IFREQ][I]
            if (M!=0):
                ARG=ARG+float(M)*FA[I]
        SC[0]=ma.sin(ARG)
        SC[1]=ma.cos(ARG)
        
        #对这个频率的幅度从后往前处理
        IA=NC[IFREQ+NFLS]
        for I in range(IALAST,IA-1,-1):
            
            #系数数目 (0 = 1st).
            J=I-IA
            
            #X/Y.
            JXY=JAXY[J]
    
            #Sin/cos.
            JSC=JASC[J]

            # T的幂次.
            JPT=JAPT[J]
    
            #总计该分量
            XYPL[JXY]=XYPL[JXY]+A[I-1]*SC[JSC]*PT[JPT]
        IALAST=IA-1
            
    #章动的周期项，日月
    
    #初始化X,Y.
    for JXY in range(2):
        XYLS[JXY]=0.0
    
    #每个频率的系数从后往前处理
    for IFREQ in range(NFLS-1,-1,-1):
        
        #获得系数方程
        ARG=0.0
        for I in range(5):
            M=MFALS[IFREQ][I]
            if (M!=0):
                ARG=ARG+float(M)*FA[I]
        SC[0]=ma.sin(ARG)
        SC[1]=ma.cos(ARG)
        
        #对这个频率的幅度从后往前处理
        IA=NC[IFREQ]
        for I in range(IALAST,IA-1,-1):
           
            #系数数目 (0 = 1st).
            J=I-IA
            #X/ Y.
            JXY=JAXY[J]
    
            #Sin/cos.
            JSC=JASC[J]

            #T的幂次.
            JPT=JAPT[J]
    
            #总计该分量
            XYLS[JXY]=XYLS[JXY]+A[I-1]*SC[JSC]*PT[JPT]
        IALAST=IA-1
                
    #CIP的单位向量分量
    X=DAS2R*(XYPR[0]+(XYLS[0]+XYPL[0])/1e6)
    Y=DAS2R*(XYPR[1]+(XYLS[1]+XYPL[1])/1e6)

    return(X,Y)

def pymPmat00(DATE1,DATE2):
    '''
    Precession matrix (including frame bias) from GCRS to a specified
    date, IAU 2000 model.

    Parameters
    ----------
    date1 : flaot
        TT as a 2-part Julian Date     
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    rbp : list(3,3)
        bias-precession matrix

    '''
    #调用pymBp00获得需要的矩阵，舍去无关项
    RB,RP,RBP=pymBp00(DATE1,DATE2)
    
    return(RBP)

def pymPrec76 (DATE01,DATE02,DATE11,DATE12):
    '''
    IAU 1976 precession model.
    
    This function forms the three Euler angles which implement general
    precession between two dates, using the IAU 1976 model (as for the
    FK5 catalog).

    Parameters
    ----------
    date01 : float
        TDB starting date    
    date02 : float
        TDB starting date    
    date11 : float
        TDB ending date    
    date12 : flaot
        TDB ending date    

    Returns
    -------
    zeta : float
        1st rotation: radians cw around z    
    z : float
        3rd rotation: radians cw around z    
    theta : float
        2nd rotation: radians ccw around y

    '''
    #1角秒对应的弧度
    DAS2R=4.848136811095359935899141e-6

    #参考历元(J2000.0), JD
    DJ00=2451545.0

    #1儒略世纪对应的天数
    DJC=36525.0
    
    #给定的开始时间与参考历元之间的时间间隔儒略世纪数.
    T0=((DATE01-DJ00)+DATE02)/DJC
    
    #计算的岁差的持续时间（结束时间-开始时间），儒略世纪数.
    T=((DATE11-DATE01)+(DATE12-DATE02))/DJC
    
    #欧拉角
    TAS2R=T*DAS2R
    W=2306.2181+(1.39656-0.000139*T0)*T0

    ZETA=(W+((0.30188-0.000344*T0)+0.017998*T)*T)*TAS2R

    Z=(W+((1.09468+0.000066*T0)+0.018203*T)*T)*TAS2R

    THETA=((2004.3109+(-0.85330-0.000217*T0)*T0)+\
           ((-0.42665-0.000217*T0)-0.041833*T)*T)*TAS2R 
    
    return(ZETA,Z,THETA)

def pymPmat76(DATE1,DATE2):
    '''
    Precession matrix from J2000.0 to a specified date, IAU 1976 model.

    Parameters
    ----------
    date1 : float
        ending date, TT    
    date2 : float
        ending date, TT

    Returns
    -------
    rmatp : list(3,3)
        precession matrix, J2000.0 -> date1+date2

    ''' 
    #参考历元(J2000.0), JD
    DJ00=2451545.0
    
    #岁差欧拉角, J2000.0到给定日期.
    ZETA,Z,THETA=pymPrec76(DJ00,0.0,DATE1,DATE2)
    
    #构建旋转矩阵
    WMAT=pymIr()
    WMAT=pymRz(-ZETA,WMAT)
    WMAT=pymRy(THETA,WMAT)
    WMAT=pymRz(-Z,WMAT)
    RMATP=pymCr(WMAT)
    
    return(RMATP)

def pymPn06 (DATE1,DATE2,DPSI,DEPS):
    '''
    Precession-nutation, IAU 2006 model:  a multi-purpose function,
    supporting classical (equinox-based) use directly and CIO-based use
    indirectly.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date       
    dpsi : float
        nutation    
    deps : float
        nutation    

    Returns
    -------
    epsa : flaot
        mean obliquity    
    rb : list(3,3)
        frame bias matrix    
    rp : list(3,3)
        precession matrix    
    rbp : list(3,3)
        bias-precession matrix    
    rn : list(3,3)
        nutation matrix    
    rbpn : list(3,3)
        GCRS-to-true matrix

    '''
    #约简的儒略日零点
    DJM0=2400000.5
    
    #参考历元(J2000.0), MJD
    DJM00=51544.5
    
    #参考架偏差-岁差角Fukushima-Williams J2000.0.
    GAMB,PHIB,PSIB,EPS=pymPfw06(DJM0,DJM00)
    
    #参考架偏差矩阵.
    RB=pymFw2m(GAMB,PHIB,PSIB,EPS)
    
    #参考架偏差-岁差 Fukushima-Williams 角.
    GAMB,PHIB,PSIB,EPS=pymPfw06(DATE1,DATE2)
    
    #参考架偏差-岁差矩阵
    RBP=pymFw2m(GAMB,PHIB,PSIB,EPS)
    
    #解出岁差矩阵
    RT=pymTr(RB)
    RP=pymRxr(RBP,RT)
    
    #基于春分点的参考架偏差-岁差-章动矩阵
    RBPN=pymFw2m(GAMB,PHIB,PSIB+DPSI,EPS+DEPS)
    
    #解出章动矩阵
    RT=pymTr(RBP)
    RN=pymRxr(RBPN,RT)
    
    #倾角，给定日期的平均值
    EPSA=EPS
    
    return(EPSA,RB,RP,RBP,RN,RBPN)

def pymPn06a(DATE1,DATE2):
    '''
    Precession-nutation, IAU 2006/2000A models:  a multi-purpose function,
    supporting classical (equinox-based) use directly and CIO-based use
    indirectly.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    dpsi : float
        nutation    
    deps : flaot
        nutation     
    epsa : flaot
        mean obliquity    
    rb : list(3,3)
        frame bias matrix    
    rp : list(3,3)
        precession matrix     
    rbp : list(3,3)
        bias-precession matrix    
    rn : list(3,3)
        nutation matrix    
    rbpn : list(3,3)
        GCRS-to-true matrix

    '''    
    #章动
    DPSI,DEPS=pymNut06a(DATE1,DATE2)
    
    #调用pymPn06获得结果
    EPSA,RB,RP,RBP,RN,RBPN=pymPn06(DATE1,DATE2,DPSI,DEPS)
    
    return(DPSI,DEPS,EPSA,RB,RP,RBP,RN,RBPN)

def pymPnm80(DATE1,DATE2):
    '''
    Form the matrix of precession/nutation for a given date, IAU 1976
    precession model, IAU 1980 nutation model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : flaot
        TT as a 2-part Julian Date

    Returns
    -------
    rmatpn : list(3,3)
        combined precession/nutation matrix

    '''
    #岁差矩阵 J2000.0到给定日期.
    RMATP=pymPmat76(DATE1,DATE2)
    
    #章动矩阵
    RMATN=pymNutm80(DATE1,DATE2)
    
    #两个矩阵结合  PN = N x P.
    RMATPN=pymRxr(RMATN,RMATP)
    
    return(RMATPN)

def pymS00a(DATE1,DATE2):
    '''
    The CIO locator s, positioning the Celestial Intermediate Origin on
    the equator of the Celestial Intermediate Pole, using the IAU 2000A
    precession-nutation model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        the CIO locator s in radians

    '''
    #参考架偏差-岁差-章动矩阵 IAU 2000A.
    RBPN=pymPnm00a(DATE1,DATE2)
    
    #分离出CIP的X,Y坐标
    X,Y=pymBpn2xy(RBPN)
    
    #基于X,Y坐标，计算CIO定位角s
    S00=pymS00(DATE1,DATE2,X,Y)
    
    return(S00)

def pymS00b(DATE1,DATE2):
    '''
    The CIO locator s, positioning the Celestial Intermediate Origin on
    the equator of the Celestial Intermediate Pole, using the IAU 2000B
    precession-nutation model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        the CIO locator s in radians

    '''
    #参考架偏差-岁差-章动矩阵 IAU 2000B.
    RBPN=pymPnm00b(DATE1,DATE2)
    
    #分离出CIP的X,Y坐标
    X,Y=pymBpn2xy(RBPN)
    
    #基于X,Y坐标，计算CIO定位角s
    S00=pymS00(DATE1,DATE2,X,Y)              
    
    return(S00)

def pymS06a(DATE1,DATE2):
    '''
    The CIO locator s, positioning the Celestial Intermediate Origin on
    the equator of the Celestial Intermediate Pole, using the IAU 2006
    precession and IAU 2000A nutation models.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    function value : float
        the CIO locator s in radians

    '''
    #参考架偏差-岁差-章动矩阵, IAU 20006/2000A.
    RBPN=pymPnm06a(DATE1,DATE2)
    
    #从天球中间极（CIP）中分解出X，Y坐标
    X,Y=pymBpn2xy(RBPN)
    
    #基于X,Y坐标，计算CIO定位角s
    S06=pymS06(DATE1,DATE2,X,Y)  
    
    return(S06)

def pymXys00a(DATE1,DATE2):
    '''
    For a given TT date, compute the X,Y coordinates of the Celestial
    Intermediate Pole and the CIO locator s, using the IAU 2000A
    precession-nutation model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    x : float
        Celestial Intermediate Pole    
    y : float
        Celestial Intermediate Pole    
    s : float
        the CIO locator s
        
    '''
    #调用pymPmN00A给出参考架偏差-岁差-章动矩阵，IAU 2000A.
    RBPN=pymPnm00a(DATE1,DATE2)

    #分离出X,Y坐标
    X,Y=pymBpn2xy(RBPN)

    #获得 s.
    S=pymS00(DATE1,DATE2,X,Y)
    
    return(X,Y,S)

def pymXys00b(DATE1,DATE2):
    '''
    For a given TT date, compute the X,Y coordinates of the Celestial
    Intermediate Pole and the CIO locator s, using the IAU 2000B
    precession-nutation model.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    x : float
        Celestial Intermediate Pole    
    y : float
        Celestial Intermediate Pole    
    s : float
        the CIO locator s

    '''
    #调用pymPmN00B给出参考架偏差-岁差-章动矩阵，IAU 2000B.
    RBPN=pymPnm00b(DATE1,DATE2)

    #分离出X,Y坐标
    X,Y=pymBpn2xy(RBPN)

    #获得 s.
    S=pymS00(DATE1,DATE2,X,Y)
    
    return(X,Y,S)

def pymXys06a(DATE1,DATE2):
    '''
    For a given TT date, compute the X,Y coordinates of the Celestial
    Intermediate Pole and the CIO locator s, using the IAU 2006
    precession and IAU 2000A nutation models.

    Parameters
    ----------
    date1 : float
        TT as a 2-part Julian Date    
    date2 : float
        TT as a 2-part Julian Date

    Returns
    -------
    x : float
        Celestial Intermediate Pole    
    y : float
        Celestial Intermediate Pole    
    s : float
        the CIO locator s

    '''
    #调用pymPmN06A给出参考架偏差-岁差-章动矩阵，IAU 2006/2000A..
    RBPN=pymPnm06a(DATE1,DATE2)

    #分离出X,Y坐标
    X,Y=pymBpn2xy(RBPN)

    #获得 s.
    S=pymS06(DATE1,DATE2,X,Y)
    
    return(X,Y,S)

